#ifndef ___ARENA_SPAWN_H___
#  define ___ARENA_SPAWN_H___

/* Yes, I'll limit the maximum number of children to this...
 * if you need more, CHANGE it!
 */
#define SPAWN_MAX_CHILDREN 32

/* ExitStatus is used to flag wheter or not the child has quit yet */
#define SPAWN_NO_EXIT_YET (-32000)

/* When spawning, you specify whether or not to wait until the child exits.
 * WAIT is just that... wait until child exits. Period.
 * ASYNC says... run child and I'll keep running ahead on my own.
 * GUI means... run child, and I'll keep the "basic" X Gui running... But will
 *              not proceed until the child exits.
 * BACKGROUND means... run child and I'll allow "ALL" events to be processed,
 *                     but I'll not proceed until child exits.
 */
typedef enum { WAIT_WAIT = 1,
	       WAIT_ASYNC = 2,
	       WAIT_NOWAIT = 2,
	       WAIT_GUI = 3,
	       WAIT_BACKGROUND = 4,
	       WAIT_BG = 4 } SPAWNWaitMode;

/* When spawning, you specify how to get rid of children on exit.
 * Some children you may want to leave running even if we go away... etc.
 */
typedef enum { KILL_NEVER = 1,
	       KILL_ALWAYS,
	       KILL_TIMEOUT,
	       KILL_WAIT } SPAWNKillMode;

/* Footprint for the SIGCHLD handlers you must build */
struct _spawn_child_info;
typedef void (*SPAWNChildHandler) (struct _spawn_child_info *c);


/* Struct to keep a list of file streams to clean up for
 * a childs connection to us!
 */
typedef struct _spawn_file_info
{
 FILE *stream;
 Bool flush_file;
 struct _spawn_file_info *next;
} SPAWNFileInfo;


/* All the info for one child process
 */
typedef struct _spawn_child_info
{
 pid_t pid;                        /* pid of the new child we create         */
 SPAWNWaitMode wait_how;           /* Which WAIT mode to use                 */
 SPAWNKillMode kill_how;           /* Which KILL mode to use                 */
 int argc;                         /* Yes, number of args                    */
 char **args;                      /* Arg list of data passed to us          */
 int exit_status;                  /* Returned status of child upon exit     */
 SPAWNChildHandler exit_handler;   /* Specify an exit handler to use         */
 Bool exit_handler_done;           /* Flags that the handler is done         */
 SPAWNFileInfo *file;              /* A chain of files to close upon kill    */
 } SPAWNChildInfo;

/* SpawnChild    See SpawnVE for an example!
 * Fork a new process and execute YOUR func() as the child...
 *
 * YOUR func() will be called with path, args and envp... which may or may not
 * actually be those things.  If your func is execve(), the they had better be!
 * Of course, you could specify any func() you want, and it might "copy" a file
 * using getc()/putc() and return!.  That function would execute as a child
 * process and exit!  Better be careful with this... That type of child
 * knows a LOT about the parent (not always a good thing in real life either).
 *
 * You also specify the type of WAIT and KILL to use on this child.
 * You MAY specify an exit handler to be called by SIGCHLDHandler() for you.
 */
/* PLEASE NOTE: as of 8.10.97 the use of the char *?name, and FILE** should
 * be NULL only.  VERY LITTLE TESTING OF THIS HAS BEEN DONE!
 */
SPAWNChildInfo *SpawnChild(void (*func) (SPAWNChildInfo *, char *, char **, char **),
			    char *path, char *args[], char *envp[],
			    SPAWNWaitMode wait, SPAWNKillMode kill,
			    char *Iname, FILE **Ipipe,
			    char *Oname, FILE **Opipe, 
			    char *Ename, FILE **Epipe,
			    SPAWNChildHandler chandler);

/* RememberFile is used to "remember" a file to be closed (with/without) buffer
 * flushing when the child is finished.  SIGCHLDHandler() will take care of
 * this for you at the time the child dies.  SpawnChild is going to remember
 * any pipes or I/O files it uses/creates for you!
 */
void SpawnRememberFile(SPAWNChildInfo *child, FILE *f, Bool flushit);


/* These are the easiest to use!  Examples of SpawnChild() usage.
 * Specify the WAIT, KILL, path, argv list, [environment] and exit handler!
 * If you need PATH= from environment, you will have to build it into envp
 * when using VE.  VP will uses the user's environment.
 */
SPAWNChildInfo *SpawnVE(SPAWNWaitMode wait, SPAWNKillMode kill,
			char *path, char *argv[], char *envp[],
			SPAWNChildHandler chandler);
SPAWNChildInfo *SpawnVP(SPAWNWaitMode wait, SPAWNKillMode kill,
			char *path, char *argv[],
			SPAWNChildHandler chandler);

/* Let's duplicate the system() function, but a little differently!
 * Pass command string as usual, but allow an optional environment...
 * If envp is NULL, we'll use the user's current environment!
 * If now exit handler is specified, that's ok!
 * Difference: Shell expansion of args is NOT performed... Much safer!
 */
int SpawnSystemCommand(char *cmd, char *envp[],
		       SPAWNWaitMode wait, SPAWNKillMode kill,
		       SPAWNChildHandler chandler);

/* The Quick and Dirty functions to "execute" as the child.
 * When debugging Arena, if you don't specify an exit handler, I'll make you
 * have one!
 */
void SpawnExecve(SPAWNChildInfo *c, char *path, char *argv[], char *envp[]);
void SpawnExecvp(SPAWNChildInfo *c, char *path, char *argv[], char *envp[]);
void SpawnExitHandler(SPAWNChildInfo *child);

/* Whenever you spawn a child with a "WAIT till done" mode (everything except
 * WAIT_ASYNC) YOU MUST "forget" the child pointer when YOU are done with it!
 * If you do NOT "forget" the child... you may run out of table space!
 */
void SpawnForgetChild(SPAWNChildInfo *child);

/* KILL A child! NOT TESTED! */
void SpawnKillOneChild(SPAWNChildInfo *child);

/* CleanUpChildFiles
 * Close up any "remembered" files associated with this child
 */
void SpawnCleanUpChildFiles(SPAWNChildInfo *child);

/* CleanUpChildren is called by Exit()
 * It will kill and clean up after each child still active.
 */ 
void SpawnCleanUpChildren(void);

/* Dummy routines... well sort of...
 * If we ever have to write something special to handle the WAIT_BACKGROUND
 * or WAIT_GUI modes.  Right now, they each call Arena's handlers!
 */
int SpawnEventHandler(SPAWNChildInfo *child);
int SpawnGuiEventHandler(SPAWNChildInfo *child);

/* Routine to "see" if the child pointer you have points to a RUNNING child! */
Bool SpawnIsChildPlaying(SPAWNChildInfo *child);

/* Very quick routine to see if child is alive... Used primarily just after
 * Spawn?? to check if child is defined and if exit status is still NO_EXIT_YET
 */
Bool SpawnIsChildAlive(SPAWNChildInfo *child);

/* See if we have A child running in WAIT_GUI mode... If so, the Gui code
 * needs to know so it won't do anything stupid (i.e. OpenDoc...)
 */
Bool SpawnIsWaitGui(void);

/* Routines to set (increment), unset (decrement) and test if a BLOCK is
 * in use.  Example: bookmark.c   Sorting of folders now occurs in ASYNC mode.
 * Don't want to try doing anything in bookmarks until any previous sorting
 * is DONE!
 */
void SpawnBlock(int *blockflag);
void SpawnUnBlock(int *blockflag);
Bool SpawnIsBlocked(int blockflag);
void SpawnWaitForBlock(int blockflag);

/* This routine will wait until the specified child has died!
 * You may specify any wait mode you wish... It can even be different
 * than that specified at creation.... WHY?!  Well, let's assume you
 * spawn(ASYNC), do some stuff, and then want to wait till death! OK?!
 */
void SpawnWaitForChild(SPAWNChildInfo *child, SPAWNWaitMode myWait);

/* Since we get command lines from the user, and Spawning seems to want an
 * argc/argv setup, MAKE IT!
 */
char **SpawnStringToRagged(char *string, int *argc);

/* Make a memory allocated duplicate of an argc/argv
 * and a method to free it!
 */
char **SpawnDupRagged(int *argc, char **argv);
void SpawnFreeRagged(int *argc, char ***argv);

#endif /* ___ARENA_SPAWN_H___   */
