/***************************************
  $Header: /home/amb/wwwoffle/RCS/monitor.c 1.8 1998/02/21 15:41:28 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.1.
  A header file for all of the programs wwwoffle, wwwoffled.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1998 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "errors.h"


static void MonitorFormPage(int fd,char *args);
static void MonitorFormParse(int fd,char *request_body);
static void MonitorFormError(int fd,char *body);
static void IllegalMonitorPage(int fd,char *path,char *args);


/*++++++++++++++++++++++++++++++++++++++
  Send to the client a page to allow monitoring using HTML.

  int fd The file descriptor of the client.

  char *path The path that was specified by the user.

  char *args The argument that was appended to the URL.

  char *request_body The body of the HTTP request.
  ++++++++++++++++++++++++++++++++++++++*/

void MonitorPage(int fd,char *path,char *args,char *request_body)
{
 if(!strcmp("/monitor/",path))
    MonitorFormPage(fd,args);
 else if(!strcmp("/monitor-request/",path))
    MonitorFormParse(fd,request_body);
 else
    IllegalMonitorPage(fd,path,args);
}


/*++++++++++++++++++++++++++++++++++++++
  The form that the user enters the details on.

  int fd The file descriptor.

  char *args The arguments that were on the request for this URL.
  ++++++++++++++++++++++++++++++++++++++*/

static void MonitorFormPage(int fd,char *args)
{
 char *head=
 "HTTP/1.0 200 WWWOFFLE Monitor Form\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Interactive Monitor Form</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Interactive Monitor Form</H1>\n"
 "You can use this form to monitor any URL.\n"
 "<p>\n"
 "<form action=\"/monitor-request/\" method=post>\n"
 "Monitor <input type=\"text\" name=\"url\" value=\"";
 char *middle=
 "\"size=60>\n"
 "<br>\n";
 char *tail1=
 "<br>\n"
 "<input type=\"submit\" value=\"Monitor Now\">\n"
 "</form>\n";
 char *tail2=
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head);
 if(args)
   {
    char *decargs=UrlDecode(args,0);
    write_string(fd,decargs);
    free(decargs);
   }
 write_string(fd,middle);
 if(MonitorInterval<=0)
    write_string(fd,"every time online\n");
 else
    write_formatted(fd,"every %d days\n",MonitorInterval);
 write_string(fd,tail1);
 if(!args)
    write_string(fd,"<p align=center>[<a href=\"/\">Back to the Welcome page</a>]</p>\n");
 write_string(fd,tail2);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the reply from the form.

  int fd The file descriptor of the client.

  char *request_body The body of the HTTP request sent by the browser.
  ++++++++++++++++++++++++++++++++++++++*/

static void MonitorFormParse(int fd,char *request_body)
{
 int i,mfd=-1;
 char *copy,*url=NULL;
 URL *Url;

 if(!request_body)
   {
    MonitorFormError(fd,NULL);
    return;
   }

 copy=(char*)malloc(strlen(request_body)+1);
 strcpy(copy,request_body);

 for(i=0;copy[i];i++)
   {
    if(i!=0 && copy[i-1]=='&')
       copy[i-1]=0;
    if(i==0 || copy[i-1]==0)
      {
       if(!strncmp("url=",&copy[i],4))
          url=&copy[i+4];
      }
   }

 if(url==NULL || *url==0)
   {
    MonitorFormError(fd,request_body);
    free(copy);
    return;
   }

 url=UrlDecode(url,1);
 Url=SplitURL(url);

 mfd=OpenMonitorSpoolFile(Url);
 if(mfd==-1)
    ServerError(fd,"Cannot open file to store monitor request");
 else
   {
    char *head=
    "HTTP/1.0 200 WWWOFFLE Monitor Requested\r\n"
    "Content-type: text/html\r\n"
    "\r\n"
    "<HTML>\n"
    "<HEAD>\n"
    "<TITLE>WWWOFFLE - Monitor Requested</TITLE>\n"
    "</HEAD>\n"
    "<BODY>\n"
    "<H1 align=center>WWWOFFLE Monitor Requested</H1>\n"
    "<p align=center>\n"
    "Your request for URL\n"
    "<br><b><tt>\n";
    char *tail=
    "\n"
    "</tt></b><br>\n"
    "to be monitored has been recorded.\n"
    "<br>\n"
    "</BODY>\n"
    "</HTML>\n";

    char *req=RequestURL(Url->name,NULL);
    int ofd=OpenOutgoingSpoolFile(0);

    if(ofd==-1)
       PrintMessage(Warning,"Cannot open outgoing spool file for monitored URL '%s' [%!s].",url);
    else
      {
       write_string(ofd,req);

       CloseOutgoingSpoolFile(ofd,Url);
      }

    write_string(mfd,req);

    write_string(fd,head);
    write_string(fd,Url->name);
    write_string(fd,tail);

    free(req);
    close(mfd);
   }

 free(copy);

 FreeURL(Url);
}


/*++++++++++++++++++++++++++++++++++++++
  An error with the form.

  int fd The file descriptor.

  char *body The browser reply that the user entered.
  ++++++++++++++++++++++++++++++++++++++*/

static void MonitorFormError(int fd,char *body)
{
 char *head=
 "HTTP/1.0 404 WWWOFFLE Monitor Form Error\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Interactive Monitor Form Error</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Interactive Monitor Form Error</H1>\n"
 "<p align=center>\n";
 char *middle1=
 "The reply from the form that your browser sent did not have a body.\n";
 char *middle2=
 "The reply from the form that your browser sent\n"
 "<br><b><tt>\n";
 char *middle3=
 "\n"
 "</tt></b><br>\n"
 "had an error and could not be parsed.\n";
 char *tail=
 "<p align=center>[<a href=\"/refresh/\">Back to the Monitor page</a>]</p>"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head);
 if(!body)
    write_string(fd,middle1);
 else
   {
    write_string(fd,middle2);
    write_string(fd,body);
    write_string(fd,middle3);
   }
 write_string(fd,tail);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the specified monitor page is illegal.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The arguments to the page.
  ++++++++++++++++++++++++++++++++++++++*/

static void IllegalMonitorPage(int fd,char *path,char *args)
{
 char *head=
 "HTTP/1.0 404 WWWOFFLE Illegal Monitor Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Illegal Interactive Monitor Page</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Illegal Interactive Monitor Page</H1>\n"
 "<p align=center>\n"
 "Your request for the monitor URL\n"
 "<br><b><tt>\n";
 char *tail=
 "\n"
 "</tt></b><br>\n"
 "is illegal, select the link below for the main interactive monitor page.\n"
 "<br>\n"
 "<a href=\"/monitor/\">/monitor/</a>\n"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head);
 if(args)
    write_formatted(fd,"/%s?%s",path,args);
 else
    write_formatted(fd,"/%s",path);
 write_string(fd,tail);
}


/*++++++++++++++++++++++++++++++++++++++
  Convert monitor requests into outgoing requests.
  ++++++++++++++++++++++++++++++++++++++*/

void RequestMonitoredPages(void)
{
 DIR *dir;
 struct dirent* ent;
 int now=time(NULL);

 /* Open the monitor subdirectory. */

 if(chdir("monitor"))
   {PrintMessage(Warning,"Cannot change to directory 'monitor' [%!s]; no files monitored.");return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory 'monitor' [%!s]; no files monitored.");chdir("..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory 'monitor' [%!s]; no files monitored.");closedir(dir);chdir("..");return;}
 ent=readdir(dir);  /* skip .. */

 /* Scan through all of the files. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(lstat(ent->d_name,&buf))
      {PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);return;}
    else if(S_ISREG(buf.st_mode) && *ent->d_name=='O')
      {
       int atimedays=(now-buf.st_atime+3600)/(24*3600);
       int mtimedays=(now-buf.st_mtime+3600)/(24*3600);

       PrintMessage(Debug,"Monitoring %s; atimedays=%d mtimedays=%d =>%s",ent->d_name,atimedays,mtimedays,
                    (MonitorInterval<=0 ||
                     atimedays>=MonitorInterval ||
                     (atimedays && mtimedays && (mtimedays%MonitorInterval)==0))?"Yes":"No");

       if(MonitorInterval<=0 ||
          atimedays>=MonitorInterval ||
          (atimedays && mtimedays && (mtimedays%MonitorInterval)==0))
         {
          char *url=FileNameToURL(ent->d_name);
          int ifd=open(ent->d_name,O_RDONLY);
          init_buffer(ifd);

          if(ifd==-1)
             PrintMessage(Warning,"Cannot open monitored file 'monitor/%s' to read [%!s].",ent->d_name);
          else
            {
             int ofd;

             chdir("..");

             ofd=OpenOutgoingSpoolFile(0);

             if(ofd==-1)
                PrintMessage(Warning,"Cannot open outgoing spool file for monitored URL '%s' [%!s].",url);
             else
               {
                char *contents=(char*)malloc(buf.st_size);
                URL *Url=SplitURL(url);

                read_data(ifd,contents,buf.st_size);
                write_data(ofd,contents,buf.st_size);

                CloseOutgoingSpoolFile(ofd,Url);

                free(contents);
                FreeURL(Url);
               }

             close(ifd);

             if(chdir("monitor"))
               {PrintMessage(Warning,"Cannot change back to directory 'monitor' [%!s]; no more files monitored.");closedir(dir);return;}
            }

          free(url);
         }
      }
   }

 closedir(dir);

 chdir("..");
}
