/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */

package com.kvisco.xsl;

import org.w3c.dom.*;

import com.kvisco.util.List;


/**
 * @author Keith Visco (kvisco@ziplink.net)
**/
public abstract class FilterBase implements MatchExpr {

    public static final int NO_OP       = 0;
    public static final int ANCESTOR_OP = 1;
    public static final int PARENT_OP   = 2;

    private int ancestryOp  = NO_OP;

    List predicates = null;
    
    
    /**
     * Creates a new FilterBase
    **/
    public FilterBase() {
        super();
        predicates = new List();
    } //-- FilterBase

    
    /**
     * Adds the given Expression to this FilterBase's predicate
     * List.
     * @param expr the Expr to add to the predicate list
    **/
    public void addPredicate(Expr expr) {
        if (predicates.contains(expr)) return;
        predicates.add(expr);
    } //-- addPredicate
    
    
    
    /**
     * Determines the priority of a PatternExpr as follows:
     * <PRE>
     *  - From the 19990421 XSL Working Draft -
     *  + If the Pattern has the form of a QName optionally preceded by
     *    the @ character, then the priority is 0.
     *  + Otherwise if the pattern consists of just a NodeTest then the
     *    priority is -1
     *  + Otherwise the priority is 1
     * </PRE>
     * @return the priority for this PatternExpr
    **/
    public abstract int getDefaultPriority();
    
    /**
     * Evaluates this FilterBase using the given context node, and
     * ProcessorState
     * @param context the context node for evaluation
     * @param ps the current ProcessorState
    **/
    public abstract NodeSet evaluate(Node context, ProcessorState ps)
        throws InvalidExprException;
        
    /**
     * Evaluates the PredicateExpr of this FilterBase against the given Node.
     * @param nodes the current NodeSet
     * @param ps the current ProcessorState
    **/
    public void evaluatePredicates(NodeSet nodes, ProcessorState ps) 
        throws InvalidExprException 
    {
        if ((predicates.size() == 0) || (nodes == null)) return;
        
        NodeSet remNodes = new NodeSet(nodes.size());
        for (int pIdx = 0; pIdx < predicates.size(); pIdx++) {
            
            Expr expr = (Expr)predicates.get(pIdx);
            
            for (int nIdx = 0; nIdx < nodes.size(); nIdx++) {
                Node node = nodes.get(nIdx);
                ExprResult exprResult = expr.evaluate(node,ps);
                //-- check for NumberResult in case we need to
                //-- check the position of the node
                switch(exprResult.getResultType()) {
                    case ExprResult.NUMBER:
                        double val = exprResult.numberValue();
                        if (val != (double)(nIdx+1))
                            remNodes.add(node);
                        break;
                    default: //-- convert to boolean
                        if (!exprResult.booleanValue())
                            remNodes.add(node);
                        break;
                }
            }
            //-- remove unmatched nodes
            for (int nIdx = 0; nIdx < remNodes.size(); nIdx++)
                nodes.remove(remNodes.get(nIdx));
            //-- clear remove list
            remNodes.clear();
        }
        
    } // evaluateTestExpr
    
    
    public int getAncestryOp() {
        return this.ancestryOp;
    } //-- getAncestryOp
    
    /**
     * Returns the List of predicates of this FilterBase
     * @return the List of predicates of this FilterBase
    **/
    public List getPredicates() {
        return predicates;
    }
    
    /**
     * Determines if the given node is matched by this MatchExpr with
     * respect to the given context node.
     * @param node the node to determine a match for
     * @param context the Node which represents the current context
     * @param ps the current ProcessorState
     * @return true if the given node is matched by this MatchExpr
    **/
    public abstract boolean matches(Node node, Node context, ProcessorState ps) 
        throws InvalidExprException;
    
    /**
     * Removes the given predicate from this FilterBase
     * @param expr the Expr to remove from the predicate list
    **/
    public void removePredicate(Expr expr) {
        predicates.remove(expr);
    } //-- removeBooleanExpr

    /**
     * Sets the AncestryOp for this FilterBase
     * @param ancestryOp the ancestry operator to use when
     * evaluating expressions
    **/
    public void setAncestryOp(int ancestryOp) {
        this.ancestryOp = ancestryOp;
    } //-- setAncestryOp
    
    /**
     * Returns the String representation of this FilterBase
     * @return the String representation of this FilterBase
    **/
    public String toString() {
        StringBuffer strbuf = new StringBuffer();
        for (int i = 0; i < predicates.size(); i++) {
            Expr expr = (Expr)predicates.get(i);    
            strbuf.append("[");
            strbuf.append(expr.toString());
            strbuf.append("]");
        }
        return strbuf.toString();
        
    } //-- toString


} //-- FilterBase