// aptcache.h  -*-c++-*-
//
//  Replacements for the Apt cache and dependency cache file classes.  Those
// are lacking in a couple of crucial ways (eg: persistent state), so I
// have to come up with my own set of routines.. :(
//
//  In order to track the appropriate package information, we must keep our
// own file (I think..apt doesn't have hooks for adding stuff..) containing
// that information, properly lock/unlock it, etc.
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

#ifndef APTCACHE_H
#define APTCACHE_H

#include <apt-pkg/depcache.h>
#include <apt-pkg/dpkginit.h>

class aptitudeDepCache:public pkgDepCache
{
public:
  struct aptitude_state
  {
    bool new_package;
    // True if the package is a "new" package.  This is a slightly more complex
    // state than dselect's "unknown" state, as it persists until the
    // 'forget new' command is executed.

    pkgCache::State::PkgSelectedState selection_state;
    // Ok, I give in.  I think I see how to implement something slightly more
    // general, but this will do for the current release.  Has the same
    // semantics as dselect's SelectedState flag, except that Unknown isn't
    // included -- making things a lot simpler.
    //  A quick digression on why this is actually a somewhat Good Idea: apt's
    // modes don't tell us anything about a package except for what's to be
    // done with it /right now/.  The dselect selection state paradigm is
    // useful because it states what the user *wants done* with a package.
    //  The limitations of this are that it makes it hard to represent things
    // like downgrading to a previous version, following the latest version
    // in a particular release, and so on.  Those will be addressed here
    // if no-one puts them in libapt first (which would be much better, since
    // otherwise apt-get and the other apt tools will have the unpleasant 
    // effect of clobbering our state)

    pkgCache::State::PkgSelectedState dselect_state;
    //  Used in an attempt to detect changes to the selected state by dselect.
    //  Hopefully this will make aptitude a little less annoying about fighting
    // over package states.  The idea is that we keep the dselect/dpkg state
    // of the package as of Aptitude's last run in this item, and if it changes
    // we assume that the user has fiddled with the package in one of those
    // tools and flip our own state.  We'll see how it goes ;-)
    //  And yes, the Right Thing[tm] to do would be to actually use the dpkg
    // status db, /except/ that I still want to extend this and I don't have
    // a good way to write back to the status database (dpkg --set-selections
    // is actually problematic..)

    bool reinstall;
    //  True if the package in question should be reinstalled
  };

private:
  aptitude_state *package_states;
  int lock;
  // The lock on the extra-info file.
public:
  aptitudeDepCache(MMap &Map, OpProgress &Prog, bool WithLock);

  bool build_selection_list(OpProgress &Prog, bool WithLock);
  void forget_new();
  // Clears all information about which packages are 'new'.

  inline aptitude_state &get_ext_state(const PkgIterator &Pkg)
  {return package_states[Pkg->ID];}

  bool save_selection_list(OpProgress &prog);
  // If the list isn't locked, is a NOP.

  void MarkInstall(const PkgIterator &Pkg, bool AutoInst);
  void MarkDelete(const PkgIterator &Pkg, bool Purge=false);
  void MarkKeep(const PkgIterator &Pkg, bool Soft=false, bool SetHold=true);
  // These just wrap the equivalent depCache functions for the UI's benefit;
  // they mark what the user wants done with each package.
  void MarkFromDselect(const PkgIterator &Pkg);
  // Marks the package based on its current status and its dselect state,
  // adjusting its selected state as appropriate.
  void SetReInstall(const PkgIterator &Pkg, bool To);
  // Wraps pkgDepCache::SetReInstall

  virtual ~aptitudeDepCache();
};

class aptitudeCacheFile
// Hack around problems in libapt.  Most of the code associated with this
// class was copied directly from libapt and reindented..
{
  MMap *Map;
  aptitudeDepCache *Cache;
  pkgDpkgLock *Lock;
public:
  // We look pretty much exactly like a pointer to a dep cache
  inline operator aptitudeDepCache &() {return *Cache;};
  inline operator aptitudeDepCache *() {return Cache;};
  inline aptitudeDepCache *operator ->() {return Cache;};
  inline aptitudeDepCache &operator *() {return *Cache;};
  inline aptitudeDepCache::StateCache &operator [](pkgCache::PkgIterator const &I) {return (*Cache)[I];};
  inline unsigned char &operator [](pkgCache::DepIterator const &I) {return (*Cache)[I];};

  inline void ReleaseLock() {Lock->Close();}

  bool Open(OpProgress &Progress, bool WithLock=true);

  aptitudeCacheFile();
  ~aptitudeCacheFile();
};

#endif
