/*
    ext2_buffer.c -- ext2 buffer cache
    Copyright (C) 1998,99 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"

/* pseudo-header */

static __inline__ int ext2_block_hash(blk_t block)
{
	unsigned long x;

	x = block ^ (block >> 8) ^ (block >> 16) ^ (block >> 24);
	return x & ((1 << ext2_hash_bits) - 1);
}

static struct ext2_buffer_head *ext2_bh_alloc   (struct ext2_buffer_cache *, blk_t);
static void                     ext2_bh_dealloc (struct ext2_buffer_head *);
static struct ext2_buffer_head *ext2_bh_find    (struct ext2_buffer_cache *, blk_t);
static void                     ext2_bh_do_read (struct ext2_buffer_head *);
static void                     ext2_bh_do_write(struct ext2_buffer_head *);
static void                     ext2_bh_hash    (struct ext2_buffer_head *);
static void                     ext2_bh_unhash  (struct ext2_buffer_head *);



static void try_to_flush(struct ext2_buffer_cache *bc)
{
	int i;

	for (i=0;i<bc->size;i++)
	{
		struct ext2_buffer_head *bh;

		bh = &bc->heads[i];
	
		if (bh->alloc && !bh->usecount && !bh->dirty)
		{
			ext2_bh_dealloc(bh);
			return;
		}
	}

	for (i=0;i<bc->size;i++)
	{
		struct ext2_buffer_head *bh;

		bh = &bc->heads[i];

		if (bh->alloc && !bh->usecount && bh->dirty)
		{
			ext2_bh_do_write(bh);
			ext2_bh_dealloc(bh);

			return;
		}
	}

	fprintf(stderr, "ext2resize: couldn't flush!\n");
}





static struct ext2_buffer_head *ext2_bh_alloc(struct ext2_buffer_cache *bc, blk_t block)
{
	struct ext2_buffer_head *bh;
	int i;

	bh = NULL;

 tryagain:
	for (i=0;i<bc->size;i++)
	{
		bh = &bc->heads[i];

		if (!bh->alloc)
			break;
	}

	if (i == bc->size)
	{
		try_to_flush(bc);
		goto tryagain;
	}

	bh = &bc->heads[i];

	bh->next = NULL;
	bh->prev = NULL;
	bh->block = block;
	bh->usecount = 0;
	bh->dirty = 0;
	bh->alloc = 1;
	bc->numalloc++;

	ext2_bh_hash(bh);

	return bh;
}

static void ext2_bh_dealloc(struct ext2_buffer_head *bh)
{
	if (bh->dirty)
		fprintf(stderr, "deallocing() a dirty buffer! %i\n", bh->block);

	ext2_bh_unhash(bh);
	bh->alloc = 0;
	bh->bc->numalloc--;
}

static struct ext2_buffer_head *ext2_bh_find(struct ext2_buffer_cache *bc, blk_t block)
{
	struct ext2_buffer_head *a;
	struct ext2_buffer_head *b;
	int			 hash;

	hash = ext2_block_hash(block);
	a = bc->hash[hash];

	if (a != NULL)
	{
		b = a;
		do
		{
			if (a->block == block)
				return a;

			a = a->next;
		} while (a != b);
	}

	return NULL;
}

static void ext2_bh_do_read(struct ext2_buffer_head *bh)
{
	ext2_read_blocks(bh->bc->fs, bh->data, bh->block, 1);
}

static void ext2_bh_do_write(struct ext2_buffer_head *bh)
{
	ext2_write_blocks(bh->bc->fs, bh->data, bh->block, 1);
	bh->dirty = 0;
}

static void ext2_bh_hash(struct ext2_buffer_head *bh)
{
	int hash;

	hash = ext2_block_hash(bh->block);
	if (bh->bc->hash[hash] == NULL)
	{
		bh->bc->hash[hash] = bh;
		bh->next = bh->prev = bh;
		return;
	}

	bh->next = bh->bc->hash[hash];
	bh->prev = bh->next->prev;
	bh->next->prev = bh;
	bh->prev->next = bh;
}

static void ext2_bh_unhash(struct ext2_buffer_head *bh)
{
	int hash;

	hash = ext2_block_hash(bh->block);

	bh->prev->next = bh->next;
	bh->next->prev = bh->prev;

	if (bh->bc->hash[hash] == bh)
	{
		if (bh->next != bh)
			bh->bc->hash[hash] = bh->next;
		else
			bh->bc->hash[hash] = NULL;
	}

	bh->next = NULL;
	bh->prev = NULL;
}







static int breadimmhits = 0;
static int breadindhits = 0;
static int breadmisses = 0;

void ext2_bcache_deinit(struct ext2_fs *fs)
{
	ext2_bcache_sync(fs);
	free(fs->bc->buffermem);
	free(fs->bc->hash);
	free(fs->bc->heads);
	free(fs->bc);

	fprintf(stderr, "direct hits %i indirect hits %i misses %i\n",
		breadimmhits, breadindhits, breadmisses);
}

void ext2_bcache_dump(struct ext2_fs *fs)
{
	int i;

	fprintf(stderr, "buffer cache dump:\n");

	for (i=0;i<(1<<ext2_hash_bits);i++)
		if (fs->bc->hash[i] != NULL)
		{
			struct ext2_buffer_head *a;
			struct ext2_buffer_head *b;

			fprintf(stderr, "%i: ", i);

			a = b = fs->bc->hash[i];
			do
			{
				fprintf(stderr, "%i ", a->block);
				a = a->next;
			} while (a != b);

			fprintf(stderr, "\n");
		}
}

void ext2_bcache_flush(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;

	if ((bh = ext2_bh_find(fs->bc, block)) == NULL)
		return;

	if (bh->usecount)
		fprintf(stderr, "flushing buffer that's in use! [%i,%i]\n", bh->block, bh->usecount);

	if (bh->dirty)
		ext2_bh_do_write(bh);

	ext2_bh_dealloc(bh);
}

/* NB flushes [first,last> buffer range. anyone a better name for 'last'? */
void ext2_bcache_flush_range(struct ext2_fs *fs, blk_t first, blk_t last)
{
	blk_t i;

	for (i=first;i<last;i++)
		ext2_bcache_flush(fs, i);
}

int ext2_bcache_init(struct ext2_fs *fs)
{
	struct ext2_buffer_cache *bc;
	int i;
	int size;

	size = ext2_buffer_cache_pool_size >> (fs->logsize - 10);

	if ((bc = (struct ext2_buffer_cache *)malloc(sizeof(struct ext2_buffer_cache))) == NULL)
	{
		fprintf(stderr, "MAError!\n");
		return 0;
	}

	if ((bc->heads = (struct ext2_buffer_head *)malloc(size * sizeof(struct ext2_buffer_head))) == NULL)
	{
		fprintf(stderr, "MAError!\n");
		free(bc);
		return 0;
	}

	if ((bc->hash = (struct ext2_buffer_head **)malloc(sizeof(struct ext2_buffer_head *) << ext2_hash_bits)) == NULL)
	{
		fprintf(stderr, "MAError!\n");
		free(bc->heads);
		free(bc);
		return 0;
	}

	if ((bc->buffermem = (unsigned char *)malloc(ext2_buffer_cache_pool_size << 10)) == NULL)
	{
		fprintf(stderr, "MAError!\n");
		free(bc->hash);
		free(bc->heads);
		free(bc);
		return 0;
	}

	bc->cache = &bc->heads[0];
	bc->fs = fs;
	bc->size = size;
	bc->numalloc = 0;

	for (i=0;i<size;i++)
	{
		bc->heads[i].data = bc->buffermem + (i << fs->logsize);
		bc->heads[i].bc = bc;
		bc->heads[i].alloc = 0;
	}

	for (i=0;i<(1<<ext2_hash_bits);i++)
		bc->hash[i] = NULL;

	fs->bc = bc;

	return 1;
}

void ext2_bcache_sync(struct ext2_fs *fs)
{
	int i;

	for (i=0;i<fs->bc->size;i++)
	{
		struct ext2_buffer_head *bh;

		bh = &fs->bc->heads[i];

		if (bh != NULL && bh->dirty)
			ext2_bh_do_write(bh);
	}
}








struct ext2_buffer_head *ext2_bcreate(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;

	if ((bh = ext2_bh_find(fs->bc, block)) != NULL)
	{
		bh->usecount++;
	}
	else
	{
		bh = ext2_bh_alloc(fs->bc, block);
		bh->usecount = 1;
	}

	memset(bh->data, 0, fs->blocksize);
	bh->dirty = 1;

	return bh;
}

struct ext2_buffer_head *ext2_bread(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;

	if ((bh = fs->bc->cache)->block == block)
	{
		breadimmhits++;
		bh->usecount++;
		return bh;
	}

	if ((bh = ext2_bh_find(fs->bc, block)) != NULL)
	{
		fs->bc->cache = bh;
		breadindhits++;
		bh->usecount++;
		return bh;
	}

	breadmisses++;

	bh = ext2_bh_alloc(fs->bc, block);
	fs->bc->cache = bh;
	bh->usecount = 1;
	ext2_bh_do_read(bh);

	return bh;
}

void ext2_brelse(struct ext2_buffer_head *bh, int forget)
{
	if (bh->usecount-- == 1 && forget)
	{
		if (bh->dirty)
			ext2_bh_do_write(bh);

		ext2_bh_dealloc(bh);
	}
}
