/*
 *  dialog - Display simple dialog boxes from shell scripts
 *
 *  AUTHOR: Savio Lam (lam836@cs.cuhk.hk)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *  HISTORY:
 *
 *  17/12/93 - Version 0.1 released.
 *
 *  19/12/93 - menu will now scroll if there are more items than can fit
 *             on the screen.
 *           - added 'checklist', a dialog box with a list of options that
 *             can be turned on or off. A list of options that are on is 
 *             returned on exit.
 *
 *  20/12/93 - Version 0.15 released.
 *
 *  29/12/93 - Incorporated patch from Patrick J. Volkerding
 *             (volkerdi@mhd1.moorhead.msus.edu) that made these changes:
 *             - increased MAX_LEN to 2048
 *             - added 'infobox', equivalent to a message box without pausing
 *             - added option '--clear' that will clear the screen
 *             - Explicit line breaking when printing prompt text can be
 *               invoked by real newline '\n' besides the string "\n"
 *           - an optional parameter '--title <string>' can be used to
 *             specify a title string for the dialog box
 *
 *  03/01/94 - added 'textbox', a dialog box for displaying text from a file.
 *           - Version 0.2 released.
 *
 *  04/01/94 - some fixes and improvements for 'textbox':
 *             - fixed a bug that will cause a segmentation violation when a
 *               line is longer than MAX_LEN characters. Lines will now be
 *               truncated if they are longer than MAX_LEN characters.
 *             - removed wrefresh() from print_line(). This will increase
 *               efficiency of print_page() which calls print_line().
 *             - display current position in the form of percentage into file.
 *           - Version 0.21 released.
 *
 *  05/01/94 - some changes for faster screen update.
 *
 *  07/01/94 - much more flexible color settings. Can use all 16 colors
 *             (8 normal, 8 highlight) of the Linux console.
 *
 *  08/01/94 - added run-time configuration using configuration file.
 *
 *  09/01/94 - some minor bug fixes and cleanups for menubox, checklist and
 *             textbox.
 *
 *  11/01/94 - added a man page.
 *
 *  13/01/94 - some changes for easier porting to other Unix systems (tested
 *             on Ultrix, SunOS and HPUX)
 *           - Version 0.3 released.
 *  04/08/95 - Complete rewriten in C++. It is no more the official
 *             dialog program. Not even sure its still work. The goal
 *	       was to turn all this into a library.
 */


#define VERSION	"C++ 0.3"

#include <stdlib.h>
#include "dialog.h"
#include "diadef.h"
#ifdef HAVE_NCURSES
#include "colors.h"
#endif


/*
 * Print program usage
 */
static void Usage(char *name)
{
	fprintf(stderr,
		"dialog version %s, by Savio Lam (lam836@cs.cuhk.hk).\n"
		"\n"
		"* Display dialog boxes from shell scripts *\n"
		"\n"
		"Usage: %s --clear\n"
		"       %s --create-rc <file>\n"
		"       %s [--title <title>] [--clear] <Box options>\n"
		"\n"
		"Box options:\n"
		"\n"
		"  --yesno     <text>\n"
		"  --msgbox    <text>\n"
		"  --infobox   <text>\n"
		"  --inputbox  <text>\n"
		"  --inputpass <text>\n"
		"  --textbox   <file> <height> <width>\n"
		"  --menu      <text> <menu height> <tag1> <item1>...\n"
		"  --checklist <text> <list height> <tag1> <item1> <status1>...\n"
		, VERSION, name, name, name);
	exit (-1);
}



int main(int argc, char *argv[])
{
	int offset = 0, clear_screen = 0, end_common_opts = 0;
	int retval = 0;
	char *title = NULL;

	if (argc < 2) {
		Usage(argv[0]);
	}else if (!strcmp(argv[1], "--create-rc")) {
		#ifdef HAVE_NCURSES
			if (argc != 3) {
				Usage(argv[0]);
			}
			create_rc(argv[2]);
			return 0;
		#else
			fprintf(stderr, "\nThis option is currently unsupported on your system.\n");
			return -1;
		#endif
	}

	while (offset < argc-1 && !end_common_opts) {    /* Common options */
		if (!strcmp(argv[offset+1], "--title")) {
			if (argc-offset < 3 || title != NULL) {    /* No two "--title" please! */
				Usage(argv[0]);
			}else {
				title = argv[offset+2];
				offset += 2;
			}
		}else if (!strcmp(argv[offset+1], "--clear")) {
			if (clear_screen) {    /* Hey, "--clear" can't appear twice! */
				Usage(argv[0]);
			}else if (argc == 2) {    /* we only want to clear the screen */
				init_dialog();
				refresh();    /* init_dialog() will clear the screen for us */
				endwin();
				return 0;
			}else {
				clear_screen = 1;
				offset++;
			}
		}else{    /* no more common options */
			end_common_opts = 1;
		}
	}

	if (argc-1 == offset) {    /* no more options */
		Usage(argv[0]);
	}else if (!strcmp(argv[offset+1], "--yesno")) {
		if (argc-offset != 5) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_yesno(title, argv[offset+2],NULL)==MENU_YES
			? 0 : 1;
	}else if (!strcmp(argv[offset+1], "--msgbox")) {
		if (argc-offset != 5) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_msgbox(title, argv[offset+2], 1);
	}else if (!strcmp(argv[offset+1], "--infobox")) {
		if (argc-offset != 5) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_msgbox(title, argv[offset+2], 0);
	}else if (!strcmp(argv[offset+1], "--textbox")) {
		if (argc-offset != 5) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_textbox(title, argv[offset+2], atoi(argv[offset+3])
			,atoi(argv[offset+4]));
	}else if (!strcmp(argv[offset+1], "--menu")) {
		int choice=0;
		if (argc-offset < 5 || ((argc-offset-3) % 2)) {
			Usage(argv[0]);
		}
		init_dialog();
		if (dialog_menu(title, argv[offset+2], help_nil,0
			,(argc-offset-3)/2, argv+offset + 3,choice)
			== MENU_OK){
			fprintf (stderr,"%s\n",argv[offset+3+choice*2]);
		}
	}else if (!strcmp(argv[offset+1], "--checklist")) {
		if (argc-offset < 9 || ((argc-offset-6) % 3)) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_checklist_p(title, argv[offset+2],NULL,
                              atoi(argv[offset+3]),
                              (argc-offset-6)/3, argv+offset + 6);
	}else if (!strcmp(argv[offset+1], "--inputbox")) {
		char inpstr[MAX_LEN+1];
		if (argc-offset != 5) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_inputbox(title, argv[offset+2], NULL, inpstr);
		if (retval == 0) fprintf (stderr,"%s",inpstr);
	}else if (!strcmp(argv[offset+1], "--inputpass")) {
		char inpstr[MAX_LEN+1];
		if (argc-offset != 5) {
			Usage(argv[0]);
		}
		init_dialog();
		retval = dialog_inputpass(title, argv[offset+2], NULL, inpstr);

		if (retval == 0) fprintf (stderr,"%s",inpstr);
	}else{
		Usage(argv[0]);
	}
	if (clear_screen) {    /* clear screen before exit */
		attr_clear(stdscr, LINES, COLS, screen_attr);
		refresh();
	}
    endwin();
    return retval;
}




