/*****************************************************************************
**
** pnpdump.c
*/

#include <stdio.h>

#ifdef __DJGPP__
#undef REALTIME
#else
#ifndef _OS2_
#include <getopt.h>
#endif /* _OS2_ */
#endif

#include <stdlib.h>
#include <string.h>		/* For strncpy */

#define TAG_DEBUG 0
#define DUMPADDR 0

#include "iopl.h"
#include "pnp.h"
#include "resource.h"
#include "res-access.h"
#include "errenum.h"
#include "callbacks.h"
#include "release.h"

#ifdef REALTIME
#include "realtime.h"
long l_realtime_timeout = 5L;
#endif

#define LARGE_LEN 65536

static char rcsid[] = "$Id: pnpdump.c,v 1.21 1999/12/09 22:28:33 fox Exp $";

static void print_resource(FILE * output_file,
			   struct resource *res,
			   int selected);


#ifndef __DJGPP__
int do_script;
char *output_script = NULL;

static void print_script_header(FILE * script_file);

static void print_script(FILE * output_file,
						 struct resource *res,
						 int selected);
#endif /* __DJGPP__ */

/* Global state variables used for printing resource information. */
unsigned long serialno = 0;
char devid[8];											   /* Copy as devidstr
														    * returns pointer to
														    * static buffer subject
														    * to change */
int nestdepth;
int curid;
int curdma, depdma;
int curio, depio;
int curint, depint;
int curmem, depmem;
int starteddeps;
char *indep = "";

/* This shouldn't be required 8-Oct-1998 */
static void
prepare_to_print(void)
{
	nestdepth = 0;
	curid = 0;
	curdma = 0;
	depdma = 0;
	curio = 0;
	depio = 0;
	curint = 0;
	depint = 0;
	curmem = 0;
	depmem = 0;
	starteddeps = 0;
	indep = "";
}

unsigned char tmp[TMP_LEN];
unsigned char large[LARGE_LEN];
static char *devstring = 0;

int showmasks = 0;
int do_autoconfig = 0;

void dumpdata(int);

static void usage(char *program_name);

int do_dumpregs = 0;
void dumpregs(int);

char *devidstr(unsigned char, unsigned char, unsigned char, unsigned char);

/****************************************************************************/
/* Callbacks needed by the libisapnp procedures                             */
/****************************************************************************/

char *st_error_messages[] = {
#define ISAPNP_ERR_CODE(code,msg) msg,
#include "errcodes.h"
};

/* Forward declarations - this way I am making sure that the typedefs and the
** actual declarations (below) match each other.
*/
fatal_error     pnpdump_fatal_error_callback;
non_fatal_error pnpdump_non_fatal_error_callback;
progress_report pnpdump_progress_report_callback;

/* The callbacks themselves */
void
pnpdump_fatal_error_callback(int in_isapnp_error)
{
  if (in_isapnp_error >= ISAPNP_E_LAST_ERROR_CODE) {
    fprintf(stderr, "Unknown fatal error - error code is %d.\n",
	    in_isapnp_error);
  }
  else {
    fprintf(stderr,"%s\n",st_error_messages[in_isapnp_error]);
  }
  exit(1);
}

void
pnpdump_non_fatal_error_callback(int in_errno, int in_isapnp_error)
{
  if (in_isapnp_error >= ISAPNP_E_LAST_ERROR_CODE) {
    fprintf(stderr, "Unknown fatal error - error code is %d: %s\n",
	    in_isapnp_error, strerror(in_errno));
  }
  else if (ISAPNP_E_PRINT_PROGRESS_REPORT_BUF == in_isapnp_error) {
    fprintf(stderr, progress_report_buf);
  }
  else {
    if (0 == in_errno) {
      fprintf(stderr,"%s\n", st_error_messages[in_isapnp_error]);
    }
    else {
		perror(st_error_messages[in_isapnp_error]);
    }
  }
}

void
pnpdump_progress_report_callback(const char *in_msg)
{
  /* Prints the progress report to stdout.
  ** In pnpdump.c it is used to actually write the isapnp.conf
  ** configuration file.
  */
  printf("%s", in_msg);
}

/****************************************************************************/

static void
banner(FILE *fp, char *prefix)
{
	fprintf(fp, 
			"%s%s\n"
			"%sRelease %s (library %s)\n"
			"%s\n"
			"%sThis is free software, see the sources for details.\n"
			"%sThis software has NO WARRANTY, use at your OWN RISK\n"
			"%s\n"
			"%sFor details of the output file format, see isapnp.conf(5)\n"
			"%s\n"
			"%sFor latest information and FAQ on isapnp and pnpdump see:\n"
			"%shttp://www.roestock.demon.co.uk/isapnptools/\n"
			"%s\n"
			"%sCompiler flags: %s\n"
			"%s(for   library: %s)\n"
			"%s\n",
			prefix, rcsid,
			prefix, ISAPNPTOOLSVER, libtoolsver,
			prefix, prefix, prefix, prefix, prefix, prefix, prefix, prefix, prefix,
			prefix, ISAPNPTOOLSCOMPILERFLAGS,
			prefix, libcompilerflags,
			prefix);
}

/*
 * Single argument is initial readport
 * Two arguments is no-of-boards and the readport to use
 */
int
main(int argc, char **argv)
{
	int i;
	char *program = argv[0];
	struct resource *resources;
	int resource_count;
	int alloc_result;

	/* Process command line options */
#ifndef __DJGPP__
	const struct option longopts[] =
	{
		{"masks", no_argument, NULL, 'm'},
		{"config", no_argument, NULL, 'c'},
		{"help", no_argument, NULL, 'h'},
		{"script", optional_argument, NULL, 's'},
		{"reset", no_argument, NULL, 'r'},
		{"dumpregs", no_argument, NULL, 'd'},
		{"debug-isolate", no_argument, NULL, 'D'},
		{"ignorecsum", no_argument, NULL, 'i'},
#ifdef REALTIME
		{"max-realtime", required_argument, NULL, 't'},
#endif
		{"version", no_argument, NULL, 'v'},
		{0, 0, 0, 0},
	};
	int opt;

	while ((opt = getopt_long(argc, argv, "cms:rdit:vDh", longopts, NULL)) != EOF)
	{
		switch (opt)
		{
		case 'c':
			do_autoconfig = 1;
			break;
		case 'm':
			showmasks = 1;
			break;
		case 'r':
			do_fullreset = 1;
			break;
		case 's':
			do_script = 1;
			output_script = optarg;
			break;
		case 'd':
			do_dumpregs = 1;
			break;
		case 'D':
			debug_isolate = 1;
			break;
		case 'h':
			usage(program);
			break;
		case 'i':
			do_ignorecsum = 1;
			break;
#ifdef REALTIME
		case 't':
			l_realtime_timeout = atol(optarg);
			break;
#else
		case 't':
			fprintf(stderr, "Realtime support not compiled in - option ignored\n");
			break;
#endif
		case 'v':
			fprintf(stderr, "Version: pnpdump from %s\n", ISAPNPTOOLSVER);
			exit(0);
			break;
		case '?':
#if 1
			fprintf(stderr, "unrecognized option.\n");
#endif
			usage(program);
			return 1;
		case ':':
#if 1
			fprintf(stderr, "missing parameter.\n");
#endif
			usage(program);
			return 1;
		default:
			fprintf(stderr,
					"?? getopt returned character code 0x%x ('%c').\n",
					opt, opt);
			return 1;
		}

	}
	argc -= optind - 1;
	argv += optind - 1;
#else
	while (argc > 1)
	{
		if (strcmp(argv[1], "--help") == 0)
		{
			usage(program);
		}
		else if (strcmp(argv[1], "--reset") == 0)
		{
			do_fullreset = 1;
		}
		else if (strcmp(argv[1], "--masks") == 0)
		{
			showmasks = 1;
		}
		else if (strcmp(argv[1], "--dumpregs") == 0)
		{
			do_dumpregs = 1;
		}
		else if (strcmp(argv[1], "--config") == 0)
		{
			do_autoconfig = 1;
		}
		else if (strcmp(argv[1], "--ignorecsum") == 0)
		{
			do_ignorecsum = 1;
		}
		else if (strcmp(argv[1], "--debug-isolate") == 0)
		{
			debug_isolate = 1;
		}
		else if (strcmp(argv[1], "--version") == 0)
		{
			fprintf(stderr, "Version: pnpdump from %s\n", ISAPNPTOOLSVER);
		}
		else
		{
			break;
		}
		argv++;
		argc--;
	}
#endif /* __DJGPP__ */

	/* Initialize callbacks */
	callbacks_init(pnpdump_fatal_error_callback,
		       pnpdump_non_fatal_error_callback,
		       pnpdump_progress_report_callback);

	/* Process arguments */
	if ((argc < 1) || (argc > 3))
	{
		usage(program);
		return 1;
	}

	/* If a number of boards and read_port have been given, don't ISOLATE */
	if (argc == 3)
	{
		boards_found = atoi(argv[1]);	/* This stops ISOLATE */
		if ((boards_found < 0) || (boards_found >= NUM_CARDS))
		{
			fprintf(stderr, "Cannot handle %d boards, recompile with NUM_CARDS bigger\n", boards_found);
			exit(1);
		}
	}
	if (argc > 1)
	{
		/* Read decimal or hex number */
		read_port = (int) strtol(argv[argc - 1], (char **) NULL, 0);
		if ((read_port < MIN_READ_ADDR) || (read_port > MAX_READ_ADDR))
		{
			fprintf(stderr, "Port address %s (0x%04x) out of range 0x203..0x3ff\n", argv[argc - 1], read_port);
			exit(1);
		}
		read_port |= 3;
	}
	if (acquire_pnp_io_privileges() != 0) {
	  perror("Unable to get io permission for WRITE_DATA");
	  exit(1);
	}
	banner(stdout, "# ");
	/* Have to do this anyway to check readport for clashes */
	alloc_system_resources();
#ifdef REALTIME
	setroundrobin(l_realtime_timeout);
#endif /* REALTIME */
	/* Now get board info */
	initiate();
	resources = NULL;
	resource_count = 0;
	for (i = 1; i <= boards_found; i++)
	{
#if 1
		read_board(i, &resources, &resource_count);
#else
		dumpdata(i);
#endif
		if(do_dumpregs)
			dumpregs(i);
	}
#ifdef REALTIME
	/* No longer need real-time scheduling */
	normal_sched();
#endif /* REALTIME */
	if (do_autoconfig)
	{
		alloc_result = alloc_resources(resources, resource_count, NULL, 0);
	}
	else
	{
		alloc_result = 0;
	}
	prepare_to_print();
	for_each_resource(resources, resource_count,
					  print_resource, stdout, alloc_result);
	/* Musn't forget this ! */
	printf("# Returns all cards to the \"Wait for Key\" state\n");
	printf("(WAITFORKEY)\n");
	/* Release resources */
	if (relinquish_pnp_io_privileges() != 0) {
	  perror("Unable to release io permission for WRITE_DATA");
	  exit(1);
	}


#ifndef __DJGPP__
	if (do_script)
	{
		FILE *script_file;

		if (output_script != NULL)
		{
			script_file = fopen(output_script, "w");
		}
		else
		{
			script_file = popen("sh", "w");
		}
		if (script_file == NULL)
		{
			perror(output_script ? output_script : "sh");
			exit(1);
		}
		if (alloc_result == 0)
		{
			fprintf(script_file, "#!/bin/sh\n" "isa-pnp-config-failed\n");
		}
		else
		{
			print_script_header(script_file);
			for_each_resource(resources, resource_count,
							  print_script, script_file, 1);
		}
		if (output_script)
		{
			fclose(script_file);
		}
		else
		{
			pclose(script_file);
		}
	}
#endif /* __DJGPP__ */
	return (0);
}

static void 
usage(char *program_name)
{
	banner(stderr, "");
	fprintf(stderr,
			"Usage: %s [OPTIONS] [[devs] readport]\n\n"
#ifndef __DJGPP__
			"   -c, --config          uncomment settings to which the devices can be set\n"
			"   -d, --dumpregs        dump standard configuration registers for each board\n"
			"   -D, --debug-isolate   output loads of extra information on isolation process\n"
			"   -h, --help            show a help summary to stderr\n"
			"   -i, --ignorecsum      ignore checksum errors when choosing the readport address\n"
			"   -r, --reset           carry out a full configuration reset\n"
			"   -m, --masks           list acceptable interrupts and DMA channels as bitmasks\n"
#ifdef REALTIME
			" -t t, --max-realtime=t  maximum real-time priority execution is t seconds\n"
#endif
			" -s scr, --script[=scr]  write a shell script to the specified file\n"
			"                         no script pipes output to a shell\n"
			"   -v, --version         print the isapnptools version number on stderr\n"
#else
			" 	--config          uncomment settings to which the devices can be set\n"
			" 	--dumpregs        dump standard configuration registers for each board\n"
			" 	--debug-isolate   output loads of extra information on isolation process\n"
			" 	--help            show a help summary to stderr\n"
			" 	--ignorecsum      ignore checksum errors when choosing the readport address\n"
			" 	--reset           carry out a full configuration reset\n"
			" 	--masks           list acceptable interrupts and DMA channels as bitmasks\n"
			" 	--version         print the isapnptools version number on stderr\n"
#endif
			"\n"
			"   devs      is the number of PnP cards that the BIOS has found\n"
			"   readport  is the address of the readport to use for Plug-And-Play access\n"
			, program_name);
	exit(0);
}

char *
devidstr(unsigned char d1, unsigned char d2, unsigned char d3, unsigned char d4)
{
	static char resstr[] = "PNP0000";
	sprintf(resstr, "%c%c%c%x%x%x%x", 'A' + (d1 >> 2) - 1, 'A' + (((d1 & 3) << 3) | (d2 >> 5)) - 1,
			'A' + (d2 & 0x1f) - 1, d3 >> 4, d3 & 0x0f, d4 >> 4, d4 & 0x0f);
	return resstr;
}

void
lengtherror(int len, char *msg)
{
	int i;
	printf("# Bad tag length for %s in 0x%02x", msg, tmp[0]);
	if (tmp[0] & 0x80)
	{
		printf(" 0x%02x 0x%02x", len & 255, len >> 8);
		for (i = 0; i < len; i++)
			printf(" 0x%02x", large[i]);
	}
	else
	{
		for (i = 1; i <= len; i++)
			printf(" 0x%02x", tmp[i]);
	}
	printf("\n");
}

#ifdef ABORT_ONRESERR
#define BREAKORCONTINUE   goto oncorruptresourcedata
#else
#define BREAKORCONTINUE   break
#endif

static void
showmask(unsigned long mask)
{
	int i;
	int firsttime = 1;

	if (showmasks)
	{
		printf(" mask 0x%lx", mask);
	}
	else
	{
		for (i = 0; mask; i++, mask >>= 1)
		{
			if (mask & 0x1)
			{
				if (!firsttime)
				{
					printf(mask == 1 ? " or " : ", ");
				}
				else
				{
					printf(" ");
					firsttime = 0;
				}
				printf("%d", i);
			}
		}
	}
}


static int IORangeCheck = 0;

static void
print_resource(FILE * output_file, struct resource *res, int selected)
{
	static int logdevno = 0;
	int type = res->type;
	int len = res->len;
	int i;
	char *comment_if_not_selected = selected ? " " : "#";

	switch (type)
	{
	case NewBoard_PSEUDOTAG:
		{
			int csn = res->start;

			logdevno = 0;
			serialno = (unsigned long)
				serial_identifier[csn][4] +
				(serial_identifier[csn][5] << 8) +
				(serial_identifier[csn][6] << 16) +
				(serial_identifier[csn][7] << 24);
			strncpy(devid, devidstr(serial_identifier[csn][0],
									serial_identifier[csn][1],
									serial_identifier[csn][2],
									serial_identifier[csn][3]), 8);

			printf("# Card %d: (serial identifier", csn);
			for (i = IDENT_LEN; i--;)
				printf(" %02x", serial_identifier[csn][i]);
			printf(")\n");
			if (serialno == 0xffffffffUL)
			{
				printf("# Vendor Id %s, No Serial Number (-1), checksum 0x%02X.\n", devid, serial_identifier[csn][8]);
			}
			else
			{
				printf("# Vendor Id %s, Serial Number %lu, checksum 0x%02X.\n", devid, serialno, serial_identifier[csn][8]);
			}
			if (res->len != IDENT_LEN)
			{
				i = 0;
				printf("# Ident byte %d, (%02x) differs from resource data (%02x)\n", i, serial_identifier[csn][i], res->data[0]);
				printf("#Assuming the card is broken and this is the start of the resource data\n");
			}
			else
			{
				for (i = 1; i < IDENT_LEN; i++)
				{
					if ((res->data[i] != serial_identifier[csn][i]) && (i < (IDENT_LEN - 1)))
						printf("# Ident byte %d, (%02x) differs from resource data (%02x)\n", i, serial_identifier[csn][i], res->data[i]);
				}
			}
		}

		break;
	case PnPVerNo_TAG:
		{
			if (len != 2)
			{
				lengtherror(len, "PnPVerNo_TAG");
				BREAKORCONTINUE;
			}
			printf("# %sVersion %d.%d, Vendor version %x.%x\n", indep,
				   res->data[0] >> 4,
				   res->data[0] & 0x0f,
				   res->data[1] >> 4,
				   res->data[1] & 0x0f);
			break;
		}
	case LogDevId_TAG:
		{
			int reg;
			static char *regfns[8] =
			{"Device capable of taking part in boot process",
			 "Device supports I/O range check register",
			 "Device supports reserved register @ 0x32",
			 "Device supports reserved register @ 0x33",
			 "Device supports reserved register @ 0x34",
			 "Device supports reserved register @ 0x35",
			 "Device supports reserved register @ 0x36",
			 "Device supports reserved register @ 0x37"};
			if ((len < 5) || (len > 6))
			{
				lengtherror(len, "LogDevId_TAG");
				BREAKORCONTINUE;
			}
			indep = "";
			if (nestdepth)
			{
				/* If we have a device name, show it (the last description string before the END DF flag is it) */
				if (serialno == 0xffffffffUL)
					printf(" (NAME \"%s/-1[%d]", devid, logdevno-1);
				else
					printf(" (NAME \"%s/%lu[%d]", devid, serialno, logdevno-1);
				if(devstring)
					printf("{%-20s}", devstring);
				printf("\")\n");
				printf("%s (ACT Y)\n", comment_if_not_selected);
				while (nestdepth)
				{
					printf(")");
					nestdepth--;
				}
				printf("\n");
			}
			printf("#\n# %sLogical device id %s\n", indep, devidstr(res->data[0], res->data[1], res->data[2], res->data[3]));
			indep = "    ";
			IORangeCheck = 0;
			for (i = 0, reg = 1; reg < 256; i++, reg <<= 1)
			{
				if (res->data[4] & reg)
				{
					printf("# %s%s\n", indep, regfns[i]);
					if(i == 1)
						IORangeCheck = 1;
				}
			}
			for (i = 0, reg = 1; reg < 256; i++, reg <<= 1)
			{
				if (res->data[5] & reg)
				{
					printf("# %sDevice supports vendor reserved register @ 0x%02x\n", indep, 0x38 + i);
				}
			}
			printf("#\n# Edit the entries below to uncomment out the configuration required.\n");
			printf("# Note that only the first value of any range is given, this may be changed if required\n");
			printf("# Don't forget to uncomment the activate (ACT Y) when happy\n\n");
			if (serialno == 0xffffffffUL)
				printf("(CONFIGURE %s/-1 (LD %d\n", devid, logdevno++);
			else
				printf("(CONFIGURE %s/%lu (LD %d\n", devid, serialno, logdevno++);
			nestdepth = 2;
			curdma = 0;
			depdma = 0;
			curio = 0;
			depio = 0;
			curint = 0;
			depint = 0;
			curmem = 0;
			depmem = 0;
			starteddeps = 0;
			break;
		}
	case CompatDevId_TAG:
		{
			if (len != 4)
			{
				lengtherror(len, "CompatDevId_TAG");
				BREAKORCONTINUE;
			}
			printf("# %sCompatible device id %s\n", indep, devidstr(res->data[0], res->data[1], res->data[2], res->data[3]));
			break;
		}
	case IRQ_TAG:
		{
			int firstirq = 0;
			char *edge = "+E";
			if ((len < 2) || (len > 3))
			{
				lengtherror(len, "IRQ_TAG");
				BREAKORCONTINUE;
			}
			if ((len >= 2) && (res->data[0] || res->data[1]))
			{
				printf("# %sIRQ", indep);
				if(res->errflags & RES_ERR_NO_IRQ)
					printf("# %s%s*** Bad resource data: No IRQ specified\n", indep, indep);
				if(res->errflags & RES_ERR_IRQ2)
					printf("# %s%s*** Bad resource data (Clarifications 4.6.2): IRQ 2 invalid, changing to 9\n", indep, indep);
				showmask(res->mask);
				printf(".\n");
				firstirq = res->value;
				if (len == 3)
				{
					if (res->data[2] & 1)
					{
						printf("# %s%sHigh true, edge sensitive interrupt\n", indep, indep);
						edge = "+E";
					}
					if (res->data[2] & 2)
					{
						printf("# %s%sLow true, edge sensitive interrupt\n", indep, indep);
						edge = "-E";
					}
					if (res->data[2] & 4)
					{
						printf("# %s%sHigh true, level sensitive interrupt\n", indep, indep);
						edge = "+L";
					}
					if (res->data[2] & 8)
					{
						printf("# %s%sLow true, level sensitive interrupt\n", indep, indep);
						edge = "-L";
					}
				}
				else
				{
					printf("# %s%sHigh true, edge sensitive interrupt (by default)\n", indep, indep);
				}
				printf("%s (INT %d (IRQ %ld (MODE %s)))\n",
					   comment_if_not_selected, curint, res->value, edge);
				curint++;
				if (!starteddeps)
					depint = curint;
			}
			else
			{
				printf("# %s*** ERROR *** No IRQ specified!\n", indep);
			}
			break;
		}
	case DMA_TAG:
		{
			int firstdma = 4;							   /* Ie, no DMA */
			if (len != 2)
			{
				lengtherror(len, "DMA_TAG");
				BREAKORCONTINUE;
			}
			if (res->mask)
			{
				printf("# %s%sDMA channel",
					   indep, (curdma == 0) ? "First " : "Next ");
				firstdma = res->value;
				showmask(res->mask);
				printf(".\n");
			}
			else
			{
				printf("# %s*** ERROR *** No DMA channel specified!\n", indep);
			}
			if ((res->data[1] & 3) == 0)
				printf("# %s%s8 bit DMA only\n", indep, indep);
			if ((res->data[1] & 3) == 1)
				printf("# %s%s8 & 16 bit DMA\n", indep, indep);
			if ((res->data[1] & 3) == 2)
				printf("# %s%s16 bit DMA only\n", indep, indep);
			printf("# %s%sLogical device is%s a bus master\n", indep, indep, res->data[2] & 4 ? "" : " not");
			printf("# %s%sDMA may%s execute in count by byte mode\n", indep, indep, res->data[1] & 8 ? "" : " not");
			printf("# %s%sDMA may%s execute in count by word mode\n", indep, indep, res->data[1] & 0x10 ? "" : " not");
			if ((res->data[1] & 0x60) == 0x00)
				printf("# %s%sDMA channel speed in compatible mode\n", indep, indep);
			if ((res->data[1] & 0x60) == 0x20)
				printf("# %s%sDMA channel speed type A\n", indep, indep);
			if ((res->data[1] & 0x60) == 0x40)
				printf("# %s%sDMA channel speed type B\n", indep, indep);
			if ((res->data[1] & 0x60) == 0x60)
				printf("# %s%sDMA channel speed type F\n", indep, indep);
			printf("%s (DMA %d (CHANNEL %d))\n", comment_if_not_selected, curdma, firstdma);
			curdma++;
			if (!starteddeps)
				depdma = curdma;
			break;
		}
	case StartDep_TAG:
		{
			if (len > 1)
			{
				lengtherror(len, "StartDep_TAG");
				BREAKORCONTINUE;
			}
			putchar('\n');
			if (!starteddeps)
				printf("# Multiple choice time, choose one only !\n\n");
			if (res->len == 0)
			{
				printf("# %sStart dependent functions: priority acceptable\n", indep);
			}
			else
				switch (res->data[0])
				{
				case 0:
					printf("# %sStart dependent functions: priority preferred\n", indep);
					break;
				case 1:
					printf("# %sStart dependent functions: priority acceptable\n", indep);
					break;
				case 2:
					printf("# %sStart dependent functions: priority functional\n", indep);
					break;
				default:
					printf("# %sStart dependent functions: priority INVALID\n", indep);
					break;
				}
			indep = "      ";
			starteddeps = 1;
			curio = depio;
			curdma = depdma;
			curint = depint;
			curmem = depmem;
			break;
		}
	case EndDep_TAG:
		{
			int i;
			if (len > 0)
			{
				lengtherror(len, "EndDep_TAG");
				BREAKORCONTINUE;
			}
			for (i = 0; i < res->end; i++)
			{
				for_each_resource(res->alternatives[i].resources,
								  res->alternatives[i].len,
								  print_resource,
								  stdout,
								  selected && (res->value == i));
			}
			indep = "    ";
			printf("\n# %sEnd dependent functions\n", indep);
			break;
		}
	case IOport_TAG:
		{
			if (len != 7)
			{
				lengtherror(len, "IOport_TAG");
				printf("# Bad tag length in 0x%02x\n", res->tag);
				BREAKORCONTINUE;
			}
			printf("# %sLogical device decodes %s IO address lines\n", indep, res->data[0] ? "16 bit" : "10 bit");
			printf("# %s%sMinimum IO base address 0x%04lx\n", indep, indep, res->start);
			printf("# %s%sMaximum IO base address 0x%04lx\n", indep, indep, res->end - 1);
			printf("# %s%sIO base alignment %ld bytes\n", indep, indep, res->step);
			if(res->errflags & RES_ERR_NO_STEP)
				printf("# %s%s*** Bad resource data: Base alignment 0 - changed to 1\n", indep, indep);
			printf("# %s%sNumber of IO addresses required: %d\n", indep, indep, res->data[6]);
#if DUMPADDR
			for (i = ((res->data[3] << 8) + res->data[2]); i <= ((res->data[5] << 8) + res->data[4]); i += res->data[5])
			{
				printf("# %s%s0x%04x..0x%04x\n", indep, indep, indep, i, i + res->data[6] - 1);
			}
#endif /* DUMPADDR */
			printf("%s (IO %d (SIZE %d) (BASE 0x%04lx)%s)\n",
				   comment_if_not_selected, curio, res->data[6], res->value, IORangeCheck ? " (CHECK)" : "");
			curio++;
			if (!starteddeps)
				depio = curio;
			break;
		}
	case FixedIO_TAG:
		{
			if (len != 3)
			{
				lengtherror(len, "FixedIO_TAG");
				printf("# Bad tag length in 0x%02x\n", res->tag);
				BREAKORCONTINUE;
			}
			printf("# %sFixed IO base address 0x%04lx\n", indep, res->start);
			printf("# %s%sNumber of IO addresses required: %ld\n", indep, indep, res->size);
			printf("%s (IO %d (SIZE %ld) (BASE 0x%04lx)%s)\n",
				   comment_if_not_selected, curio, res->size, res->start, IORangeCheck ? " (CHECK)" : "");
			curio++;
			if (!starteddeps)
				depio = curio;
			break;
		}
	case MemRange_TAG:
		{
			char width = 'w';

			if (len != 9)
			{
				lengtherror(len, "MemRange_TAG");
				printf("# %sInvalid length for memory range tag 0x%02x\n", indep, res->tag);
				BREAKORCONTINUE;
			}
			printf("# %sMemory is %s\n", indep, res->data[0] & 1 ? "writeable" : "non-writeable (ROM)");
			printf("# %sMemory is %s\n", indep, res->data[0] & 2 ? "read cacheable, write-through" : "non-cacheable");
			printf("# %sMemory decode supports %s\n", indep, res->data[0] & 4 ? "range length" : "high address");
			if ((res->data[0] & 0x18) == 0x00)
			{
				width = 'b';
				printf("# %smemory is 8-bit only\n", indep);
			}
			if ((res->data[0] & 0x18) == 0x08)
				printf("# %smemory is 16-bit only\n", indep);
			if ((res->data[0] & 0x18) == 0x10)
				printf("# %smemory is 8-bit and 16-bit\n", indep);
			if (res->data[0] & 0x20)
				printf("# %smemory is shadowable\n", indep);
			if (res->data[0] & 0x40)
				printf("# %smemory is an expansion ROM\n", indep);
			printf("# %sMinimum memory base address 0x%06lx\n", indep, res->start);
			printf("# %sMaximum memory base address 0x%06lx\n", indep, res->end - 1);
			printf("# %sRange base alignment mask 0xff%04lx bytes\n", indep, res->step);
			printf("# %sRange length %ld bytes\n", indep, res->size);
#ifdef DUMPADDR
#if 0
			***untested ***
				for (i = ((res->data[2] << 16) + (res->data[1] << 8));
					 i <= ((res->data[4] << 16) + (res->data[3] << 8));
					 i += ((res->data[6] << 8) + res->data[5]))
			{
				printf("# %s%s0x%06x..0x%06x\n", indep, indep,
					 i, i + (res->data[8] << 16) + (res->data[7] << 8) - 1);
			}
#endif
#endif /* DUMPADDR */
			printf("# Choose UPPER = Range, or UPPER = Upper limit to suit hardware\n");
			printf("%s (MEM %d (BASE 0x%06lx) (MODE %cu) (UPPER 0x%06lx))\n",
				   comment_if_not_selected,
				   curmem, res->start, width, res->start + res->size);
			printf("# (MEM %d (BASE 0x%06lx) (MODE %cr) (UPPER 0x%06lx))\n", curmem, res->start, width, res->size);	/* XXX AJR */
			curmem++;
			if (!starteddeps)
				depmem = curmem;
			break;
		}
	case ANSIstr_TAG:
		{
			printf("# %sANSI string -->", indep);
			/* Remember the device name */
			if(devstring)
				free(devstring);
			devstring = (char *)malloc(len + 1);
			if(!devstring)
			{
				fprintf(stderr, "Out of memory to store board device string\n");
				exit(1);
			}
			for (i = 0; i < len; i++)
				devstring[i] = res->data[i];
			devstring[i] = 0;
			printf("%s<--\n", devstring);
			break;
		}
	case UNICODEstr_TAG:
		{
			printf("# %sUNICODE string -->", indep);
			/* Remember the device name */
			if(devstring)
				free(devstring);
			devstring = (char *)malloc(len + 1);
			if(!devstring)
			{
				fprintf(stderr, "Out of memory to store board device string\n");
				exit(1);
			}
			for (i = 0; i < len; i++)
				putchar(devstring[i] = res->data[i]);
			devstring[i] = 0;
			printf("<--\n");
			break;
		}
	case VendorShort_TAG:
	case VendorLong_TAG:
		{
			printf("# %sVendor defined tag: ", indep);
			printf(" %02x", res->tag);
			for (i = 0; i < len; i++)
				printf(" %02x", res->data[i]);
			putchar('\n');
			break;
		}
	case End_TAG:
		{
			char *indep = "";
			if (nestdepth)
			{
				/* If we have a device name, show it (the last description string before the END DF flag is it) */
				if (serialno == 0xffffffffUL)
					printf(" (NAME \"%s/-1[%d]", devid, logdevno-1);
				else
					printf(" (NAME \"%s/%lu[%d]", devid, serialno, logdevno-1);
				if(devstring)
					printf("{%-20s}", devstring);
				printf("\")\n");
				printf("%s (ACT Y)\n", comment_if_not_selected);
				while (nestdepth)
				{
					printf(")");
					nestdepth--;
				}
				printf("\n");
			}
			/* Cancel any device names */
			if(devstring)
				free(devstring);
			devstring = 0;
			printf("# %sEnd tag... Checksum 0x%02x (%s)\n\n", indep, csum, (csum % 256) ? "BAD" : "OK");
			break;
		}
	case Mem32Range_TAG:
	case FixedMem32Range_TAG:
		{
			if (len != 17)
			{
				lengtherror(len, "Mem32Range_TAG or FixedMem32Range_TAG");
				BREAKORCONTINUE;
			}
			printf("# %s32-bit MemRange tag %02x ...\n", indep, type);
			break;
		}
	default:
		{
			printf("# %sUnknown tag %02x ...\n", indep, type);
			BREAKORCONTINUE;
		}
	}
	return;
#ifdef ABORT_ONRESERR
  oncorruptresourcedata:
	printf("# Resource data dump aborted\n");
	exit(1);
#endif
}

#ifndef __DJGPP__
static void
print_script_header(FILE * output_file)
{
	fprintf(output_file,
			"#!/bin/sh\n"
			"prefixes=\"/etc/pnp/config-scripts/isa /usr/share/pnp/config-scripts/isa\"\n"
			"\n"
			"function pnp_isa_configure {\n"
			"    local i id dir found_it\n"
			"    i=0\n"
			"    while [ \"${dev_ids[$i]}\" != \"\" ] ; do\n"
			"        id=\"${dev_ids[$i]}\"\n"
			"        for dir in $prefixes ; do\n"
			"	    if [ -r $dir/$id ] ; then\n"
			"	        source $dir/$id\n"
			"		return $?\n"
			"	    fi\n"
			"	done\n"
			"	i=`expr $i + 1`\n"
			"    done\n"
			"    # Failed to find anything\n"
			"    return 1\n"
			"}\n"
	);
}

static void
print_script(FILE * output_file, struct resource *res, int selected)
{
	switch (res->type)
	{
	case NewBoard_PSEUDOTAG:
		{
			int csn = res->start;

			serialno = (unsigned long)
				serial_identifier[csn][4] +
				(serial_identifier[csn][5] << 8) +
				(serial_identifier[csn][6] << 16) +
				(serial_identifier[csn][7] << 24);
			strncpy(devid, devidstr(serial_identifier[csn][0],
									serial_identifier[csn][1],
									serial_identifier[csn][2],
									serial_identifier[csn][3]), 8);
		}
		break;
	case PnPVerNo_TAG:
		break;
	case LogDevId_TAG:
		{
			if (nestdepth)
			{
				fprintf(output_file, "pnp_isa_configure\n\n");
				nestdepth = 0;
			}
			fprintf(output_file,
			  "unset irq irq_flags dma dma_flags io_start io_len io_flags\n"
					"unset mem_start mem_len mem_flags dev_ids\n"
					"board_id=%s\n"
					"dev_ids[0]=%s\n",
					devid,
					devidstr(res->data[0], res->data[1],
							 res->data[2], res->data[3]));
			nestdepth = 2;
			curdma = 0;
			depdma = 0;
			curio = 0;
			depio = 0;
			curint = 0;
			depint = 0;
			curmem = 0;
			depmem = 0;
			curid = 2;
			starteddeps = 0;
			break;
		}
	case CompatDevId_TAG:
		{
			fprintf(output_file,
					"dev_ids[%d]=%s\n",
					curid,
					devidstr(res->data[0], res->data[1],
							 res->data[2], res->data[3]));
			curid++;
			break;
		}
	case IRQ_TAG:
		{
			fprintf(output_file,
					"irq[%d]=%ld ; irq_flags[%d]=0x%02x\n",
					curint, res->value, curint, res->data[2]);
			curint++;
			if (!starteddeps)
				depint = curint;
			break;
		}
	case DMA_TAG:
		{
			fprintf(output_file,
					"dma[%d]=%ld ; dma_flags[%d]=0x%02x\n",
					curdma, res->value, curdma, res->data[1]);
			curdma++;
			if (!starteddeps)
				depdma = curdma;
			break;
		}
	case StartDep_TAG:
		{
			starteddeps = 1;
			curio = depio;
			curdma = depdma;
			curint = depint;
			curmem = depmem;
			break;
		}
	case EndDep_TAG:
		{
			for_each_resource(res->alternatives[res->value].resources,
							  res->alternatives[res->value].len,
							  print_script,
							  output_file, 1);
			break;
		}
	case IOport_TAG:
		{
			fprintf(output_file,
					"io_start[%d]=0x%04lx ; "
					"io_len[%d]=%-2ld ; "
					"io_flags[%d]=0x%02x\n",
					curio, res->value,
					curio, res->size,
					curio, res->data[0]);
			curio++;
			if (!starteddeps)
				depio = curio;
			break;
		}
	case FixedIO_TAG:
		{
			fprintf(output_file,
					"io_start[%d]=0x%04lx ; io_len[%d]=%ld ; "
					"io_flags[%d]=FIXED_IO\n",
					curio, res->start, curio, res->size, curio);
			curio++;
			if (!starteddeps)
				depio = curio;
			break;
		}
	case MemRange_TAG:
		{
			fprintf(output_file,
					"mem_start[%d]=0x%04lx ; mem_len[%d]=%ld\n ; "
					"mem_flags[%d]=0x%04x\n",
					curmem, res->value, curmem, res->size,
					curmem, res->data[0]);
			curmem++;
			if (!starteddeps)
				depmem = curmem;
			break;
		}
	case ANSIstr_TAG:
		break;
	case UNICODEstr_TAG:
		break;
	case VendorShort_TAG:
	case VendorLong_TAG:
		break;
	case End_TAG:
		{
			if (nestdepth)
			{
				fprintf(output_file, "pnp_isa_configure\n\n");
				nestdepth = 0;
			}
			break;
		}
	case Mem32Range_TAG:
		{
			/* STUB: Need to add support for this. */
			break;
		}
	case FixedMem32Range_TAG:
		{
			/* STUB: Need to add support for this. */
			break;
		}
	default:
		{
			fprintf(output_file, "# %sUnknown tag %02x ...\n", indep, res->type);
			BREAKORCONTINUE;
		}
	}
	return;
#ifdef ABORT_ONRESERR
  oncorruptresourcedata:
	fprintf(output_file, "# Resource data dump aborted\n");
	exit(1);
#endif
}
#endif /* __DJGPP__ */


#if 0
void
dumpdata(int csn)
{
	int i;
	/* #define OLD_READ_STRATEGY */
#ifdef OLD_READ_STRATEGY
	struct resource res;
#endif /* OLD_READ_STRATEGY */
	unsigned long serialno = (unsigned long) serial_identifier[csn][4] + (serial_identifier[csn][5] << 8) +
	(serial_identifier[csn][6] << 16) + (serial_identifier[csn][7] << 24);
	struct resource *res_array;
	int array_len;
	int alloc_result;

	strncpy(devid, devidstr(serial_identifier[csn][0],
							serial_identifier[csn][1],
							serial_identifier[csn][2],
							serial_identifier[csn][3]), 8);

	nestdepth = 0;
	curdma = 0;
	depdma = 0;
	curio = 0;
	depio = 0;
	curint = 0;
	depint = 0;
	curmem = 0;
	depmem = 0;
	starteddeps = 0;

	Wake(csn);
	printf("# Card %d: (serial identifier", csn);
	for (i = IDENT_LEN; i--;)
		printf(" %02x", serial_identifier[csn][i]);
	printf(")\n");
	if (serialno == 0xffffffffUL)
	{
		printf("# Vendor Id %s, No Serial Number (-1), checksum 0x%02X.\n", devid, serial_identifier[csn][8]);
	}
	else
	{
		printf("# Vendor Id %s, Serial Number %lu, checksum 0x%02X.\n", devid, serialno, serial_identifier[csn][8]);
	}
	/*
	 * Check for broken cards that don't reset their resource pointer
	 * properly
	 */
	for (i = 0; i < TMP_LEN; i++)
		tmp[i] = 0;
	/* Get the first byte */
	if (read_resource_data(&tmp[0]) != 0)
		return;
	/* Just check the first byte, if these match we assume it's ok */
	if (tmp[0] != serial_identifier[csn][0])
	{
		printf("# Ident byte %d, (%02x) differs from resource data (%02x)\n", i, serial_identifier[csn][i], tmp[0]);
		printf("#Assuming the card is broken and this is the start of the resource data\n");
		unread_resource_data(tmp[0]);
		goto broken;			/* Wouldn't need this if people read the spec */
	}
	/*
	 * Read resource data past serial identifier - As it should be spec
	 * section 4.5 para 2
	 */
	for (i = 1; i < IDENT_LEN; i++)
	{
		/* Use tmp[0] as a temporary vbl */
		if (read_resource_data(&tmp[0]) != 0)
			return;
		/*
		 * Checksum byte is not guaranteed, but the previous bytes should
		 * match
		 */
		if ((tmp[0] != serial_identifier[csn][i]) && (i < (IDENT_LEN - 1)))
			printf("# Ident byte %d, (%02x) differs from resource data (%02x)\n", i, serial_identifier[csn][i], tmp[0]);
	}
	/* Now for the actual resource data */
	csum = 0;
  broken:
	res_array = NULL;
	array_len = 0;
	read_board_resources(NULL, 0, &res_array, &array_len);
	if (do_autoconfig)
	{
		alloc_result = alloc_resources(res_array, array_len, NULL, 0);
	}
	else
	{
		alloc_result = 0;
	}
	for_each_resource(res_array, array_len,
					  print_resource, stdout, alloc_result);
}
#endif

void
dumpregs(int csn)
{
	int logical_device;
	int i;
	int addr;
	int desc;
	unsigned long serialno = (unsigned long) serial_identifier[csn][4] + (serial_identifier[csn][5] << 8) +
	(serial_identifier[csn][6] << 16) + (serial_identifier[csn][7] << 24);
	strncpy(devid, devidstr(serial_identifier[csn][0],
							serial_identifier[csn][1],
							serial_identifier[csn][2],
							serial_identifier[csn][3]), 8);

	Wake(csn);
	printf("# Configuration registers for card %d: (serial identifier", csn);
	for (i = IDENT_LEN; i--;)
		printf(" %02x", serial_identifier[csn][i]);
	printf(")\n");
	if (serialno == 0xffffffffUL)
	{
		printf("# Vendor Id %s, No Serial Number (-1), checksum 0x%02X.\n", devid, serial_identifier[csn][8]);
	}
	else
	{
		printf("# Vendor Id %s, Serial Number %lu, checksum 0x%02X.\n", devid, serialno, serial_identifier[csn][8]);
	}
	ADDRESS(0x06);
	printf("#\n# Card Select Number register [0x06]: %d\n", READ_DATA);
	printf("#\n# Vendor defined card level registers 0x20..2f:");
	for (addr = 0x20; addr < 0x30; addr++)
	{
		ADDRESS(addr);
		printf(" %02x", READ_DATA);
	}
	putchar('\n');
	for (logical_device = 0; logical_device < 256; logical_device++)
	{
		LOGICALDEVICENUMBER;
		WRITE_DATA(logical_device);
		if (READ_DATA != logical_device)
			break;
		printf("#\n# Logical device %d\n", logical_device);
		ADDRESS(0x30);
		printf("# Activate       register [0x30]: %s\n", READ_DATA ? "* Activated *" : "Inactive");
		ADDRESS(0x31);
		printf("# IO range check register [0x31]: ");
		tmp[0] = READ_DATA;
		if(tmp[0] & 2)
			printf("IO range check active, reads return %02x", tmp[0] & 1 ? 0x55 : 0xAA);
		else
			printf("Inactive");
		printf("\n# Logical device Control reserved reg 0x32..37:");
		for (addr = 0x32; addr < 0x38; addr++)
		{
			ADDRESS(addr);
			printf(" %02x", READ_DATA);
		}
		printf("\n# Logical device Control Vendor defnd 0x38..40:");
		for (addr = 0x38; addr < 0x40; addr++)
		{
			ADDRESS(addr);
			printf(" %02x", READ_DATA);
		}
		putchar('\n');
		for (addr = 0x40, desc = 0; addr < 0x60; addr += 8, desc++)
		{
			printf("# 24 bit Memory descriptor %d at %02x..%02x: ", desc, addr, addr + 4);
			for (i = 0; i < 5; i++)
			{
				ADDRESS(addr + i);
				tmp[i] = READ_DATA;
			}
			printf("Base address 0x%02x%02x00 %s 0x%02x%02x00, %d bit\n", tmp[0], tmp[1],
				   tmp[2] & 1 ? "...." : "size",
				   tmp[3], tmp[4], tmp[2] & 2 ? 16 : 8);
		}
		for (addr = 0x76, desc = 0; addr < 0xb0; addr += 16, desc++)
		{
			printf("# 32 bit Memory descriptor %d at %02x..%02x: ", desc, addr, addr + 8);
			for (i = 0; i < 9; i++)
			{
				ADDRESS(addr + i);
				tmp[i] = READ_DATA;
			}
			printf("Base address 0x%02x%02x%02x%02x %s 0x%02x%02x%02x%02x, %d bit\n", tmp[0], tmp[1], tmp[2], tmp[3],
				   tmp[4] & 1 ? "...." : "size",
				   tmp[5], tmp[6], tmp[7], tmp[8], tmp[8] & 4 ? 32 : (tmp[8] & 2 ? 16 : 8));
			addr &= 0xf0;
		}
		for (addr = 0x60, desc = 0; addr < 0x70; addr += 2, desc++)
		{
			printf("# IO descriptor %d at %02x..%02x: ", desc, addr, addr + 1);
			for (i = 0; i < 2; i++)
			{
				ADDRESS(addr + i);
				tmp[i] = READ_DATA;
			}
			printf("Base address 0x%02x%02x\n", tmp[0], tmp[1]);
		}
		for (addr = 0x70, desc = 0; addr < 0x74; addr += 2, desc++)
		{
			printf("# Interrupt descriptor %d at %02x..%02x: ", desc, addr, addr + 1);
			for (i = 0; i < 2; i++)
			{
				ADDRESS(addr + i);
				tmp[i] = READ_DATA;
			}
			printf("Interrupt level %d, active %s, %s triggered\n", tmp[0], tmp[1] & 2 ? "high" : "low", tmp[1] & 1 ? "level" : "edge");
		}
		for (addr = 0x74; addr < 0x76; addr++)
		{
			printf("# DMA descriptor %d at %02x: ", addr - 0x74, addr);
			ADDRESS(addr);
			tmp[0] = READ_DATA;
			printf("DMA channel %d\n", tmp[0]);
		}
	}
	putchar('\n');
}

/* End of pnpdump.c */
