/*
C routines of the RTS

Copyright (C) 1991-99 Free Software Foundation, Inc.

Authors: Jukka Virtanen <jtv@hut.fi>
         Peter Gerwinski <peter@gerwinski.de>
         Frank Heckenbach <frank@pascal.gnu.de>

This file is part of the GNU Pascal Library. The GNU Pascal
Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

The GNU Pascal Library is distributed in the hope that it will
be useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; see the file COPYING.LIB.
If not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "rts.h"
#include <math.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

double       _p_sin               PROTO ((double));
double       _p_sinh              PROTO ((double));
double       _p_cos               PROTO ((double));
double       _p_cosh              PROTO ((double));
double       _p_arctan            PROTO ((double));
double       _p_arctan2           PROTO ((double, double));
double       _p_sqrt              PROTO ((double));
double       _p_ln                PROTO ((double));
double       _p_exp               PROTO ((double));
double       _p_expon             PROTO ((double, double));
long double  _pp_sin              PROTO ((long double));
long double  _pp_cos              PROTO ((long double));
long double  _pp_arctan           PROTO ((long double));
long double  _pp_sqrt             PROTO ((long double));
long double  _pp_ln               PROTO ((long double));
long double  _pp_exp              PROTO ((long double));
long double  _pp_expon            PROTO ((long double, long double));

void         _p_sleep             PROTO ((int));
void         _p_unix_time_to_time PROTO ((UnixTimeType, int *, int *, int *, int *, int *, int *));
UnixTimeType _p_time_to_unix_time PROTO ((int, int, int, int, int, int));
UnixTimeType _p_get_unix_time     PROTO ((int *));
void         _p_init_time         PROTO ((void));
int          _p_get_cpu_time      PROTO ((int *));

Boolean      _p_WIfExited         PROTO ((int));
int          _p_WExitStatus       PROTO ((int));
Boolean      _p_WIfSignaled       PROTO ((int));
int          _p_WTermSig          PROTO ((int));
Boolean      _p_WIfStopped        PROTO ((int));
int          _p_WStopSig          PROTO ((int));
int          _p_csystem           PROTO ((char *));
void         _p_csetenv           PROTO ((char *, char *, char *, Boolean));
void         _p_cunsetenv         PROTO ((char *, char*));
char        *_p_c_strsignal       PROTO ((int));
Boolean      _p_kill              PROTO ((int, int));
int          _p_waitpid           PROTO ((int, int *, Boolean));
Boolean      _p_setpgid           PROTO ((int, int));
Boolean      _p_tcsetpgrp         PROTO ((int, int));
int          _p_tcgetpgrp         PROTO ((int));
Boolean      _p_cgetpwnam         PROTO ((char *, TCPasswordEntry *));
Boolean      _p_cgetpwuid         PROTO ((int, TCPasswordEntry *));
int          _p_cgetpwent         PROTO ((TCPasswordEntry **));

int _p_SIGHUP  = SIGHUP,  _p_SIGINT    = SIGINT,    _p_SIGQUIT   = SIGQUIT,
    _p_SIGILL  = SIGILL,  _p_SIGABRT   = SIGABRT,   _p_SIGFPE    = SIGFPE,
    _p_SIGKILL = SIGKILL, _p_SIGSEGV   = SIGSEGV,   _p_SIGPIPE   = SIGPIPE,
    _p_SIGALRM = SIGALRM, _p_SIGTERM   = SIGTERM,   _p_SIGUSR1   = SIGUSR1,
    _p_SIGUSR2 = SIGUSR2, _p_SIGCHLD   = SIGCHLD,   _p_SIGCONT   = SIGCONT,
    _p_SIGSTOP = SIGSTOP, _p_SIGTSTP   = SIGTSTP,   _p_SIGTTIN   = SIGTTIN,
    _p_SIGTTOU = SIGTTOU, _p_SIGTRAP   = SIGTRAP,   _p_SIGEMT    = SIGEMT,
    _p_SIGBUS  = SIGBUS,  _p_SIGSYS    = SIGSYS,    _p_SIGSTKFLT = SIGSTKFLT,
    _p_SIGURG  = SIGURG,  _p_SIGIO     = SIGIO,     _p_SIGXCPU   = SIGXCPU,
    _p_SIGXFSZ = SIGXFSZ, _p_SIGVTALRM = SIGVTALRM, _p_SIGPROF   = SIGPROF,
    _p_SIGPWR  = SIGPWR,  _p_SIGLOST   = SIGLOST,   _p_SIGWINCH  = SIGWINCH;

static void
check_errno ()
{
  switch (errno)
    {
      case 0      : break;
      case ERANGE : _p_error (700); /* Overflow in exponentiation */
      case EDOM   : _p_error (701); /* In `x pow y', x must be >= 0 if y < 0 and y is not an integer */
      default     : _p_internal_error_integer (922, (long int) errno); /* undocumented errno value %d in exponentiation */
    }
}

double
_p_sin (x)
double x;
{
  return sin (x);
}

double
_p_sinh (x)
double x;
{
  return sinh (x);
}

double
_p_cos (x)
double x;
{
  return cos (x);
}

double
_p_cosh (x)
double x;
{
  return cosh (x);
}

double
_p_arctan (x)
double x;
{
  return atan (x);
}

double
_p_arctan2 (y, x)
double y, x;
{
  return atan2 (y, x);
}

double
_p_sqrt (x)
double x;
{
  if (x < 0.0) _p_error (708); /* argument of `Sqrt' is < 0 */
  return sqrt (x);
}

double
_p_ln (x)
double x;
{
  if (x <= 0.0) _p_error (707); /* argument of `Ln' is <= 0 */
  return log (x);
}

double
_p_exp (x)
double x;
{
  double ret;
  errno = 0;
  ret = exp (x);
  check_errno ();
  return ret;
}

double
_p_expon (x, y)
double x, y;
{
  double rval;
  errno = 0;
  rval = pow (x, y);
  check_errno ();
  return rval;
}

#ifndef HAVE_SINL
#define sinl sin
#endif

#ifndef HAVE_COSL
#define cosl cos
#endif

#ifndef HAVE_SQRTL
#define sqrtl sqrt
#endif

#ifndef HAVE_LOGL
#define logl log
#endif

#ifndef HAVE_EXPL
#define expl exp
#endif

#ifndef HAVE_POWL
#define powl pow
#endif

#ifndef HAVE_ATANL
#define atanl atan
#endif

long double
_pp_sin (x)
long double x;
{
  return sinl (x);
}

long double
_pp_cos (x)
long double x;
{
  return cosl (x);
}

long double
_pp_arctan (x)
long double x;
{
  return atanl (x);
}

long double
_pp_sqrt (x)
long double x;
{
  if (x < 0.0) _p_error (708); /* argument of `Sqrt' is < 0 */
  return sqrtl (x);
}

long double
_pp_ln (x)
long double x;
{
  if (x <= 0.0) _p_error (707); /* argument of `Ln' is <= 0 */
  return logl (x);
}

long double
_pp_exp (x)
long double x;
{
  long double ret;
  errno = 0;
  ret = expl (x);
  check_errno ();
  return ret;
}

long double
_pp_expon (x, y)
long double x, y;
{
  long double rval;
  errno = 0;
  rval = powl (x, y);
  check_errno ();
  return rval;
}

void
_p_sleep (seconds)
int seconds;
{
  int i = seconds;
  do
    i = sleep (i);
  while (i != 0);
}

void
_p_unix_time_to_time (Time, Year, Month, Day, Hour, Minute, Second)
UnixTimeType Time;
int *Year, *Month, *Day, *Hour, *Minute, *Second;
{
  time_t seconds = (time_t) Time;
  struct tm *gnu = localtime (&seconds);
  *Year   = gnu->tm_year;
  *Month  = gnu->tm_mon + 1; /* 1 = January */
  *Day    = gnu->tm_mday;
  *Hour   = gnu->tm_hour;
  *Minute = gnu->tm_min;
  *Second = gnu->tm_sec;
  if (*Year < 1000) *Year += 1900;
}

UnixTimeType
_p_time_to_unix_time (Year, Month, Day, Hour, Minute, Second)
int Year, Month, Day, Hour, Minute, Second;
{
  struct tm gnu; /* It is a trademark after all }:--)= */
  memset (&gnu, 0, sizeof (gnu));
  gnu.tm_isdst = -1;
  gnu.tm_year  = Year - 1900;
  gnu.tm_mon   = Month - 1; /* 1 = January */
  gnu.tm_mday  = Day;
  gnu.tm_hour  = Hour;
  gnu.tm_min   = Minute;
  gnu.tm_sec   = Second;
  return (UnixTimeType) mktime (&gnu);
}

UnixTimeType
_p_get_unix_time (MicroSecond)
int *MicroSecond;
{
#ifdef HAVE_GETTIMEOFDAY
  struct timeval tval;
  if (!gettimeofday (&tval, 0))
    {
      if (MicroSecond) *MicroSecond = tval.tv_usec;
      return tval.tv_sec;
    }
#elif defined(HAVE_TIME)
  if (MicroSecond) *MicroSecond = 0;
  return (UnixTimeType) time ((time_t *) 0);
#endif
  if (MicroSecond) *MicroSecond = 0;
  return - 1;
}

#ifdef HAVE_GETRUSAGE
int
_p_get_cpu_time (MicroSecond)
int *MicroSecond;
{
  int ms;
  struct rusage u;
  getrusage (RUSAGE_SELF, &u);
  ms = u.ru_utime.tv_usec + u.ru_stime.tv_usec;
  if (MicroSecond) *MicroSecond = ms % 1000000;
  return (int) u.ru_utime.tv_sec + u.ru_stime.tv_sec + ms / 1000000;
}

void
_p_init_time ()
{
}
#else
static clock_t initial_clocks = 0;

int
_p_get_cpu_time (MicroSecond)
int *MicroSecond;
{
  clock_t clocks = clock () - initial_clocks;
  if (MicroSecond) *MicroSecond = ((int) ((((long long int) clocks) * 1000000) / CLOCKS_PER_SEC)) % 1000000;
  return (int) (clocks / CLOCKS_PER_SEC);
}

void
_p_init_time ()
{
  initial_clocks = clock ();
}
#endif

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifndef WIFEXITED
#define WIFEXITED(S) (((S) & 0xff) == 0)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(S) (((S) & 0xff00) >> 8)
#endif
#ifndef WIFSIGNALED
#define WIFSIGNALED(S) (((S) & 0xff) != 0 && ((S) & 0xff) != 0x7f)
#endif
#ifndef WTERMSIG
#define WTERMSIG(S) ((S) & 0x7f)
#endif
#ifndef WIFSTOPPED
#define WIFSTOPPED(S) (((S) & 0xff) == 0x7f)
#endif
#ifndef WSTOPSIG
#define WSTOPSIG(S) (((S) & 0xff00) >> 8)
#endif

Boolean
_p_WIfExited (Status)
int Status;
{
  return WIFEXITED (Status) ? TRUE : FALSE;
}

int
_p_WExitStatus (Status)
int Status;
{
  return WEXITSTATUS (Status);
}

Boolean
_p_WIfSignaled (Status)
int Status;
{
  return WIFSIGNALED (Status) ? TRUE : FALSE;
}

int
_p_WTermSig (Status)
int Status;
{
  return WTERMSIG (Status);
}

Boolean
_p_WIfStopped (Status)
int Status;
{
  (void) Status;
  return WIFSTOPPED (Status) ? TRUE : FALSE;
}

int
_p_WStopSig (Status)
int Status;
{
  (void) Status;
  return WSTOPSIG (Status);
}

int
_p_csystem (CmdLine)
char *CmdLine;
{
  return system (CmdLine);
}

/* The environment handling of the different libcs is a whole big mess:
   - The environment passed to main() may or may not survive putenv() calls
     (e.g. not under DJGPP).
   - There may or may not be an environ or __environ variable containing
     the current environment.
   - If it exists, assignments to it can cause segfaults (e.g. glibc).
   - The putenv() function (POSIX) expects a static string, so the caller
     has to make a copy, but libc does not free it when the variable is
     unset or overwritten.
   - OTOH, there's no guarantee if and when the caller may free the string
     after it's not needed any more.
   - Functions like execl() or system() access the (internal) current
     environment. Furthermore, system() is quite system specific (esp.
     on Unix vs. Dos systems), so it can't be easily reprogrammed using
     another environment.
   This function tries to make the best out of the situation, so don't
   complain that it's a whole big mess as well. ;*/
void _p_csetenv (char *VarName, char *Value, char *NewEnvCString, Boolean UnSet)
{
  (void) VarName;
  (void) Value;
  (void) NewEnvCString;
  (void) UnSet;
#ifdef HAVE_UNSETENV
  if (UnSet)
    unsetenv (VarName);
  else
#endif
#ifdef HAVE_SETENV
    setenv (VarName, Value, 1);
#elif defined(HAVE_PUTENV)
    {
#ifdef HAVE_GETENV
      char *OldValue = getenv (VarName);
      if ((!OldValue && !UnSet) || _p_strcmp (Value, OldValue) != 0)
#endif
        putenv (_p_strdup (NewEnvCString));
    }
#endif
}

#ifdef HAVE_STRSIGNAL
extern char *strsignal PROTO ((int));
#endif

char *
_p_c_strsignal (signal)
int signal;
{
  #ifdef HAVE_STRSIGNAL
  return strsignal (signal);
  #else
  static char buf [100];
  sprintf (buf, "signal #%i", signal);
  return buf;
  #endif
}

Boolean
_p_kill (PID, Signal)
int PID; int Signal;
{
  return (kill (PID, Signal) == 0) ? TRUE : FALSE;
}

int
_p_waitpid (PID, WStatus, Block)
int PID; int *WStatus; Boolean Block;
{
  int wpid;
  #ifdef HAVE_WAITPID
  do
    wpid = waitpid (PID, WStatus, Block ? 0 : WNOHANG);
  while
    #ifdef EINTR
    (wpid < 0 && errno == EINTR);
    #else
    (0);
    #endif
  #else
  wpid = - 1;
  #endif
  if (wpid <= 0) *WStatus = 0;
  return wpid;
}

Boolean
_p_setpgid (Process, ProcessGroup)
int Process; int ProcessGroup;
{
  #ifdef HAVE_SETPGID
  return (setpgid (Process, ProcessGroup) == 0) ? TRUE : FALSE;
  #else
  return FALSE;
  #endif
}

Boolean
_p_tcsetpgrp (File, ProcessGroup)
int File; int ProcessGroup;
{
  #ifdef HAVE_TCSETPGRP
  return (tcsetpgrp (File, ProcessGroup) == 0) ? TRUE : FALSE;
  #else
  (void) File;
  (void) ProcessGroup;
  return FALSE;
  #endif
}

int
_p_tcgetpgrp (File)
int File;
{
  #ifdef HAVE_TCGETPGRP
  return tcgetpgrp (File);
  #else
  (void) File;
  return -1;
  #endif
}

static void
_p_pwl2c (const struct passwd *p, TCPasswordEntry *Entry)
{
  Entry -> UserName = _p_strdup (p -> pw_name);
  Entry -> RealName =
    #ifdef HAVE_PW_GECOS
    _p_strdup (p -> pw_gecos);
    #else
    NULL;
    #endif
  Entry -> Password =
    #ifdef HAVE_PW_PASSWD
    _p_strdup (p -> pw_passwd);
    #else
    NULL;
    #endif
  Entry -> HomeDirectory = _p_strdup (p -> pw_dir);
  Entry -> Shell = _p_strdup (p -> pw_shell);
  Entry -> UID = p -> pw_uid;
  Entry -> GID = p -> pw_gid;
}

Boolean
_p_cgetpwnam (char *UserName, TCPasswordEntry *Entry)
{
  #ifdef HAVE_GETPWNAM
  struct passwd *p = getpwnam (UserName);
  if (!p) return FALSE;
  _p_pwl2c (p, Entry);
  return TRUE;
  #else
  return FALSE;
  #endif
}

Boolean
_p_cgetpwuid (int UID, TCPasswordEntry *Entry)
{
  #ifdef HAVE_GETPWUID
  struct passwd *p = getpwuid (UID);
  if (!p) return FALSE;
  _p_pwl2c (p, Entry);
  return TRUE;
  #else
  return FALSE;
  #endif
}

int
_p_cgetpwent (TCPasswordEntry **Entries)
{
  int Count = 0;
  #if defined(HAVE_GETPWENT) && defined(HAVE_SETPWENT) && defined(HAVE_ENDPWENT)
  struct passwd *p;
  setpwent ();
  *Entries = NULL;
  do
    {
      p = getpwent ();
      if (!p) break;
      _p_gpc_realloc ((void **) Entries, (++Count) * sizeof (TCPasswordEntry));
      _p_pwl2c (p, &((*Entries) [Count - 1]));
    }
  while (1);
  endpwent ();
  #endif
  return Count;
}
