/*
Routines to output various things.

Copyright (C) 1991-99 Free Software Foundation, Inc.

Authors: Jukka Virtanen <jtv@hut.fi>
         Peter Gerwinski <peter@gerwinski.de>
         Frank Heckenbach <frank@pascal.gnu.de>

This file is part of the GNU Pascal Library. The GNU Pascal
Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

The GNU Pascal Library is distributed in the hope that it will
be useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; see the file COPYING.LIB.
If not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "rts.h"
#include "fdr.h"

void
_p_internal_fwrite (ptr, size, presult, File)
    const void *ptr; size_t size; size_t *presult; FDR File;
{
  size_t result;
  if (File->WriteFunc)
    {
      result = File->WriteFunc ((File->WriteFunc == _p_f_write) ? m_FILNUM (File) : File->PrivateData, ptr, size);
      if (_p_inoutres && !_p_inoutres_str) IOERROR_FILE (_p_inoutres, File,);
    }
  else
    {
      result = 0;
      IOERROR_FILE (466, File,); /* error when writing to `%s' */
    }
  if (presult)
    *presult = result;
  else
    if (!_p_inoutres && result != size)
      IOERROR_FILE (467, File,); /* could not write all the data to `%s' */
}

static void
_p_write_to_buf (File, ptr, size)
FDR File; const char *ptr; size_t size;
{
  size_t a;
  if (_p_inoutres) return;
  a = File->BufSize - File->BufPos;
  if (size < a) a = size;
  if (a > 0)
    {
      _p_move (ptr, File->BufPtr + File->BufPos, a);
      File->BufPos += a;
      ptr += a;
      size -= a;
    }
  if (size == 0) return;
  if (File->Flags & READ_WRITE_STRING_MASK)
    {
      if (File->Flags & TRUNCATE_STRING_MASK)
        return;
      else
        IOERROR (582,); /* string capacity exceeded in `WriteStr' */
    }
  _p_internal_fwrite (File->BufPtr, File->BufPos, NULL, File);
  if (size <= File->BufSize)
    {
      _p_move (ptr, File->BufPtr, size);
      File->BufPos = size;
    }
  else
    {
      _p_internal_fwrite (ptr, size, NULL, File);
      File->BufPos = 0;
    }
}

void
_p_write_flush (File)
FDR File;
{
  if (_p_inoutres) return;
  if (File->BufPos != 0)
    _p_internal_fwrite (File->BufPtr, File->BufPos, NULL, File);
  _p_clearbuffer (File);
  if (tst_FLUSH(File))
    _p_flush(File);
}

/* pad with spaces */
#define PADSIZE 32
static char const blanks[PADSIZE] =
{' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
 ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' '};

static void
_p_write_pad (File,count)
FDR File; int count;
{
  register int i;
  for (i = count; i > 0 ; i -= PADSIZE)
    _p_write_to_buf (File, blanks, (i >= PADSIZE) ? PADSIZE : i);
}

static void
_p_write_padded (File, buf, length, width, clip)
FDR File; char *buf; int length, width, clip;
{
  int pad_left = 0, pad_right = 0;
  if (width != _p_low_integer)
    {
      int abs_width, pad;
      abs_width = (width >= 0) ? width : - width;
      if (length > abs_width)
        {
          pad = 0;
          if (clip) length = abs_width;
        }
      else
        pad = abs_width - length;
      if (width >= 0)
        pad_left = pad;
      else
        {
          switch (File->Flags & NEG_WIDTH_MASKS)
            {
              case NEG_WIDTH_ERROR_MASK:  IOERROR (580,); /* fixed field width cannot be negative */
              case NEG_WIDTH_LEFT_MASK:   pad_right = pad;
                                          break;
              case NEG_WIDTH_CENTER_MASK: pad_left = pad / 2;
                                          pad_right = pad - pad_left;
            }
        }
    }
  _p_write_pad (File, pad_left);
  _p_write_to_buf (File, buf, length);
  _p_write_pad (File, pad_right);
}

/* Sufficient width to hold a signed long long in decimal representation */
#define MAX_LONG_WIDTH (sizeof(long long) * 64/*BITS_PER_UNIT*/ / 3 + 2)

#define DEFWRITEINT(fnname,type,conv_fn)            \
void                                                \
fnname (File, num, width)                           \
FDR File; type num; int width;                      \
{                                                   \
  char buf[MAX_LONG_WIDTH], *buf_begin;             \
  int negative = num < 0;                           \
  if (negative) num = - num;                        \
  buf_begin = conv_fn (num, buf + MAX_LONG_WIDTH);  \
  if (negative) *(--buf_begin) = '-';               \
  _p_write_padded (File, buf_begin, buf + MAX_LONG_WIDTH - buf_begin, width, 0); \
}
DEFWRITEINT  (_p_write_integer,  signed int,         _p_card_to_decimal)
DEFWRITEINT  (_p_write_longint,  signed long long,   _p_longcard_to_decimal)

#define DEFWRITEUINT(fnname,type,conv_fn)           \
void                                                \
fnname (File, num, width)                           \
FDR File; type num; int width;                      \
{                                                   \
  char buf[MAX_LONG_WIDTH], *buf_begin;             \
  buf_begin = conv_fn (num, buf + MAX_LONG_WIDTH);  \
  _p_write_padded (File, buf_begin, buf + MAX_LONG_WIDTH - buf_begin, width, 0); \
}
DEFWRITEUINT (_p_write_cardinal, unsigned int,       _p_card_to_decimal)
DEFWRITEUINT (_p_write_longcard, unsigned long long, _p_longcard_to_decimal)

void
_p_write_real (File, num, width, prec)
FDR File; long double num; int width, prec;
{
  char *buf;
  int buf_size;
  if (prec < 0 && prec != _p_low_integer)
    IOERROR (581,); /* fixed real fraction field width cannot be negative */
  buf = _p_longreal_to_decimal (num, width, prec,
        width != _p_low_integer,
        (File->Flags & REAL_NOBLANK_MASK) == 0,
        (File->Flags & REAL_CAPITAL_EXP_MASK) != 0, &buf_size);
  _p_write_padded (File, buf, _p_strlen(buf), width, 0);
  if (buf_size)
    _p_dispose (buf);
}

void
_p_write_char (File, ch, width)
FDR File; char ch; int width;
{
  _p_write_padded (File, &ch, sizeof (ch), width, 0);
}

void
_p_write_boolean (File, b, width)
FDR File; int b; int width;
{
  const char *str_val = b ? TRUE_str : FALSE_str;
  _p_write_padded (File, str_val, _p_strlen (str_val), width, 1);
}

void
_p_write_string (File, s, length, width)
FDR File; char *s; int length; int width;
{
  if (s == NULL)
    length = 0;
  else if (length < 0)  /* CString */
    length = strlen (s);
  _p_write_padded (File, s, length, width, File->Flags & CLIP_STRING_MASK);
}

void
_p_writeln (File)
FDR File;
{
  char newline = NEWLINE;
  _p_write_to_buf (File, &newline, sizeof (newline));
}

void
_p_write_init (File, Flags)
FDR File;
int Flags;
{
  _p_ok_WRITE (File);
  File->BufSize = FILE_BUFSIZE;
  File->BufPos = 0;
  File->Flags = Flags;
}

void
_p_writestr_init (File, s, Capacity, Flags)
FDR File;
char *s;
int Capacity, Flags;
{
  File->BufPtr = s;
  File->BufSize = Capacity;
  File->BufPos = 0;
  File->Flags = Flags;
}

int
_p_writestr_getlength (File)
FDR File;
{
  return File->BufPos;
}
