/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_browse.c,v 1.4 1999/11/11 04:47:33 arodrig6 Exp $
 * 
 *	Function:	- program browser panel in builder dialog
 */

#define _NO_PROTO

#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/TextF.h>

#include <stdlib.h>
#include <dirent.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "all_list.h"
#include "lam.h"
#include "xmpi.h"

/*
 * global functions
 */
void			xmpi_browse_build();
void			xmpi_browse_fill();

/*
 * external functions
 */
extern char		*getworkdir();

/*
 * local functions
 */
static void		browse_dir_cb();
static void		browse_select_cb();
static int		fnamecmp();

/*
 * local variables
 */
static Widget		list_w = 0;	/* directory contents list */
static Widget		text_w = 0;	/* current directory text field */

/*
 *	xmpi_browse_build
 *
 *	Function:	- builds file browse panel
 *	Accepts:	- parent widget
 */
void
xmpi_browse_build(parent_w)

Widget			parent_w;

{
	Widget		mgr_w;
	char		*cwd;
	XmString	xstr;

	mgr_w = XtVaCreateWidget("browse_mgr",
		xmPanedWindowWidgetClass, parent_w,
		XmNsashWidth, 1,
		XmNsashHeight, 1,
		XmNseparatorOn, False,
		XmNleftAttachment, XmATTACH_POSITION,
		XmNleftPosition, 1,
		XmNleftOffset, 5,
		XmNrightAttachment, XmATTACH_POSITION,
		XmNrightPosition, 2,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);

	xstr = XmStringCreateSimple("Browse Programs");
	XtVaCreateManagedWidget("banner",
				xmLabelWidgetClass, mgr_w,
				XmNlabelString, xstr,
				NULL);
	XmStringFree(xstr);

	text_w = XtVaCreateManagedWidget("browse_text",
		xmTextFieldWidgetClass, mgr_w,
		NULL);
	XtAddCallback(text_w, XmNactivateCallback, browse_dir_cb, NULL);

	list_w = XmCreateScrolledList(mgr_w, "browse_list", NULL, 0);
	XtVaSetValues(list_w,
		XmNscrollBarDisplayPolicy, XmSTATIC,
		XmNscrollHorizontal, False,
		XmNselectionPolicy, XmBROWSE_SELECT,
		NULL);
	XtAddCallback(list_w, XmNdefaultActionCallback,
			browse_select_cb, NULL);
	XtAddCallback(list_w, XmNbrowseSelectionCallback,
			browse_select_cb, NULL);
	XtManageChild(list_w);
/*
 * Initialize the text field to the current directory.
 * Fill the list with the current directory contents.
 */
	if ((cwd = getworkdir())) {
		XmTextFieldSetString(text_w, cwd);
		XmTextFieldSetInsertionPosition(text_w, strlen(cwd));
		xmpi_run_set_prog("", cwd);
		free(cwd);
		xmpi_browse_fill(list_w);
	}

	xmpi_nosash(mgr_w);
	XtManageChild(mgr_w);
}

/*
 *	browse_dir_cb
 *
 *	Function:	- take action on the selected directory
 *	Accepts:	- widget
 *			- client data
 *			- callback ptr
 */
static void
browse_dir_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmAnyCallbackStruct	*p_callback;

{
	char		*dirpath;

	dirpath = XmTextFieldGetString(w);

	if ((dirpath == 0) || (*dirpath == '\0')) {
		XtFree(dirpath);
		return;
	}
/*
 * Change to new directory.
 */
	if (chdir(dirpath)) {
		xmpi_error(w, dirpath);
		XtFree(dirpath);
		return;
	}

	xmpi_run_set_prog("", dirpath);
	xmpi_browse_fill(list_w);
	XtFree(dirpath);
}

/*
 *	browse_select_cb
 *
 *	Function:	- selects a program or new directory
 *	Accepts:	- widget
 *			- client data
 *			- callback ptr
 */
static void
browse_select_cb(w, cdata, cbs)

Widget			w;
XtPointer		cdata;
XmListCallbackStruct	*cbs;

{
	char		*filename;
	char		*p;
	struct stat	fileinfo;

	XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &filename);

	if (lstat(filename, &fileinfo)) {
		xmpi_error(w, filename);
		XtFree(filename);
		return;
	}
/*
 * If the file is a sym-link, stat() the real file.
 */
	if (S_ISLNK(fileinfo.st_mode)) {
		if (stat(filename, &fileinfo)) {
			xmpi_error(w, filename);
			XtFree(filename);
			return;
		}
	}
/*
 * If the file is a directory, change to it.
 */
	if (S_ISDIR(fileinfo.st_mode) &&
			(cbs->reason == (int) XmCR_DEFAULT_ACTION)) {

		if (chdir(filename)) {
			xmpi_error(w, filename);
			XtFree(filename);
			return;
		}

		if ((p = getworkdir()) == 0) {
			xmpi_error(w, "Cannot get current directory");
			XtFree(filename);
			return;
		}

		XmTextFieldSetString(text_w, p);
		XmTextFieldSetInsertionPosition(text_w, strlen(p));
		xmpi_run_set_prog("", p);
		xmpi_browse_fill(list_w);
		free(p);
	}
/*
 * Write the selected file into the program text area.
 */
	else if (!S_ISDIR(fileinfo.st_mode)) {

		if ((p = getworkdir()) == 0) {
			xmpi_error(w, "Cannot get current directory");
			XtFree(filename);
			return;
		}

		xmpi_run_set_prog(filename, p);
		free(p);
	}

	XtFree(filename);
}

/*
 *	xmpi_browse_fill
 *
 *	Function:	- fills a list with the contents of the
 *			  current directory
 *	Accepts:	- list widget
 */
void
xmpi_browse_fill(w)

Widget			w;

{
	DIR		*pdir;		/* opened directory */
	struct dirent	*pfile;		/* directory entry ptr */
	XmString	xstr;		/* Motif string */
	struct stat	fileinfo;	/* file status info */
	LIST		*sortlist;	/* list of sorted filenames */
	int		fnamelen;	/* filename length */
	char		*fname;		/* filename */
	char		**p;		/* favourite pointer */
/*
 * Delete all current list entries.
 */
	XmListDeselectAllItems(w);
	XmListDeleteAllItems(w);
/*
 * Open the directory.
 */
	if ((pdir = opendir(".")) == 0) xmpi_fail("xmpi (opendir)");
/*
 * Initialize the list used for sorting.
 */
	sortlist = al_init((int4) sizeof(char *), fnamecmp);

	if (sortlist == 0) xmpi_fail("xmpi (al_init)");
/*
 * Fill the list with the directory's files.
 */
	for (pfile = readdir(pdir); pfile; pfile = readdir(pdir)) {
/*
 * Insert the file in the sorted list.
 * Add a path delimiter if the file is a directory.
 */
		if (lstat(pfile->d_name, &fileinfo)) {
			xmpi_error(w, pfile->d_name);
			continue;
		}
/*
 * If a sym-link, stat() the real file.
 * Ignore any error and accept the link anyway.
 */
		if (S_ISLNK(fileinfo.st_mode)) {
			stat(pfile->d_name, &fileinfo);
		}

		fnamelen = strlen(pfile->d_name);
		if (S_ISDIR(fileinfo.st_mode)) {
			++fnamelen;
		}

		fname = malloc((unsigned) fnamelen + 1);

		if (fname == 0) xmpi_fail("xmpi (malloc)");

		strcpy(fname, pfile->d_name);
		if (S_ISDIR(fileinfo.st_mode)) {
			strcat(fname, STRSDIR);
		}

		if (! al_insert(sortlist, (char *) &fname))
			xmpi_fail("xmpi (al_insert)");
	}
/*
 * Loop adding the sorted files to the scrolled list.
 */
	p = (char **) al_top(sortlist);
	while (p) {
		xstr = XmStringCreateSimple(*p);
		XmListAddItem(w, xstr, 0);
		XmStringFree(xstr);
		free(*p);
		p = (char **) al_next(sortlist, (char *) p);
	}

	al_free(sortlist);
	closedir(pdir);
}

/*
 *	fnamecmp
 *
 *	Function:	- compare two filename entries in the sorted list
 *			- special cases: "./" is first, "../" is second
 *	Accepts:	- ptr to the two filenames
 *	Returns:	- -ve/0/+ve (the usual <, ==, > comparison)
 */
static int
fnamecmp(f1, f2)

char			**f1, **f2;

{
	int		ret;

	if  (strcmp(*f1, "./") == 0) {
		ret = -1;
	} else if (strcmp(*f2, "./") == 0) {
		ret = 1;
	} else if  (strcmp(*f1, "../") == 0) {
		ret = -1;
	} else if  (strcmp(*f2, "../") == 0) {
		ret = 1;
	} else {
		ret = strcmp(*f1, *f2);
	}

	return(ret);
}
