/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *      $Id: xmpi_dtype.c,v 1.5 1999/11/10 21:32:33 arodrig6 Exp $
 * 
 *	Function:	- Datatype subwindow
 */

#define _NO_PROTO
#define XMPI_NORANK	(-1)			/* indicates no process */

#include <Xm/Text.h>
#include <Xm/Form.h>
#include <Xm/DialogS.h>

#include <stdio.h>
#include <stdlib.h>

#include "mpitrace.h"
#include "xmpi.h"
#include "xmpi_dbase.h"

#include "Bitmaps/blank.xbm"
#include "Bitmaps/light.xbm"
#include "Bitmaps/tack.xbm"
#include "Bitmaps/mesg.xbm"

/*
 * public functions
 */
void			xmpi_dt_create();
void			xmpi_dt_update();
void			xmpi_dt_clear();
void			xmpi_dt_msg();
void			xmpi_dt_proc();
void			xmpi_dt_setrank();
void			xmpi_dt_popdown();
void			xmpi_dt_busy();
void			xmpi_dt_unbusy();

/*
 * external functions
 */
extern void		*xmpi_sys_dtype();
extern void		xmpi_dtypeprint();

/*
 * private functions
 */
static void		createpixlabels();
static void		tack_cb();
static void		print_type();

/*
 * external variables
 */
extern int		fl_dbase;		/* driven by dbase flag */
extern struct _gps	*xmpi_app_procs;	/* application GPS array */

/*
 * local variables
 */
static int		dt_proc = -1;		/* process rank */
static int		dt_dtype = -1;		/* datatype label */
static char		fmtbuf[2048];		/* large buffer */
static Pixmap		blankpix = XmUNSPECIFIED_PIXMAP;
static Pixmap		trafficpix = XmUNSPECIFIED_PIXMAP;
static Pixmap		mesgpix = XmUNSPECIFIED_PIXMAP;
static Pixmap		tackpix = XmUNSPECIFIED_PIXMAP;
static Pixmap		armtackpix = XmUNSPECIFIED_PIXMAP;
static Widget		dtw = 0;
static Widget		dtw_form;
static Widget		dtw_dtype;
static Widget		dtw_hdr;
static Widget		dtw_pix;
static int		dtw_rank = XMPI_NORANK;
static int		dtw_show = XMPI_SHOWNONE;

/*
 *	xmpi_dt_msg
 *
 *	Function:	- set the message datatype
 *	Accepts:	- process global rank
 *			- focus entry
 *			- ignore cache flag (1 => ignore cache)
 */
void
xmpi_dt_msg(rank, pfocus, nocache)

int			rank;
struct xmfocus		*pfocus;
int			nocache;

{
	struct xmmsg	*pmsg;			/* message entry */

	if (dtw_show != XMPI_SHOWMESG || dtw_rank != rank) return;
/*
 * If no messages, blank the datatype.
 */
	pmsg = pfocus->xmf_proc->xmp_curmsg;

	if (pmsg == 0 || pmsg->xmm_dtype == 0) {
		dt_dtype = -1;
		XmTextSetString(dtw_dtype, " ");
	}
/*
 * If the datatype is not in the cache, fetch it and display it.
 */
	else if (nocache || pmsg->xmm_dtype != dt_dtype
			|| pmsg->xmm_gsrc != dt_proc) {

		dt_dtype = pmsg->xmm_dtype;
		dt_proc = pmsg->xmm_gsrc;

		print_type(dt_proc, dt_dtype, fmtbuf);
		XmTextSetString(dtw_dtype, fmtbuf);
	}
}

/*
 *	xmpi_dt_proc
 *
 *	Function:	- set the process datatype
 *	Accepts:	- process global rank
 *			- focus entry
 *			- ignore cache flag (1 => ignore cache)
 */
void
xmpi_dt_proc(rank, pfocus, nocache)

int			rank;
struct xmfocus		*pfocus;
int			nocache;

{
	struct xmproc	*pproc;			/* process entry */

	if (dtw_show != XMPI_SHOWPROC || dtw_rank != rank) return;
/*
 * If process running or done, blank the datatype.
 */
	pproc = pfocus->xmf_proc;

	if (pproc->xmp_state == XMPI_SRUN
		|| pproc->xmp_state == XMPI_SUNDEF || pproc->xmp_dtype == 0) {

		dt_dtype = -1;
		XmTextSetString(dtw_dtype, " ");
	}
/*
 * If datatype not in cache, fetch it and display it.
 */ 
	else if (nocache || pproc->xmp_dtype != dt_dtype || rank != dt_proc) {

		dt_dtype = pproc->xmp_dtype;
		dt_proc = rank;

		print_type(dt_proc, dt_dtype, fmtbuf);
		XmTextSetString(dtw_dtype, fmtbuf);
	}
}

/*
 *	xmpi_dt_create
 *
 *	Function:	- create and initialize Datatype subwindow
 *	Accepts:	- parent window
 */
void
xmpi_dt_create(parent_w)

Widget			parent_w;

{
	Widget		lbl_w;
	Widget		w;

	if (dtw != 0) return;
/*
 * Create the Datatype window.
 */
	dtw = XtVaCreatePopupShell("dt_win",
		xmDialogShellWidgetClass, parent_w,
		XmNdefaultPosition, False,
		XmNmappedWhenManaged, False,
		XmNdeleteResponse, XmUNMAP,
		XmNtitle, "XMPI Datatype",
		NULL);
/*
 * Create the high-level form.
 */
	dtw_form = XtVaCreateWidget("dt_mgr",
		xmFormWidgetClass, dtw,
		XmNmarginHeight, 2,
		XmNmarginWidth, 2,
		XmNverticalSpacing, 2,
		NULL);
/*
 * Create the label area window.
 */
	lbl_w = XtVaCreateWidget("dt_label",
		xmFormWidgetClass, dtw_form,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
/*
 * tack button (cache and re-use pixmaps)
 */
	if (tackpix == XmUNSPECIFIED_PIXMAP) {
		w = xmpi_mkpixbutton(lbl_w, tack_cb, NULL,
			(void (*)()) 0, NULL, (void (*)()) 0, NULL,
			tack_bits, tack_width, tack_height, False);

		XtVaGetValues(w, XmNlabelPixmap, &tackpix,
			XmNarmPixmap, &armtackpix, NULL);
	} else {
		w = xmpi_mkpixbutton(lbl_w, tack_cb, NULL,
			(void (*)()) 0, NULL, (void (*)()) 0, NULL,
			NULL, 0, 0, False);

		XtVaSetValues(w, XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, tackpix,
			XmNarmPixmap, armtackpix, NULL);
	}

	XtVaSetValues(w, XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM, NULL);

	xmpi_add_pophelp(w, "Dismiss datatype window");
/*
 * header label
 */
	dtw_hdr = xmpi_mklabel(lbl_w, "banner", XmALIGNMENT_CENTER, False);

	XtVaSetValues(dtw_hdr,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNleftOffset, 3,
		NULL);

	sprintf(fmtbuf, "%s", "DT Test");
	xmpi_setlabel(dtw_hdr, fmtbuf);

	XtManageChild(lbl_w);
/*
 * Create the pixmaps and datatype pixmap window.
 */
	createpixlabels(dtw_form);
	dtw_pix = xmpi_mklabel(dtw_form, "traffic_lbl",
		XmALIGNMENT_CENTER, False);
	XtVaSetValues(dtw_pix,
		XmNlabelType, XmPIXMAP,
		XmNlabelPixmap, blankpix,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, lbl_w,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNbottomOffset, 2,
		XmNleftAttachment, XmATTACH_FORM,
		XmNleftOffset, 2,
		NULL);
/*
 * Create the datatype scrolled window.
 */
	dtw_dtype = XtVaCreateManagedWidget("dt_dtype",
		xmTextWidgetClass, dtw_form,
		XmNeditable, False,
		XmNeditMode, XmMULTI_LINE_EDIT,
		XmNautoShowCursorPosition, False,
		XmNcursorPositionVisible, False,
		XmNcolumns, 32,
		XmNrows, 8,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, lbl_w,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNbottomOffset, 2,
		XmNrightAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, dtw_pix,
		XmNleftOffset, 2,
		NULL);

	XtManageChild(dtw_form);

#ifdef AIX
/*
 * AIX Motif 1.2 does not seem to honor the alignment resource
 * of child labels unless we force a resize!?
 */
        {
                Dimension       w;
                XtVaGetValues(dtw, XmNwidth, &w, NULL);
                XtVaSetValues(dtw, XmNwidth, w+2, NULL);
        }
#endif
}

/*
 *	xmpi_dt_update
 *
 *	Function:	- updates the datatype display window
 *	Accepts:	- rank
 *			- focus entry
 *			- show process or message type
 */
void
xmpi_dt_update(rank, pfocus, show)

int			rank;
struct xmfocus		*pfocus;
int			show;

{
	int		nocache;	/* set on if cache is no good */
	struct xmproc	*pproc;		/* process entry */

/*
 * Update the rank in the header.  Keep the format consistent with
 * xmpi_fo_update().
 */
	if (rank != dtw_rank) {
		pproc = pfocus->xmf_proc;
		if ((pproc->xmp_state == XMPI_SRUN) ||
				(pproc->xmp_state == XMPI_SUNDEF)) {
			sprintf(fmtbuf, "%d   %s", rank, pproc->xmp_prog);
		} else {
			sprintf(fmtbuf, "%d / %d   %s", rank,
				pproc->xmp_lrank, pproc->xmp_prog);
		}
		xmpi_setlabel(dtw_hdr, fmtbuf);
	}
/*
 * Update the pixmap.
 */
	if (show != dtw_show) {
		if (show == XMPI_SHOWMESG) {
			XtVaSetValues(dtw_pix, XmNlabelPixmap, mesgpix, NULL);
		} else if (show == XMPI_SHOWPROC) {
			XtVaSetValues(dtw_pix,
				XmNlabelPixmap, trafficpix, NULL);
		} else {
			XtVaSetValues(dtw_pix, XmNlabelPixmap, blankpix, NULL);
		}
	}
/*
 * Update state. Must be done before text update.
 */
	nocache = (dtw_rank != rank || dtw_show != show);
	dtw_show = show;
	dtw_rank = rank;
/*
 * Update the data type text.
 */
	if (show == XMPI_SHOWMESG) {
		xmpi_dt_msg(rank, pfocus, nocache);
	}
	else if (show == XMPI_SHOWPROC) {
		xmpi_dt_proc(rank, pfocus, nocache);
	}

	XtPopup(dtw, XtGrabNone);
	XtManageChild(dtw_form);
}		

/*
 *	xmpi_dt_clear
 *
 *	Function:	- clear and pop down Datatype window
 */
void
xmpi_dt_clear()

{
	if (dtw == 0) return;

	dtw_rank = -1;
	dtw_show = -1;
	xmpi_setlabel(dtw_hdr, "");
	XmTextSetString(dtw_dtype, "");
	XtVaSetValues(dtw_pix, XmNlabelPixmap, blankpix, NULL);
	XtPopdown(dtw);
}

/*
 *	xmpi_dt_setrank
 *
 *	Function:	- set rank in Datatype window header
 *	Accepts:	- rank
 *			- text form of rank
 */
void
xmpi_dt_setrank(rank, text)

int			rank;
char			*text;

{
	if (dtw_rank == rank) xmpi_setlabel(dtw_hdr, text);
}

/*
 *	xmpi_dt_popdown
 *
 *	Function:	- unmap the Datatype window
 */
void
xmpi_dt_popdown()

{
	if (dtw) XtPopdown(dtw);
}

/*
 *	tack_cb
 *
 *	Function:	- callback to unmap the Datatype window
 */
static void
tack_cb()

{
	XtPopdown(dtw);
}

/*
 *	createpixlabels
 *
 *	Function:	- create pixmaps for labels
 *			- blank, traffic light & message
 *	Accepts:	- parent widget
 */
static void
createpixlabels(w)

Widget			w;

{
	unsigned int	depth;			/* screen depth */
	Display		*disp;			/* display */
	Drawable	root;			/* root window */
	Pixel		fg, bg;			/* fore/background colours */

	if (blankpix != XmUNSPECIFIED_PIXMAP) return;

	disp = XtDisplay(w);
	root = RootWindowOfScreen(XtScreen(w)),
	depth = DefaultDepthOfScreen(XtScreen(w));
	XtVaGetValues(w, XmNforeground, &fg, XmNbackground, &bg, NULL);
/*
 * Create the static pixmaps.
 */
	trafficpix = XCreatePixmapFromBitmapData(disp, root,
		(char*)light_bits, light_width, light_height, fg, bg, depth);

	mesgpix = XCreatePixmapFromBitmapData(disp, root,
		(char*)mesg_bits, mesg_width, mesg_height, fg, bg, depth);

	blankpix = XCreatePixmapFromBitmapData(disp, root,
		(char*)blank_bits, blank_width, blank_height, fg, bg, depth);
}

/*
 *	xmpi_dt_busy
 *
 *	Function:	- set the busy cursor for the Datatype window
 */
void
xmpi_dt_busy()

{
	if (dtw) xmpi_busy_widget(dtw);
}

/*
 *	xmpi_dt_unbusy
 *
 *	Function:	- set the normal cursor for the Datatype window
 */
void
xmpi_dt_unbusy()

{
	if (dtw) xmpi_unbusy_widget(dtw);
}

/*
 *	print_type
 *
 *	Function:	- textual representation of a datatype
 *	Accepts:	- process which defined the datatype
 *			- datatype label
 *			- textual representation (out)
 */
static void
print_type(proc, dtype, buf)

int			proc;
int			dtype;
char			*buf;

{
	void		*trace;

	if (dtype < 0) {
		sprintf(buf, "invalid datatype: %d", dtype);
	}
	else {
		trace = (fl_dbase) ? xmpi_db_getdtype(proc, dtype)
				: xmpi_sys_dtype(xmpi_app_procs + proc, dtype);

		if (trace == (void *) -1) {
			sprintf(buf, "unknown datatype: %d", dtype);
		} else {
			xmpi_dtypeprint(dtype, trace, XMPI_DTINDENT, "", buf);
			if (!fl_dbase && trace) free((char *) trace);
		}
	}
}
