/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_tr_dump.c,v 1.4 1999/11/08 06:20:26 bbarrett Exp $
 *
 *	Function:	- dump traces into a trace file
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <Xm/Xm.h>
#include <Xm/SelectioB.h>

#include "lam.h"
#include "xmpi.h"

/*
 * global functions
 */
void			xmpi_tr_dump();
void			xmpi_tr_express();
void			xmpi_tr_cleanup();

/*
 * local functions
 */
static void		dump_cb();
static int		dump();

/*
 * extern variables
 */
extern int		xmpi_app_nprocs;	/* number of procs in appl. */
extern struct _gps	*xmpi_app_procs;	/* appl. GPS array */
extern char             tracefile[];            /* name of trace file */

/*
 * local variables
 */
static char		tempfile[XMPI_PATHMAX];

/*
 *	xmpi_tr_dump
 *
 *	Function:	- dump current application's traces into a file
 *	Accepts:	- parent widget
 */
void
xmpi_tr_dump(parent)

Widget			parent;

{
	Widget		dialog_w;		/* file name prompt dialog */
	XmString	str1, str2;		/* various strings */
	Arg		args[3];

	if (xmpi_app_procs == 0) {
		errno = 0;
		xmpi_error(parent,
			"No application for which to dump traces.");
		return;
	}
	
	str1 = XmStringCreateSimple("Enter trace file name.");

	if (!tempfile[0]) {
		sprintf(tempfile, "%s/lam-xmpi%d.lamtr", TMP, (int)getpid());
	}

	str2 = XmStringCreateSimple(tempfile);

	XtSetArg(args[0], XmNselectionLabelString, str1);
	XtSetArg(args[1], XmNautoUnmanage, False);
	XtSetArg(args[2], XmNtextString, str2);

	dialog_w = XmCreatePromptDialog(parent, "prompt_pop", args, 3);

	XmStringFree(str1);
	XmStringFree(str2);

	XtUnmanageChild(XmSelectionBoxGetChild(dialog_w,
					XmDIALOG_HELP_BUTTON));

	XtAddCallback(dialog_w, XmNokCallback, dump_cb, NULL);
	XtAddCallback(dialog_w, XmNcancelCallback,
				(XtCallbackProc) XtDestroyWidget, NULL);

	XtManageChild(dialog_w);
	XtPopup(XtParent(dialog_w), XtGrabNone);
}

/*
 *	dump_cb
 *
 *	Function:	- dumps traces (callback)
 *	Accepts:	- callback data
 */
static void
dump_cb(w, data, cbs)

Widget			w;
XtPointer		data;
XmSelectionBoxCallbackStruct
			*cbs;

{
	struct stat 	finfo;			/* file information */
	char		*fname;			/* file name */
	int		exists = 1;		/* file exists? */

	XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &fname);

	if (! (fname && *fname)) {
		errno = 0;
		xmpi_error(w, "No filename specified.");
		return;
	}
/*
 * Check if file already exists and if so ask user if they want to overwrite
 * it or not.
 */
	if (stat(fname, &finfo)) {
		if (errno != ENOENT) {
			xmpi_error(w, fname);
			XtFree(fname);
			return;
		}
		exists = 0;
	}

	if (exists && !xmpi_yesno(w, "File exists. Overwrite?")) {
		XtFree(fname);
		return;
	}

	(void) dump(fname);

	XtFree(fname);
	XtDestroyWidget(w);
}

/*
 *	dump
 *
 *	Function:	- dumps traces
 *	Accepts:	- file name
 *	Returns		- 0 else LAMERROR
 */
static int
dump(file)

char *			file;

{
	int		fd;			/* trace file descriptor */
/*
 * Open the file and dump the traces into it.
 */
	fd = open(file, O_WRONLY | O_CREAT | O_TRUNC, 0644);

	if (fd < 0) {
		xmpi_error(0, file);
		return(LAMERROR);
	}

	xmpi_busy();
	xmpi_ctl_setinfo("Collecting traces...");

	if (xmpi_sys_trace(fd, xmpi_app_procs, xmpi_app_nprocs)) {
		xmpi_ctl_resetinfo();
		xmpi_unbusy();
		xmpi_error(0, "Collecting traces");
		close(fd);
		return(LAMERROR);
	}

	xmpi_ctl_resetinfo();
	xmpi_unbusy();
	close(fd);
	return(0);
}

/*
 *	xmpi_tr_express
 *
 *	Function:	- dumps traces and views them
 *	Accepts:	- parent widget
 */
void
xmpi_tr_express(parent)

Widget			parent;

{
        FILE            *dbfs;                  /* trace file stream */

	if (!tempfile[0]) {
		sprintf(tempfile, "%s/lam-xmpi%d.lamtr", TMP, (int)getpid());
	}

	if (dump(tempfile)) return;

	strncpy(tracefile, tempfile, XMPI_PATHMAX);
        dbfs = fopen(tracefile, "r");

	if (dbfs == 0) {
                xmpi_error(0, tracefile);
        } else {
		xmpi_parse_tracefile(dbfs);
        }
}

/*
 *	xmpi_tr_cleanup
 *
 *	Function:	- removes temporary trace file, if any
 *	Accepts:	- parent widget
 */
void
xmpi_tr_cleanup()

{
	if (tempfile[0]) {
		unlink(tempfile);
		tempfile[0] = 0;
	}
}
