/*******************************************************************************
 * vi:ts=4:sw=4
 *	original Ogasawara Hiroyuki (COR.)
 *  original Atsushi Nakamura
 ******************************************************************************/
#ifdef KANJI

#ifdef MSDOS
#include	<io.h>
#endif
#ifdef NT
# include <windows.h>
# undef DELETE
#endif
#include	"vim.h"
#include	"globals.h"
#include	"proto.h"
#include	"param.h"
#include	"ops.h"
#include	"kanji.h"
#include	"jptab.h"

#define	JP_EUC_G2		0x8e

static int_u euctosjis		__ARGS((char_u, char_u));
static int_u sjistoeuc		__ARGS((char_u, char_u));
static char_u *kanjiin		__ARGS((char_u));
static char_u *asciiin		__ARGS((char_u));
static char_u *kanain		__ARGS((char_u));
static int jisx0201rto0208	__ARGS((char_u, char_u, char_u *, char_u *));

static char_u	kanji_map_sjis[]= {
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0
};

static char_u	kanji_map_euc[]= {
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0
};

	int
ISkanji(code)
int			code;
{
	if (code >= 0x100)
		return 0;
	return(kanji_map_sjis[code & 0xff] & 1);
}

	int
ISkana(code)
int			code;
{
	if (code >= 0x100)
		return 0;
	return(kanji_map_sjis[code & 0xff] & 2);
}

	int
ISdisp(code)
int			code;
{
	if (code >= 0x100)
		return 0;
	return(kanji_map_sjis[code & 0xff]);
}

/* input pos : 1..strlen
   return  0 : not kanji
	   1 : kanji first byte
	   2 : kanji last byte
*/
	int
ISkanjiPosition(ptr, pos)
char_u	*	ptr;
int			pos;
{
	int	kanji = 0;

	for (; *ptr && pos--; ptr++)
	{
		if (kanji == 1)
			kanji= 2;
		else
		{
			if (kanji_map_sjis[*ptr] & 1)
				kanji= 1;
			else
				kanji= 0;
		}
	}
	return(kanji);
}

	int
ISkanjiPointer(ptr, p)
char_u	*	ptr;
char_u	*	p;
{
	return(ISkanjiPosition(ptr, p - ptr + 1));
}

	int
ISkanjiCol(lnum, col)
linenr_t	lnum;
colnr_t		col;
{
	return(ISkanjiPosition(ml_get_buf(curbuf, lnum, FALSE), col + 1));
}


	int
ISkanjiCur()
{
	return(ISkanjiPosition(ml_get_buf(curbuf, curwin->w_cursor.lnum, FALSE),
						curwin->w_cursor.col + 1));
}

	int
ISkanjiFpos(po)
FPOS	*	po;
{
	return(ISkanjiPosition(ml_get_buf(curbuf, po->lnum, FALSE), po->col + 1));
}

	int
vcol2col(wp, lnum, maxcol, wantcol)
WIN		*	wp;
linenr_t	lnum;
colnr_t		maxcol;
int		*	wantcol;
{
	char_u	*	line;
	char_u	*	ptr;
	char_u		c;
	colnr_t		vcol;

	ptr = line = ml_get_buf(wp->w_buffer, lnum, FALSE);

	vcol = 0;
	while ((c = *ptr) != NUL)
	{
		if (kanji_map_sjis[c] & 1)
		{
			vcol += 2;
			if (vcol > maxcol)
				break;
			ptr += 2;
			continue;
		}
		vcol += chartabsize(c, vcol);
		if (vcol > maxcol)
			break;
		ptr ++;
	}
	if (wantcol != NULL)
	{
		if (c == NUL)
			*wantcol = ptr - line;
		else if (kanji_map_sjis[c] & 1)
			*wantcol = ptr - line + 1;
		else
			*wantcol = ptr - line;
	}
	return(ptr - line);
}

/*
 *	Japanese Character;
 */
	int_u
sjistojis(high, low)
char_u		high;
char_u		low;
{
	if (high <= 0x9f)
		high -= 0x71;
	else
		high -= 0xb1;
	high = high * 2 + 1;
	if (low > 0x7f)
		low--;
	if (low >= 0x9e)
	{
		low -= 0x7d;
		high++;
	}
	else
	{
		low -= 0x1f;
	}
	return(((int_u)high << 8) | (low & 0xff));
}

	static int_u
sjistoeuc(high, low)
char_u		high;
char_u		low;
{
	return(sjistojis(high, low) | 0x8080);
}

	int_u
jistosjis(high, low)
char_u		high;
char_u		low;
{
	if (high & 1)
		low += 0x1f;
	else
		low += 0x7d;
	if (low >= 0x7f)
		low++;
	high = ((high - 0x21) >> 1) + 0x81;
	if (high > 0x9f)
		high += 0x40;
	return(((int_u)high << 8) | (low & 0xff));
}

	static int_u
euctosjis(high, low)
char_u		high;
char_u		low;
{
	return(jistosjis(high & 0x7f, low & 0x7f));
}

/*
 * return kanji shift-in string
 */
	static char_u *
kanjiin(code)
	char_u	code;
{
	switch(code) {
	case JP_JIS:		return "\033$B";
	default:			return "";
	}
}

/*
 * return kanji shift-out string
 */
	static char_u *
asciiin(code)
	char_u	code;
{
	switch(code) { 
	case JP_JIS:		return "\033(B";
	default:			return "";
	}
}

/*
 * return kana shift-in string
 */
	static char_u *
kanain(code)
	char_u	code;
{
	switch(code) { 
	case JP_JIS:		return "\033(I";
	default:			return "";
	}
}

	char_u	*
JPdisp(now, mode, code)
	int_u	*now;
	int		mode;
	int		code;		/* kanji code */
{
	static	char_u	buffer[32];
	char_u			*p;

	buffer[0] = NUL;
	switch (*now) {
	case JP_ASCII:
		switch (mode) {
		case JP_KANJI:
			p = kanjiin(code);
			if (p)
				STRCAT(buffer, p);
			break;
		case JP_KANA:
			p = kanain(code);
			if (p)
				STRCAT(buffer, p);
			break;
		}
		break;
	case JP_KANJI:
		switch (mode) {
		case JP_ASCII:
			p = asciiin(code);
			if (p)
				STRCAT(buffer, p);
			break;
		case JP_KANA:
			p = kanain(code);
			if (p)
				STRCAT(buffer, p);
			break;
		}
		break;
	case JP_KANA:
		switch (mode) {
		case JP_ASCII:
			p = asciiin(code);
			if (p)
				STRCAT(buffer, p);
			break;
		case JP_KANJI:
			p = kanjiin(code);
			if (p)
				STRCAT(buffer, p);
			break;
		}
		break;
	}
	*now = mode;
	return(buffer);
}

/*
 * convert SJIS letter into suitable letter.
 */
	void
kanjito(k1, k2, code)
	char_u	*k1, *k2;
	char	code;
{
	int_u		kanji;

	switch(code) {
	case JP_JIS:
		kanji = sjistojis(*k1, *k2);
		*k1 = (kanji & 0xff00) >> 8;
		*k2 = kanji & 0xff;
		break;
	case JP_EUC:
		kanji = sjistoeuc(*k1, *k2);
		*k1 = (kanji & 0xff00) >> 8;
		*k2 = kanji & 0xff;
	    break;
#ifdef UCODE
	case JP_UTF8:
	case JP_WIDE:
		multi2wide(k1, k2, 2);
		break;
#endif
	default:
		break;
	}
}

	void
kanato(k1, k2, code)
	char_u	*k1, *k2;
	char	code;
{
	switch(code) {
	case JP_JIS:
		*k1 &= 0x7f;
		*k2 = NUL;
		break;
	case JP_EUC:
		*k2 = *k1;
		*k1 = JP_EUC_G2;
	    break;
#ifdef UCODE
	case JP_UTF8:
	case JP_WIDE:
		multi2wide(k1, k2, 1);
		break;
#endif
	default:
		*k2 = NUL;
		break;
	}
}

/*
 *	Japanese Character class;
 *					Make sure this routine is consistent with search.c:cls().
 *
 * 	for Japanese
 *		3 - alphabet, digits
 *		4 - japanese hiragana
 *		5 - japanese katakana
 *		6 - symbols
 *		7 - other multi-char letter
 */
	int
jpcls(c, k)
char_u		c;
char_u		k;
{
	if (c == ' ' || c == '\t' || c == NUL)
		return 0;
	if (kanji_map_sjis[c] & 1)
	{
		int		ret;
		ret = sjistojis(c, k);
		c = ((int_u)ret & 0xff00) >> 8;
		k =  (int_u)ret & 0xff;
		ret = jptab[c].cls1;
		if (ret == JPC_KIGOU)
			ret = jptab[k].cls2;
		if (ret == JPC_KIGOU2)
			ret = JPC_KIGOU;
		return ret;
	}
	if (isidchar(c))
		return 1;
	return -1;
}

/* 
 *	isjppunc(c, k) returns whether a kanji character ck necessary KINSOKU
 *	processing or not.
 */
	int
isjppunc(c, k, type)
char_u		c;
char_u		k;
int			type;
{
	int		jis;

	jis = sjistojis(c, k);
	c = ((int_u)jis & 0xff00) >> 8;
	k =  (int_u)jis & 0xff;
	switch(jptab[(int)c].cls1)
	{
	case JPC_KIGOU:
		return type ? jptab[(int)k].punccsym : jptab[(int)k].puncosym;
	case JPC_HIRA:
	case JPC_KATA:
		return type ? jptab[(int)k].puncckana : FALSE;
	default:
		return FALSE;
	}
}

/* 
 *	isaspunc(c, type) returns whether an ascii character ck necessary KINSOKU
 *	processing or not.
 */
	int
isaspunc(c, type)
char_u		c;
int			type;
{
	return(type ? jptab[(int)c].punccasc: jptab[(int)c].puncoasc);
}

/* 
 *	isjsend(*cp) returns whether a JIS character *cp separates
 *	sentences or not.
 */
	int
isjsend(cp)
char_u	*	cp;
{
	int	kanji	= sjistojis(cp[0], cp[1]);
	int	k1		= (kanji & 0xff00) >> 8;
	int	k2		= kanji & 0xff;
	return k1 == '!' && jptab[k2].stcend;
}

/* 
 *	jptocase(&c, &k, tocase)
 *		modify c & k to case tocase
 *			tocase == UPPER : to upper
 *			tocase == LOWER : to lower
 *			tocase == others : swap case
 */
	void
jptocase(cp, kp, tocase)
char_u	*	cp;
char_u	*	kp;
int			tocase;
{
	char_u		k;
	int_u		kanji = sjistoeuc(*cp, *kp);

	*cp	= (kanji & 0xff00) >> 8;
	*kp = kanji & 0xff;
	k = *kp & 0x7f;
	switch(jptab[*cp & 0x7f].cls1) {
	case JPC_ALNUM:
		if (tocase != LOWER && islower(k))
			*kp = TO_UPPER(k) | 0x80;
		if (tocase != UPPER && isupper(k))
			*kp = TO_LOWER(k) | 0x80;
		break;
	case JPC_KIGOU:
		if (  (tocase != LOWER && jptab[k].scase == JLOS)
		   || (tocase != UPPER && jptab[k].scase == JUPS))
			*kp = jptab[k].swap;
		break;
	case JPC_KATA:
		if (tocase != -1)
			*cp = JP1_HIRA | 0x80;
		break;
	case JPC_HIRA:
		if (tocase != 1)
			*cp = JP1_KATA | 0x80;
		break;
	default:
		break;
	}
	kanji = euctosjis(*cp, *kp);
	*cp	= (kanji & 0xff00) >> 8;
	*kp = kanji & 0xff;
}

/*
 *
 *
 */
#ifdef UCODE
	static int
judge_sjis_euc(ptr)
char_u	*	ptr;
{
	if (((0xa1 <= ptr[0] && ptr[0] <= 0xfe)
				&& (0xa1 <= ptr[1] && ptr[1] <= 0xfe))
			|| (ptr[0] == 0x8e && (0xa1 <= ptr[1] && ptr[1] <= 0xdf)))
		return(TRUE);/* EUC */
	else if ((((0x81 <= ptr[0] && ptr[0] <= 0x9f)
					|| (0xe0 <= ptr[0] && ptr[0] <= 0xef))
				&& ((0x40 <= ptr[1] && ptr[1] <= 0x7e)
					|| (0x80 <= ptr[1] && ptr[1] <= 0xfc)))
			|| (0xa1 <= ptr[0] && ptr[0] <= 0xdf))
		return(TRUE);/* SJIS */
	return(FALSE);
}

	static int
judge_ucs(ptr)
char_u	*	ptr;
{
	short_u	ucs;
	char_u	dst[2];

	if (ptr[0] < 0xe0)
		ucs = ((ptr[0] & 0x1f) << 6) | (ptr[1] & 0x3f);
	else
		ucs = ((ptr[0] & 0x0f) << 12) | ((ptr[1] & 0x3f) << 6) | (ptr[2] & 0x3f);
	dst[0] = ucs & 255;
	dst[1] = ucs >> 8;
	if (wide2multi(dst, 2, FALSE) == 1 && dst[0] == '?')
		return(FALSE);
	return(TRUE);
}
#endif

/*
 *
 *
 */
	int
judge_jcode(origcode, ptr, size)
char_u	*	origcode;
char_u	*	ptr;
long		size;
{
	char	code;
	int		i;
	int		bfr  = FALSE;	/* Kana Moji */
	int		bfk  = 0;		/* EUC Kana */
	int		sjis = 0;
	int		euc  = 0;
#ifdef UCODE
	int		utf8 = 0;
	int		bfu  = 0;
#endif

	code = '\0';
#ifdef UCODE
	if ((ptr[0] == 0xff && ptr[1] == 0xfe)
							|| (ptr[0] == 0xfe && ptr[1] == 0xff))
	{
		code = 'U';		/* UNICODE */
		goto breakBreak;
	}
	if (*origcode == 'U')
		*origcode = 'S';

	/* valid UTF-8 or not */
	i = 0;
	while (i < size)
	{
		if (ptr[i] < 0x80)
			i++;
		else if (ptr[i] < 0xc0)
		{
			/* malformed */
			utf8 = 0;
			break;
		}
		else if (ptr[i] < 0xe0)
		{
			if (size - i > 1)
			{
				if (ptr[i + 1] >= 0x80 && ptr[i + 1] < 0xc0)
				{
					if (judge_ucs(&ptr[i]))
						code = 'T';
					if (judge_sjis_euc(&ptr[i]))
						;
					else
						utf8++;
				}
				else
				{
					/* malformed */
					utf8 = 0;
					break;
				}
			}
			i += 2;
		}
		else if (ptr[i] < 0xf0)
		{
			if (size - i > 2)
			{
				if (ptr[i + 1] >= 0x80 && ptr[i + 1] < 0xc0
						&& ptr[i + 2] >= 0x80 && ptr[i + 2] < 0xc0)
				{
					if (judge_ucs(&ptr[i]))
						code = 'T';
					if (judge_sjis_euc(&ptr[i]))
						;
					else
						utf8++;
				}
				else
				{
					/* malformed */
					utf8 = 0;
					break;
				}
			}
			i += 3;
		}
		else if (ptr[i] < 0xf8)
		{
			/* valid but not supported */
			if (size - i > 3)
			{
				if (!(ptr[i + 1] >= 0x80 && ptr[i + 1] < 0xc0
						&& ptr[i + 2] >= 0x80 && ptr[i + 2] < 0xc0
						&& ptr[i + 3] >= 0x80 && ptr[i + 3] < 0xc0))
				{
					/* malformed */
					utf8 = 0;
					break;
				}
			}
			i += 4;
		}
		else if (ptr[i] < 0xfc)
		{
			/* valid but not supported */
			if (size - i > 4)
			{
				if (!(ptr[i + 1] >= 0x80 && ptr[i + 1] < 0xc0
						&& ptr[i + 2] >= 0x80 && ptr[i + 2] < 0xc0
						&& ptr[i + 3] >= 0x80 && ptr[i + 3] < 0xc0
						&& ptr[i + 4] >= 0x80 && ptr[i + 4] < 0xc0))
				{
					/* malformed */
					utf8 = 0;
					break;
				}
			}
			i += 5;
		}
		else if (ptr[i] < 0xfe)
		{
			/* valid but not supported */
			if (size - i > 5)
			{
				if (!(ptr[i + 1] >= 0x80 && ptr[i + 1] < 0xc0
						&& ptr[i + 2] >= 0x80 && ptr[i + 2] < 0xc0
						&& ptr[i + 3] >= 0x80 && ptr[i + 3] < 0xc0
						&& ptr[i + 4] >= 0x80 && ptr[i + 4] < 0xc0
						&& ptr[i + 5] >= 0x80 && ptr[i + 5] < 0xc0))
				{
					/* malformed */
					utf8 = 0;
					break;
				}
			}
			i += 6;
		}
		else
		{
			/* malformed */
			utf8 = 0;
			break;
		}
	}
	if (utf8 > 1)
	{
		code = 'T';
		goto breakBreak;
	}
#endif /* UCODE */

	i = 0;
	while (i < size)
	{
		if (ptr[i] == '\033' && (size - i >= 3))
		{
			if ((ptr[i+1] == '$' && ptr[i+2] == 'B')
						 || (ptr[i+1] == '(' && ptr[i+2] == 'B'))
			{
				code = 'J';
#ifdef USE_OPT
				if (!(p_opt & OPT_NO_JIS))
					i += 3;
				else
#endif
				goto breakBreak;
			}
			else if ((ptr[i+1] == '$' && ptr[i+2] == '@')
						 || (ptr[i+1] == '(' && ptr[i+2] == 'J'))
			{
				code = 'J';
#ifdef USE_OPT
				if (!(p_opt & OPT_NO_JIS))
					i += 3;
				else
#endif
				goto breakBreak;
			}
			else if (ptr[i+1] == '(' && ptr[i+2] == 'I')
			{
				code = 'J';
				i += 3;
			}
			else if (ptr[i+1] == ')' && ptr[i+2] == 'I')
			{
				code = 'J';
				i += 3;
			}
			else
				i++;
			bfr = FALSE;
			bfk = 0;
		}
		else
		{
			if (ptr[i] < 0x20)
			{
#ifndef UCODE
				if (bfr == TRUE)
				{
					code = 'S';
					goto breakBreak;
				}
#endif
				bfr = FALSE;
				bfk = 0;
				/* ?? check kudokuten ?? && ?? hiragana ?? */
				if ((i >= 2) && (ptr[i-2] == 0x81)
						&& (0x41 <= ptr[i-1] && ptr[i-1] <= 0x49))
				{
					code = 'S';
					sjis += 100;	/* kudokuten */
				}
				else if ((i >= 2) && (ptr[i-2] == 0xa1)
						&& (0xa2 <= ptr[i-1] && ptr[i-1] <= 0xaa))
				{
					code = 'E';
					euc  += 100;	/* kudokuten */
				}
				else if ((i >= 2) && (ptr[i-2] == 0x82) && (0xa0 <= ptr[i-1]))
					sjis += 40;		/* hiragana */
				else if ((i >= 2) && (ptr[i-2] == 0xa4) && (0xa0 <= ptr[i-1]))
					euc  += 40;		/* hiragana */
#ifdef UCODE
				else if ((0xa1 <= ptr[i-2] && ptr[i-2] <= 0xfe)
								&& (0xa1 <= ptr[i-1] && ptr[i-1] <= 0xfe))
					;	/* EUC */
				else if (0x8e == ptr[i-2]
								&& (0xa1 <= ptr[i-1] && ptr[i-1] <= 0xdf))
					;	/* EUC */
				else if (((0x81 <= ptr[i-2] && ptr[i-2] <= 0x9f)
								|| (0xe0 <= ptr[i-2] && ptr[i-2] <= 0xef))
						&& ((0x40 <= ptr[i-1] && ptr[i-1] <= 0x7e)
								|| (0x80 <= ptr[i-1] && ptr[i-1] <= 0xfc)))
					;	/* SJIS */
				else if ((i >= 3) && (ptr[i-3] & 0xf0) == 0xe0
											&& (ptr[i-2] & 0xc0) == 0x80
											&& (ptr[i-1] & 0xc0) == 0x80)
				{
					code = 'T';
					utf8  += 30;
				}
				else if ((i >= 2) && (ptr[i-2] & 0xe0) == 0xc0
											&& (ptr[i-1] & 0xc0) == 0x80)
				{
					code = 'T';
					utf8  += 10;
				}
#endif
			}
			else
			{
				/* ?? check hiragana or katana ?? */
				if ((size - i > 1) && (ptr[i] == 0x82) && (0xa0 <= ptr[i+1]))
					sjis++;	/* hiragana */
				else if ((size - i > 1) && (ptr[i] == 0x83)
						&& (0x40 <= ptr[i+1] && ptr[i+1] <= 0x9f))
					sjis++;	/* katakana */
				else if ((size - i > 1) && (ptr[i] == 0xa4) && (0xa0 <= ptr[i+1]))
					euc++;	/* hiragana */
				else if ((size - i > 1) && (ptr[i] == 0xa5) && (0xa0 <= ptr[i+1]))
					euc++;	/* katakana */
#ifdef UCODE
				if (bfu)
					bfu--;
				else
#endif
				if (bfr == TRUE)
				{
					if ((i >= 1) && (0x40 <= ptr[i] && ptr[i] <= 0xa0) && ISkanji(ptr[i-1]))
					{
						code = 'S';
						goto breakBreak;
					}
					else if ((i >= 1) && (0x81 <= ptr[i-1] && ptr[i-1] <= 0x9f) && ((0x40 <= ptr[i] && ptr[i] < 0x7e) || (0x7e < ptr[i] && ptr[i] <= 0xfc)))
					{
						code = 'S';
						goto breakBreak;
					}
					else if ((i >= 1) && (0xfd <= ptr[i] && ptr[i] <= 0xfe) && (0xa1 <= ptr[i-1] && ptr[i-1] <= 0xfe))
					{
						code = 'E';
						goto breakBreak;
					}
					else if ((i >= 1) && (0xfd <= ptr[i-1] && ptr[i-1] <= 0xfe) && (0xa1 <= ptr[i] && ptr[i] <= 0xfe))
					{
						code = 'E';
						goto breakBreak;
					}
					else if ((i >= 1) && (ptr[i] < 0xa0 || 0xdf < ptr[i]) && (0x8e == ptr[i-1]))
					{
						code = 'S';
						goto breakBreak;
					}
					else if (ptr[i] <= 0x7f)
					{
						code = 'S';
						goto breakBreak;
					}
					else
					{
						if (0xa1 <= ptr[i] && ptr[i] <= 0xa6)
							euc++;	/* sjis hankaku kana kigo */
						else if (0xa1 <= ptr[i] && ptr[i] <= 0xdf)
							;	/* sjis hankaku kana */
						else if (0xa1 <= ptr[i] && ptr[i] <= 0xfe)
							euc++;
						else if (0x8e == ptr[i])
							euc++;
						else if (0x20 <= ptr[i] && ptr[i] <= 0x7f)
							sjis++;
						bfr = FALSE;
						bfk = 0;
					}
				}
#ifdef UCODE
				else if ((size - i > 3) && (ptr[i] & 0xf0) == 0xe0
											&& (ptr[i+1] & 0xc0) == 0x80
											&& (ptr[i+2] & 0xc0) == 0x80
										&& ((ptr[i+3] & 0x80) == 0x00
											|| (ptr[i+3] & 0xf0) == 0xe0
											|| (ptr[i+3] & 0xe0) == 0xc0)
						&& !((0xa1 <= ptr[i] && ptr[i] <= 0xfe)
								&& (0xa1 <= ptr[i+1] && ptr[i+1] <= 0xfe)
								&& (0xa1 <= ptr[i+2] && ptr[i+2] <= 0xfe)))
				{
					utf8++;
					bfu = 2;
					bfk = 0;
				}
				else if ((size - i > 2) && (ptr[i] & 0xe0) == 0xc0
											&& (ptr[i+1] & 0xc0) == 0x80
										&& ((ptr[i+2] & 0x80) == 0x00
											|| (ptr[i+2] & 0xf0) == 0xe0
											|| (ptr[i+2] & 0xe0) == 0xc0)
						&& !((0xa1 <= ptr[i] && ptr[i] <= 0xfe)
								&& (0xa1 <= ptr[i+1] && ptr[i+1] <= 0xfe)))
				{
					utf8++;
					bfu = 1;
					bfk = 0;
				}
#endif
				else if (0x8e == ptr[i])
				{
					if (size - i <= 1)
						;
					else if (0xa1 <= ptr[i+1] && ptr[i+1] <= 0xdf)
					{
						/* EUC KANA or SJIS KANJI */
						if (bfk == 1)
							euc += 100;
						bfk++;
						i++;
					}
					else
					{
						/* SJIS only */
						code = 'S';
						goto breakBreak;
					}
				}
				else if (0x81 <= ptr[i] && ptr[i] <= 0x9f)
				{
					/* SJIS only */
					code = 'S';
					goto breakBreak;
				}
				else if (0xfd <= ptr[i] && ptr[i] <= 0xfe)
				{
					/* EUC only */
					code = 'E';
					goto breakBreak;
				}
				else if (ptr[i] <= 0x7f)
					;
				else
				{
					bfr = TRUE;
					bfk = 0;
				}
			}
			i++;
		}
	}
	if (code == '\0')
	{
		code = *origcode;
#ifdef UCODE
		if (utf8 > sjis && utf8 > euc)
			code = 'T';
		else
#endif
		if (sjis > euc)
			code = 'S';
		else if (sjis < euc)
			code = 'E';
	}
breakBreak:
	return(code);
}

	int				/* return the length of dst */
kanjiconvsfrom(ptr, ptrlen, dst, dstlen, tail, code, charsetp)
	char_u	*ptr;
	int		ptrlen;
	char_u	*dst;
	int		dstlen;
	char	*tail;
	char	code;
	int		*charsetp;
{
	char_u	*dtop;
	int		c;
	int		charset;

	dtop = dst;
	if (tail)
		tail[0] = NUL;

	switch (code) {
	case JP_EUC:
	case JP_SJIS:
	case JP_JIS:
#ifdef UCODE
	case JP_UTF8:
#endif
		if (charsetp)
			charset = *charsetp;
		else
			charset = JP_ASCII;
		while (ptrlen) {
			if (dst - dtop >= dstlen)
				return -1;
			ptrlen --;

			if (*ptr & 0x80)
			{
				charset = JP_ASCII;
				switch (code) {
				case JP_EUC:
					switch (kanji_map_euc[*ptr]) {
					case 1:		/* kanji */
						if (ptrlen >= 1 && ptr[1] != NUL)
						{					/* JIS0208 char. */
							*dst++ = (euctosjis(ptr[0], ptr[1]) & 0xff00) >> 8;
							*dst++ = euctosjis(ptr[0], ptr[1]) & 0xff;
							ptr += 2;
							ptrlen --;
						}
						else if (tail && ptrlen == 0)
						{					/* not completed  */
							tail[0] = *ptr;
							tail[1] = NUL;
						}
						else
						{					/* Illegal char  */
							*(dst - 1) &= 0x7f;
						}
						continue;
					case 2:
						if (ptr[1] != NUL)
						{
							if (p_jkc && ISkana(ptr[1]))
							{
								char c1 = NUL;
								char c2 = NUL;

								if ((ptrlen >= 3) && (ptr[2] == JP_EUC_G2))
								{
									if (jisx0201rto0208(ptr[1], ptr[3], &c1, &c2))
									{
										ptr += 4;
										*dst++ = c1;
										*dst++ = c2;
										ptrlen -= 3;
									}
									else
									{
										ptr += 2;
										*dst++ = c1;
										*dst++ = c2;
										ptrlen --;
									}
								}
								else
								{
									(void)jisx0201rto0208(ptr[1], NUL, &c1, &c2);
									ptr += 2;
									*dst++ = c1;
									*dst++ = c2;
									ptrlen --;
								}
							}
							else
							{
								*dst++  = * ++ptr;
								ptr++;
								ptrlen --;
							}
							continue;
						}
						if (tail && ptrlen == 0)
						{					/* not completed  */
							tail[0] = *ptr;
							tail[1] = NUL;
							continue;
						}
						break;
					default:
						if (!kanji_map_sjis[*ptr])
							*dst ++ = *ptr ++;
						else
						{
							*dst = *ptr & 0x7f;
							ptr++;
							dst++;
						}
						break;
					}
					break;
				case JP_SJIS:
					c = *dst ++ = *ptr ++;
					if (kanji_map_sjis[c] & 1)
					{
						if (ptrlen >= 1 && *ptr != NUL)
						{
							* dst++ = * ptr++;
							ptrlen --;
						}
						else if (tail && ptrlen == 0)
						{
							tail[0] = c;
							tail[1] = NUL;
							dst--;
						}
						else
						{
							*(dst - 1) &= 0x7f;
						}
						continue;
					}
					else if (p_jkc && ISkana(c))
					{					/* JIS X 0201R 8bit encoding */
						char c1 = NUL;

						if (jisx0201rto0208(c, ptrlen ? *ptr : NUL, dst - 1, &c1))
						{	/* 2 characters -> double byte character. */
							* dst++ = c1;
							ptr ++;
							ptrlen --;
						}
						else if (c1)
							* dst++ = c1;
						continue;
					}
					break;
#ifdef UCODE
				case JP_UTF8:
					c = *ptr++;
					if (c < 0xc0)
					{ /* malformed */
						*dst++ = '?';
					}
					else if (c < 0xe0 && ptrlen >= 1)
					{
						short_u ucs = ((c & 0x1f) << 6) | (ptr[0] & 0x3f);
						int len;
						dst[0] = ucs & 255;
						dst[1] = ucs >> 8;
						len = wide2multi(dst, 2, FALSE);
						dst += len;
						ptr += 1;
						ptrlen--;
					}
					else if (c < 0xf0 && ptrlen >= 2)
					{
						short_u ucs = ((c & 0x0f) << 12) |
							((ptr[0] & 0x3f) << 6) | (ptr[1] & 0x3f);
						int len;
						dst[0] = ucs & 255;
						dst[1] = ucs >> 8;
						len = wide2multi(dst, 2, FALSE);
						dst += len;
						ptr += 2;
						ptrlen -= 2;
					}
					else if (c < 0xf8 && ptrlen >= 3)
					{
						ptr += 3;
						ptrlen -= 3;
						*dst++ = '?';
					}
					else if (c < 0xfc  && ptrlen >= 4)
					{
						ptr += 4;
						ptrlen -= 4;
						*dst++ = '?';
					}
					else if (c < 0xfe  && ptrlen >= 5)
					{
						ptr += 5;
						ptrlen -= 5;
						*dst++ = '?';
					}
					else
					{ /* malformed */
						*dst++ = '?';
					}
					if (tail && ptrlen == 0)
					{					/* not completed  */
						tail[0] = *ptr;
						tail[1] = NUL;
						continue;
					}
					break;
#endif /* UCODE */
				default:
					*dst ++ = *ptr ++;
					break;
				}
			}
			else
			{
				c = *dst ++ = *ptr ++;
#ifdef USE_OPT
				if (c == ESC && (code == JP_JIS || !(p_opt & OPT_NO_JIS)))
#else
				if (c == ESC)
#endif
				{
					dst --;

					if (ptrlen == 0)
					{
						if (tail && code == JP_JIS)
						{
							tail[0] = c;
							tail[1] = NUL;
						}
						else
							dst++;
						continue;
					}

					if (*ptr == '(')
					{
						ptrlen --;
						ptr ++;

						if (ptrlen == 0)
						{
							if (tail && code == JP_JIS)
							{
								tail[0] = c;
								tail[1] = '(';
								tail[2] = NUL;
							}
							else
							{
								dst ++;
								* dst++ = '(';
							}
							continue;
						}

						if (*ptr == 'I')
						{						/* JIS X 0201R ISO2022 encoding */
							charset = JP_KANA;
							ptr ++;
							ptrlen --;
							continue;
						}
						else if (*ptr == 'J' || *ptr == 'H' || *ptr == 'B')
						{						/* ASCII/JIS In */
							charset = JP_ASCII;
							ptr ++;
							ptrlen --;
							continue;
						}
						else
						{
							dst ++;
							* dst++ = '(';
							continue;
						}
					}

					if (*ptr == '$')
					{
						ptrlen --;
						ptr ++;

						if (ptrlen == 0)
						{
							if (tail && code == JP_JIS)
							{
								tail[0] = c;
								tail[1] = '$';
								tail[2] = NUL;
							}
							else
							{
								dst ++;
								* dst++ = '$';
							}
							continue;
						}

						if (*ptr == '@' || *ptr == 'B')		/* Kanji In */
						{
							charset = JP_KANJI;
							ptrlen --;
							ptr ++;
							continue;
						}
						else
						{
							dst ++;
							* dst ++ = '$';
							continue;
						}
					}
					else
						dst++;
				}
				switch (charset)
				{
					case JP_ASCII:
						break;

					case JP_KANA:
						if (p_jkc)
						{					/* JIS X 0201R 8bit encoding */
							char c1 = NUL;

							if (jisx0201rto0208(c, ptrlen ? *ptr : NUL, dst - 1, &c1))
							{	/* 2 characters -> double byte character. */
								* dst++ = c1;
								ptr ++;
								ptrlen --;
							}
							else if (c1)
								* dst++ = c1;
							continue;
						}
						else
						{
							dst[-1] |= 0x80;
						}
						break;

					default: /* JP_KANJI */
						if (ptrlen == 0)
						{
							if (tail && code == JP_JIS)
							{
								tail[0] = c;
								tail[1] = NUL;
								dst --;
							}
						}
						else if (c > ' ' && *ptr > ' ')
						{
							ptrlen --;
							dst[-1] = (jistosjis(ptr[-1], ptr[0]) & 0xff00) >> 8;
							dst[0]  = jistosjis(ptr[-1], ptr[0]) & 0xff;
							dst++;
							ptr++;
						}
						else if (c == NUL || c == '\n' || c == '\r')
							charset = JP_ASCII;
				}
			}
		}
		if (charsetp)
			*charsetp = charset;
		break; /* return dst - dtop; */
#ifdef UCODE
	case JP_WIDE:		/* UNICODE */
#endif
	default:
		while (ptrlen)
		{
			*dst ++ = *ptr ++;
			ptrlen --;
			if (dst - dtop >= dstlen)
				return -1;
		}
		break; /* return dst - dtop; */
	}
	return dst - dtop;
}

#ifdef UCODE
# define TO_UTF8(ucs, ptr) \
				if (ucs < 0x80) { *ptr ++ = ucs; } \
				else if (ucs < 0x800) { \
					*ptr ++ = (ucs >> 6) | 0xc0; \
					*ptr ++ = (ucs & 0x3f) | 0x80; \
				} else { \
					*ptr ++ = (ucs >> 12) | 0xe0; \
					*ptr ++ = ((ucs >> 6) & 0x3f) | 0x80; \
					*ptr ++ = (ucs & 0x3f) | 0x80; }
#endif

	char_u *
kanjiconvsto(ptr, code)
	char_u	*ptr;
	char_u	code;
{
	char_u	*top, *ptr2;
	char_u	*cp;
	int_u	kanji;
	int		nshift;

	if (ptr == NULL)
		return ptr;

	top = ptr;

	switch (code) {
	case JP_SJIS:
		nshift = STRLEN(ptr) + 1;
		top = alloc(nshift);
		memcpy((char *)top, (char *)ptr, nshift);
		top[nshift - 1] = '\0';
		return top;
#ifdef UCODE
	case JP_UTF8:
		for(nshift = 0; *ptr; ptr++)
		{
			switch (kanji_map_sjis[*ptr]) {
			case 1:
				nshift += 2;
				ptr++;
				break;
			case 2:
				nshift += 2;
				break;
			}
		}
		ptr = top;
		top = ptr2 = alloc(STRLEN(top) + nshift + 1);
		while (*ptr)
		{
			char_u buf[2];
			switch (kanji_map_sjis[*ptr]) {
			case 1:		/* kanji */
				buf[0] = ptr[0];
				buf[1] = ptr[1];
				multi2wide(buf, buf + 1, 2);
				kanji = buf[0] | (buf[1] << 8);
				TO_UTF8(kanji, ptr2);
				ptr += 2;
				break;
			case 2:		/* kana */
				buf[0] = ptr[0];
				buf[1] = '\0';
				multi2wide(buf, buf + 1, 1);
				kanji = buf[0] | (buf[1] << 8);
				TO_UTF8(kanji, ptr2);
				ptr += 1;
				break;
			default:	/* ascii */
				*ptr2 ++ = *ptr ++;
				break;
			}
		}
		*ptr2 = NUL;
		return top;
	case JP_WIDE:		/* UNICODE */
		for (nshift = 0; *ptr; ptr++)
		{
			switch (kanji_map_sjis[*ptr]) {
			case 1:
				ptr++;
				break;
			default:
				nshift++;
				break;
			}
		}
		ptr = top;
		top = ptr2 = alloc(STRLEN(top) + nshift + 2);
		while (*ptr)
			switch (kanji_map_sjis[*ptr]) {
			case 1:		/* kanji */
				ptr2[0] = ptr[0];
				ptr2[1] = ptr[1];
				multi2wide(&ptr2[0], &ptr2[1], 2);
				ptr  += 2;
				ptr2 += 2;
				break;
			default:
				ptr2[0] = ptr[0];
				ptr2[1] = 0;
				multi2wide(&ptr2[0], &ptr2[1], 1);
				ptr  += 1;
				ptr2 += 2;
				break;
			}
		ptr2[0] = ptr[0];
		ptr2[1] = 0;
		multi2wide(&ptr2[0], &ptr2[1], 1);
		return top;
#endif /* UCODE */
	case JP_EUC:
		for(nshift = 0; *ptr; ptr++)
		{
			switch (kanji_map_sjis[*ptr]) {
			case 1:
				ptr++;
				break;
			case 2:
				nshift++;
				break;
			}
		}
		ptr = top;
		top = ptr2 = alloc(STRLEN(top) + nshift + 1);
		while (*ptr)
			switch (kanji_map_sjis[*ptr]) {
			case 1:		/* kanji */
				kanji = sjistoeuc(ptr[0], ptr[1]);
				*ptr2 ++ = (kanji & 0xff00) >> 8;
				*ptr2 ++ = kanji & 0xff;
				ptr += 2;
				break;
			case 2:		/* kana */
				*ptr2 ++ = JP_EUC_G2;
				*ptr2 ++ = *ptr ++;
				break;
			default:	/* ascii */
				*ptr2 ++ = *ptr ++;
				break;
			}
		*ptr2 = NUL;
		return top;
	case JP_JIS:
		kanji = 0;
		for(nshift = 0; *ptr; ptr++)
		{
			switch (kanji_map_sjis[*ptr]) {
			case 1:
				ptr++;
				if (kanji != 1)
					nshift++;
				kanji = 1;
				break;
			case 2:
				if (kanji != 2)
					nshift++;
				kanji = 2;
				break;
			default:
				if (kanji != 0)
					nshift++;
				kanji = 0;
				break;
			}
		}
		if (nshift)
			nshift++;		/* eol code */

		ptr = top;
		top = ptr2 = alloc(STRLEN(top) + (nshift * 3 * 2/*safe*/) + 1);
		kanji = JP_ASCII;
		while (*ptr)
		{
			switch (kanji_map_sjis[*ptr]) {
			case 1:		/* kanji */
				cp = JPdisp(&kanji, JP_KANJI, code);
				for(; *cp;)
					*ptr2++ = *cp++;
				*ptr2 ++ = (sjistojis(ptr[0], ptr[1]) & 0xff00) >> 8;
				*ptr2 ++ = sjistojis(ptr[0], ptr[1]) & 0xff;
				ptr += 2;
				break;
			case 2:		/* kana */
				cp = JPdisp(&kanji, JP_KANA, code);
				for(; *cp;)
					*ptr2++ = *cp++;
				*ptr2 ++ = *ptr & 0x7f;
				ptr ++;
				break;
			default:	/* ascii */
				cp = JPdisp(&kanji, JP_ASCII, code);
				for(; *cp;)
					*ptr2++ = *cp++;
				*ptr2 ++ = *ptr ++;
				break;
			}
		}
		cp = JPdisp(&kanji, JP_ASCII, code);
		for(; *cp;)
			*ptr2++ = *cp++;
		*ptr2 = NUL;
		return top;
	}
	return top;
}

	char *
fileconvsfrom(org)
	char_u	*org;
{
	static char		fnamebuf[2][MAXPATHL];
	static int		cnt = 0;
	char		*	fname;
	int				jkc = p_jkc;
#ifdef MSDOS
	char_u		*	p;
	char_u		*	t;
	int				cygnus = FALSE;
#endif

	if (org == NULL)
		return(NULL);
	p_jkc = FALSE;
	fname = &fnamebuf[++cnt & 1][0];
	fname[kanjiconvsfrom(org, strlen(org),
					fname, MAXPATHL, NULL, toupper(JP_SYS), NULL)] = NUL;
#ifdef MSDOS
	t = p = (char_u *)fname;
	if (p[0] == '/' && p[1] == '/' && isalpha(p[2]) && p[3] == '/')
	{
		cygnus = TRUE;
		t[0] = p[2];
		t[1] = ':';
		t[2] = '/';
		memmove(&t[3], &p[4], strlen(p) - 3);
	}
	while (*p)
	{
		if (ISkanji(*p))
		{
			*t++ = *p++;
			*t++ = *p++;
		}
		else if (cygnus && p[0] == '\\')
			p++;
		else if (p != (char_u *)fname && p[0] == '\\' && p[1] == '\\')
			p++;
		else
			*t++ = *p++;
	}
	*t = '\0';
#endif
	p_jkc = jkc;
	return(fname);
}

	char *
fileconvsto(org)
	char_u	*org;
{
	static char		fnamebuf[2][MAXPATHL];
	static int		cnt = 0;
	char		*	fname;
	char_u		*	p;
#ifdef MSDOS
	char_u		*	t;
	int				cygnus = FALSE;
#endif

	if (org == NULL)
		return(NULL);
	fname = &fnamebuf[++cnt & 1][0];
#ifdef MSDOS
	strcpy(fname, org);
	t = p = (char_u *)fname;
	if (p[0] == '/' && p[1] == '/' && isalpha(p[2]) && p[3] == '/')
	{
		cygnus = TRUE;
		t[0] = p[2];
		t[1] = ':';
		t[2] = '/';
		memmove(&t[3], &p[4], strlen(p) - 3);
	}
	while (*p)
	{
		if (ISkanji(*p))
		{
			*t++ = *p++;
			*t++ = *p++;
		}
		else if (cygnus && p[0] == '\\')
			p++;
		else if (p != (char_u *)fname && p[0] == '\\' && p[1] == '\\')
			p++;
		else
			*t++ = *p++;
	}
	*t = '\0';
	org = fname;
#endif
	p = kanjiconvsto(org, toupper(JP_SYS));
	strcpy(fname, p);
	free(p);
	return(fname);
}

	void
binaryconvsfrom(lnum, code, tail, ptr, len, dst)
	linenr_t lnum;
	char	code;
	int		*tail;
	char_u	*ptr;
	int		len;
	char_u	*dst;
{
	char_u	*wk = ptr;
	int		i;

	memset(dst, '\0', 81);
	sprintf(dst, "%08x: ", lnum * 16);
	dst += 10;

	for (i = 0; i < 16; ++i)
	{
		if (len <= i)
		{
			*dst++ = ' ';
			*dst++ = ' ';
		}
		else
		{
			sprintf(dst, "%02x", *wk);
			dst += 2;
			++wk;
		}
		if (i & 1)
			*dst++ = ' ';
	}
	*dst++ = ';';
	for (i = 0; i < len;)
	{
		if (*tail)
		{
			*tail = FALSE;
			goto normal;
		}
		switch (code) {
		case JP_EUC:
			if ((kanji_map_euc[*ptr] & 1) && (0xa1 <= ptr[1] && ptr[1] <= 0xfe))
			{
				*dst++ = (euctosjis(ptr[0], ptr[1]) & 0xff00) >> 8;
				*dst++ = euctosjis(ptr[0], ptr[1]) & 0xff;
				ptr += 2;
				i += 2;
				if (i > len)
					*tail = TRUE;
			}
			else if ((kanji_map_euc[*ptr] & 2) && (0xa1 <= ptr[1] && ptr[1] <= 0xdf))
			{
				*dst++ = '.';
				*dst++ = ptr[1];
				ptr += 2;
				i += 2;
				if (i > len)
					*tail = TRUE;
			}
			else
				goto normal;
			break;
		case JP_SJIS:
			if ((kanji_map_sjis[*ptr] & 1)
					&& ((0x40 <= ptr[1] && ptr[1] <= 0x7e) || (0x80 <= ptr[1] && ptr[1] <= 0xfc)))
			{
				*dst++ = *ptr++;
				*dst++ = *ptr++;
				i += 2;
				if (i > len)
					*tail = TRUE;
			}
			else if (kanji_map_sjis[*ptr] & 2)
			{
				*dst++ = *ptr++;
				i += 1;
			}
			else
				goto normal;
			break;
#ifdef NT
		case JP_WIDE:		/* UNICODE */
			{
				int			wlen;
				char_u		buf[2];
				int			nochar;

				if (ptr[0] == 0xff && ptr[1] == 0xfe)
				{
					*tail = TRUE;
					goto normal;
				}
				wlen = WideCharToMultiByte(p_cpage, WC_COMPOSITECHECK, (LPCWSTR)ptr, 1, (LPSTR)buf, 2, NULL, &nochar);
				if (nochar)
					goto normal;
				if (wlen == 2)
				{
					*dst++ = buf[0];
					*dst++ = buf[1];
				}
				else if (buf[0] < ' ')
				{
					*dst++ = '.';
					*dst++ = '.';
				}
				else
				{
					*dst++ = buf[0];
					*dst++ = '.';
				}
				ptr += 2;
				i += 2;
				if (i > len)
					*tail = TRUE;
			}
			break;
#endif
		default:
normal:
			if (*ptr < ' ' || 0x7f <= *ptr)
				*dst++ = '.';
			else
				*dst++ = *ptr;
			ptr++;
			i++;
			break;
		}
	}
}

	char_u *
binaryconvsto(code, ptr, len)
	char	code;
	char_u	*ptr;
	int		*len;
{
	char_u				value = 0;
	int					cnt = 0;
	char_u				*top;
	int					quote = FALSE;

	*len = 0;
	if ((strchr(ptr, ':') > strchr(ptr, '"')) && strchr(ptr, '"') != NULL)
		;
	else if ((top = strchr(ptr, ':')) != NULL)
		ptr = top + 1;
	top = alloc(STRLEN(ptr) + 1);

	while (*ptr)
	{
		if (quote)
		{
			if (*ptr == '"')
				quote = FALSE;
			else if (ptr[0] == '\\' && ptr[1] == '"')
			{
				top[(*len)++] = '"';
				ptr++;
			}
			else if (code == JP_JIS)
				top[(*len)++] = *ptr;
			else
			{
				char_u		buf[3];
				char_u		*tmpptr;
				char_u		*p;

				if (ISkanji(*ptr))
				{
					buf[0] = ptr[0];
					buf[1] = ptr[1];
					buf[2] = NUL;
					p = tmpptr = kanjiconvsto(buf, code);
					while (*p)
						top[(*len)++] = *p++;
					free(tmpptr);
					ptr++;
				}
				else if (ISkana(*ptr))
				{
					buf[0] = ptr[0];
					buf[1] = NUL;
					p = tmpptr = kanjiconvsto(buf, code);
					while (*p)
						top[(*len)++] = *p++;
					free(tmpptr);
				}
				else
					top[(*len)++] = *ptr;
			}
		}
		else
		{
			switch (*ptr) {
			case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
				value = (value << 4) + (*ptr - '0');
				cnt++;
				break;
			case 'a': case 'A':
				value = (value << 4) + 10
								+ (*ptr == 'a' ? *ptr - 'a' : *ptr - 'A');
				cnt++;
				break;
			case 'b': case 'B':
				value = (value << 4) + 10
								+ (*ptr == 'b' ? *ptr - 'a' : *ptr - 'A');
				cnt++;
				break;
			case 'c': case 'C':
				value = (value << 4) + 10
								+ (*ptr == 'c' ? *ptr - 'a' : *ptr - 'A');
				cnt++;
				break;
			case 'd': case 'D':
				value = (value << 4) + 10
								+ (*ptr == 'd' ? *ptr - 'a' : *ptr - 'A');
				cnt++;
				break;
			case 'e': case 'E':
				value = (value << 4) + 10
								+ (*ptr == 'e' ? *ptr - 'a' : *ptr - 'A');
				cnt++;
				break;
			case 'f': case 'F':
				value = (value << 4) + 10
								+ (*ptr == 'f' ? *ptr - 'a' : *ptr - 'A');
				cnt++;
				break;
			case ';':
				if (cnt == 1)
				{
					top[(*len)++] = value;
					cnt = 0;
					value = 0;
				}
				return(top);
			case '"':
				if (cnt == 1)
				{
					top[(*len)++] = value;
					cnt = 0;
					value = 0;
				}
				quote = TRUE;
				break;
			case ' ':
			case '\t':
			case '\n':
			case '\r':
			default:
				if (cnt == 1)
				{
					top[(*len)++] = value;
					cnt = 0;
					value = 0;
				}
				break;
			}
			if (cnt == 2)
			{
				top[(*len)++] = value;
				cnt = 0;
				value = 0;
			}
		}
		ptr++;
	}
	return(top);
}

/*
 * JIS X 0201 Right hand set(hankaku kana) <-> JIS X 0208(zenkaku)
 *
 */
	static int
jisx0201rto0208(src0, src1, dst0, dst1)
	char_u src0, src1, *dst0, *dst1;
{
	char_u	c, y;
	char_u	*x0201p, z;
	int		conv;

	src0 |= 0x80;
	src1 |= 0x80;
	c = (char_u)src0;
	x0201p = (char_u *)jisx0201r + 2 * (c - 0xa0);

	if (! ISkanji(y = *x0201p))
	{
		*dst0 = y;
		*dst1 = NUL;
		return FALSE;
	}

	z = *(x0201p + 1);
	conv = FALSE;
	if		((char_u)src1 == 0xdf &&		/* maru */
				(c >= 0xca && c <= 0xce))	/* ha - ho */
	{
		z += 2;
		conv = TRUE;
	}
	else if ((char_u)src1 == 0xde)			/* dakuten */
	{
		conv = TRUE;
		if (   (c >= 0xb6 && c <= 0xc4)		/* ka - to */
			|| (c >= 0xca && c <= 0xce) )	/* ha - ho */
			z ++;
		else if (c == 0xb3)					/* u -> vu*/
			z = 0x94;
		else
			conv = FALSE;
	}

	*dst0 = y;
	*dst1 = z;

	return conv ? TRUE : FALSE;
}

/*
 * compare two strings, ignoring case
 * return 0 for match, 1 for difference
 */
	int
jp_strnicmp(s1, s2, len)
	char_u	*s1;
	char_u	*s2;
	size_t	len;
{
	char_u	*	str[2];
	char_u		wrk[2][2];
	int			slen[2];
	char_u	*	s;
	char_u	*	w;
	int		*	l;
	size_t		wlen;
	int			i;
	int			class;
	int			flg = TRUE;
	size_t		tlen = len;

	wlen = len;
	s = s1;
	w = s2;
	while (wlen)
	{
		if ((*s == '\t' && *w == ' ') || (*s == ' ' && *w == '\t'))
			;
		else if (TO_UPPER(*s) != TO_UPPER(*w))
		{
			flg = FALSE;
			break;
		}
		if (*s == NUL)
			break;
		++s;
		++w;
		--wlen;
	}
	if (flg)
		return tlen;						/* strings match */
	str[0]	= s1;
	str[1]	= s2;
	tlen	= 0;
	while (len > 0)
	{
		if (*str[1] == NUL)
			return tlen;
		if (*str[0] == NUL)
			return 0;
		for (i = 0; i < 2; i++)
		{
			s = str[i];
			w = &wrk[i][0];
			l = &slen[i];
			*l = 1;
			if (ISkanji(*s))
			{
				*l = 2;
				w[0] = *s;
				w[1] = *(s + 1);
			}
			else if (ISkana(*s))
			{
				if (jisx0201rto0208(*s, *(s + 1), &w[0], &w[1]))
					*l = 2;
			}
			else if (isalpha(*s))
			{
				if (islower(*s))
				{
					w[0] = 0x82;
					w[1] = 0x60 + (*s - 'a');
				}
				else
				{
					w[0] = 0x82;
					w[1] = 0x60 + (*s - 'A');
				}
			}
			else if (isdigit(*s))
			{
				w[0] = 0x82;
				w[1] = 0x4f + (*s - '0');
			}
			else
			{
				char_u	**	cnv = (char_u **)Asconv;
				char	*	c;

				while (*cnv[0])
				{
					c = *cnv;
					if (c[0] == *s)
					{
						w[0] = c[1];
						w[1] = c[2];
						break;
					}
					cnv++;
				}
			}
			class = jpcls(w[0], w[1]);
			if (class == JPC_HIRA || class == JPC_ALNUM)
				jptocase(&w[0], &w[1], UPPER);
		}
		if (!(wrk[0][0] == wrk[1][0] && wrk[0][1] == wrk[1][1]))
			return 0;
		str[0] += slen[0];
		str[1] += slen[1];
		len -= slen[1];
		tlen += slen[0];
	}
	return tlen;							/* strings match */
}

#ifdef UCODE
int ucs2sjis __ARGS((unsigned short ucs, unsigned char* sjis));
void sjis2ucs __ARGS((unsigned char* sjis, int len, unsigned char* ucs));

	int
wide2multi(ptr, size, first)
	char_u	*	ptr;
	int			size;
	int			first;
{
	char_u	*	p = ptr;
	char_u		buf[2];
	int			len;
	int			total = 0;
	int			i;
	static int	rev = 0;

	if (first)
	{
		short_u		w = 0xfeff;		/* BOM */

		memmove((char *)buf, &w, 2);
		rev = 1;
		if (p[0] == buf[0] && p[1] == buf[1])
			rev = 0;
		p += 2;
		size -= 2;
	}
	for (i = 0; i < size; i += 2)
	{
		if (rev) {
			buf[0]	= p[0];
			p[0]	= p[1];
			p[1]	= buf[0];
		}
#ifdef NT
		len = WideCharToMultiByte(p_cpage, WC_COMPOSITECHECK, (LPCWSTR)p, 1, (LPSTR)buf, 2, NULL, NULL);
#else
		len = ucs2sjis((*(p+1) << 8) | *p, buf);
#endif
		memmove((char *)&ptr[total], buf, len);
		p += 2;
		total += len;
	}
	return(total);
}

	void
multi2wide(k1, k2, len)
	char_u	*	k1;
	char_u	*	k2;
	int			len;
{
	char_u	buf[2];
	char_u	wbuf[2/*sizeof(WCHAR)*/];

	buf[0] = *k1;
	buf[1] = *k2;
#ifdef NT
	MultiByteToWideChar(p_cpage, MB_PRECOMPOSED, (LPSTR)buf, len, (LPWSTR)wbuf, 1);
#else
	sjis2ucs(buf, len, wbuf);
#endif
	*k1 = wbuf[0];
	*k2 = wbuf[1];
}
#endif /* UCODE */
#endif	/* KANJI */
