/*-------------------------------------------------------------------------*/
/* selection.c --- xcircuit routines handling element selection etc.	   */
/* Copyright (c) 1998  Tim Edwards, Johns Hopkins University       	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "Xw/Xw.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

extern short eventmode;	/* keep track of the mode the screen is in */
extern Display	*dpy;   /* Works well to make this globally accessible */
extern Window win;
extern int *appcolors;
extern Cursor	appcursors[NUM_CURSORS];
extern Clientdata areastruct;
extern char _STR[150];
extern short attachto;
extern short textpos, textend;
extern long finddist();

#define RADFAC 0.0174532925199   /* pi / 180 */

/*------------------------------------------------------------------------*/
/* Declarations of functions defined in selection.c	 	  	  */
/*------------------------------------------------------------------------*/

short *selectobject(short, objectptr);
XtEventHandler trackselarea();

/*----------------------------------------------------------------------------*/
/* Look at select stack to see if there are any selects; call select, if not. */
/*----------------------------------------------------------------------------*/

Boolean checkselect(value)
  short value;
{
   short *check;

   if (areastruct.selects == 0) objectselect(value);
   if (areastruct.selects == 0) return False;
   for (check = areastruct.selectlist; check < areastruct.selectlist +
	areastruct.selects; check++)
      if (SELECTTYPE(check) & value) break;
   if (check == areastruct.selectlist + areastruct.selects) return False;
   else return True;
}

/*----------------------*/
/* Draw a selected item */
/*----------------------*/

void geneasydraw(short instance, short mode, objectptr curobj)
{
   genericptr elementptr = *(curobj->plist + instance);

   switch ((*(curobj->plist + instance))->type) {
      case OBJECT:
         UDrawObject((objinstptr)elementptr, 
	    ((objinstptr)elementptr)->thisobject, SINGLE, mode);
         break;
      case LABEL:
         UDrawString(areastruct.topobject, (labelptr)elementptr, 0);
	 break;
      case ARC:
         UDrawArc(areastruct.topobject, (arcptr)elementptr);
	 break;
      case POLYGON:
         UDrawPolygon(areastruct.topobject, (polyptr)elementptr);
	 break;
      case SPLINE:
	 UDrawSpline(areastruct.topobject, (splineptr)elementptr);
	 break;
      case PATH:
	 UDrawPath(areastruct.topobject, (pathptr)elementptr);
   }
}

/*-------------------------------------------------*/
/* Draw a selected item, including selection color */
/*-------------------------------------------------*/

void gendrawselected(short *newselect, objectptr curobj)
{
   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, BACKGROUND);
   geneasydraw(*newselect, DOFORALL, curobj);
   XSetFunction(dpy, areastruct.gc, GXxor);
   XSetForeground(dpy, areastruct.gc, SELECTCOLOR ^ BACKGROUND);
   geneasydraw(*newselect, DOFORALL, curobj);
}

/*---------------------------------------------------*/
/* Allocate or reallocate memory for a new selection */
/*---------------------------------------------------*/

short *allocselect()
{
   short *newselect;

   if (areastruct.selects == 0)
      areastruct.selectlist = (short *) malloc(sizeof(short));
   else
      areastruct.selectlist = (short *) realloc(areastruct.selectlist,
	 (areastruct.selects + 1) * sizeof(short));

   newselect = areastruct.selectlist + areastruct.selects;
   areastruct.selects++;

   return newselect;
}

/*-------------------------------------------------*/
/* Set Options menu according to 1st selection	   */
/*-------------------------------------------------*/

setoptionmenu()
{
   short      *mselect;
   labelptr   mlabel;

   if (areastruct.selects == 0) {
      setallstylemarks(areastruct.style);
      setcolormark(areastruct.color);
      setdefaultfontmarks();
      return;
   }

   for (mselect = areastruct.selectlist; mselect < areastruct.selectlist +
	areastruct.selects; mselect++) {
      setcolormark(SELTOCOLOR(mselect));
      switch(SELECTTYPE(mselect)) {
	 case ARC:
	    setallstylemarks(SELTOARC(mselect)->style);
	    return;
	 case POLYGON:
	    setallstylemarks(SELTOPOLY(mselect)->style);
	    return;
	 case SPLINE:
            setallstylemarks(SELTOSPLINE(mselect)->style);
	    return;
	 case PATH:
            setallstylemarks(SELTOPATH(mselect)->style);
	    return;
	 case LABEL:
	    mlabel = SELTOLABEL(mselect);
	    setfontmarks((unsigned short)(*(mlabel->string + 1)) - FONT_START,
		mlabel->justify);
	    return;
      }
   }
}

/*--------------------------------------------*/
/* Search for selection among path components */
/*--------------------------------------------*/

Boolean pathselect(curgen, class)
  genericptr *curgen;
  short class;
{
   /*----------------------------------------------------------------------*/
   /* wirelim is the distance, in user-space units, at which an element is */
   /* considered for selection.						   */
   /*									   */
   /* wirelim = A + B / (scale + C)					   */
   /*									   */
   /* where A = minimum possible distance (expands range at close scale)   */
   /*       C = minimum possible scale    (contract range at far scale)	   */
   /*	    B   makes wirelim approx. 25 at default scale of 0.5, which	   */
   /*		is an empirical result.					   */
   /*----------------------------------------------------------------------*/

   float	wirelim = 2 + 11.5 / (*areastruct.vscale + 0.05);
   long		sqrwirelim = (int)(wirelim * wirelim);

   long		newdist;
   Boolean	selected = False;

   if ((*curgen)->type == (class & ARC)) {

      /* look among the arcs */

      fpointlist currentpt;
      XPoint nearpt[3];

      nearpt[2].x = nearpt[0].x = (short)(TOARC(curgen)->points[0].x);
      nearpt[2].y = nearpt[0].y = (short)(TOARC(curgen)->points[0].y);
      for (currentpt = TOARC(curgen)->points + 1; currentpt < TOARC(curgen)->points
              + TOARC(curgen)->number; currentpt++) {
	 nearpt[1].x = nearpt[0].x;
	 nearpt[1].y = nearpt[0].y;
	 nearpt[0].x = (short)(currentpt->x);
	 nearpt[0].y = (short)(currentpt->y);
	 newdist = finddist(&nearpt[0], &nearpt[1], &areastruct.save);
         if (newdist <= sqrwirelim) break;
      }
      if ((!(TOARC(curgen)->style & UNCLOSED)) && (newdist > sqrwirelim))
	 newdist = finddist(&nearpt[0], &nearpt[2], &areastruct.save);

      if (newdist <= sqrwirelim) selected = True;
   }

   else if ((*curgen)->type == (class & SPLINE)) {

      /* look among the splines --- look at polygon representation */

      fpointlist currentpt;
      XPoint nearpt[2];

      nearpt[0].x = (short)(TOSPLINE(curgen)->points[0].x);
      nearpt[0].y = (short)(TOSPLINE(curgen)->points[0].y);
      newdist = finddist(&(TOSPLINE(curgen)->ctrl[0]), &(nearpt[0]),
		   &areastruct.save);
      if (newdist > sqrwirelim) {
         for (currentpt = TOSPLINE(curgen)->points; currentpt <
		  TOSPLINE(curgen)->points + INTSEGS; currentpt++) {
	    nearpt[1].x = nearpt[0].x;
	    nearpt[1].y = nearpt[0].y;
	    nearpt[0].x = (short)(currentpt->x);
	    nearpt[0].y = (short)(currentpt->y);
	    newdist = finddist(&nearpt[0], &nearpt[1], &areastruct.save);
            if (newdist <= sqrwirelim) break;
	 }
	 if (newdist > sqrwirelim) {
	    newdist = finddist(&nearpt[0], &(TOSPLINE(curgen)->ctrl[3]),
			&areastruct.save);
            if ((!(TOSPLINE(curgen)->style & UNCLOSED)) && (newdist > sqrwirelim))
	       newdist = finddist(&(TOSPLINE(curgen)->ctrl[0]),
		     &(TOSPLINE(curgen)->ctrl[3]), &areastruct.save);
	 }
      }

      if (newdist <= sqrwirelim) selected = True;
   }

   else if ((*curgen)->type == (class & POLYGON)) {

      /* finally, look among the polygons */

      pointlist currentpt;

      for (currentpt = TOPOLY(curgen)->points; currentpt < TOPOLY(curgen)
		->points + TOPOLY(curgen)->number - 1; currentpt++) {
	 newdist = finddist(currentpt, currentpt + 1, &areastruct.save);
	 if (newdist <= sqrwirelim) break;
      }
      if ((!(TOPOLY(curgen)->style & UNCLOSED)) && (newdist > sqrwirelim))
	 newdist = finddist(currentpt, TOPOLY(curgen)->points,
		&areastruct.save);

      if (newdist <= sqrwirelim) selected = True;
   }
   return selected;
}

/*--------------------------------------*/
/* Select one of the elements on-screen */
/*--------------------------------------*/

short *selectobject(short class, objectptr selobj)
{
   short	*newselect, *foundtmp, *foundlist;

   genericptr	*curgen;

   XPoint 	bboxpts[4], newboxpts[4];
   XEvent 	event;
   XButtonEvent *buttonevent = (XButtonEvent *)(&event);
   Boolean	selected, unselect = False;
   short	stval, i, j, found = 0;
   long		newdist;
   Arg		wargs[2];

   XDefineCursor(dpy, win, QUESTION);

   /* Definition for unselecting an element */

   if (class < 0) {
      unselect = True;
      class = -class;
   }

   /* Loop through all elements found underneath the cursor */

   for (curgen = selobj->plist; curgen < selobj->plist + selobj->parts; curgen++) {

      /* when selection element to attach to, don't select the element */
      /* to be attached. . .					       */

      if (attachto == 1 && eventmode != COPY2_MODE && eventmode !=
	    PRESS_MODE && curgen == EDITPART) continue;

      selected = False;

      /* check for position inside the bounding box of one of the objects */

      if ((*curgen)->type == (class & OBJECT)) {

#ifdef SCHEMA
#define lowerleft lleft2
#define width width2
#define height height2
#endif

         bboxpts[0].x = bboxpts[1].x = TOOBJINST(curgen)->thisobject->lowerleft.x;
         bboxpts[2].x = bboxpts[3].x = TOOBJINST(curgen)->thisobject->lowerleft.x + 
	    TOOBJINST(curgen)->thisobject->width;
         bboxpts[0].y = bboxpts[3].y = TOOBJINST(curgen)->thisobject->lowerleft.y;
         bboxpts[1].y = bboxpts[2].y = TOOBJINST(curgen)->thisobject->lowerleft.y + 
	    TOOBJINST(curgen)->thisobject->height;
         UTransformPoints(bboxpts, newboxpts, 4, TOOBJINST(curgen)->position,
		TOOBJINST(curgen)->scale, TOOBJINST(curgen)->rotation);

#ifdef SCHEMA
#undef lowerleft
#undef width
#undef height
#endif
         /* Look for an intersect of the boundingbox and pointer position. */
         /* This is a simple matter of rotating the pointer position with  */
         /*  respect to the origin of the bounding box segment, as if the  */
         /*  segment were rotated to 0 degrees.  The sign of the resulting */
         /*  point's y-position is the same for all bbox segments if the   */
         /*  pointer position is inside the bounding box.		   */

         stval = sign((newboxpts[0].x - newboxpts[3].x) * (areastruct.save.y - 
	    newboxpts[3].y) - (newboxpts[0].y - newboxpts[3].y) * (areastruct.save.x 
	    - newboxpts[3].x));

         for (j = 1; j < 4; j++) {
	    stval += sign((newboxpts[j].x - newboxpts[j - 1].x) * (areastruct.save.y
	    - newboxpts[j - 1].y) - (newboxpts[j].y - newboxpts[j - 1].y) * 
	    (areastruct.save.x - newboxpts[j - 1].x));
         }

         if (abs(stval) == 4) selected = True;
      }

      else if ((*curgen)->type == (class & LABEL)) {

         /* now look among the labels */

	 short tmplength = ULength(TOLABEL(curgen)->string, 0.0, 0, 0);

         bboxpts[0].x = bboxpts[1].x = (TOLABEL(curgen)->justify & NOTLEFT ?
	    (TOLABEL(curgen)->justify & RIGHT ? -tmplength : -tmplength / 2) : 0);
	 bboxpts[2].x = bboxpts[3].x = bboxpts[0].x + tmplength;
         bboxpts[0].y = bboxpts[3].y = (TOLABEL(curgen)->justify & NOTBOTTOM ?
	    (TOLABEL(curgen)->justify & TOP ? -TEXTHEIGHT : -HALFHEIGHT) : 0);
	 bboxpts[1].y = bboxpts[2].y = bboxpts[0].y + TEXTHEIGHT;

#ifdef SCHEMA
	 if (TOLABEL(curgen)->pin) {
	     int i;
	     for (i = 0; i < 4; i++) 
		pinadjust(TOLABEL(curgen)->justify, &(bboxpts[i].x),
			&(bboxpts[i].y), 1);
	 }
#endif

         UTransformPoints(bboxpts, newboxpts, 4, TOLABEL(curgen)->position,
		TOLABEL(curgen)->scale, TOLABEL(curgen)->rotation);

         /* check for point inside bounding box, as for objects */

         stval = sign((newboxpts[0].x - newboxpts[3].x) * (areastruct.save.y -
            newboxpts[3].y) - (newboxpts[0].y - newboxpts[3].y) * (areastruct.save.x 
	    - newboxpts[3].x));

         for (j = 1; j < 4; j++)
            stval += sign((newboxpts[j].x - newboxpts[j - 1].x) * (areastruct.save.y
            - newboxpts[j - 1].y) - (newboxpts[j].y - newboxpts[j - 1].y) *
            (areastruct.save.x - newboxpts[j - 1].x));

         if (abs(stval) == 4) selected = True;
      }


      else if ((*curgen)->type == (class & PATH)) {

         /* Check among the paths */

	 genericptr *pathp;
	 Boolean lselect;

	 /* Accept path if any subcomponent of the path is accepted */
	 /* Record which part was selected in the "subeditpart" variable */

 	 for (pathp = TOPATH(curgen)->plist; pathp < TOPATH(curgen)->plist +
		TOPATH(curgen)->parts; pathp++)
	    if (pathselect(pathp, SPLINE|ARC|POLYGON)) {
	       selected = True;
	       areastruct.editsubpart = (short)(pathp - TOPATH(curgen)->plist);
	       break;
	    }
      }

      /* Check among polygons, arcs, and curves */

      else selected = pathselect(curgen, class);

      /* check if this object has already been selected */

      for (newselect = areastruct.selectlist; newselect <
             areastruct.selectlist + areastruct.selects; newselect++) 
         if (*newselect == (short)(curgen - selobj->plist)) {
	    if (!unselect) selected = False;
	    break;
	 }

      if (unselect && (newselect == areastruct.selectlist + areastruct.selects))
	 selected = False;

      /* Add this object to the list of things found under the cursor */

      if (selected && (!unselect)) {

         if (found == 0)
            foundlist = (short *) malloc(sizeof(short));
         else
            foundlist = (short *) realloc(foundlist, (found + 1) * sizeof(short));

         newselect = foundlist + found;
         found++;
     
         if (selected) *newselect = (short)(curgen - selobj->plist);
      }

      /* mechanism for unselecting elements under the cursor */

      else if (selected) {
         short *desel;

         XSetFunction(dpy, areastruct.gc, GXcopy);
         XTopSetForeground(GSELTOCOLOR(selobj, newselect));
         geneasydraw(*newselect, DEFAULTCOLOR, selobj);
         areastruct.selects--;
         for (desel = newselect; desel < areastruct.selectlist +
	        areastruct.selects; desel++)
	    *desel = *(desel + 1);
         if (areastruct.selects == 0) free (areastruct.selectlist);
      }
      else newselect = NULL;

   } /* for-loop over all parts */

   /* Various options depending on how many things were found */

   switch (found) {
      case 0: 
	 newselect = NULL;
	 break;

      case 1: 
	 newselect = allocselect();
	 *newselect = *foundlist;
	 free(foundlist);
	 gendrawselected(newselect, selobj);
	 break;

      default: {
	 XSetFunction(dpy, areastruct.gc, GXcopy);
	 for (foundtmp = foundlist; foundtmp < foundlist + found; foundtmp++) {
            XSetForeground(dpy, areastruct.gc, QUERYCOLOR);
            geneasydraw(*foundtmp, DOFORALL, selobj);
	    sprintf(_STR, "Click to accept/reject: %d of %d",
		(int)(foundtmp - foundlist + 1), found);
	    Wprintf(_STR);

	    for(;;) {  	/* This flushes the X event queue */
	       XNextEvent(dpy, &event); 
	       if (event.type != ButtonPress && event.type != ButtonRelease
		    && event.type != MotionNotify && event.type != KeyPress)
	          XtDispatchEvent(&event);
	       else if (event.type == ButtonPress) break;
	    }
	    if (buttonevent->button == Button3) {
               XTopSetForeground(GSELTOCOLOR(selobj, foundtmp));
               geneasydraw(*foundtmp, DEFAULTCOLOR, selobj);
	    }
	    else {
	       newselect = allocselect();
               *newselect = *foundtmp;
               XSetForeground(dpy, areastruct.gc, SELECTCOLOR);
               geneasydraw(*newselect, DOFORALL, selobj);
	    }
         }
      }  break;
   }

   if (found > 1) {
      short *nselect;

      XDefineCursor(dpy, win, CIRCLE);
      if (eventmode == PRESS_MODE)
	 Wprintf("Click and hold button 1 to continue move.");
      else
	 Wprintf("Selection complete");

      for(;;) {
	 XNextEvent(dpy, &event); 
	 if (event.type != ButtonPress && event.type != ButtonRelease &&
		event.type != MotionNotify && event.type != KeyPress)
	    XtDispatchEvent(&event);
	 else if ((event.type == ButtonPress) && (eventmode == PRESS_MODE))
	    break;
	 else if ((event.type == ButtonRelease) && (eventmode != PRESS_MODE))
	    break;
      }
      free(foundlist);

      /* make it work right for smooth xor'ing */

      XSetFunction(dpy, areastruct.gc, GXcopy);
      XSetForeground(dpy, areastruct.gc, BACKGROUND);
      for (nselect = areastruct.selectlist; nselect < areastruct.selectlist
	   + areastruct.selects; nselect++)
         geneasydraw(*nselect, DOFORALL, selobj);
      XSetFunction(dpy, areastruct.gc, GXxor);
      XSetForeground(dpy, areastruct.gc, SELECTCOLOR ^ BACKGROUND);
      for (nselect = areastruct.selectlist; nselect < areastruct.selectlist
	   + areastruct.selects; nselect++)
         geneasydraw(*nselect, DOFORALL, selobj);
   }

   setoptionmenu();

   if (eventmode == DELETE_MODE)
      XDefineCursor(dpy, win, SCISSORS);
   else
      XDefineCursor(dpy, win, CROSS);

   /* snap the coordinates */

   u2u_snap(&areastruct.save);
   return newselect;
}

/*----------------------------------------------------------------*/
/* select arc, curve, and polygon objects from a defined box area */
/*----------------------------------------------------------------*/

Boolean areaelement(curgen)
  genericptr *curgen;
{
   Boolean selected;
   pointlist    currentpt;

   switch((*curgen)->type) {

      case(ARC):
	   /* check center of arcs */

      	   selected = (TOARC(curgen)->position.x < areastruct.save.x) &&
	  	(TOARC(curgen)->position.x > areastruct.origin.x) &&
		(TOARC(curgen)->position.y < areastruct.save.y) &&
		(TOARC(curgen)->position.y > areastruct.origin.y);
	   break;

      case(POLYGON):
	   /* check each point of the polygons */

	   selected = False;
           for (currentpt = TOPOLY(curgen)->points; currentpt <
		TOPOLY(curgen)->points + TOPOLY(curgen)->number; currentpt++) {
              if ((currentpt->x < areastruct.save.x) && (currentpt->x >
	           areastruct.origin.x) && (currentpt->y < areastruct.save.y) &&
	           (currentpt->y > areastruct.origin.y)) {
	         selected = True;
		 break;
	      }
           }
	   break;

      case(SPLINE):
	   /* check each control point of the spline */
	
           selected = ((TOSPLINE(curgen)->ctrl[0].x < areastruct.save.x) && 
	   	(TOSPLINE(curgen)->ctrl[0].x > areastruct.origin.x) &&
		(TOSPLINE(curgen)->ctrl[0].y < areastruct.save.y) &&
		(TOSPLINE(curgen)->ctrl[0].y > areastruct.origin.y))
		|| ((TOSPLINE(curgen)->ctrl[3].x < areastruct.save.x) &&
		(TOSPLINE(curgen)->ctrl[3].x > areastruct.origin.x) &&
		(TOSPLINE(curgen)->ctrl[3].y < areastruct.save.y) &&
		(TOSPLINE(curgen)->ctrl[3].y > areastruct.origin.y));
	   break;
   }
   return selected;
}

/*--------------------------------------------*/
/* select all objects from a defined box area */
/*--------------------------------------------*/

void selectarea()
{
   short	*newselect;
   genericptr   *curgen, *pathgen;
   Boolean	selected;

   /* put points of area bounding box into proper order */

   if (areastruct.origin.y > areastruct.save.y) {
      short tmp;
      tmp = areastruct.origin.y;
      areastruct.origin.y = areastruct.save.y;
      areastruct.save.y = tmp;
   }
   if (areastruct.origin.x > areastruct.save.x) {
      short tmp;
      tmp = areastruct.origin.x;
      areastruct.origin.x = areastruct.save.x;
      areastruct.save.x = tmp;
   }
 
   for (curgen = objectdata->plist; curgen < objectdata->plist
	+ objectdata->parts; curgen++) {

      switch((*curgen)->type) {
	case(OBJECT):
           /* check for instance of object inside area box */
           selected = (TOOBJINST(curgen)->position.x < areastruct.save.x) &&
		(TOOBJINST(curgen)->position.x > areastruct.origin.x) &&
		(TOOBJINST(curgen)->position.y < areastruct.save.y) &&
		(TOOBJINST(curgen)->position.y > areastruct.origin.y);
	   break;

        case(LABEL): {
	   XPoint urpt, llpt;
	   labelptr slab = TOLABEL(curgen);
	   short tmplength;
	   selected = False;

   	   /* translate select box into the coordinate system of the label */
	   InvTransformPoints(&areastruct.origin, &llpt, 1, slab->position,
		  slab->scale, slab->rotation);
	   InvTransformPoints(&areastruct.save, &urpt, 1, slab->position,
		  slab->scale, slab->rotation);
#ifdef SCHEMA
	   if (slab->pin) {
	      pinadjust(slab->justify, &llpt.x, &llpt.y, -1);
	      pinadjust(slab->justify, &urpt.x, &urpt.y, -1);
	   }
#endif
	   if (urpt.y < llpt.y) {
	      short tmpval = urpt.y;
	      urpt.y = llpt.y;
	      llpt.y = tmpval;
	   }
	   if (urpt.x < llpt.x) {
	      short tmpval = urpt.x;
	      urpt.x = llpt.x;
	      llpt.x = tmpval;
	   } 
	   if (urpt.y > 0 && llpt.y < TEXTHEIGHT) {
	      short xadj;
	      tmplength = ULength(slab->string, 0.0, 0, 0);
	      xadj = (slab->justify & NOTLEFT ? (slab->justify & RIGHT ?
		        tmplength : tmplength >> 1) : 0);
	      urpt.x += xadj;
	      llpt.x += xadj;
	      if (urpt.x > 0 && llpt.x < tmplength) {
	         textend = ULength(slab->string, 0.0, 0, llpt.x);
	         textpos = ULength(slab->string, 0.0, 0, urpt.x);
		 if ((textend > 1) && (*(slab->string + textend - 1) == TEXT_ESC))
		    textend--;
		 /* printf("Characters %d to %d bounded\n", textend, textpos);  */
		 selected = True;
	      }
	   }
	   } break;
	   
	case(PATH):
	   /* check position point of each subpart of the path */

	   selected = False;
	   for (pathgen = TOPATH(curgen)->plist; pathgen < TOPATH(curgen)->plist
		  + TOPATH(curgen)->parts; pathgen++) {
	      if (areaelement(pathgen)) selected = True;
	   }
	   break;

	default:
	   selected = areaelement(curgen);
	   break;
      }

      /* check if this part has already been selected */

      if (selected)
         for (newselect = areastruct.selectlist; newselect <
              areastruct.selectlist + areastruct.selects; newselect++)
            if (*newselect == (short)(curgen - objectdata->plist))
                  selected = False;
	
      /* add to list of selections */

      if (selected) {
         newselect = allocselect();
         *newselect = (short)(curgen - objectdata->plist);
      }
   }
   setoptionmenu();

   /* if none or > 1 label has been selected, cancel any textpos placement */

   if (!checkselect(LABEL) || areastruct.selects != 1 ||
	(areastruct.selects == 1 && SELECTTYPE(areastruct.selectlist) != LABEL)) {
      textpos = textend = 0;
   }

   /* Drawing of selected objects will take place when drawarea() is */
   /* executed after the button release.			     */
}

/*------------------------*/
/* start deselection mode */
/*------------------------*/

void startdesel(w, clientdata, nulldata)
  Widget w;
  caddr_t clientdata, nulldata;
{
   if (eventmode == NORMAL_MODE) {
      eventmode = DESEL_MODE;
      Wprintf("Click on element to deselect");
   }
}

/*------------------------------------------*/
/* deselect all objects on the select stack */
/*------------------------------------------*/

void deselect(w, clientdata, calldata)
  Widget w;
  caddr_t clientdata;
  caddr_t calldata;
{
   objectselect(-ANY);

   /* return Option menu to default state */

   setoptionmenu();
}

void objectdeselect()
{
   short *lastselect;

   if (areastruct.selects == 0) return;

   /* reset the graphics state */

   XSetFunction(dpy, areastruct.gc, GXcopy);

   for (lastselect = areastruct.selectlist; lastselect < areastruct.selectlist 
	+ areastruct.selects; lastselect++) {
      XTopSetForeground(SELTOCOLOR(lastselect)); 
      easydraw(*lastselect, SELTOCOLOR(lastselect));
   }

   areastruct.selects = 0;
   free(areastruct.selectlist);

   setallstylemarks(areastruct.style);
   setcolormark(areastruct.color);
   setdefaultfontmarks();
}

/*------------------------------------------------------------------------*/
/* Select the nearest polygon, searching down the hierarchy if necessary. */
/* Return pointers to the polygon and the object containing the polygon.  */
/*------------------------------------------------------------------------*/

objectptr recurselect(polyptr *polyreturn, objectptr selobj)
{
   short *selectobj;
   objinstptr selinst;
   objectptr selreturn;
   XPoint savesave, tmppt;

   if (areastruct.selects > 0) {
      areastruct.selects = 0;
      free(areastruct.selectlist);
   }

   if ((selectobj = selectobject(POLYGON, selobj)) != NULL) {
      *polyreturn = TOPOLY(selobj->plist + (*selectobj));
      areastruct.selects = 0;
      free(areastruct.selectlist);
      /* printf("polygon found in object %s\n", selobj->name); */
      selreturn = selobj;
   }
   else if ((selectobj = selectobject(OBJECT, selobj)) != NULL) {
      if ((*(selobj->plist + *selectobj))->type == OBJECT) {
         selinst = TOOBJINST(selobj->plist + (*selectobj));	
         /* Translate areastruct.save into object's own coordinate system */
         savesave.x = areastruct.save.x;
         savesave.y = areastruct.save.y;
         InvTransformPoints(&areastruct.save, &tmppt, 1, selinst->position,
		selinst->scale, selinst->rotation);
         areastruct.save.x = tmppt.x;
         areastruct.save.y = tmppt.y;
         /* printf("objinst %s found in object %s; searching recursively\n",
		selinst->thisobject->name, selobj->name); */
         /* printf("cursor position originally (%d, %d); in new object is (%d, %d)\n",
		savesave.x, savesave.y,
		areastruct.save.x, areastruct.save.y); */
	 UPushCTM();
	 UPreMultCTM(DCTM, selinst->position, selinst->scale, selinst->rotation);
         selreturn = recurselect(polyreturn, selinst->thisobject);
         UPopCTM();
         areastruct.save.x = savesave.x;
         areastruct.save.y = savesave.y;
      }
   }
   else {
      /* printf("nothing found in object %s\n", selobj->name); */
      selreturn = (objectptr)NULL;
   }
   return selreturn;
}

/*----------------------------------*/
/* Start drawing a select area box. */
/*----------------------------------*/

XtTimerCallbackProc startselect(id, clientdata)
  XtIntervalId  id;
  caddr_t       clientdata;
{
   if (eventmode == NORMAL_MODE) eventmode = SELAREA_MODE;
   areastruct.origin.x = areastruct.save.x;
   areastruct.origin.y = areastruct.save.y;
   UDrawBox(areastruct.origin, areastruct.save);
   XtAddEventHandler(areastruct.area, ButtonMotionMask, False, 
	(XtEventHandler)trackselarea, NULL);
}

/*-------------------------*/
/* Track a select area box */
/*-------------------------*/

XtEventHandler trackselarea(w, clientdata, event)
  Widget w;
  XtPointer clientdata;
  XMotionEvent *event;
{
   Window nullwin;
   int    nullint, xpos, ypos;
   XPoint       newpos;
   unsigned int   nullui;

   XQueryPointer(dpy, win, &nullwin, &nullwin, &nullint, &nullint, &xpos, &ypos,
	&nullui);
   window_to_user(xpos, ypos, &newpos);
   if (newpos.x == areastruct.save.x && newpos.y == areastruct.save.y) return;

   UDrawBox(areastruct.origin, areastruct.save);
   UDrawBox(areastruct.origin, newpos);

   areastruct.save.x = newpos.x;
   areastruct.save.y = newpos.y;
}

/*-------------------------------------------------------------------------*/
