/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     bright.c                                                       *
 *            Funktionen zum Ermitteln von Helligkeitsextrema                *
 * Author:    Andreas Tille                                                  *
 * Date:      05.05.1998                                                     *
 * Copyright: Andreas Tille, 1999; Gnu Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdio.h>
#include <math.h>

#include "paul.h"
#include "names.h"

unsigned long brightness(unsigned char *buf, int x, int y, int step, int storepix)
/* Sum of all pixels in a cefined image section
 * --- Parameter: ---
 * unsigned char *buf         : buffer with pixels
 * int            x           : width of section to calculate
 * int            y           : height of section to calculate
 * int            step        : step to the beginning of the next line (== width of original image)
 * int            storepix    : number of bytes used to store the image (3 for 24 bit, 1 for 8 bit)
 * --- Return: ---
 * unsigned long  brightness(): brightness of the section
 */
{
  register int            sx = x*storepix, sy = step*storepix;
  register unsigned long  bright = 0;
  register unsigned char *ap, *fip, *yp, *fiyp;

  for ( fiyp = (yp = buf) + y*sy; yp < fiyp; yp += sy )
    for ( fip = (ap = yp) + sx; ap < fip; ap++ )   /* here storepix is not used just add them all */
       bright += *ap;

  return bright;
}

unsigned long Xsum(unsigned char *buf, int len, int storepix)
/* calculate sum of a string of bytes
 * --- Parameter: ---
 * unsigned char *buf   : values of brightness (24 Bit)
 * int      len         : number of bytes in the whole string (must not be multiplied with storepix)
 * --- Return: ---
 * unsigned long Xsum() : sum
 */
{
  register int            sp = storepix;
  register unsigned char *ap, *fip;
  register unsigned long  sum = 0;

  for ( fip = (ap = buf) + len; ap < fip; ap += sp )
    sum += (unsigned long)(*ap);

  return sum;
}

static gfloat Deviation(unsigned char *buf, int len, int storepix, gfloat mean, int n)
/* standard deviation
 */
{
  register int            sp = storepix;
  register unsigned char *ap, *fip;
  gfloat                  dev = 0.0;

  for ( fip = (ap = buf) + len; ap < fip; ap += sp )
    dev += (gfloat)(*ap) * (gfloat)(*ap);
  return sqrt((dev - mean * (mean / (gfloat)n))/n);
}


unsigned long Ysum(unsigned char *buf, int len, int step)
/* calculate sum of bytes, which are stored in the distance step (24-bit-version)
 * that means it calculates the sum of a column, if one row has step bytes (=storepix*width)
 * --- Parameter: ---
 * unsigned char *buf  : values of brightness (24 Bit)
 * int      len        : number of pixels to sum up
 * int      step       : distance of the pixels (width of original * storepix)
 * --- Return: ---
 * unsigned long Ysum(): sum
 */
{
  register int      s = 3*step;
  register unsigned long sum = 0;   
  register unsigned char *ap, *fip;

  for ( fip = (ap = buf) + len*s; ap < fip; ap += s )
    sum += *ap + *(ap+1) + *(ap+2);
  return sum;
}

int AnalyseBrightness(PAUL *p)
/* Calculate brightness of image sets
 * --- Parameter: ---
 * PAUL *p    : list of images
 * char *file : name of the file to store the brightnesses in
 */
{
  FILE          *fp;
#ifdef PHYSICAL_PAUL
  char          *time, leer[] = "---  ", *fs;
#endif
  char          *chan[3] = {_("red"), _("green"), _("blue")}, *ch;
  double         d_time;
  gfloat        *bright, *dev, *bp, *dp;
  PICTURE       *bild;
  GList         *pl;
  unsigned char *ap;
  int            storepix, nfiles, n;

  g_return_val_if_fail ( IS_PAUL(p), RET_ERR );
  g_return_val_if_fail ( p->piclist, RET_ERR );
  g_return_val_if_fail ( BILD(p->piclist), RET_ERR );

  if ( !(p->opt->bright) ) p->opt->bright = g_strdup(BRIGHTFILE);

  g_return_val_if_fail ( (fp = fopen(p->opt->bright, "at")), RET_ERR );
  bright = g_new( gfloat, 2 * (nfiles = NBILDER(p->piclist)) );
  dev    = bright + nfiles;

#ifdef PHYSICAL_PAUL
  if ( PG ) fs = g_strdup(GetCharSpec(PG->spec, PH_f_s));
  else      fs = g_strdup(leer);
#endif
      
  for ( bild = BILD(pl = p->piclist), bp = bright, dp = dev; pl; bild = BILD(pl = pl->next), bp++, dp++ ) {
#ifdef PHYSICAL_PAUL
    if ( Chunk2Double(bild->spec, ChunkRelTime, &d_time) == RET_OK ) 
      time = g_strdup_printf("%g", d_time);
    else
      time = g_strdup(leer);
#endif

    ap       = bild->DATA;
    storepix = 1;
    ch       = _("all");
    n = bild->size * bild->storepix;
    if ( p->opt->greps < 3 ) {
      ap      += p->opt->greps;
      storepix = bild->storepix;
      ch       = chan[p->opt->greps];
      n        = bild->size;
    }

    *bp = (gfloat)Xsum(ap, bild->storepix * bild->size, storepix);
    *dp = Deviation(ap, bild->storepix * bild->size, storepix, *bp, n);

#ifdef PHYSICAL_PAUL
    fprintf(fp, "%10s  %10.6g  %10.6g  %8s  %s (%s)\n", time, *bp/(gfloat)n, *dp, fs, bild->file, ch);
    FREE(time);
#else
    fprintf(fp, "%10.6g  %10.6g  %s (%s)\n", *bp/(gfloat)n, *dp, bild->file, ch);
#endif
  }
#ifdef PHYSICAL_PAUL
  FREE(fs);
#endif
  fclose(fp);
  return RET_OK;
}



