/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "GuiColormap.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"


#define LOAD		0
#define SAVE		1
#define BANDSIZE	15



/*
 *  constructor: initialize everything 
 */
GuiColormap::GuiColormap (Gtk_Widget *parent, int maxcolors)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ GuiColormap\n");

	SanityCheck::bailout ((!parent), "parent==NULL", "GuiColormap::GuiColormap");
	SanityCheck::bailout ((maxcolors<2), "maxcolors<2", "GuiColormap::GuiColormap");

	parent->realize ();
	p_colormap = parent->get_colormap ();
	p_colors = new GdkColor[maxcolors];
	d_ncolors = 0;
	d_maxcolors = maxcolors;

	SanityCheck::bailout ((!this->allocNamedColor (&d_black, "black")), 
				"can't alloc color black", "FileIO::open");
	SanityCheck::bailout ((!this->allocNamedColor (&d_white, "white")), 
				"can't alloc color white", "FileIO::open");

}


/*
 *  destructor: free allocated memory
 */
GuiColormap::~GuiColormap ()
{
	if (p_colors)
		delete [] p_colors;

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- GuiColormap\n");
}


/*
 *  getColor: get the color at the specified index
 */
GdkColor *GuiColormap::getColor (int index)
{
	if (index < d_ncolors && index >= 0)
		return (&(p_colors[index]));
	else
		{
		g_print ("Warning: GuiColormap index (%d) out of range (0, %d)\n", index, d_ncolors-1);
		return NULL;
		}
}


/*
 *  getColorCount: return the number of colors in the colormap
 */
int GuiColormap::getColorCount ()
{
	return (d_ncolors);
}


/*
 *  addColor: add a color to the colormap
 */
bool GuiColormap::addColor (GdkColor *col, int r, int g, int b) 
{
	if (allocColor (col, r, g, b))
		{
		p_colors[d_ncolors++] = *col;
		return (d_ncolors);
		}
	else
		return (-1);
}


/*
 *  addColor: add a color to the colormap
 */
bool GuiColormap::addColor (GdkColor	*col)
{
	if (allocColor (col))
		{
		p_colors[d_ncolors++] = *col;
		return (d_ncolors);
		}
	else
		return (-1);
}


/*
 *  addNamedColor: add a named color to the colormap
 */
bool GuiColormap::addNamedColor (GdkColor	*col, 
				char		*colname)
{
	if (allocNamedColor (col, colname))
		{
		p_colors[d_ncolors++] = *col;
		return (d_ncolors);
		}
	else
		return (-1);
}


/*
 *  addNamedColor: allocate a color 
 */
bool GuiColormap::allocColor (GdkColor *col, int r, int g, int b) 
{
	col->red=r;
	col->green=g;
	col->blue=b;
	if (gdk_color_alloc (p_colormap, col))
		return (TRUE);
	else
		return (FALSE);
}


/*
 *  addNamedColor: allocate a color 
 */
bool GuiColormap::allocColor ( 	GdkColor	*col)
{
	if (gdk_color_alloc (p_colormap, col))
		return (TRUE);
	else
		return (FALSE);
}


/*
 *  addNamedColor: allocate a named color 
 */
bool GuiColormap::allocNamedColor (	GdkColor	*col, 
					char		*colname)
{
	if (	gdk_color_parse (colname, col) && 
		gdk_color_alloc (p_colormap, col))
		return (TRUE);
	else 
		{
		SanityCheck::warning ("parent==NULL", "GuiColormap::GuiColormap");
		printf (_("Unable to allocate color [%s] ... Exiting\n"), colname);
		return (FALSE);
		}
}


