/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "HeightFieldCore.h" 
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"



/*
 *  constructor: take a hf pointer and set everything appropriately
 */
HeightFieldCore::HeightFieldCore (PTYPE *hf, int xsize, int ysize)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldCore\n");

	p_hf=NULL;
	d_max=d_min=d_avg=0;
	d_sealevel=0.33;

	setData (hf, xsize, ysize);
}


/*
 *  constructor: create a new height field of specified size 
 */
HeightFieldCore::HeightFieldCore (int xsize, int ysize)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldCore\n");

	SanityCheck::bailout ((xsize<0), "xsize<0", "HeightFieldCore::HeightFieldCore()");
	SanityCheck::bailout ((ysize<0), "ysize<0", "HeightFieldCore::HeightFieldCore()");

	int	size = xsize * ysize;
	PTYPE	*hf = NULL;

	p_hf=NULL;
	d_max=d_min=d_avg=0;
	d_sealevel=0.33;

	if (size > 0)
		{
		hf = new PTYPE[size];
		memset (hf, 0, size);
		setData (hf, xsize, ysize);
		}
	else
		setData (NULL, 0, 0);
}


/*
 *  destructor: clean up, free allocated memory
 */
HeightFieldCore::~HeightFieldCore ()
{
	if (p_hf)
		delete [] p_hf;

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldCore\n");
}


/*
 *  SetData : delete and current HF data and replace it with whatever 
 * 	is passed in. 
 */
void HeightFieldCore::setData (PTYPE *hf, int xsize, int ysize, char *dummy)
{
	if (p_hf)
		delete [] p_hf;

	 if (hf)
		{
		SanityCheck::bailout ((xsize<1), "xsize<1", "HeightFieldCore::setData()");
		SanityCheck::bailout ((ysize<1), "ysize<1", "HeightFieldCore::setData()");

		p_hf=hf;
		d_xsize = xsize;
		d_ysize = ysize;
		d_size = xsize * ysize;
		gatherStatistics ();
		}
	else
		{
		SanityCheck::bailout ((xsize), "xsize!=0", "HeightFieldCore::setData()");
		SanityCheck::bailout ((ysize), "ysize!=0", "HeightFieldCore::setData()");

		p_hf=NULL;
		d_xsize=d_ysize=d_size=0;
		d_max=d_min=d_avg=0;
		}
}


/*
 *  SetSealevel: Set the sealevel but make sure it never equals or exceeds 
 * 	the actual elevation values for the height field. If it does, 
 * 	certain equations will results in divede by 0s and dump core.
 */
void HeightFieldCore::setSealevel (PTYPE sealevel)
{
	d_sealevel = sealevel;

	if (d_sealevel <= d_min)
		d_sealevel = d_min+0.000001;
	if (d_sealevel >= d_max)
		d_sealevel = d_max-0.000001;
}



/*
 *  gatherStatistics: gather some general statistics about the Height Field
 */
int HeightFieldCore::gatherStatistics ()
{
	PTYPE	h=0; 

	SanityCheck::bailout ((!p_hf), "p_hf==NULL", "HeightFieldCore::gatherStatistics");

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Surveying Height Field ...\n"); 

	rescale ();

	for (int i=0; i<(int)d_size; i++)
		h += p_hf[i];
	d_avg=h/d_size;

	return 0;
}


/*
 *  pMinMax: return 2 pointers to the min and max height cells
 */
void HeightFieldCore::pMinMax (PTYPE *pfloor, PTYPE *pceil)
{
	*pfloor = *pceil = p_hf[0];

	for (int i=1; i<d_size; i++)
		{
		if (p_hf[i] < *pfloor)
			*pfloor = p_hf[i];
		else
		if (p_hf[i] > *pceil)
			*pceil = p_hf[i];
		}
}


/*
 *  rescale: rescale hf to lie between values floor and ceil inclusive 
 */
int HeightFieldCore::rescale (float floor, float ceil)	// default: (0.0, 1.0)
{
	int		i;
	double		cfactor;
	char 		buf[80];

	pMinMax (&d_min, &d_max);

	sprintf (buf, "Rescaling %f, %f ---> ", d_min, d_max);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	if (d_min == d_max)
		{
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "fixing constant Height Field\n");
		for (i=0; i<d_size; i++)
			p_hf[i] = 0;
		p_hf[0] = 1;
		return -1;
		}

	if (d_min==floor && d_max==ceil)
		{
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "OK\n");
		return 1;
		}

	cfactor=(double)((ceil-floor)/(d_max-d_min));
	for (i=0; i<d_size; i++)
		p_hf[i] = (p_hf[i]-d_min)*cfactor;

	d_min=floor;
	d_max=ceil;

	sprintf (buf, "%f, %f\n", floor, ceil);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	return 0;
}


/*
 *  Stat: display the statistics
 */
int HeightFieldCore::printStatistics ()
{

	printf ("* * * * * * * * * * * Height Field Statistics * * * * * * * * * * *\n");
	printf ("Height Field		: %dx%d, %d\n", d_xsize, d_ysize, d_size);
	printf ("Min Max, Avg		: %f, %f, %f\n", d_min, d_max, d_avg);
	fflush (stdout);
	return 0;
}

