/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* ******************************************************************** */
/* *** HeightFieldDraw is based on code from John Beale's excellent *** */
/* ************************ hf-lab package. *************************** */
/* ******************************************************************** */


#include <math.h>
#include <stdlib.h>
#include "HeightFieldDraw.h"
#include "GuiColormapBands.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "MenuDefs.h"
#include "TFOptions.h"
#include "tf_flexarrCOORD2.h"
#include "tf_glistFlexArr.h"


void resize (int x, int y);
static double ATAN2 (double y, double x);


/*
 *  constructor: initialize everything, pass in the main terraform window
 * 		 to draw into.  
 */
HeightFieldDraw::HeightFieldDraw (HeightField 		*HF, 	// Height Field 
				  TFWindow		*tfWin)	// Drawable
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldDraw\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldDraw::HeightFieldDraw");
	SanityCheck::bailout ((!HF->getData()), "HF->getData()==NULL", "HeightFieldDraw::HeightFieldDraw");
	SanityCheck::bailout ((!tfWin), "tfWin==NULL", "HeightFieldDraw::HeightFieldDraw");

	p_HF = HF;
	p_HFM = NULL;
	p_tfWin = tfWin;
	p_drawArea = dynamic_cast<GuiBufferedDrawingArea*>(p_tfWin);
	p_drawArea->setSync (p_HF->getWidth()*5);
	p_RGBbuf = NULL;

	p_cMap = NULL;
	d_ncolors=0;
	d_nLandColors=0;
	d_nWaterColors=0;
	d_nbands=0;
	d_bandsize=0;
	d_bandLandOffset=0;

	d_erasewin=TRUE; 
	d_scale=1;
	b_fastWire=FALSE;

	resetView ();
}


/*
 *  constructor: initialize everything, pass in a buffered drawing area 
 * 		 to draw to. This typically signifies that we're working 
 * 		 on a preview and can't do window ops (tfWin==NULL).  
 */
HeightFieldDraw::HeightFieldDraw (HeightField 		*HF, 	// Height Field 
				  GuiBufferedDrawingArea *da)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldDraw\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldDraw::HeightFieldDraw");
	SanityCheck::bailout ((HF->getData()==NULL), "HF->getData()==NULL", "HeightFieldDraw::HeightFieldDraw");
	SanityCheck::bailout ((!da), "da==NULL", "HeightFieldDraw::HeightFieldDraw");

	p_HF = HF;
	p_HFM = NULL;
	p_tfWin = NULL;
	p_drawArea = da;
	p_drawArea->setSync (p_HF->getWidth()*5);
	p_RGBbuf = NULL;

	p_cMap = NULL;
	d_ncolors=0;
	d_nLandColors=0;
	d_nWaterColors=0;
	d_nbands=0;
	d_bandsize=0;
	d_bandLandOffset=0;

	d_erasewin=TRUE; 
	d_scale=1;
	b_fastWire=FALSE;

	resetView ();
}

/*
 *  destructor: delete any (model) data we have 
 */
HeightFieldDraw::~HeightFieldDraw ()
{
	if (p_HFM)
		delete p_HFM;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "--- HeightFieldDraw\n");
}


/*
 *  setColormap: set the colormap we will use to draw 
 */
int HeightFieldDraw::setColormap (GuiColormap *colorMap) 
{
	SanityCheck::bailout ((!colorMap), "colorMap==NULL", "HeightFieldDraw::setColormap");
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Setting colormap ...\n"); 

	if (p_cMap == colorMap)
		return 1;

	p_cMap = colorMap;
	d_ncolors=p_cMap->getColorCount()-1;
	if (!colorMap->d_linear)
		{
		GuiColormapBands	*cMapBands=NULL;

		cMapBands = static_cast<GuiColormapBands*>(colorMap);
		d_nbands = cMapBands->getNumBands();
		d_bandsize = cMapBands->getBandSize();
		d_bandLandOffset = cMapBands->getLandBandOffset();

		d_nWaterColors = d_bandLandOffset*cMapBands->getBandSize();
		d_nLandColors = d_ncolors-d_nWaterColors;
		}
	else
		{
		d_nbands = d_bandsize = d_bandLandOffset = 0;
		d_nWaterColors = d_ncolors*p_HF->getSealevel();
		d_nLandColors = d_ncolors-d_nWaterColors;
		}
	
	SanityCheck::bailout ((!p_cMap->d_linear && (!d_nbands || !d_bandsize || !d_bandLandOffset)), 
			"non-linear colormap without band specs", 
			"HeightFieldDraw::setColormap");

	return 0;
}


/*
 *  setSync: set the sync interval
 */
void HeightFieldDraw::setSync (int syncRate)
{
	p_drawArea->setSync (syncRate);
}


/*
 *  setMode: set the drawing mode
 */
int HeightFieldDraw::setMode (char *s)
{
	int	oldShader = d_shader;
	char 	buf[80];

	SanityCheck::bailout ((!s), "s==NULL", "HeightFieldDraw::setMode");

	/* *** figure out which mode we should set *** */
	if (!strcmp (s, MENU_VIEW_2DPLANE) || !strcmp (s, "0"))
		{
		if (d_shader == WIRE || d_shader == HEIGHT)
			d_wireResolution=d_resolution;
		d_shader=PLANE;
		d_resolution=1;
		}
	else
	if (!strcmp (s, MENU_VIEW_3DWIRE) || !strcmp (s, "1"))
		{
		d_shader=WIRE;
		d_resolution=d_wireResolution;
		}
	else
	if (!strcmp (s, MENU_VIEW_3DHEIGHT) || !strcmp (s, "2"))
		{
		d_shader=HEIGHT;
		d_resolution=d_wireResolution;
		}
	else
	if (!strcmp (s, MENU_VIEW_3DLIGHT) || !strcmp (s, "3"))
		{
		if (d_shader == WIRE || d_shader == HEIGHT)
			d_wireResolution=d_resolution;
		d_shader=LIGHT;
		d_resolution=1;
		}
	else
		{
		cerr << "Error: Invalid mode [" << s << "] supplied ...\n";
		d_shader=PLANE;
		return (-1);
		}

	if (d_shader == oldShader)
		return (1);

	sprintf (buf, "Mode set to %s\n", s);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	return (0);
}



/*
 *  setScale: set the scale at which we'll look at the height field
 */
int HeightFieldDraw::setScale (float s, int *x, int *y)
{
	SanityCheck::bailout ((!p_tfWin), "p_tfWin==NULL", "HeightFieldDraw::setScale");

	/* *** figure out which mode we should set *** */
	if (s==0.25 || s==0.5 || s==1 || s==2 || s==3 || s==4 || s==5)
		{
		if ((p_HF->getWidth()*s <= MAX_SCREEN_SIZE) &&
		    (p_HF->getHeight()*s <= MAX_SCREEN_SIZE) &&
		    (p_HF->getWidth()*s >= MIN_SCREEN_SIZE) &&
		    (p_HF->getHeight()*s >= MIN_SCREEN_SIZE))
			{
			d_xgsize=(int)(p_HF->getWidth()*s);// size of graphics window
			d_ygsize=(int)(p_HF->getHeight()*s);// size of graphics window
			}
		else
			return 1;

		d_scale = s;
		p_tfWin->setSize (d_xgsize, d_ygsize);

		// avoid bombing out the first time around 
		// if (p_cMap)
		//	draw ();    	// this is called in TFDialogOptions.cc

		if (x) *x=d_xgsize;
		if (y) *y=d_ygsize;

		}
	else
		{
		if (x) *x=d_xgsize;
		if (y) *y=d_ygsize;
		cerr << "Error: Invalid scale [" << s << "] supplied ...\n";
		cerr << "Valid scale factors are: 1, 2, 3\n"; 
		return (-1);
		}

	return (TRUE);
}


/*
 *  alterScale: either double or half the current display scale
 */
int HeightFieldDraw::alterScale (float s, int *x, int *y)
{
	SanityCheck::bailout ((!p_tfWin), "p_tfWin==NULL", "HeightFieldDraw::setScale");

	if ((s != 0.5) && (s != 1) && (s != 2))
		return -1;
	
	if ((p_HF->getWidth()*s <= MAX_SCREEN_SIZE) &&
	    (p_HF->getHeight()*s <= MAX_SCREEN_SIZE) &&
	    (p_HF->getWidth()*s >= MIN_SCREEN_SIZE) &&
	    (p_HF->getHeight()*s >= MIN_SCREEN_SIZE) &&
	    (d_scale*s >= MIN_SCALE_FACTOR) && (d_scale*s <= MAX_SCALE_FACTOR))
		{
		d_scale = d_scale*s;
		d_xgsize=(int)(p_HF->getWidth()*d_scale);// size of graphics window
		d_ygsize=(int)(p_HF->getHeight()*d_scale);// size of graphics window
		p_tfWin->setSize (d_xgsize, d_ygsize);

		// avoid bombing out the first time around 
		// if (p_cMap)
		//	draw ();

		if (x) *x=d_xgsize;
		if (y) *y=d_ygsize;
		return 0;
		}
	else
		return 1;
}


/*
 *  setFastWire: set the fast wire mode on or off
 */
void HeightFieldDraw::setFastWire (bool fastWire)
{
	b_fastWire = fastWire;
	
}


/*
 *  setFastOptions: set the fast wire mode on or off
 */
void HeightFieldDraw::setFastOptions (bool halfWireRes, bool halfYscale, bool doRotZ)
{
	if (halfWireRes)
		d_fastWireResAdj = 2;
	else
		d_fastWireResAdj = 1;

	if (halfYscale)
		d_fastYscaleAdj = 0.5;
	else
		d_fastYscaleAdj = 1;

	d_fastDoRotZ = doRotZ;
	if (b_fastWire)
		draw ();
}


/*
 *  setEyePos: set the eye position
 */
void HeightFieldDraw::setEyePos (double _x_, double _y_, double _z_)
{
	d_eye.x=_x_;
	d_eye.y=_y_;
	d_eye.z=_z_;
}


/*
 *  setLeightPos: set the light position
 */
void HeightFieldDraw::setLightPos (double _x_, double _y_, double _z_)
{
	d_light.x=_x_;
	d_light.y=_y_;
	d_light.z=_z_;
}


/*
 *  setYScale: set the Y scaling factor 
 */
void HeightFieldDraw::setYScale (double yscale)
{
	d_ysf=yscale;
}


/*
 *  setSeaFill: should we draw the sea as filled?
 */
void HeightFieldDraw::setSeaFill (int sfill)
{
	b_sea=sfill;
}


/*
 *  setResolution: set the wireframe d_resolution
 */
void HeightFieldDraw::setResolution (int i)
{
	if (i < 0)
		{
		cerr << "Sanity Check Failed: Resolution (" << i << ") out of range ...\n";
		cerr << "Valid Range is > 0\n";
		exit (-1);
		}
	d_resolution=i;
	if (d_shader == WIRE || d_shader == HEIGHT)
		d_wireResolution=d_resolution;
}


/*
 *  resetView: reset all view parameters back to their default settings 
 */
void HeightFieldDraw::resetView ()
{
	SanityCheck::bailout ((!p_HF), "HF==NULL", "HeightFieldDraw::resetView");
	SanityCheck::bailout ((!p_HF->getData()), "HF->getData()==NULL", "HeightFieldDraw::resetView");
	SanityCheck::bailout ((!p_drawArea), "p_drawArea==NULL", "HeightFieldDraw::resetView");

	d_wireResolution=d_resolution=10;       // low d_resolution == high detail
	setMode(MENU_VIEW_3DWIRE);
	d_fastWireResAdj=2;
	d_fastYscaleAdj=0.5;
	d_fastDoRotZ=FALSE;
	b_drawContour=TRUE;

	d_xgsize=(int)(p_HF->getWidth()*d_scale);
	d_ygsize=(int)(p_HF->getHeight()*d_scale);

	d_slMin=1.0;		// minimum slope (shading offset)
	d_slScale=1.0;		// slope scale factor (shading scale)
	b_sea=TRUE;

	d_sxmid = d_xgsize/2;  	// middle of screen in screen coords 
	d_symid = d_ygsize/2;
	d_zsf = 0.3;		// z coordinate scale factor 
	d_ysf = 0.3;		// vertical (y) coordinate scale factor 
	d_zeps = 0.001;
	d_eyeinc = 0.8;		// viewpoint 0,0,0 in +z direction 
	//d_eyeinc = 0.1;	// viewpoint 0,0,0 in +z direction 
	d_bright = 0.8;  	// lightsource intensity 
	d_maxcon = FALSE;	// maximum contrast 

	d_light.x=1;		// light source position 
	d_light.y=0;
	d_light.z=-1;

	d_eye.x=0;		// eye position
	d_eye.y=ceil(p_HF->getMax())*-2;
	d_eye.z=-3.5;

	if (p_HFM)
		{
		delete p_HFM;
		p_HFM = NULL;
		}
	// only create a new HFM if we're in the TFWindow (no fastWire
	// needed for preview drawing)
	if (p_tfWin)
		p_HFM = new HeightFieldModel (p_HF);
}


/*
 *  updateParams: update Parameters from TFOptions
 */
void HeightFieldDraw::updateParams ()
{
	b_sea = TFOptions::s_fillSea;
	b_drawContour = TFOptions::s_drawDerivedData;
	d_scale = TFOptions::s_scale;
	setSync (TFOptions::s_syncRate);
	d_resolution = TFOptions::s_resolution;
	d_ysf = TFOptions::s_yscale;
	p_HF->setSealevel (TFOptions::s_sealevel);

	if (TFOptions::s_fastHalfRes)
		d_fastWireResAdj = 2.0;
	else
		d_fastWireResAdj = 1.0;

	if (TFOptions::s_fastHalfYscale)
		d_fastYscaleAdj = 0.5;
	else
		d_fastYscaleAdj = 1.0;
}


/*
 *  clear: clear the window 
 */
void HeightFieldDraw::clear ()
{
	p_drawArea->setColors (&(p_cMap->d_black));
	p_drawArea->drawRectangle (0, 0, d_xgsize, d_ygsize, TRUE);

}


/*
 *  draw: setup and call the appropriate drawing routines
 */
int HeightFieldDraw::draw (bool changedHF)
{
	D		lmag;         	// magnitude of a vector 

	SanityCheck::bailout ((!p_cMap), "p_cMap==NULL", "HeightFieldDraw::draw");
	SanityCheck::bailout ((d_shader>LIGHT), "shade mode out of range", "HeightFieldDraw::draw");

	d_cmin = 0; 
	d_cmax = d_ncolors-1;

	/* *** calculate the positive and negative range *** */
	d_nrange = ABS (p_HF->getMin() - p_HF->getSealevel());
	d_nrange *= 1.01;				// make sure we don't overflow
	d_prange = ABS (p_HF->getMax() - p_HF->getSealevel());
	d_prange *= 1.01;				// when checking max and min

	d_xgsize=(int)(p_HF->getWidth()*d_scale);
	d_ygsize=(int)(p_HF->getHeight()*d_scale);
	if (p_tfWin)
		p_tfWin->setSize (d_xgsize, d_ygsize);

	if (d_erasewin)
		clear ();

	if (d_shader == PLANE) 			// Wireframe 
		drawhf2d ();
	else
	if (b_fastWire && d_shader==WIRE)
		{
		SanityCheck::bailout ((!p_HFM), "p_HFM==NULL", "HeightFieldDraw::draw");
		if (changedHF)
			p_HFM->refresh (p_cMap, d_resolution*d_fastWireResAdj, 
					d_ysf*d_fastYscaleAdj, b_sea);
		drawhf3dFast ();
		}
	else
		{
		//if (TRACE_DEBUG_2)
		//	printSettings ();

		/* *** assign class variables used in further calculations *** */
		lmag = sqrt(d_light.x*d_light.x + d_light.y*d_light.y + d_light.z*d_light.z);
		d_light.x /= lmag;
		d_light.y /= lmag;
		d_light.z /= lmag;

		d_zcenter = -d_eye.z;
		d_ycorr = 2*d_eye.y/(d_zcenter);
		d_sxmid = d_xgsize / 2;  		// middle of screen in screen coords 
		d_symid = d_ygsize / 2;
		d_yoff = (int) (d_symid*(d_ycorr + d_eye.y/d_zcenter)); 

		/* *** do the actual drawing *** */
		drawhf3d ();
		}

	p_drawArea->setColors (&(p_cMap->d_black));
	p_drawArea->sync ();
	gdk_flush ();
	return(0);
}



/* ********************************************************************* */
/* ******************* execute the command in token ******************** */
/* ********************************************************************* */

/*
int HeightFieldDraw::P_Execute ()
{
	int	redraw=TRUE;
	char	*c=P_Tok(0);

	while (*c)
		{
		if (*c == '+') 				// y scaling factor
			d_ysf *= 1.1;
		else if (*c == '-' || *c == '_') 
			ysf /= 1.1;
		else if (*c == 'r') 			// x axis (right, left)
			d_eye.x -= d_eyeinc;
		else if (*c == 'l') 
			eye.x += d_eyeinc;
		else if (*c == 'u') 			// y axis (up, down)
			eye.y -= d_eyeinc;
		else if (*c == 'd') 
			eye.y += d_eyeinc;
		else if (*c == 'b') 			// z axis (back, forward)
			eye.z -= d_eyeinc;
		else if (*c == 'f') 
			eye.z += d_eyeinc;
		else if (*c == '>') 			// brightness 
			bright *= 1.1;
		else if (*c == '<') 
			bright /= 1.1;
		else if (*c == '*') 			// scale 
			{ 
			d_xgsize*=d_scalefact; 
			d_ygsize*=d_scalefact; 
			X_WResize (d_xgsize, d_ygsize);
			redraw=FALSE;
			}
		else if (*c == '/') 
			{ 
			d_xgsize/=d_scalefact; 
			d_ygsize/=d_scalefact;
			X_WResize (d_xgsize, d_ygsize);
			redraw=FALSE;
			}
		else if (*c == 'm') 
			d_maxcon = !d_maxcon;
		else if (!P_TokArgCmp ("verbose", 1)) 
			{
			p_HF->verbose=!p_HF->verbose;
			redraw=FALSE;
			}
		else if (*c == 'l') 
			{
			double 	lmag;
			d_maxcon = FALSE;      // reset contrast limits 
			sscanf(++c ,"%lf,%lf,%lf",&light.x,&light.y,&light.z);
			lmag = sqrt(light.x*light.x + light.y*light.y + light.z*light.z);
			light.x /= lmag;
			light.y /= lmag;
			light.z /= lmag;
			}
		else if (*c == 'e') 
			{
			d_maxcon = FALSE;      // reset eye position 
			sscanf(++c,"%lf,%lf,%lf",&eye.x,&eye.y,&eye.z);
			if (eye.z > -1.01) 
				eye.z = -1.01;
			}
		else if (*c == 'f') 
			{
			eye.z += 0.5;
			if (eye.z >= -1.1) 
				eye.z = -1.1;
			}
		else if ((*c == 'h') || (*c == '?')) 
			{
			help();
			redraw=FALSE;
			}
		else if (*c == 's') 
			{
			d_maxcon = FALSE;
			if (setMode (++c))
				redraw=FALSE;
			break;
			}
		else if (*c == 'S')
			{
			b_sea=!b_sea;
			}
		else if (*c == 'p') 
			{
			sscanf (++c, "%d", &d_resolution);
			//d_resolution = (int)atol((const char*) *(++c));
			if (d_resolution < 1) 
				d_resolution = 1;
			}
		else if (*c == 'R') ;		// Redraw
		else if (*c == 'v')
			{
			//print_vars ();
			p_HF->verbose=!p_HF->verbose;
			redraw=FALSE;
			}
		else if (*c == 'q')
			{
			PARSE_DONE=TRUE;
			}
		else if (*c == 'C')
			use_color=!use_color;
		else if (*c == 'c')
			{
			int y;		// needed for Off2Posx()
			X_WClear ();
			for (int i=0; 
				 i<colors->ColorsUsed() && i<X_WGetWidth(); 
				 i++)
				 X_WFillRectangle (p_HF->Off2Posx(i), p_HF->Off2Posy(i), 
					1, X_WGetHeight(), 
					colors->ColorEntry (i));
			redraw=FALSE;
			}
		else 
			{
			printf ("Unrecognized token: [%s][", c);
			redraw=FALSE;
			break;
			}
		c++;
		}

	if (redraw)
		X_WRedraw (this->X_WGetWindow ());
	return (0);
}
*/


/*
 *  rotateFastView: pass on parameters to p_HFM
 */
int HeightFieldDraw::rotateFastView (float a, float b, float c)
{
	SanityCheck::bailout ((!p_HFM), "p_HFM==NULL", "HeightFieldDraw::rotateFastView");

	if (b_fastWire)
		{
		p_HFM->transform (a, b, c);
		p_HFM->transformD32 ();
		return 0;
		}	
	return 1;
}


/*
 *  draw2dContourLines: draw a height field in 2D (top view). We don't 
 * 	actually draw but just change the p_RGBbuf buffer. 
 */
int HeightFieldDraw::draw2dContourLines ()
{
	SanityCheck::bailout ((!p_RGBbuf), "d_RBGbuf==NULL", "HeightFieldDraw::draw2dContourLines");
	SanityCheck::bailout ((!p_HF->getContourList()), "p_HF->getContourList==NULL", "HeightFieldDraw::draw2dContourLines");

	Glib_GList	*contourList = p_HF->getContourList();
	int 		limMain=contourList->length(),
			limLine, 
			boff, xx, yy, xloc, yloc;
	TFFlexArrayCoord2 *faLine=NULL;
	COORD2		*coord2;

	for (int i=0; i<limMain; i++)
		{
		faLine = static_cast<TFFlexArrayCoord2*>(contourList->nth(i));
		SanityCheck::bailout ((!faLine), "faLine==NULL", "HeightFieldDraw::draw2dContourLine");
		limLine = faLine->getSize();

		for (int j=0; j<limLine; j++)
			{
			coord2 = static_cast<COORD2*>(faLine->El(j));
	    		for (yy=0; yy<(int)d_scale; yy++)
			    for (xx=0; xx<(int)d_scale; xx++)
				{
				xloc = (int)(coord2->x*d_scale+xx);
				yloc = (int)(coord2->y*d_scale+yy);

				boff = (int)(yloc*p_HF->getWidth()*d_scale*3 + xloc*3);
				p_RGBbuf[boff++] = 0;
				p_RGBbuf[boff++] = 0;
				p_RGBbuf[boff] = 0;
				}
			}
		}

	return 0;
}


/*
 *  drawHF2d: draw a height field in 2D (top view)
 */
int HeightFieldDraw::drawhf2d()
{
	int			x, y;  		// x, y 
	GdkColor		*col;
	guchar			*buf;
	int			boff=0, 
				llim = (int)(p_HF->getWidth()*3*d_scale), 
				lim = (int)(p_HF->getSize()*3*(d_scale*d_scale));
	bool			wholeScreen = FALSE;

	if (!p_HF->getData())
		return (-1);

	// if we have contour lines, we alloc space for the entire picture 
	// so we can overlay the existing picture with the contour lines
	if (b_drawContour && p_HF->getContourList())
		wholeScreen = TRUE;

	if (wholeScreen)
		{
		buf = new guchar[lim];
		p_RGBbuf = buf;
		}
	else
		{
		buf = new guchar[llim];
		p_RGBbuf = NULL;
		}

	for (y=0; y<p_HF->getHeight(); y++)	
	    for (int yy=0; yy<(int)d_scale; yy++)
		{
		if (!wholeScreen)
			boff=0;

		for (x=0; x<p_HF->getWidth();  x++)
			{
			col = getColor2d(x,y);
			for (int xx=0; xx<(int)d_scale; xx++)
				{
				// prevent core dump if color !found
				if (col)
					{
					buf[boff++] = col->red/255;
					buf[boff++] = col->green/255;
					buf[boff++] = col->blue/255;
					}
				else
					{
					// FIXME
					//printf ("%d, %d  --> %f\n", x, y, p_HF->getEl(x, y));
					buf[boff++] = 0;
					buf[boff++] = 0;
					buf[boff++] = 0;
					}
				}
			}

		// draw this line
		if (!wholeScreen)
			p_drawArea->drawRGB (0, (int)(y*d_scale+yy), 
				(int)(p_HF->getWidth()*d_scale), 1, buf, llim);
		}

	// update image buffer with contour line points and then draw
	if (wholeScreen)	// p_HF->getUpdateTime()==d_clTimestamp
		{
		draw2dContourLines ();
		// draw entire screen 
		p_drawArea->drawRGB (0, 0, (int)(p_HF->getWidth()*d_scale), 
			(int)(p_HF->getHeight()*d_scale), buf, llim);
		p_RGBbuf = NULL;
		}

	delete [] buf;

	return (0);
}


/*
 *  drawHF3dFast: draw fast wireframe by drawing precomputed triangles
 */
int HeightFieldDraw::drawhf3dFast ()
{
	int 	lim,
		cIndex,
		width = d_xgsize,		// size of graphics area
		height = d_ygsize,		// size of graphics area
		xsize = p_HF->getWidth(),		// size of HF
		ysize = p_HF->getHeight();		// size of HF
	PT2	p1, p2, p3;

	p_drawArea->setColors (p_cMap->getColor (d_ncolors-1));
	//p_HFM->transformD32 ();
	lim = p_HFM->getNumTri ();
	for (int i=0; i<lim; i++)
		{
		p_HFM->get2DPoints (i, &p1, &p2, &p3, &cIndex);
		p_drawArea->setColors (p_cMap->getColor (cIndex));
/*
		char	buf[80];
		sprintf (buf, "3DFast: %d\t(%f,%f), (%f,%f), (%f,%f)\n", i, 
			  p1.x, p1.y, p2.x, p2.y, p3.x, p3.y);
		if (GlobalTrace::trace (GlobalTrace::TRACE_DEBUG, buf);
		p_drawArea->drawLine (
			int (width/2 + p1.x*(float(width)/float(xsize))),
			int (height/10 - p2.y*(float(height)/float(ysize))),
			int (width/2 + p2.x*(float(width)/float(xsize))),
			int (height/10 - p2.y*(float(height)/float(ysize))));
*/
		p_drawArea->drawLine (
			int (width/2 + p2.x*(float(width)/float(xsize))),
			int (height/10 - p2.y*(float(height)/float(ysize))),
			int (width/2 + p3.x*(float(width)/float(xsize))),
			int (height/10 - p3.y*(float(height)/float(ysize))));
		p_drawArea->drawLine (
			int (width/2 + p3.x*(float(width)/float(xsize))),
			int (height/10 - p3.y*(float(height)/float(ysize))),
			int (width/2 + p1.x*(float(width)/float(ysize))),
			int (height/10 - p1.y*(float(height)/float(xsize))));
		//if (i==2)
		//	i=lim;
		}
	return 0;
}


/*
 *  drawHF3d: draw a height field in 3D perspective as a series of triangles
 */
int HeightFieldDraw::drawhf3d ()
{
	int 	x,z,izi;
	int 	ymin, ymax, xmin, xmax;
	int 	noffx, noffz;
	D 	xd2, yd2, x0, y0, z0, x1, y1, z1, y2, y3;

	xd2 = p_HF->getWidth() / 2.0;
	yd2 = p_HF->getHeight() / 2.0;
	ymin = 0;
	ymax = p_HF->getHeight()-1;
	xmin = 0;
	xmax = p_HF->getWidth()-1;
	if (!d_maxcon) 
		{
		d_cmax = 0; 
		d_cmin = d_ncolors;	// initialize global lims to extrema 
  		}

	/* *** loop over the two array indices: z = into the screen,  *** */
	/* *** x = to right across the screen, y = vertically upwards *** */
	for (z=ymin; z<ymax; z+=d_resolution) 
		{
		izi = (ymax - z) + ymin;
		if ((z+d_resolution) < ymax) 
			noffz = d_resolution;
		else 
			noffz = ymax-z;
		for (x=xmin; x<xmax; x+=d_resolution) 
			{
			if ((x+d_resolution) < xmax) 
				noffx = d_resolution;
			else 
				noffx = xmax-x;
			x0 = (x-xd2)/xd2; z0 = (izi-yd2)/yd2; 
			x1 = (x+noffx-xd2)/xd2; z1 = (izi-noffz-yd2)/yd2; 
			if (b_sea)
				{
				y0 = (D) p_HF->getElmodSea(x,z);
				y1 = (D) p_HF->getElmodSea(x+noffx,z+noffz);
				y2 = (D) p_HF->getElmodSea(x+noffx,z);
				y3 = (D) p_HF->getElmodSea(x,z+noffz);
				tri3d(x0,z0,x1,z1,y0,y1,y2,y3,p_HF->Pos2Off(x,z)); 
				}
			else
				{
				y0 = (D) p_HF->getElmod(x,z);
				y1 = (D) p_HF->getElmod(x+noffx,z+noffz);
				y2 = (D) p_HF->getElmod(x+noffx,z);
				y3 = (D) p_HF->getElmod(x,z+noffz);
				tri3d(x0,z0,x1,z1,y0,y1,y2,y3,p_HF->Pos2Off(x,z)); 
				}
			// if (d_shader==4 || d_shader==5) { col = (int) (d_ncolors * p_HF->getEl (x*p_HF->getWidth()/d_xgsize,z*p_HF->getHeight()/d_ygsize)); }
			tri3d(x0,z0,x1,z1,y0,y1,y2,y3,p_HF->Pos2Off(x,z)); 
			} 
		} 

	return (0);
} 


/*
 *  tri3d: project two triangles, given 3D coordinates, on a 2D viewscreen
 *  ===============================================================
 *  project two triangles, given 3D coordinates, on a 2D viewscreen
 *  uses gobal vars:
 *   ysf : vertical heightfield scaling factor
 *   ex,ey,ez : location of eye (viewpoint)
 *   d_symid : y coordinate midpoint
 *   ycorr : y correction factor
 *   yoff  : y offset in screen units 
 *   zcenter : center to 'rotate' appparent hf around
 *   ncolors  : maximum color value 
 *   bright  : lightsource intensity 
 * ===============================================================
 *  shader: 0 = wireframe, 1 = height color, 2 = slope, 3 = lightsource
 */
int HeightFieldDraw::tri3d(D x0, D z0,D x1,D z1,D y0,D y1,D y2,D y3, int offset)
{
	D	zmod0, zmod1, slope1, slope2; 
	int 	c1=0, c2=0, c3=0;		// colors 
	D	nx,ny,nz,nmag;   		// normal vector 
	D	ctheta1, ctheta2;          	// cos(theta)  theta = angle between light and normal */
	PTYPE	elv = p_HF->getEl(offset), 
		sl = p_HF->getSealevel();
	int	land_bands = d_nbands-d_bandLandOffset;

	/* three sets of points in 2D -- but must alloc 4 for edge lines */ 
	//PT2	p[4];
	//XPoint p[4];
	GdkPoint p[4];

/*
	printf("t3d %1.3f %1.3f %1.3f %1.3f",x0,z0,x1,z1);
	printf(" %1.3f %1.3f %1.3f %1.3f ",y0,y1,y2,y3);
	printf(" * %d %d ",d_shader,col); 
	fflush(stdout);
*/

	/* *** x0,y0,z0 should be in the range -1..1 		*** */
	/* *** transform to eye coordinate (eye.x,eye.y,eye.z) 	*** */
	y0 *= -d_ysf; y1 *= -d_ysf; y2 *= -d_ysf; y3 *= -d_ysf;                          
	x0 -= d_eye.x; y0 -= d_eye.y; z0 -= d_eye.z;
	x1 -= d_eye.x; y1 -= d_eye.y; z1 -= d_eye.z;
	y2 -= d_eye.y; y3 -= d_eye.y; 
	if ((z1 <= d_zeps) || (z0 <= d_zeps)) 
		return (1);  			// foreground clipping 


	if (d_shader == WIRE) 			// Wireframe 
		{ 
		if (p_cMap->d_linear)
			{
			// white, we hope. brightest color, anyway 
			c1=(int)(d_ncolors*(elv/(p_HF->getMax()-(p_HF->getMin()))));
			c2=0;
			}
		else
			{
			if (elv <= sl)
				{
				c1=(int)(ABS((sl-(sl-elv))/d_nrange*(d_bandLandOffset)));
				c1*=d_bandsize;
				}
			else
				{
				c1=(int)(ABS(((elv-sl)/d_prange) * (land_bands)));
				c1*=d_bandsize;
				//c1+=(d_bandsize*(d_bandLandOffset+1));
				c1+=(d_bandsize*(d_bandLandOffset));
				}
			c2=0;
			}
		}
	else
	if (d_shader == HEIGHT) 			// Height, before transform 
		{
		if (p_cMap->d_linear)
			{
			c1=(int)(d_ncolors*(elv/(p_HF->getMax()-(p_HF->getMin()))));
			c2=0;
			}
		else
			{
			if (elv <= sl)
				{
				c2=(int)(ABS((sl-(sl-elv))/d_nrange*(d_bandLandOffset)));
				c2*=d_bandsize;
				}
			else
				{
				c2=(int)(ABS(((elv-sl)/d_prange) * (land_bands)));
				c2*=d_bandsize;
				//c2+=(d_bandsize*(d_bandLandOffset+1));
				c2+=(d_bandsize*(d_bandLandOffset));
				}
			c1=c2+d_bandsize/2;
			}
		} 
	else 
	if (d_shader == LIGHT) 			// compute normal, dot product 
		{				// with lightsource 
		nx = (z0-z1)*(y2-y0);
		ny = (z1-z0)*(x1-x0);
		nz = (x1-x0)*(y2-y1);
		nmag = sqrt(nx*nx + ny*ny + nz*nz);
		/* printf("nxyz %1.3f %1.3f %1.3f  nmag %1.3f ",nx,ny,nz,nmag); fflush(stdout); */
		if (nmag==0) 
			ctheta1 = 0;
		else 
			ctheta1 = (nx*d_light.x + ny*d_light.y + nz*d_light.z) / nmag;

		nx = -(z0-z1)*(y3-y1);
		/* ny = (z1-z0)*(x1-x0); */   /* duplicates line above */
		nz = (x0-x1)*(y3-y0);
		nmag = sqrt(nx*nx + ny*ny + nz*nz);
		if (nmag==0) 
			ctheta2 = 0;
		else
			ctheta2 = (nx*d_light.x + ny*d_light.y + nz*d_light.z) / nmag;

		if (p_cMap->d_linear)
			{
			c1 = (int) ((d_ncolors>>1)*(d_bright*(ctheta1 + 1.0)));
			c2 = (int) ((d_ncolors>>1)*(d_bright*(ctheta2 + 1.0)));
			}
		else
			{
			float cbi; 	// color band index, color band offset

			if (ctheta1 < 0)
				ctheta1 = 0;
			if (ctheta2 < 0)
				ctheta2 = 0;

			if (elv <= sl)
				{
				cbi=(int)(ABS(((sl-(sl-elv))/d_nrange*(d_bandLandOffset))));
				cbi*=d_bandsize;
				}
			else
				{
				cbi=(int)(ABS(((elv-sl)/d_prange) * (land_bands)));
				cbi*=d_bandsize;
				cbi+=(d_bandsize*(d_bandLandOffset));
				}
			c1=(int)(cbi+d_bandsize*ctheta1);
			c2=(int)(cbi+d_bandsize*ctheta2);

			if (c1 < 0 || c2 < 0)
				{
				printf ("C1 = %d, C2 = %d\n", c1, c2);
				printf ("cbi = %d\n", (int)(ABS(((sl-(sl-elv))/d_nrange*(d_bandLandOffset)))));
				printf ("ctheta1 = %f, ctheta2 = %f\n", ctheta1, ctheta2);
				exit (0);
				}
			}
		}

	if (d_maxcon &&(d_cmax!=d_cmin)) 		// if maximize_contrast is TRUE... 
		{
/*
		c1 = (int) (((D)d_ncolors/(d_cmax-d_cmin))*(c1-d_cmin));
		c2 = (int) (((D)d_ncolors/(d_cmax-d_cmin))*(c2-d_cmin));
*/
		c1 = (int) (((D)d_ncolors/d_cmax)*(c1));
		c2 = (int) (((D)d_ncolors/d_cmax)*(c2));
		} 

	char 	t[80];

	sprintf (t, "Colormap Lookup RangeError: no color %d in c1", c1);
	if (SanityCheck::warning ((c1 < 0 || c1 > d_ncolors), 
			t, "HeightFieldGrid::trid3d"))
		{
		if (c1 < 0)
			c1 = 0;
		else 
			c1 = d_ncolors-1;
		}

	sprintf (t, "Colormap Lookup RangeError: no color %d in c2", c2);
	if (SanityCheck::warning ((c2 < 0 || c2 > d_ncolors), 
			t, "HeightFieldGrid::trid3d"))
		{
		if (c2 < 0)
			c2 = 0;
		else 
			c2 = d_ncolors-1;
		}

	if (!d_maxcon) 
		{
		if (c1 > d_cmax) 
			d_cmax = c1;
		if (c1 < d_cmin) 
			d_cmin = c1;
		}

	zmod0 = z0 * d_zsf; 		// z1 multiplied by z scale factor
	zmod1 = z1 * d_zsf;

	p[0].x = (int) (d_sxmid*x0/zmod0) + d_sxmid;  	// x,y,z 0 
	p[0].y = (int) (d_symid*y0/zmod0) + d_symid + d_yoff;
 
	p[1].x = (int) (d_sxmid*x1/zmod1) + d_sxmid;  	// x,y,z 1 
	p[1].y = (int) (d_symid*y1/zmod1) + d_symid + d_yoff;

	p[2].x = (int) (d_sxmid*x1/zmod0) + d_sxmid; 	// x,y,z 2 
	p[2].y = (int) (d_symid*y2/zmod0) + d_symid + d_yoff;

	/* *** we're off the sceen. no need to do anything else *** */
	if ( ((p[0].x<0) && (p[1].x<0) && (p[2].x<0)) ||
	     ((p[0].x>d_xgsize) && (p[1].x>d_xgsize) && (p[2].x>d_xgsize)) ||
	     ((p[0].y<0) && (p[1].y<0) && (p[2].y<0)) ||
	     ((p[0].y>d_ygsize) && (p[1].y>d_ygsize) && (p[2].y>d_ygsize))  )
		return (1);
		
	if ( ((p[0].x==p[1].x) && (p[0].y==p[1].y)) ||
	     ((p[0].x==p[2].x) && (p[0].y==p[2].y)) ||
	     ((p[1].x==p[2].x) && (p[1].y==p[2].y)) )  
		{

		//gdk_draw_points (da->window, da->style->black_gc, p, 1);
		p_drawArea->drawPoint (p[0].x, p[0].y);

		}

	else 
		{
		if (d_shader == HEIGHT)
			{
			p_drawArea->drawBorderedPolygon (p, 3, 
					p_cMap->getColor(c2), 
					p_cMap->getColor(c1));

			}
		else if (d_shader == WIRE) 
			{
			if (p_cMap->d_linear)
				c3=(int)(d_ncolors*(elv/(p_HF->getMax()-(p_HF->getMin()))));
			else
				c3=c1;
			p_drawArea->setColors (p_cMap->getColor (c3));
			p_drawArea->drawPolygon (p, 3, FALSE);
			} 
		else 
			{
			p_drawArea->drawBorderedPolygon (p, 3, 
					p_cMap->getColor(c1), 
					p_cMap->getColor(c1));
			
			}
		}

	p[1].x = (int) (d_sxmid*x0/zmod0) + d_sxmid;	// x,y,z 0 
	p[1].y = (int) (d_symid*y0/zmod0) + d_symid + d_yoff;

	p[0].x = (int) (d_sxmid*x1/zmod1) + d_sxmid;	// x,y,z 1 
	p[0].y = (int) (d_symid*y1/zmod1) + d_symid + d_yoff;

	p[2].x = (int) (d_sxmid*x0/zmod1) + d_sxmid; 	// x,y,z 3 
	p[2].y = (int) (d_symid*y3/zmod1) + d_symid + d_yoff; 

	if( ((p[0].x==p[1].x) && (p[0].y==p[1].y)) ||
	    ((p[0].x==p[2].x) && (p[0].y==p[2].y)) ||
	    ((p[1].x==p[2].x) && (p[1].y==p[2].y)) )  
		{

		// gdk_draw_points (da->window, da->style->black_gc, p, 1);
		p_drawArea->drawPoint (p[0].x, p[0].y);
		}
	else 
		{
		if (d_shader == HEIGHT)  
			{
			p_drawArea->drawBorderedPolygon (p, 3, 
					p_cMap->getColor(c1), 
					p_cMap->getColor(c2));
			}
		else if (d_shader == WIRE) 
			{
			if (p_cMap->d_linear)
				c3=(int)(d_ncolors*(elv/(p_HF->getMax()-(p_HF->getMin()))));
			else
				c3=c2;
			p_drawArea->setColors (p_cMap->getColor (c1));
			p_drawArea->drawPolygon (p, 3, FALSE);
					//p_cMap->getColor(c3), 
					//p_cMap->getColor(c1));
			} 
		else 
			{
			p_drawArea->drawBorderedPolygon (p, 3, 
					p_cMap->getColor(c2), 
					p_cMap->getColor(c2));
			}
		}

	return (0);
} 


/*
 *  printSettings: print the settings 
 */
void HeightFieldDraw::printSettings ()
{
	printf ("*************** HFD Parameters *************\n");
	printf ("Graph Size		: %d, %d\n", d_xgsize, d_ygsize);
	printf ("Eye Coords		: %f, %f, %f\n", d_eye.x,  d_eye.y, d_eye.z);
	printf ("Light Vector		: %f, %f, %f\n", d_light.x, d_light.y, d_light.z);
	printf ("Scale, Vertical Scale	: %f, %f\n", d_scale, d_ysf);
	printf ("Lightsource Intensity	: %f\n", d_bright);
	printf ("Contrast Maximization	: %s\n", (d_maxcon ? "ON" : "OFF"));
	printf ("Color min, max		: min=%d, max=%d\n", d_cmin, d_cmax);
	printf ("Grid Spacing		: %d\n", d_resolution);
	printf ("Shading Mode		: %d\n", d_shader);
	printf ("Clear			: %s\n", (d_erasewin ? "ON":"OFF"));
	printf ("HF Colors, Bands, C/B	: %d, %d, %d\n", d_ncolors, d_nbands, d_bandsize);
	printf ("Land cols begin@band	: %d\n\n", d_bandLandOffset);
	fflush (stdout);
}



/* 
 * help: print the help for commands into the window 
 */
void HeightFieldDraw::help ()              /* print help message */
{
	printf ("*************** hf3d Help *************\n");
	printf ("You can enter whitespace delimeted commands into the display window\n");
	printf ("Shading: S0=wireframe, S1=height-shading, S2=slope, S3=lightsource\n");
	printf ("Vector to lightsource: Lx,y,z (affects S3 mode only)\n");
	printf ("Eye Position: Ex,y,z\n");
	printf ("Eye Movement: R=right, L=left, U=Up, D=Down, B=back, F=forward (delta=0.1)\n");
	printf ("Contrast: M=toggle 'maximize contrast' mode\n");
	printf ("Brightness: '>'=brighter, '<'=darker (delta=10%%)\n");
	printf ("Verstical Scaling: '+'=increase, '-'=decrease (delta=10%%)\n");
	printf ("Resolution: Pn=draw only every nth line, for faster preview.\n");
	printf ("Settings: V\n");
	printf ("Quit: q\n");
}


static double ATAN2 (double y, double x)
{
	return ( ((x==0&&y==0) ? 0 : atan2(y,x)) );
}


/*
 * getColorElv: returns the GdkColor value for a given elevation, using the
 * current settings for color map, ocean is filled, etc.  most of this code
 * is shamelessly extracted from the original drawhf2d() function.
 */
GdkColor *HeightFieldDraw::getColorElv(PTYPE elv)
{
	int			icol;		// color index for a given point
	float			elvRatio;	// elevation ratio

	if (!p_HF->getData())
		return ((GdkColor *)NULL);

	if (p_cMap->d_linear)
		icol=(int)(elv*d_ncolors);
	else
		{
		if (elv <= p_HF->getSealevel())
			{
			elvRatio = elv/d_nrange;
			icol=(int) (elvRatio * d_nWaterColors);
			}
		else
			{
			elvRatio = (elv-p_HF->getSealevel())/d_prange;
			icol=(int) (elvRatio * d_nLandColors + d_nWaterColors);
			}
		}
	return (GdkColor *)(p_cMap->getColor(icol));
}


/* 
 * getColor2d: return the GdkColor value corresponding to an x,y location on
 * the heightfield, using the current settings for color map, ocean is filled,
 * etc.
 */
GdkColor *HeightFieldDraw::getColor2d(int x, int y)
{
	return getColorElv (p_HF->getEl(x,y));
}
