/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef _FLEX_ARRAY_H
#define _FLEX_ARRAY_H

#include "glib.h"
#include "string.h"
#include "GlobalSanityCheck.h"


/*
 *  FlexArray: an array which automatically extends (using ever bigger memory 
 *	chunks) as new data is extended and provides fast access.
 */
class FlexArray
	{
	public:
				FlexArray (int initialSize=100, float growthFactor=2.0);
		virtual		~FlexArray ();
		void 		append (void *ptr);
		void 		*El (int i);
		void		*first ();
		void		*last ();
		void		trim ();
		int		remove (int loc, int n=1 );
		int		removeLast (int n=1);
		int		getSize () { return d_nElements; }
		void		*getData () { return d_array; }

	protected:
		void		extendArray ();
		virtual void	clearArray () 		{;}

		void			**d_array;
		int			d_nElements,
					d_len,
					d_inc;
		float			d_incFactor;

	// C++ sugar
	public:
		bool operator==(const FlexArray & rhs)
			{
			if (d_array && rhs.d_array)
				return (d_array==rhs.d_array);
			return (FALSE);
			}
	};


inline FlexArray::FlexArray(int initialSize, float growthFactor)
{
	d_array = NULL;
	d_inc = initialSize;
	d_incFactor = growthFactor;
	d_nElements = 0;
	d_len = 0;
}


// user must clear array before destroying it!
inline FlexArray::~FlexArray()
{
	if (d_array)
		{
		clearArray ();
		delete [] d_array;
		}
}


inline void *FlexArray::El (int i)
{
	if (i > d_nElements-1)
		return NULL;
	else
		return d_array[i];
}


inline void *FlexArray::first ()
{
	if (!d_array)
		return NULL;
	else
		return d_array[0];
}


inline void *FlexArray::last ()
{
	if (!d_array)
		return NULL;
	else
		return d_array[d_nElements-1];
}



inline void FlexArray::append (void *ptr)
{
	if (d_nElements >= d_len)
		extendArray ();
	d_array[d_nElements] = ptr;
	d_nElements++;
}


inline int FlexArray::remove (int loc, int n)
{
	if (d_nElements <= loc)
		return -1;

	if (loc+n > d_nElements)
		return -1;

	if (loc == d_nElements && n==1)
		d_array[d_nElements] = NULL;
	else
		{
		void	**newArr = NULL;
		int	off = loc+n;

		newArr = new void*[d_len];
		memcpy (newArr, d_array, loc*sizeof(void*));
		memcpy (newArr+loc, d_array+off, (d_len-off)*sizeof(void*));
		if (d_array)
			delete [] d_array;
		d_array = newArr;
		}

	d_nElements-=n;
	return 0;
}


inline int FlexArray::removeLast (int n)
{
	if (!d_nElements)
		return -1;

	for (int i=d_nElements-n; i<d_nElements; i++)
		d_array[i] = NULL;

	d_nElements-=n;
	return 0;
}
		

// shrink array so that it's as big as the # of elements it holds
inline void FlexArray::trim ()
{
	void	**newArr = NULL;

	if (!d_array)
		return;

	if (d_nElements == d_len)
		return;

	newArr = new void*[d_nElements];
	memcpy (newArr, d_array, d_nElements*sizeof(void*));
	delete [] d_array;
	d_len = d_nElements;
	d_array = newArr;
}


inline void FlexArray::extendArray ()
{
	void	**newArr = NULL;
	int	blen = d_nElements*sizeof(void*); // byte length of used area

	//printf ("Allocating %d array cells\n", d_inc);
	newArr = new void*[d_len+d_inc];
	memcpy (newArr, d_array, blen);
	memset (newArr+d_len, 0, d_inc*sizeof(void*));
	if (d_array)
		delete [] d_array;
	d_len += d_inc;
	d_inc = (int)(d_inc*d_incFactor);
	d_array = newArr;
}

#endif // _FLEX_ARRAY_H
