
/*
 * some svga rotines
 *
 *   (c) 1996 Gerd Knorr <kraxel@goldbach.in-berlin.de>
 *
 * svga_guess_mode     - looks for (and switches on) a video mode
 * svga_display_image  - copyes a image to the screen
 * svga_gray_palette   - set vga palette for grayscaled display (256 colos)
 * svga_dither_palette - set vga palette for dithered color (256 colors)
 * svga_cls            - clears the screen
 * svga_show           - a more complex display function,
 *                       allowes scrolling with cursor keys etc...
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <vga.h>
#include <vgagl.h>

#include "svga.h"

/* Set this to use HiColor per default. If not defined, HiColor is
 * used only if the whole image fits in
 */
#define USE_HICOLOR 0

#define BANKSIZE    0x10000

unsigned char  *vgamem;

/* ---------------------------------------------------------------------- */

vga_modeinfo   *
svga_guess_mode(int width, int height, int colors, int mode)
{
    /* best mode found: width,height,colors,mode-nr */
    int             bx = 0, by = 0, bc = 0, bnr = 0;
    vga_modeinfo   *vgamode;
    int             i;

    for (i = 1; NULL != (vgamode = vga_getmodeinfo(i)); i++) {
	if (!vga_hasmode(i))
	    continue;		/* not supported by hardware */
	if (vgamode->flags & 4)
	    continue;		/* they don't work */
	if (vgamode->colors < 256)
	    continue;		/* less than 256 colors */
	if (mode != 0 && mode != i)
	    continue;		/* user forced a mode */
	if (colors != 0 && colors != vgamode->colors)
	    continue;		/* user selected color depth */

	/* TrueColor (and HiColor) are allowed only if the whole
	 * image fits in, as long the user does not explicit sets
	 * color depth or graphics mode */
#if USE_HICOLOR
	if (vgamode->colors == 16 * 1024 * 1024 && (mode | colors) == 0)
#else
	if (vgamode->colors >= 32 * 1024 && (mode | colors) == 0)
#endif
	    if (vgamode->width < width || vgamode->height < height)
		continue;

	/* "current best" has more colors, try next */
	if (bc > vgamode->colors)
	    continue;

	/* "current best" has less colors, take it */
	if (bc < vgamode->colors) {
	    bx = vgamode->width, by = vgamode->height, bc = vgamode->colors, bnr = i;
	    continue;
	}
	/* image does not fit into "current best", and we have a larger
	 * one, take it */
	if ((bx < width && bx < vgamode->width) ||
	    (by < height && by < vgamode->height)) {
	    bx = vgamode->width, by = vgamode->height, bc = vgamode->colors, bnr = i;
	    continue;
	}
	/* this one is smaller than the "current best", but the image still
	 * fit in, take it */
	if ((vgamode->width < bx || vgamode->height < by) &&
	    vgamode->width >= width && vgamode->height >= height) {
	    bx = vgamode->width, by = vgamode->height, bc = vgamode->colors, bnr = i;
	    continue;
	}
    }

    if (bnr) {
	vga_setmode(bnr);
	gl_setcontextvga(bnr);
	vgamem = vga_getgraphmem();
	return vga_getmodeinfo(bnr);
    } else
	return NULL;
}

void
svga_list_modes()
{
    vga_modeinfo   *vgamode;
    int             i;

    for (i = 1; NULL != (vgamode = vga_getmodeinfo(i)); i++) {
	if (!vga_hasmode(i))
	    continue;		/* not supported by hardware */
	if (vgamode->flags & 4)
	    continue;		/* they don't work */
	if (vgamode->colors < 256)
	    continue;		/* less than 256 colors */

	printf("%2d: %dx%d, %d colors\n",
	       i, vgamode->width, vgamode->height, vgamode->colors);
    }
}

/* ---------------------------------------------------------------------- */

void
svga_display_image(vga_modeinfo * vgamode, char *image,
		   int width, int height, int xoff, int yoff)
{
    int             dwidth = MIN(width, vgamode->width);
    int             dheight = MIN(height, vgamode->height);
    int             data, video, bank, offset, bytes;

    bytes = vgamode->bytesperpixel;

    /* offset for image data (image > screen, select visible area) */
    offset = (yoff * width + xoff) * bytes;

    /* offset for video memory (image < screen, center image) */
    video = 0, bank = 0;
    if (width < vgamode->width)
	video += bytes * (vgamode->width - width) / 2;
    if (height < vgamode->height)
	video += bytes * vgamode->width * (vgamode->height - height) / 2;
    while (video > BANKSIZE)
	bank++, video -= BANKSIZE;
    vga_setpage(bank);

    /* go ! */
    for (data = 0;
	 data < width * height * bytes && data / width / bytes < dheight;
	 data += width * bytes, video += vgamode->linewidth) {
	if (video >= BANKSIZE) {
	    vga_setpage(++bank);
	    video -= BANKSIZE;
	}
	if (video + dwidth * bytes >= BANKSIZE) {
	    /* zeile mit bank grenze */
	    memcpy(&vgamem[video], &image[data + offset], BANKSIZE - video);
	    vga_setpage(++bank);
	    video -= BANKSIZE;
	    memcpy(vgamem, &image[data + offset - video], dwidth * bytes + video);
	} else {
	    memcpy(&vgamem[video], &image[data + offset], dwidth * bytes);
	}
    }
}

/* ---------------------------------------------------------------------- */

void
svga_gray_palette()
{
    int             i;

    for (i = 0; i < 256; i++)
	vga_setpalette(i, i >> 2, i >> 2, i >> 2);
}

void
svga_dither_palette(int r, int g, int b)
{
    int             rs, gs, bs, i;

    rs = 256 / (r - 1);
    gs = 256 / (g - 1);
    bs = 256 / (b - 1);
    for (i = 0; i < 256; i++)
	vga_setpalette(i,
		       (rs * ((i / (g * b)) % r)) >> 2,
		       (gs * ((i / b) % g)) >> 2,
		       (bs * ((i) % b)) >> 2);
}

void
svga_cls(vga_modeinfo * vgamode)
{
    int             size = vgamode->width * vgamode->height * vgamode->bytesperpixel;
    int             bank = 0;

    while (size >= BANKSIZE) {
	vga_setpage(bank++);
	memset(vgamem, 0, BANKSIZE);
	size -= BANKSIZE;
    }
    vga_setpage(bank++);
    memset(vgamem, 0, size);
}

/* ---------------------------------------------------------------------- */

int
svga_show(vga_modeinfo * vgamode, char *image,
	  int width, int height, int timeout)
{
    int             disp = 1, left = 0, top = 0, rc;
    char            key[11], nr = 0;
    fd_set          set;
    struct timeval  limit;

    /* start with centered image, if larger than screen */
    if (width >= vgamode->width)
	left = (width - vgamode->width) / 2;
    if (height >= vgamode->height)
	top = (height - vgamode->height) / 2;

    for (;;) {
	if (disp) {
	    disp = 0;
	    svga_display_image(vgamode, image, width, height, left, top);
	}
	if (0 != timeout) {
	    FD_ZERO(&set);
	    FD_SET(0, &set);
	    limit.tv_sec = timeout;
	    limit.tv_usec = 0;
	    rc = select(1, &set, NULL, NULL, &limit);
	    if (0 == rc)
		return KEY_TIMEOUT;
	}
	rc = read(0, key, 10);
	if (rc < 1) {
	    printf("%d %s\n",rc,strerror(errno));
	    /* EOF */
	    return KEY_EOF;
	}
	key[rc] = 0;

	if (rc == 1 && (*key == 'g' || *key == 'G')) {
	    return nr;
	}
	if (rc == 1 && *key >= '0' && *key <= '9') {
	    nr = nr * 10 + (*key - '0');
	} else {
	    nr = 0;
	}

	if (rc == 1 && (*key == 'q' || *key == 'Q' ||
			*key == 'e' || *key == 'E' ||
			*key == ' ' || *key == '\x1b')) {
	    if (*key == ' ')
		return KEY_SPACE;
	    if (*key == '\x1b')
		return KEY_ESC;
	    return KEY_Q;
	} else if (0 == strcmp(key, "\x1b[A") && height > vgamode->height) {
	    disp = 1;
	    top -= 50;
	    if (top < 0)
		top = 0;
	} else if (0 == strcmp(key, "\x1b[B") && height > vgamode->height) {
	    disp = 1;
	    top += 50;
	    if (top + vgamode->height > height)
		top = height - vgamode->height;
	} else if (0 == strcmp(key, "\x1b[D") && width > vgamode->width) {
	    disp = 1;
	    left -= 50;
	    if (left < 0)
		left = 0;
	} else if (0 == strcmp(key, "\x1b[C") && width > vgamode->width) {
	    disp = 1;
	    left += 50;
	    if (left + vgamode->width > width)
		left = width - vgamode->width;
	} else if (0 == strcmp(key, "\x1b[5~")) {
	    return KEY_PGUP;
	} else if (0 == strcmp(key, "\x1b[6~")) {
	    return KEY_PGDN;
#if 0
	} else {
	    /* testing: find key codes */
	    int             i;

	    vga_setmode(TEXT);
	    fprintf(stderr, "key: `");
	    for (i = 0; i < rc; i++)
		fprintf(stderr, "%s%c",
			key[i] < 0x20 ? "^" : "",
			key[i] < 0x20 ? key[i] + 0x40 : key[i]);
	    fprintf(stderr, "'\n");
	    exit(0);
#endif
	}
    }
}
