// ---------------------------------------------------------------------------
// - Instance.hpp                                                            -
// - aleph engine - aleph instance class definition                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_INSTANCE_HPP
#define  ALEPH_INSTANCE_HPP

#ifndef  ALEPH_CLASS_HPP
#include "Class.hpp"
#endif

namespace aleph {

  /// The Instance class is used as an instance of the Class class. When an
  /// instance is created, all data members are bounded in the instance local
  /// nameset. By default, all data members are set to nil. If a constructor
  /// is invoked with arguments, the data members are set with them.
  /// @author amaury darsch

  class Instance : public Object {
  private:
    /// the meta class
    Class* p_class;
    /// the super instance
    Object* p_super;
    /// the super flag
    bool    d_const;
    /// the instance local set
    Localset* p_iset;

  public:
    /// create a new instance with a class object and arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param args   the arguments to bind
    /// @param cls    the class object to bind
    Instance (Interp* interp, Nameset* nset, Cons* args, Class* cls);

    /// destroy this instance
    ~Instance (void);

    /// @return the class name
    String repr (void) const;

    /// set the super instance value
    /// @param object the super object to set
    /// @param flag   the super const flag
    Object* setsuper (Object* object, const bool flag);

    /// set an object as a const in the instance local set
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to define as const
    /// @param object the object to set
    Object* cdef (Interp* interp, Nameset* nset, const String& name, 
		  Object* object);

    /// set an object in the instance local set
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the name to define as const
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, const String& name, 
		  Object* object);

    /// evaluate an instance member by name
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param name   the member name to evaluate in this object
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply an instance method by name with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name, 
		   Cons* args);

    /// apply an instance method by object with a set of arguments
    /// @param interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, Object* object, Cons* args);

  private:
    // make the copy constructor private
    Instance (const Instance&);
    // make the assignment operator private
    Instance& operator = (const Instance&);
  };
}

#endif
