// ---------------------------------------------------------------------------
// - Boolean.cpp                                                             -
// - standard object library - boolean class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Method.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new boolean - the initial value is false
  
  Boolean::Boolean (void) {
    d_value = false;
  }
  
  // create a boolean from a native value
  
  Boolean::Boolean (const bool value) {
    d_value = value;
  }
  
  // create a boolean from a string

  Boolean::Boolean (const String& value) {
    if (value == "false") 
      d_value = false;
    else if (value == "true")
      d_value = true;
    else
      throw Exception ("literal-error","illegal boolean value",value);
  }

  // copy constructor for this boolean

  Boolean::Boolean (const Boolean& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Boolean::repr (void) const {
    return "Boolean";
  }

  // return a literal representation of this boolean
  
  String Boolean::toLiteral (void) const {
    return toString ();
  }

  // return a string representation of this boolean
  
  String Boolean::toString (void) const {
    return d_value ? "true" : "false";
  }

  // assign a boolean with a native value

  Boolean& Boolean::operator = (const bool value) {
    d_value = value;
    return *this;
  }
  
  // assign a boolean with a boolean
  
  Boolean& Boolean::operator = (const Boolean& value) {
    d_value = value.d_value;
    return *this;
  }
  
  // compare a boolean with a native value
  
  bool Boolean::operator == (const bool value) const {
    return (d_value == value);
  }
  
  // compare two boolean
  
  bool Boolean::operator == (const Boolean& value) const {
    return (d_value == value.d_value);
  }
  
  // compare a boolean with a native value
  
  bool Boolean::operator != (const bool value) const {
    return (d_value != value);
  }
  
  // compare two boolean
  
  bool Boolean::operator != (const Boolean& value) const {
    return (d_value != value.d_value);
  }

  // operate this boolean with another object

  Object* Boolean::oper (t_oper type, Object* object) {
    Boolean* bobj = dynamic_cast <Boolean*> (object);
    switch (type) {
    case Object::EQL:
      if (bobj != nilp) return new Boolean (d_value == bobj->d_value);
      break;
    case Object::NEQ:
      if (bobj != nilp) return new Boolean (d_value != bobj->d_value);
      break;
    default:
      throw Exception ("operator-error", "unsupported boolean operator");
    }
    throw Exception ("type-error", "invalid operand with boolean",
		     Object::repr (object));
  }

  // create a new boolean in a generic way

  Object* Boolean::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Boolean;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with boolean constructor");
    // try to map the boolean argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Boolean;

    // try a boolean object
    Boolean* bval = dynamic_cast <Boolean*> (obj);
    if (bval != nilp) return new Boolean (*bval);

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Boolean (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with boolean constructor",
		     obj->repr ());
  }

  // set an object to this boolean

  Object* Boolean::vdef (Interp* interp, Nameset* nset, Object* object) {
    Boolean* bobj = dynamic_cast <Boolean*> (object);
    if (bobj != nilp) {
      d_value = bobj->d_value;
      return this;
    }
    throw Exception ("type-error", "invalid object with boolean vdef",
		     Object::repr (object));
  }

  // evaluate this boolean with a member name

  Object* Boolean::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this boolean with a method name

  Object* Boolean::apply (Interp* interp, Nameset* nset, const String& name,
			  Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) delete argv;
    if ((name == "to-string") && (argc == 0)) 
      return new String (toLiteral ());

    // dispatch one argument
    if ((name == "==") && (argc == 1)) {
      Object* result = oper (Object::EQL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "!=") && (argc == 1)) {
      Object* result = oper (Object::NEQ, argv->get (0));
      delete argv;
      return result;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
