// ---------------------------------------------------------------------------
// - Library.cpp                                                             -
// - standard object library - shared library class implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cdl.hxx"
#include "String.hpp"
#include "Vector.hpp"
#include "Method.hpp"
#include "Library.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new shared library object

  Library::Library (const String& name) {
    char* lname = name.toChar ();
    p_handle = c_dlopen (lname);
    delete [] lname;
    if (p_handle == nilp) 
      throw Exception ("name-error", "cannot open dynamic library", name);
  }

  // delete this shared library by closing it

  Library::~Library (void) {
    c_dlclose (p_handle);
  }
  
  // return the class name

  String Library::repr (void) const {
    return "Library";
  }

  // create a new shared library in a generic way

  Object* Library::mknew (Vector* argv) {
    long len = (argv == nilp) ? 0 : argv->length ();
    if (len != 0)
      throw Exception ("argument-error", 
		       "invalid number of arguments with library");
    String name = argv->getstring (0);
    return new Library (name);
  }

  // evaluate a library member name

  Object* Library::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply in this library with a given name
  Object* Library::apply (Interp* interp, Nameset* nset, const String& name,
			  Cons* args) {
    typedef Object* (*t_func) (Interp*, Nameset*, Cons*);

    // try to find the symbol
    char* sname = name.toChar ();
    void* sym   = c_dlsym (sname, p_handle);
    delete [] sname;
    if (sym == nilp) 
      throw Exception ("name-error", "cannot find dynamic symbol", name);
    // map the symbol to a procedure and call it
    t_func func = (t_func) sym;
    return func (interp, nset, args);
  }
}
