// ---------------------------------------------------------------------------
// - Real.cpp                                                                -
// - standard object library - real class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cmath.hxx"
#include "cstring.hxx"
#include "Real.hpp"
#include "Method.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // the default precision
  t_real Real::d_precision = 0.00001;

  // create a new default real

  Real::Real (void) {
    d_value = 0.0;
  }

  // create a new real from a native real

  Real::Real (const t_real value) {
    d_value = value;
  }

  // create a new real from an Integer

  Real::Real (const Integer& value) {
    d_value = value.d_value;
  }

  // create a new real from a string

  Real::Real (const String& value) {
    bool status = false;
    const char* data = value.toChar ();
    d_value = c_atod (data,status);
    delete [] data;
    if (status == false) 
      throw Exception ("literal-error", "illegal string real number",value);
  }
  
  // copy constructor for this real

  Real::Real (const Real& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Real::repr (void) const {
    return "Real";
  }

  // return a literal representation of this real

  String Real::toLiteral (void) const {
    return toString ();
  }

  // return a string representation of this real

  String Real::toString (void) const {
    char* buffer = c_dtoa (d_value);
    String result (buffer);
    delete [] buffer;
    return result;
  }

  // return a format string of this real

  String Real::format (const long precision) const {
    if (precision < 0) 
      throw Exception ("precision-error",
		       "invalid negative precision with real format");
    char* buffer = c_dtoap (d_value, precision);
    String result (buffer);
    delete [] buffer;
    return result;
  }

  // return the double representation

  t_real Real::toReal (void) const {
    return d_value;
  }

  // get an integer from this real
  
  t_long Real::toInteger (void) const {
    return (t_long) d_value;
  }
  
  // set an real with a value

  Real& Real::operator = (const double value) {
    d_value = value;
    return *this;
  }

  // set an real with a value

  Real& Real::operator = (const Real& value) {
    d_value = value.d_value;
    return *this;
  }

  // compare an real with a native value
  
  bool Real::operator == (const long value) const {
    return (d_value == value);
  }
  
  // compare an real with a native value
  
  bool Real::operator != (const long value) const {
    return (d_value != value);
  }

  // compare two reals

  bool Real::operator == (const Real& value) const {
    return (d_value == value.d_value);
  }

  bool Real::operator != (const Real& value) const {
    return (d_value != value.d_value);
  }

  bool Real::operator < (const Real& value) const {
    return (d_value < value.d_value);
  }

  bool Real::operator <= (const Real& value) const {
    return (d_value <= value.d_value);
  }

  bool Real::operator > (const Real& value) const {
    return (d_value > value.d_value);
  }

  bool Real::operator >= (const Real& value) const {
    return (d_value >= value.d_value);
  }

  // add two double reals together from two double reals

  Real operator + (const Real& x, const Real& y) {
    return Real (x.d_value + y.d_value);
  }

  // add a real to this one

  Real& Real::operator += (const Real& x) {
    d_value += x.d_value;
    return *this;
  }

  // substract a real to this one

  Real& Real::operator -= (const Real& x) {
    d_value -= x.d_value;
    return *this;
  }

  // substract two double reals together from two double reals

  Real operator - (const Real& x, const Real& y) {
    return Real (x.d_value - y.d_value);
  }

  // compute the opposite of a real

  Real operator - (const Real& x) {
    return Real (-x.d_value);
  }

  // multiply two double reals together from two double reals

  Real operator * (const Real& x, const Real& y) {
    return Real (x.d_value * y.d_value);
  }

  // multiply a real with this one

  Real& Real::operator *= (const Real& x) {
    d_value *= x.d_value;
    return *this;
  }

  // divide two double reals together from two double reals

  Real operator / (const Real& x, const Real& y) {
    return Real (x.d_value / y.d_value);
  }

  // return true if the number is nan

  bool Real::isnan (void) const {
    return c_isnan (d_value);
  }

  // return the ceiling of this number

  Real Real::ceiling (void) const {
    double result = c_ceiling (d_value);
    return Real (result);
  }

  // return the floor of this number

  Real Real::floor (void) const {
    double result = c_floor (d_value);
    return Real (result);
  }

  // return the absolute value of this number
  
  Real Real::abs (void) const {
    double result = c_abs (d_value);
    return Real (result);
  }
  
  // return the remainder of this number with its argument
  
  Real Real::mod (const Real& x) const {
    double result = c_mod (d_value,x.d_value);
    return Real (result);
  }
  
  // return the square root of this real
  
  Real Real::sqrt (void) const {
    bool   status = false;
    double result = c_sqrt (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with sqrt call");
    return Real (result);
  }
  
  // return the natural logarithm of this real
  
  Real Real::log (void) const {
    bool   status = false;
    double result = c_log (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with log call");
    return Real (result);
  }
  
  // return the exponential of this number
  
  Real Real::exp (void) const {
    double result = c_exp (d_value);
    return Real (result);
  }
  
  // return the power of this number with the argument
  
  Real Real::pow (const Real& x) const {
    double result = c_pow (d_value,x.d_value);
    return Real (result);
  }
  
  // return the sine of this number
  
  Real Real::sin (void) const {
    double result = c_sin (d_value);
    return Real (result);
  }
  
  // return the cosine of this number
  
  Real Real::cos (void) const {
    double result = c_cos (d_value);
    return Real (result);
  }
  
  // return the tangent of this number
  
  Real Real::tan (void) const {
    double result = c_tan (d_value);
    return Real (result);
  }
  
  // return the arc sine of this number
  
  Real Real::asin (void) const {
    bool   status = false;
    double result = c_asin (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with asin call");
    return Real (result);
  }
  
  // return the arc cosine of this number
  
  Real Real::acos (void) const {
    bool   status = false;
    double result = c_acos (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with acos call");
    return Real (result);
  }
  
  // return the arc tangent of this number
  
  Real Real::atan (void) const {
    bool   status = false;
    double result = c_atan (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with atan call");
    return Real (result);
  }
  
  // return the hyperbolic sine of this number
  
  Real Real::sinh (void) const {
    bool   status = false;
    double result = c_sinh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with sinh call");
    return Real (result);
  }
  
  // return the hyperbolic cosine of this number
  
  Real Real::cosh (void) const {
    bool   status = false;
    double result = c_cosh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with cosh call");
    return Real (result);
  }

  // return the hyperbolic tangent of this number

  Real Real::tanh (void) const {
    bool   status = false;
    double result = c_tanh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with tanh call");
    return Real (result);
  }
  
  // return the hyperbolic arc sine of this number

  Real Real::asinh (void) const {
    bool   status = false;
    double result = c_asinh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with asinh call");
    return Real (result);
  }
  
  // return the hyperbolic arc cosine of this number
  
  Real Real::acosh (void) const {
    bool   status = false;
    double result = c_acosh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with acosh call");
    return Real (result);
  }

  // return the hyperbolic arc tangent of this number
  
  Real Real::atanh (void) const {
    bool   status = false;
    double result = c_atanh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with atanh call");
    return Real (result);
  }

  // operate this real with another object

  Object* Real::oper (t_oper type, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    Real*    robj = dynamic_cast <Real*>    (object);
    switch (type) {
    case Object::ADD:
      if (iobj != nilp) return new Real (d_value + iobj->d_value);
      if (robj != nilp) return new Real (d_value + robj->d_value);
      break;
    case Object::SUB:
      if (iobj != nilp) return new Real (d_value - iobj->d_value);
      if (robj != nilp) return new Real (d_value - robj->d_value);
      break;
    case Object::MUL:
      if (iobj != nilp) return new Real (d_value * iobj->d_value);
      if (robj != nilp) return new Real (d_value * robj->d_value);
      break;
    case Object::DIV:
      if (iobj != nilp) return new Real (*this / *iobj);
      if (robj != nilp) return new Real (*this / *robj);
      break;
    case Object::EQL:
      if (iobj != nilp) return new Boolean (d_value == iobj->d_value);
      if (robj != nilp) return new Boolean (d_value == robj->d_value);
      break;
    case Object::NEQ:
      if (iobj != nilp) return new Boolean (d_value != iobj->d_value);
      if (robj != nilp) return new Boolean (d_value != robj->d_value);
      break;
    case Object::GEQ:
      if (iobj != nilp) return new Boolean (d_value >= iobj->d_value);
      if (robj != nilp) return new Boolean (d_value >= robj->d_value);
      break;
    case Object::GTH:
      if (iobj != nilp) return new Boolean (d_value > iobj->d_value);
      if (robj != nilp) return new Boolean (d_value > robj->d_value);
      break;
    case Object::LEQ:
      if (iobj != nilp) return new Boolean (d_value <= iobj->d_value);
      if (robj != nilp) return new Boolean (d_value <= robj->d_value);
      break;
    case Object::LTH:
      if (iobj != nilp) return new Boolean (d_value < iobj->d_value);
      if (robj != nilp) return new Boolean (d_value < robj->d_value);
      break;
    case Object::MINUS:
      return new Real (-d_value);
      break;
    }
    throw Exception ("type-error", "invalid operand with real",
		     Object::repr (object));
  }

  // create a new real in a generic way

  Object* Real::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Real;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with real constructor");
    // try to map the real argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Real;

    // try an integer object
    Integer* ival = dynamic_cast <Integer*> (obj);
    if (ival != nilp) return new Real (ival->toInteger ());

    // try a real object
    Real* rval = dynamic_cast <Real*> (obj);
    if (rval != nilp) return new Real (*rval);

    // try a character object
    Character* cval = dynamic_cast <Character*> (obj);
    if (cval != nilp) return new Real (cval->toCharacter ());

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Real (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with real constructor",
		     obj->repr ());
  }

  // set an object to this real

  Object* Real::vdef (Interp* interp, Nameset* nset, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    if (iobj != nilp) {
      d_value = iobj->d_value;
      return this;
    }
    Real* robj = dynamic_cast <Real*> (object);
    if (robj != nilp) {
      d_value = robj->d_value;
      return this;
    }
    throw Exception ("type-error", "invalid object with real vdef",
		     Object::repr (object));
  }

  // evaluate this real with a member name

  Object* Real::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this real with a method name

  Object* Real::apply (Interp* interp, Nameset* nset, const String& name,
		       Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) delete argv;
    if ((name == "abs")     && (argc == 0)) return new Real    (abs     ());
    if ((name == "log")     && (argc == 0)) return new Real    (log     ());
    if ((name == "exp")     && (argc == 0)) return new Real    (exp     ());
    if ((name == "sin")     && (argc == 0)) return new Real    (sin     ());
    if ((name == "cos")     && (argc == 0)) return new Real    (cos     ());
    if ((name == "tan")     && (argc == 0)) return new Real    (tan     ());
    if ((name == "asin")    && (argc == 0)) return new Real    (asin    ());
    if ((name == "acos")    && (argc == 0)) return new Real    (acos    ());
    if ((name == "atan")    && (argc == 0)) return new Real    (atan    ());
    if ((name == "sinh")    && (argc == 0)) return new Real    (sinh    ());
    if ((name == "cosh")    && (argc == 0)) return new Real    (cosh    ());
    if ((name == "tanh")    && (argc == 0)) return new Real    (tanh    ());
    if ((name == "asinh")   && (argc == 0)) return new Real    (asinh   ());
    if ((name == "acosh")   && (argc == 0)) return new Real    (acosh   ());
    if ((name == "atanh")   && (argc == 0)) return new Real    (atanh   ());
    if ((name == "sqrt")    && (argc == 0)) return new Real    (sqrt    ());
    if ((name == "floor")   && (argc == 0)) return new Real    (floor   ());
    if ((name == "ceiling") && (argc == 0)) return new Real    (ceiling ());
    if ((name == "isnan")   && (argc == 0)) return new Boolean (isnan   ());

    if ((name == "to-string") && (argc == 0)) 
      return new String (toLiteral ());
    if ((name == "++") && (argc == 0)) {
      d_value++;
      return this;
    }
    if ((name == "--") && (argc == 0)) {
      d_value--;
      return this;
    }
    if ((name == "zero-p") && (argc == 0)) {
      bool result = (d_value == 0.0);
      delete argv;
      return new Boolean (result);
    }

    // dispatch one argument
    if ((name == "+") && (argc == 1)) {
      Object* result = oper (Object::ADD, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "-") && (argc == 1)) {
      Object* result = oper (Object::SUB, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "*") && (argc == 1)) {
      Object* result = oper (Object::MUL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "/") && (argc == 1)) {
      Object* result = oper (Object::DIV, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "+=") && (argc == 1)) {
      t_real val = argv->getireal (0);
      d_value += val;
      delete argv;
      return this;
    }
    if ((name == "-=") && (argc == 1)) {
      t_real val = argv->getireal (0);
      d_value -= val;
      delete argv;
      return this;
    }
    if ((name == "*=") && (argc == 1)) {
      t_real val = argv->getireal (0);
      d_value *= val;
      delete argv;
      return this;
    }
    if ((name == "/=") && (argc == 1)) {
      t_real val = argv->getireal (0);
      if (val == 0) throw Exception ("divide-error", "division by zero");
      d_value = d_value / val;
      delete argv;
      return this;
    }
    if ((name == "==") && (argc == 1)) {
      Object* result = oper (Object::EQL, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "?=") && (argc == 1)) {
      t_real val = argv->getreal (0);
      t_real pos = (d_value >= val) ? d_value - val : val - d_value;
      bool result = (pos <= Real::d_precision);
      delete argv;
      return new Boolean (result);
    }
    if ((name == "!=") && (argc == 1)) {
      Object* result = oper (Object::NEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "<") && (argc == 1)) {
      Object* result = oper (Object::LTH, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "<=") && (argc == 1)) {
      Object* result = oper (Object::LEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == ">") && (argc == 1)) {
      Object* result = oper (Object::GTH, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == ">=") && (argc == 1)) {
      Object* result = oper (Object::GEQ, argv->get (0));
      delete argv;
      return result;
    }
    if ((name == "format") && (argc == 1)) {
      long precision = argv->getint (0);
      Object* result = new String (format (precision));
      delete argv;
      return result;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
