// ---------------------------------------------------------------------------
// - Terminal.hpp                                                            -
// - aleph standard library - terminal class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_TERMINAL_HPP
#define  ALEPH_TERMINAL_HPP

#ifndef  ALEPH_INPUTTERM_HPP
#include "InputTerm.hpp"
#endif

#ifndef  ALEPH_OUTPUTTERM_HPP
#include "OutputTerm.hpp"
#endif

namespace aleph {
  /// The Terminal class implements the functionnality of a terminal. The class
  /// holds one input stream and two output streams. The associated streams can
  /// be retreived by using the appropriate get method. By default the class
  /// works on a line reading basis. A primary and secondary prompt can be
  /// associated with it. When a call to readline is made, the terminal set the
  /// input streams with that line. The eol character is not returned.
  /// @author amaury darsch

  class Terminal : public InputTerm, public OutputTerm {
  private:
    /// the readline buffer
    Buffer d_lbuf;
    /// the primary prompt
    String d_prt1;
    /// the secondary prompt
    String d_prt2;

  public:
    /// create a new default terminal 
    Terminal (void);

    /// destroy this terminal 
    ~Terminal (void);

    /// @return the class name
    String repr (void) const;

    /// read a line from this terminal
    /// @param pflag use primary or secondary prompt
    /// @return a line read from this terminal
    String readline (bool pflag);

    /// set the primary prompt
    /// @param value the prompt to set
    void setpprompt (const String& value); 

    /// set the secondary prompt
    /// @param value the prompt to set
    void setsprompt (const String& value); 

    /// @return the primary prompt
    inline String getpprompt (void) const {
      return d_prt1;
    }

    /// @return the secondary prompt
    inline String getsprompt (void) const {
      return d_prt1;
    }

    /// generate a new terminal
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// evaluate an terminal member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this terminal with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the copy constructor private
    Terminal (const Terminal&);
    // make the assignment operator private
    Terminal& operator = (const Terminal&);
  };
}

#endif
