// ---------------------------------------------------------------------------
// - cio.cxx                                                                 -
// - standard system library - c i/o function implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "cstring.hxx"
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

namespace aleph {

  // return the default input stream associated with this process

  int c_stdin (void) {
    return STDIN_FILENO;
  }

  // return the default output stream associated with this process

  int c_stdout (void) {
    return STDOUT_FILENO;
  }

  // return the default error stream associated with this process

  int c_stderr (void) {
    return STDERR_FILENO;
  }

  // check if the name corresponds to a file name and can be accessed.
  // it does not necessarily means that it can be opened.

  bool c_isfile (const char* name) {
    int         status;
    struct stat buffer;
  
    if (c_strlen (name) == 0) return false;
  
    status = stat (name,&buffer);
    if ((status == 0) && S_ISREG (buffer.st_mode)) return true;
    return false;
  }

  // check if the name corresponds to a directory name and can be accessed.
  // it does not necessarily means that it can be opened.

  bool c_isdir (const char* name) {
    int         status;
    struct stat buffer;
  
    if (c_strlen (name) == 0) return false;
  
    status = stat (name,&buffer);
    if ((status == 0) && S_ISDIR (buffer.st_mode)) return true;
    return false;
  }

  // open a file for reading. A file descriptor is returned on success or
  // -1 in case of error.

  int c_openr (const char* name) {
    int fd = open (name,O_RDONLY,0);
    return (fd < 0) ? -1 : fd;
  }

  // open a file for writing. A file descriptor is returned on success or
  // -1 in case of error.

  int c_openw (const char* name) {
    int fd = open (name, O_WRONLY|O_CREAT|O_TRUNC,
		         S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
    return (fd < 0) ? -1 : fd;
  }

  // open a file for read write. A file descriptor is returned on success or
  // -1 in case of error.

  int c_openrw (const char* name) {
    int fd = open (name,O_RDWR);
    return (fd < 0) ? -1 : fd;
  }

  // wait for a certain number of characters, eventually time out

  bool c_select (const int sid, const long size, const long tout) {   
    // initialize the set
    fd_set  set;
    FD_ZERO (&set);
    FD_SET  (sid, &set);
     
    // initialize the timeout
    struct timeval timeout;
    timeout.tv_sec  = tout / 1000;
    timeout.tv_usec = (tout - timeout.tv_sec) * 1000;

    // wait on select
    int status = 0;
    if (tout == 0)
      status = select (FD_SETSIZE, &set, NULL, NULL, NULL);
    else
      status = select (FD_SETSIZE, &set, NULL, NULL, &timeout);

    // compute result
    return (status == 1) ? true : false;
  }

  // Read n bytes in a buffer. The buffer must be pre-allocated. The number
  // of bytes read is returned

  long c_read (const int sid, char* buffer, const long count) {
  
    // standard checks
    if ((sid < 0) || (buffer == 0)) return -1;
  
    // read n bytes and return
    long result = read (sid,buffer,count);
    return result;
  }

  // Write n bytes from a buffer. The buffer must be pre-allocated. The number
  // of bytes written is returned

  long c_write (const int sid, const char* buffer, const long count) {
  
    // standard checks
    if ((sid < 0) || (buffer == 0)) return -1;
    if (count == 0) return 0;
  
    // write n bytes and return
    long result = write (sid,buffer,count);
    return result;
  }

  // Write an integer to the derstination stream

  long c_write (const int sid, long value) {
    // convert the integer first
    char* buffer = c_ltoa (value);
    long count    = c_write (sid,buffer,c_strlen (buffer));
    delete [] buffer;
    return count;
  }

  // Close this stream.

  void c_close (const int sid) {
    if (sid >= 0) close (sid);
  }

  // return the size of a file

  long c_fsize (const int sid) {
    struct stat buffer;
  
    // check that we have a regular file
    int status = fstat (sid,&buffer);
    if ((status != 0) || (!S_ISREG (buffer.st_mode))) return -1;
  
    // return the file size
    return buffer.st_size;
  }

  // set a file at a certain position

  bool c_lseek (const int sid, const long pos) {
    off_t status = lseek (sid, (off_t) pos, SEEK_SET);
    if (status == (off_t) -1) return false;
    return true;
  }

  // remove the file given its name.

  void c_rm (const char* name) {
    if (name == 0) return;
    unlink (name);
  }

  // create a temporary name for a file name

  char* c_tmpname (void) {
    return tmpnam (NULL);
  }

  // remove the file extension from a file name

  char* c_rmext (const char* name) {
    long size = c_strlen (name);
    if (size == 0) return 0;
  
    char* str = c_strdup (name);
    char* ptr = str + (size - 1);
    while (ptr-- != str) 
      if (*ptr == '.') *ptr = '\0';
    if ((size == 1) && (str[0] == '.')) str = '\0';
  
    // restore ptr - return new string and delete prt
    char* result = c_strdup (str);
    delete [] str;
    return result;
  }
}
