/*
 *  Expat (XML Parser Toolkit) for Ruby
 *  Jun 28, 1999 yoshidam version 0.5.15 support start/endDoctypeDecl
 *  Jun 28, 1999 yoshidam version 0.5.14 support setParamEntityParsing
 *  Apr 28, 1999 yoshidam version 0.5.11 support notStandalone
 *  Mar 29, 1998 yoshidam version 0.5.9  optimize for Ruby 1.3
 *  Mar  8, 1998 yoshidam version 0.5.7  support start/endNamespaceDecl
 *  Jan 14, 1998 yoshidam version 0.5.4  support start/endCdataSection
 *  Jan 10, 1998 yoshidam version 0.5.3  support encoding map
 *  Nov 24, 1998 yoshidam version 0.5.0  support TEST version of expat
 *  Nov  5, 1998 yoshidam version 0.4.18 mIDs are initialized in Init_xmlparser
 *  Oct 28, 1998 yoshidam version 0.4.17 mIDs are stored into static vars
 *  Oct 13, 1998 yoshidam version 0.4.12 debug and speed up myEncodingConv
 *  Oct  7, 1998 yoshidam version 0.4.11 hold internal object into ivar
 *  Sep 18, 1998 yoshidam version 0.4.6
 *  Sep  8, 1998 yoshidam version 0.4.4
 *  Sep  3, 1998 yoshidam version 0.4.3
 *  Sep  1, 1998 yoshidam version 0.4.2
 *  Aug 28, 1998 yoshidam version 0.4.1
 *  Aug 22, 1998 yoshidam version 0.4.0
 *  Jul  6, 1998 yoshidam version 0.2
 *  Jun 30, 1998 yoshidam version 0.1
 *
 *  XML_ENC_PATH: path of encoding map for Perl
 *  NEW_EXPAT: switch for TEST version of expat
 */

#include "ruby.h"
#include "rubyio.h"
#include <stdio.h>
#include "xmlparse.h"
#ifdef XML_ENC_PATH
#  include <limits.h>
#  include <sys/stat.h>
#  include "encoding.h"
#  ifndef PATH_MAX
#    define PATH_MAX 256
#  endif
#endif

static VALUE eXMLParserError;
static VALUE cXMLParser;
static VALUE cXMLEncoding;
static ID id_map;
static ID id_startElementHandler;
static ID id_endElementHandler;
static ID id_characterDataHandler;
static ID id_processingInstructionHandler;
static ID id_defaultHandler;
static ID id_defaultExpandHandler;
static ID id_unparsedEntityDeclHandler;
static ID id_notationDeclHandler;
static ID id_externalEntityRefHandler;
static ID id_unknownEncoding;
static ID id_convert;
#ifdef NEW_EXPAT
static ID id_commentHandler;
static ID id_startCdataSectionHandler;
static ID id_endCdataSectionHandler;
static ID id_startNamespaceDeclHandler;
static ID id_endNamespaceDeclHandler;
static ID id_notStandaloneHandler;
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
static ID id_startDoctypeDeclHandler;
static ID id_endDoctypeDeclHandler;
#endif

#define GET_PARSER(obj, parser) \
  Data_Get_Struct(obj, XMLParser, parser)

typedef struct _XMLParser {
  XML_Parser parser;
  int iterator;
  int defaultCurrent;
#ifdef NEW_EXPAT
  const XML_Char** lastAttrs;
#endif
} XMLParser;

enum {
  XML_DEFAULT,
  XML_START_ELEM,
  XML_END_ELEM,
  XML_CDATA,
  XML_PI,
  XML_UNPARSED_ENTITY_DECL,
  XML_NOTATION_DECL,
  XML_EXTERNAL_ENTITY_REF,
#ifdef NEW_EXPAT
  XML_COMMENT,
  XML_START_CDATA,
  XML_END_CDATA,
  XML_START_NAMESPACE_DECL,
  XML_END_NAMESPACE_DECL,
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
  XML_START_DOCTYPE_DECL,
  XML_END_DOCTYPE_DECL,
#endif
  XML_UNKNOWN_ENCODING
};
    

/* destructor */
static void
XMLParser_free(XMLParser* parser)
{
  /* fprintf(stderr, "Delete XMLParser: %p\n", parser->parser); */
  if (parser->parser) {
    XML_ParserFree(parser->parser);
    parser->parser = NULL;
  }
  free(parser);
}


/* Event handlers for iterator */
static void
iterStartElementHandler(void *recv,
		      const XML_Char *name, const XML_Char **atts)
{
  XMLParser* parser;
  VALUE attrhash;

  GET_PARSER(recv, parser);
#ifdef NEW_EXPAT
  parser->lastAttrs = atts;
#endif
  attrhash = rb_hash_new();
  while (*atts) {
    const char* key = *atts++;
    const char* val = *atts++;
    rb_hash_aset(attrhash, rb_str_new2((char*)key), rb_str_new2((char*)val));
  }
  
  rb_yield(rb_ary_new3(3, INT2FIX(XML_START_ELEM),
		    rb_str_new2((char*)name), attrhash));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterEndElementHandler(void *recv,
		    const XML_Char *name)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_END_ELEM),
		    rb_str_new2((char*)name), Qnil));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterCharacterDataHandler(void *recv,
		       const XML_Char *s,
		       int len)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_CDATA),
		    Qnil, rb_str_new((char*)s, len)));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterProcessingInstructionHandler(void *recv,
			       const XML_Char *target,
			       const XML_Char *data)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_PI),
		    rb_str_new2((char*)target), rb_str_new2((char*)data)));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterDefaultHandler(void *recv,
		   const XML_Char *s,
		   int len)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_DEFAULT),
		    Qnil, rb_str_new((char*)s, len)));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    /* XML_DefaultCurrent shoould not call in defaultHandler */
    /* XML_DefaultCurrent(parser->parser); */
  }
}

void
iterUnparsedEntityDeclHandler(void *recv,
			    const XML_Char *entityName,
			    const XML_Char *base,
			    const XML_Char *systemId,
			    const XML_Char *publicId,
			    const XML_Char *notationName)
{
  XMLParser* parser;
  VALUE valary;

  valary = rb_ary_new3(4, (base ? rb_str_new2((char*)base) : Qnil),
		    rb_str_new2((char*)systemId),
		    (publicId ? rb_str_new2((char*)publicId) : Qnil),
		    rb_str_new2((char*)notationName));
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_UNPARSED_ENTITY_DECL),
		    rb_str_new2((char*)entityName),
		    valary));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

void
iterNotationDeclHandler(void *recv,
		      const XML_Char *notationName,
		      const XML_Char *base,
		      const XML_Char *systemId,
		      const XML_Char *publicId)
{
  XMLParser* parser;
  VALUE valary;

  valary = rb_ary_new3(3, (base ? rb_str_new2((char*)base) : Qnil),
		    (systemId ? rb_str_new2((char*)systemId) : Qnil),
		    (publicId ? rb_str_new2((char*)publicId) : Qnil));
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_NOTATION_DECL),
		    rb_str_new2((char*)notationName),
		    valary));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

int
iterExternalEntityRefHandler(XML_Parser xmlparser,
			   const XML_Char *context,
			   const XML_Char *base,
			   const XML_Char *systemId,
			   const XML_Char *publicId)
{
  XMLParser* parser;
  VALUE recv;
  VALUE valary;
  VALUE ret;

  valary = rb_ary_new3(3, (base ? rb_str_new2((char*)base) : Qnil),
	     (systemId ? rb_str_new2((char*)systemId) : Qnil),
	     (publicId ? rb_str_new2((char*)publicId) : Qnil));
  recv = (VALUE)XML_GetUserData(xmlparser);
  GET_PARSER(recv, parser);
  ret = rb_yield(rb_ary_new3(3, INT2FIX(XML_EXTERNAL_ENTITY_REF),
			  (context ? rb_str_new2((char*)context) : Qnil),
			  valary));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
  /* The error status in this iterator block should be returned
     by the exception. */
  return 1;
}

#ifdef NEW_EXPAT
static void
iterCommentHandler(void *recv,
		   const XML_Char *s)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_COMMENT),
		    Qnil, rb_str_new2((char*)s)));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterStartCdataSectionHandler(void *recv)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_START_CDATA), Qnil, Qnil));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterEndCdataSectionHandler(void *recv)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_END_CDATA), Qnil, Qnil));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterStartNamespaceDeclHandler(void *recv,
			      const XML_Char *prefix,
			      const XML_Char *uri)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_START_NAMESPACE_DECL),
		       (prefix ? rb_str_new2((char*)prefix) : Qnil),
		       (uri ? rb_str_new2((char*)uri) : Qnil)));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterEndNamespaceDeclHandler(void *recv,
			    const XML_Char *prefix)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_END_NAMESPACE_DECL),
		       (prefix ? rb_str_new2((char*)prefix) : Qnil),
		       Qnil));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}
#endif

#ifdef HAVE_XML_SETPARAMENTITYPARSING
static void
iterStartDoctypeDeclHandler(void *recv,
			    const XML_Char *doctypeName)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_START_DOCTYPE_DECL),
		       rb_str_new2((char*)doctypeName),
		       Qnil));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}

static void
iterEndDoctypeDeclHandler(void *recv)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_yield(rb_ary_new3(3, INT2FIX(XML_END_DOCTYPE_DECL),
		       Qnil,
		       Qnil));
  if (parser->defaultCurrent) {
    parser->defaultCurrent = 0;
    XML_DefaultCurrent(parser->parser);
  }
}
#endif

/* Event handlers for instance method */
static void
myStartElementHandler(void *recv,
		      const XML_Char *name, const XML_Char **atts)
{
  XMLParser* parser;
  VALUE attrhash;

  GET_PARSER(recv, parser);
#ifdef NEW_EXPAT
  parser->lastAttrs = atts;
#endif
  attrhash = rb_hash_new();
  while (*atts) {
    const char* key = *atts++;
    const char* val = *atts++;
    rb_hash_aset(attrhash, rb_str_new2((char*)key), rb_str_new2((char*)val));
  }
  rb_funcall((VALUE)recv, id_startElementHandler, 2,
	     rb_str_new2((char*)name), attrhash);
}

static void
myEndElementHandler(void *recv,
		    const XML_Char *name)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_endElementHandler, 1,
	     rb_str_new2((char*)name));
}

static void
myCharacterDataHandler(void *recv,
		       const XML_Char *s,
		       int len)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_characterDataHandler, 1,
	     rb_str_new((char*)s, len));
}

static void
myProcessingInstructionHandler(void *recv,
			       const XML_Char *target,
			       const XML_Char *data)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_processingInstructionHandler, 2,
	     rb_str_new2((char*)target), rb_str_new2((char*)data));
}

static void
myDefaultHandler(void *recv,
		 const XML_Char *s,
		 int len)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_defaultHandler, 1,
	     rb_str_new((char*)s, len));
}

#ifdef NEW_EXPAT
static void
myDefaultExpandHandler(void *recv,
		 const XML_Char *s,
		 int len)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_defaultExpandHandler, 1,
	     rb_str_new((char*)s, len));
}
#endif

void
myUnparsedEntityDeclHandler(void *recv,
			    const XML_Char *entityName,
			    const XML_Char *base,
			    const XML_Char *systemId,
			    const XML_Char *publicId,
			    const XML_Char *notationName)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_unparsedEntityDeclHandler, 5,
	     rb_str_new2((char*)entityName),
	     (base ? rb_str_new2((char*)base) : Qnil),
	     rb_str_new2((char*)systemId),
	     (publicId ? rb_str_new2((char*)publicId) : Qnil),
	     rb_str_new2((char*)notationName));
}

void
myNotationDeclHandler(void *recv,
		      const XML_Char *notationName,
		      const XML_Char *base,
		      const XML_Char *systemId,
		      const XML_Char *publicId)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_notationDeclHandler, 4,
	     rb_str_new2((char*)notationName),
	     (base ? rb_str_new2((char*)base) : Qnil),
	     (systemId ? rb_str_new2((char*)systemId) : Qnil),
	     (publicId ? rb_str_new2((char*)publicId) : Qnil));
}

int
myExternalEntityRefHandler(XML_Parser xmlparser,
			   const XML_Char *context,
			   const XML_Char *base,
			   const XML_Char *systemId,
			   const XML_Char *publicId)
{
  XMLParser* parser;
  VALUE recv;
  VALUE ret;
  recv = (VALUE)XML_GetUserData(xmlparser);
  GET_PARSER(recv, parser);
  ret = rb_funcall(recv, id_externalEntityRefHandler, 4,
		   (context ? rb_str_new2((char*)context): Qnil),
		   (base ? rb_str_new2((char*)base) : Qnil),
		   (systemId ? rb_str_new2((char*)systemId) : Qnil),
		   (publicId ? rb_str_new2((char*)publicId) : Qnil));
  /* The error status in this handler should be returned
     by the exception. */
  return Qnil;
}

#ifdef NEW_EXPAT
static void
myCommentHandler(void *recv,
		 const XML_Char *s)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_commentHandler, 1,
	     rb_str_new2((char*)s));
}

static void
myStartCdataSectionHandler(void *recv)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_startCdataSectionHandler, 0);
}

static void
myEndCdataSectionHandler(void *recv)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_endCdataSectionHandler, 0);
}

static void
myStartNamespaceDeclHandler(void *recv,
			    const XML_Char *prefix,
			    const XML_Char *uri)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_startNamespaceDeclHandler, 2,
	     (prefix ? rb_str_new2((char*)prefix) : Qnil),
	     (uri ? rb_str_new2((char*)uri) : Qnil));
}

static void
myEndNamespaceDeclHandler(void *recv,
			  const XML_Char *prefix)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_endNamespaceDeclHandler, 1,
	     (prefix ? rb_str_new2((char*)prefix) : Qnil));
}

static int
myNotStandaloneHandler(void *recv)
{
  XMLParser* parser;
  VALUE v;

  GET_PARSER(recv, parser);
  v = rb_funcall((VALUE)recv, id_notStandaloneHandler, 0);
  Check_Type(v, T_FIXNUM);
  return FIX2INT(v);
}
#endif

#ifdef HAVE_XML_SETPARAMENTITYPARSING
static void
myStartDoctypeDeclHandler(void *recv,
			  const XML_Char *doctypeName)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_startDoctypeDeclHandler, 1,
	     rb_str_new2((char*)doctypeName));
}

static void
myEndDoctypeDeclHandler(void *recv)
{
  XMLParser* parser;
  GET_PARSER(recv, parser);
  rb_funcall((VALUE)recv, id_endDoctypeDeclHandler, 0);
}
#endif

static VALUE
XMLEncoding_map(VALUE obj, VALUE i)
{
  return i;
}

static VALUE
XMLEncoding_convert(VALUE obj, VALUE str)
{
  return INT2FIX('?');
}

static int
myEncodingConv(void *data, const char *s)
{
  VALUE v;
  int len;
  unsigned char c;
  int slen = RSTRING(rb_ivar_get((VALUE)data,
				 id_map))->ptr[*(unsigned char*)s];

  v = rb_funcall((VALUE)data, id_convert, 1, rb_str_new((char*)s, -slen));
  switch (TYPE(v)) {
  case T_FIXNUM:
    return FIX2INT(v);
  case T_STRING:
    len = RSTRING(v)->len;
    if (len == 1) {
      return (unsigned char)*(RSTRING(v)->ptr);
    }
    else if (len >= 2) {
      return (unsigned char)*(RSTRING(v)->ptr) |
	(unsigned char)*(RSTRING(v)->ptr + 1) << 8;
    }
  }
  return 0;
}

#if 0
static int
iterUnknownEncodingHandler(void *recv,
			   const XML_Char *name,
			   XML_Encoding *info)
{
  XMLParser* parser;
  VALUE ret;

  if (!rb_method_boundp(CLASS_OF((VALUE)recv), id_unknownEncoding, 0))
    return 0;

  GET_PARSER(recv, parser);
  ret = rb_yield(rb_ary_new3(3, INT2FIX(XML_UNKNOWN_ENCODING),
			  rb_str_new2((char*)name), Qnil));
  if (TYPE(ret) == T_OBJECT && rb_obj_is_kind_of(ret, cXMLEncoding)) {
    int i;
    ID mid = rb_intern("map");
    VALUE cmap = rb_str_new(NULL, 256);
    rb_ivar_set(ret, id_map, cmap);

    for (i = 0; i < 256; i++) {
      VALUE m = rb_funcall(ret, mid, 1, INT2FIX(i));
      RSTRING(cmap)->ptr[i] = info->map[i] = FIX2INT(m);
    }
    /* protect object form GC */
    rb_ivar_set(recv, rb_intern("_encoding"), ret);
    info->data = (void*)ret;
    info->convert = myEncodingConv;
    return 1;
  }

  return 0;
}
#endif

#ifdef XML_ENC_PATH
/*
 * Encoding map functions come from XML::Parser Version 2.19
 *
 * Copyright 1998 Larry Wall and Clark Cooper
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the same terms as Perl itself.
 */
static Encinfo*
getEncinfo(char* data, int size)
{
  Encmap_Header* header = (Encmap_Header*)data;
  unsigned short prefixes_size;
  unsigned short bytemap_size;
  Encinfo* ret;
  int i;
  PrefixMap* prefixes;
  unsigned short *bytemap;

  if (size < sizeof(Encmap_Header) || ntohl(header->magic) != ENCMAP_MAGIC)
    return NULL;
  prefixes_size = ntohs(header->pfsize);
  bytemap_size = ntohs(header->bmsize);
  if (size != (sizeof(Encmap_Header) +
	       prefixes_size * sizeof(PrefixMap) +
	       bytemap_size * sizeof(unsigned short)))
    return NULL;
  if ((ret = (Encinfo*)malloc(sizeof(Encinfo))) == NULL) {
    return NULL;
  }
  ret->prefixes_size = prefixes_size;
  ret->bytemap_size = bytemap_size;
  for (i = 0; i < 256; i++)
    ret->firstmap[i] = ntohl(header->map[i]);
  prefixes = (PrefixMap*)(data + sizeof(Encmap_Header));
  bytemap = (unsigned short*)(data + sizeof(Encmap_Header)
			      + sizeof(PrefixMap)*prefixes_size);
  if ((ret->prefixes =
       (PrefixMap*)malloc(sizeof(PrefixMap)*prefixes_size)) == NULL) {
    free(ret);
    return NULL;
  }
  if ((ret->bytemap =
       (unsigned short*)malloc(sizeof(unsigned short)*bytemap_size)) == NULL) {
    free(ret->prefixes);
    free(ret);
    return NULL;
  }
  for (i = 0; i < prefixes_size; i++, prefixes++) {
    ret->prefixes[i].min = prefixes->min;
    ret->prefixes[i].len = prefixes->len;
    ret->prefixes[i].bmap_start = ntohs(prefixes->bmap_start);
    memcpy(ret->prefixes[i].ispfx, prefixes->ispfx,
	   sizeof(prefixes->ispfx) + sizeof(prefixes->ischar));
  }
  for (i = 0; i < bytemap_size; i++)
    ret->bytemap[i] = ntohs(bytemap[i]);

  return ret;
}

static int
convertEncoding(Encinfo* enc, const char* seq)
{
  PrefixMap* curpfx;
  int count;
  int index = 0;

  for (count = 0; count < 4; count++) {
    unsigned char byte = (unsigned char)seq[count];
    unsigned char bndx;
    unsigned char bmsk;
    int offset;

    curpfx = &enc->prefixes[index];
    offset = ((int)byte) - curpfx->min;
    if (offset < 0)
      break;
    if (offset >= curpfx->len && curpfx->len != 0)
      break;

    bndx = byte >> 3;
    bmsk = 1 << (byte & 0x7);

    if (curpfx->ispfx[bndx] & bmsk) {
      index = enc->bytemap[curpfx->bmap_start + offset];
    }
    else if (curpfx->ischar[bndx] & bmsk) {
      return enc->bytemap[curpfx->bmap_start + offset];
    }
    else
      break;
  }

  return -1;
}

static void
releaseEncoding(Encinfo* enc)
{
  if (enc) {
    if (enc->prefixes)
      free(enc->prefixes);
    if (enc->bytemap)
      free(enc->bytemap);
    free(enc);
  }
}

static Encinfo*
findEncoding(const char* encname)
{
  FILE* fp;
  Encinfo* enc;
  struct stat st;
  int size;
  int len;
  char file[PATH_MAX] = "\0";
  const char* p;
  char* buf;

  /* make map file path */
  if (XML_ENC_PATH != NULL)
    strcpy(file, XML_ENC_PATH);
  len = strlen(file);
  if (len > 0 && file[len - 1] != '/')
    file[len++] = '/';
  for (p = encname; *p; p++, len++) {
    file[len] = tolower(*p);
  }
  file[len] = '\0';
  strcat(file, ".enc");

  if ((fp = fopen(file, "rb")) == NULL) {
    return NULL;
  }

  /* get file length */
  fstat(fileno(fp), &st);
  size = st.st_size;

  if ((buf = (char*)malloc(size)) == NULL) {
    fclose(fp);
    return NULL;
  }

  fread(buf, 1, size, fp);
  fclose(fp);
  enc = getEncinfo(buf, size);
  free(buf);
  return enc;
}

#endif

static int
myUnknownEncodingHandler(void *recv,
			 const XML_Char *name,
			 XML_Encoding *info)
{
  XMLParser* parser;
  VALUE ret;
  if (!rb_method_boundp(CLASS_OF((VALUE)recv), id_unknownEncoding, 0))
#ifndef XML_ENC_PATH
    return 0;
#else
  {
    Encinfo* enc;

    if ((enc = findEncoding(name)) != NULL) {
      memcpy(info->map, enc->firstmap, sizeof(int)*256);
      info->data = enc;
      info->convert = (int(*)(void*,const char*))convertEncoding;
      info->release = (void(*)(void*))releaseEncoding;
      return 1;
    }
    else
      return 0;
  }
#endif

  GET_PARSER(recv, parser);
  ret = rb_funcall((VALUE)recv, id_unknownEncoding, 1, rb_str_new2((char*)name));
  if (TYPE(ret) == T_OBJECT && rb_obj_is_kind_of(ret, cXMLEncoding)) {
    int i;
    ID mid = rb_intern("map");
    VALUE cmap = rb_str_new(NULL, 256);
    rb_ivar_set(ret, id_map, cmap);

    for (i = 0; i < 256; i++) {
      VALUE m = rb_funcall(ret, mid, 1, INT2FIX(i));
      RSTRING(cmap)->ptr[i] = info->map[i] = FIX2INT(m);
    }
    /* protect object form GC */
    rb_ivar_set((VALUE)recv, rb_intern("_encoding"), ret);
    info->data = (void*)ret;
    info->convert = myEncodingConv;
    return 1;
  }

  return 0;
}

/* constructor */
static VALUE
XMLParser_new(int argc, VALUE* argv, VALUE klass)
{
  XMLParser* parser;
  VALUE obj;
  VALUE arg1;
  VALUE arg2;
  VALUE arg3;
  int count;
  char* encoding = NULL;
#ifdef NEW_EXPAT
  char* nssep = NULL;
#endif
  char* context = NULL;
  XMLParser* rootparser = NULL;

  count = rb_scan_args(argc, argv, "03", &arg1, &arg2, &arg3);
  if (count == 1) {
    /* new(encoding) */
    if (TYPE(arg1) != T_NIL) {
      Check_Type(arg1, T_STRING); /* encoding */
      encoding = RSTRING(arg1)->ptr;
    }
  }
  else if (count == 2) {
    /* new(encoding, nschar) */
    /* new(parser, context) */
#ifdef NEW_EXPAT
    if (TYPE(arg1) != T_DATA) {
      if (TYPE(arg1) != T_NIL) {
	Check_Type(arg1, T_STRING); /* encoding */
	encoding = RSTRING(arg1)->ptr;
      }
      Check_Type(arg2, T_STRING); /* nschar */
      nssep = RSTRING(arg2)->ptr;
    }
    else {
#endif
      Check_Type(arg1, T_DATA); /* parser */
      GET_PARSER(arg1, rootparser);
      if (!NIL_P(arg2)) {
	Check_Type(arg2, T_STRING); /* context */
	context = RSTRING(arg2)->ptr;
      }
#ifdef NEW_EXPAT
    }
#endif
  }
  else if (count == 3) {
    /* new(parser, context, encoding) */
    Check_Type(arg1, T_DATA); /* parser */
    GET_PARSER(arg1, rootparser);
    if (!NIL_P(arg2)) {
      Check_Type(arg2, T_STRING); /* context */
      context = RSTRING(arg2)->ptr;
    }
    Check_Type(arg3, T_STRING); /* encoding */
    encoding = RSTRING(arg3)->ptr;
  }

  /* create object */
  obj = Data_Make_Struct(klass, XMLParser,
			 NULL, XMLParser_free, parser);
  /* create parser */
  if (rootparser == NULL) {
#ifdef NEW_EXPAT
    if (nssep == NULL)
      parser->parser = XML_ParserCreate(encoding);
    else
      parser->parser = XML_ParserCreateNS(encoding, nssep[0]);
#else
    parser->parser = XML_ParserCreate(encoding);
#endif
  }
  else {
    parser->parser = XML_ExternalEntityParserCreate(rootparser->parser,
    						    context, encoding);
    /* clear all inhrited handlers,
       because handlers should be set in "parse" method  */
    XML_SetElementHandler(parser->parser, NULL, NULL);
    XML_SetCharacterDataHandler(parser->parser, NULL);
    XML_SetProcessingInstructionHandler(parser->parser, NULL);
    XML_SetDefaultHandler(parser->parser, NULL);
    XML_SetUnparsedEntityDeclHandler(parser->parser, NULL);
    XML_SetNotationDeclHandler(parser->parser, NULL);
    XML_SetExternalEntityRefHandler(parser->parser, NULL);
#ifdef NEW_EXPAT
    XML_SetCommentHandler(parser->parser, NULL);
    XML_SetCdataSectionHandler(parser->parser, NULL, NULL);
    XML_SetNamespaceDeclHandler(parser->parser, NULL, NULL);
    XML_SetNotStandaloneHandler(parser->parser, NULL);
#endif
  }
  if (!parser->parser)
    rb_raise(eXMLParserError, "cannot create parser");

  /* setting up internal data */
  XML_SetUserData(parser->parser, (void*)obj);
#ifdef NEW_EXPAT
  parser->lastAttrs = NULL;
#endif

  rb_obj_call_init(obj, argc, argv);

  return obj;
}

/* parse method */
static VALUE
XMLParser_parse(int argc, VALUE* argv, VALUE obj)
{
  XMLParser* parser;
  int ret;
  XML_StartElementHandler start = NULL;
  XML_EndElementHandler   end = NULL;
#ifdef NEW_EXPAT
  XML_StartCdataSectionHandler startC = NULL;
  XML_EndCdataSectionHandler   endC = NULL;
  XML_StartNamespaceDeclHandler startNS = NULL;
  XML_EndNamespaceDeclHandler endNS = NULL;
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
  XML_StartDoctypeDeclHandler startDoctype = NULL;
  XML_EndDoctypeDeclHandler endDoctype = NULL;
#endif
  VALUE str;
  VALUE isFinal;
  int final = 1;
  int count;
  int fromStream = 0;
  ID mid = rb_intern("gets");
  ID linebuf = rb_intern("_linebuf");

  count = rb_scan_args(argc, argv, "02", &str, &isFinal);
  /* If "str" has public "gets" method, it will be considered *stream* */
  if (!rb_obj_is_kind_of(str, rb_cString) &&
      rb_method_boundp(CLASS_OF(str), mid, 1)) {
    fromStream = 1;
  }
  else if (!NIL_P(str)) {
    Check_Type(str, T_STRING);
  }
  if (count >= 2) {
    if (isFinal == Qtrue)
      final = 1;
    else if (isFinal == Qfalse)
      final = 0;
    else
      rb_raise(rb_eTypeError, "not valid value");
  }

  GET_PARSER(obj, parser);

  parser->iterator = rb_iterator_p();

  /* Setup event handlers */

  /* Call as iterator */
  if (parser->iterator) {
    XML_SetElementHandler(parser->parser,
			  iterStartElementHandler, iterEndElementHandler);
    XML_SetCharacterDataHandler(parser->parser,
				iterCharacterDataHandler);
    XML_SetProcessingInstructionHandler(parser->parser,
					iterProcessingInstructionHandler);
    /* check dummy default handler */
#ifdef NEW_EXPAT
    if (rb_method_boundp(CLASS_OF(obj), id_defaultExpandHandler, 0))
      XML_SetDefaultHandlerExpand(parser->parser, iterDefaultHandler);
    else
#endif
    if (rb_method_boundp(CLASS_OF(obj), id_defaultHandler, 0))
      XML_SetDefaultHandler(parser->parser, iterDefaultHandler);

    if (rb_method_boundp(CLASS_OF(obj), id_unparsedEntityDeclHandler, 0))
      XML_SetUnparsedEntityDeclHandler(parser->parser,
				       iterUnparsedEntityDeclHandler);
    if (rb_method_boundp(CLASS_OF(obj), id_notationDeclHandler, 0))
      XML_SetNotationDeclHandler(parser->parser,
				 iterNotationDeclHandler);
    if (rb_method_boundp(CLASS_OF(obj), id_externalEntityRefHandler, 0))
      XML_SetExternalEntityRefHandler(parser->parser,
				      iterExternalEntityRefHandler);
#ifdef NEW_EXPAT
    if (rb_method_boundp(CLASS_OF(obj), id_commentHandler, 0))
      XML_SetCommentHandler(parser->parser, iterCommentHandler);

    if (rb_method_boundp(CLASS_OF(obj), id_startCdataSectionHandler, 0))
      startC = iterStartCdataSectionHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endCdataSectionHandler, 0))
      endC = iterEndCdataSectionHandler;
    if (startC || endC)
      XML_SetCdataSectionHandler(parser->parser, startC, endC);

    if (rb_method_boundp(CLASS_OF(obj), id_startNamespaceDeclHandler, 0))
      startNS = iterStartNamespaceDeclHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endNamespaceDeclHandler, 0))
      endNS = iterEndNamespaceDeclHandler;
    if (startNS || endNS)
      XML_SetNamespaceDeclHandler(parser->parser, startNS, endNS);
    if (rb_method_boundp(CLASS_OF(obj), id_notStandaloneHandler, 0))
      XML_SetNotStandaloneHandler(parser->parser, myNotStandaloneHandler);
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
    if (rb_method_boundp(CLASS_OF(obj), id_startDoctypeDeclHandler, 0))
      startDoctype = iterStartDoctypeDeclHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endDoctypeDeclHandler, 0))
      endDoctype = iterEndDoctypeDeclHandler;
    if (startDoctype || endDoctype)
      XML_SetDoctypeDeclHandler(parser->parser, startDoctype, endDoctype);
#endif
    /* Call non-iterator version of UnknownEncoding handler,
       because the porcedure block often returns the unexpected value. */
    XML_SetUnknownEncodingHandler(parser->parser,
				  myUnknownEncodingHandler,
				  (void*)obj);
  }
  /* Call as not iterator */
  else {
    if (rb_method_boundp(CLASS_OF(obj), id_startElementHandler, 0))
      start = myStartElementHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endElementHandler, 0))
      end = myEndElementHandler;
    if (start || end)
      XML_SetElementHandler(parser->parser, start, end);
    if (rb_method_boundp(CLASS_OF(obj), id_characterDataHandler, 0))
      XML_SetCharacterDataHandler(parser->parser,
				  myCharacterDataHandler);
    if (rb_method_boundp(CLASS_OF(obj),
			 id_processingInstructionHandler, 0))
      XML_SetProcessingInstructionHandler(parser->parser,
					  myProcessingInstructionHandler);
#ifdef NEW_EXPAT
    if (rb_method_boundp(CLASS_OF(obj), id_defaultExpandHandler, 0))
      XML_SetDefaultHandlerExpand(parser->parser, myDefaultExpandHandler);
    else
#endif
    if (rb_method_boundp(CLASS_OF(obj), id_defaultHandler, 0)) {
      XML_SetDefaultHandler(parser->parser, myDefaultHandler);
    }
    if (rb_method_boundp(CLASS_OF(obj), id_unparsedEntityDeclHandler, 0))
      XML_SetUnparsedEntityDeclHandler(parser->parser,
				     myUnparsedEntityDeclHandler);
    if (rb_method_boundp(CLASS_OF(obj), id_notationDeclHandler, 0))
      XML_SetNotationDeclHandler(parser->parser,
			       myNotationDeclHandler);
    if (rb_method_boundp(CLASS_OF(obj), id_externalEntityRefHandler, 0))
      XML_SetExternalEntityRefHandler(parser->parser,
				    myExternalEntityRefHandler);
#ifdef NEW_EXPAT
    if (rb_method_boundp(CLASS_OF(obj), id_commentHandler, 0))
      XML_SetCommentHandler(parser->parser, myCommentHandler);

    if (rb_method_boundp(CLASS_OF(obj), id_startCdataSectionHandler, 0))
      startC = myStartCdataSectionHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endCdataSectionHandler, 0))
      endC = myEndCdataSectionHandler;
    if (startC || endC)
      XML_SetCdataSectionHandler(parser->parser, startC, endC);

    if (rb_method_boundp(CLASS_OF(obj), id_startNamespaceDeclHandler, 0))
      startNS = myStartNamespaceDeclHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endNamespaceDeclHandler, 0))
      endNS = myEndNamespaceDeclHandler;
    if (startNS || endNS)
      XML_SetNamespaceDeclHandler(parser->parser, startNS, endNS);
    if (rb_method_boundp(CLASS_OF(obj), id_notStandaloneHandler, 0))
      XML_SetNotStandaloneHandler(parser->parser, myNotStandaloneHandler);
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
    if (rb_method_boundp(CLASS_OF(obj), id_startDoctypeDeclHandler, 0))
      startDoctype = myStartDoctypeDeclHandler;
    if (rb_method_boundp(CLASS_OF(obj), id_endDoctypeDeclHandler, 0))
      endDoctype = myEndDoctypeDeclHandler;
    if (startDoctype || endDoctype)
      XML_SetDoctypeDeclHandler(parser->parser, startDoctype, endDoctype);
#endif
    XML_SetUnknownEncodingHandler(parser->parser,
				  myUnknownEncodingHandler,
				  (void*)obj);
  }

  /* Parse from stream (probably slightly slow) */
  if (fromStream) {
    VALUE buf;

    do {
      buf = rb_funcall(str, mid, 0);
      if (!NIL_P(buf)) {
	Check_Type(buf, T_STRING);
	rb_ivar_set(obj, linebuf, buf); /* protect buf from GC (reasonable?)*/
	ret = XML_Parse(parser->parser,
			RSTRING(buf)->ptr, RSTRING(buf)->len, 0);
      }
      else {
	ret = XML_Parse(parser->parser, NULL, 0, 1);
      }
      if (!ret) {
	int err = XML_GetErrorCode(parser->parser);
	const char* errStr = XML_ErrorString(err);
	rb_raise(eXMLParserError, (char*)errStr);
      }
    } while (!NIL_P(buf));
    return Qnil;
  }

  /* Parse string */
  if (!NIL_P(str))
    ret = XML_Parse(parser->parser,
		    RSTRING(str)->ptr, RSTRING(str)->len, final);
  else
    ret = XML_Parse(parser->parser, NULL, 0, final);
  if (!ret) {
    int err = XML_GetErrorCode(parser->parser);
    const char* errStr = XML_ErrorString(err);
    rb_raise(eXMLParserError, (char*)errStr);
  }

  return Qnil;
}

/* done method */
static VALUE
XMLParser_done(VALUE obj)
{
  XMLParser* parser;

  GET_PARSER(obj, parser);
  if (parser->parser) {
    XML_ParserFree(parser->parser);
    parser->parser = NULL;
  }
  return Qnil;
}

/* defaultCurrent method */
static VALUE
XMLParser_defaultCurrent(VALUE obj)
{
  XMLParser* parser;

  GET_PARSER(obj, parser);
  if (!(parser->iterator)) {
    XML_DefaultCurrent(parser->parser);
  }
  else {
    parser->defaultCurrent = 1;
  }
  return Qnil;
}

/* line method */
static VALUE
XMLParser_getCurrentLineNumber(VALUE obj)
{
  XMLParser* parser;
  int line;

  GET_PARSER(obj, parser);
  line = XML_GetCurrentLineNumber(parser->parser);

  return INT2FIX(line);
}

/* column method */
static VALUE
XMLParser_getCurrentColumnNumber(VALUE obj)
{
  XMLParser* parser;
  int column;

  GET_PARSER(obj, parser);
  column = XML_GetCurrentColumnNumber(parser->parser);

  return INT2FIX(column);
}

/* byte index method */
static VALUE
XMLParser_getCurrentByteIndex(VALUE obj)
{
  XMLParser* parser;
  long pos;

  GET_PARSER(obj, parser);
  pos = XML_GetCurrentByteIndex(parser->parser);

  return INT2FIX(pos);
}

/* set URI base */
static VALUE
XMLParser_setBase(VALUE obj, VALUE base)
{
  XMLParser* parser;
  int ret;

  Check_Type(base, T_STRING);
  GET_PARSER(obj, parser);
  ret = XML_SetBase(parser->parser, RSTRING(base)->ptr);

  return INT2FIX(ret);
}

#ifdef NEW_EXPAT
static VALUE
XMLParser_getSpecifiedAttributes(VALUE obj)
{
  XMLParser* parser;
  int count;
  const XML_Char** atts;
  VALUE attrhash;

  GET_PARSER(obj, parser);
  atts = parser->lastAttrs;
  if (!atts)
    return Qnil;
  count = XML_GetSpecifiedAttributeCount(parser->parser)/2;
  attrhash = rb_hash_new();
  while (*atts) {
    const char* key = *atts++;
    atts++;
    rb_hash_aset(attrhash, rb_str_new2((char*)key),
		 (count-- > 0) ? Qtrue: Qfalse);
  }

  return attrhash;
}

static VALUE
XMLParser_getCurrentByteCount(VALUE obj)
{
  XMLParser* parser;

  GET_PARSER(obj, parser);
  return INT2FIX(XML_GetCurrentByteCount(parser->parser));
}
#endif

#ifdef XML_DTD
static VALUE
XMLParser_setParamEntityParsing(VALUE obj, VALUE parsing)
{
  XMLParser* parser;
  int ret;

  Check_Type(parsing, T_FIXNUM);
  GET_PARSER(obj, parser);
  ret = XML_SetParamEntityParsing(parser->parser, FIX2INT(parsing));

  return INT2FIX(ret);
}
#endif

void
Init_xmlparser()
{
  VALUE mXML;

  eXMLParserError = rb_define_class("XMLParserError", rb_eException);
  cXMLParser = rb_define_class("XMLParser", rb_cObject);
  cXMLEncoding = rb_define_class("XMLEncoding", rb_cObject);

  /* Class name aliases */
  mXML = rb_define_module("XML");
  rb_define_const(mXML, "ParserError", eXMLParserError);
  rb_define_const(mXML, "Parser", cXMLParser);
  rb_define_const(mXML, "Encoding", cXMLEncoding);

  rb_define_singleton_method(cXMLParser, "new", XMLParser_new, -1);
  rb_define_method(cXMLParser, "parse", XMLParser_parse, -1);
  rb_define_method(cXMLParser, "done", XMLParser_done, 0);
  rb_define_method(cXMLParser, "defaultCurrent",  XMLParser_defaultCurrent, 0);
  rb_define_method(cXMLParser, "line", XMLParser_getCurrentLineNumber, 0);
  rb_define_method(cXMLParser, "column", XMLParser_getCurrentColumnNumber, 0);
  rb_define_method(cXMLParser, "byteIndex", XMLParser_getCurrentByteIndex, 0);
  rb_define_method(cXMLParser, "setBase", XMLParser_setBase, 1);
#ifdef NEW_EXPAT
  rb_define_method(cXMLParser, "getSpecifiedAttributes",
		   XMLParser_getSpecifiedAttributes, 0);
  rb_define_method(cXMLParser, "byteCount", XMLParser_getCurrentByteCount, 0);
#endif
#ifdef XML_DTD
  rb_define_method(cXMLParser, "setParamEntityParsing",
		   XMLParser_setParamEntityParsing, 1);
#endif

  rb_define_method(cXMLEncoding, "map", XMLEncoding_map, 1);
  rb_define_method(cXMLEncoding, "convert", XMLEncoding_convert, 1);

  rb_define_const(cXMLParser, "START_ELEM", INT2FIX(XML_START_ELEM));
  rb_define_const(cXMLParser, "END_ELEM", INT2FIX(XML_END_ELEM));
  rb_define_const(cXMLParser, "CDATA", INT2FIX(XML_CDATA));
  rb_define_const(cXMLParser, "PI", INT2FIX(XML_PI));
  rb_define_const(cXMLParser, "DEFAULT", INT2FIX(XML_DEFAULT));
  rb_define_const(cXMLParser, "UNPARSED_ENTITY_DECL",
		  INT2FIX(XML_UNPARSED_ENTITY_DECL));
  rb_define_const(cXMLParser, "NOTATION_DECL",
		  INT2FIX(XML_NOTATION_DECL));
  rb_define_const(cXMLParser, "EXTERNAL_ENTITY_REF",
		  INT2FIX(XML_EXTERNAL_ENTITY_REF));
#ifdef NEW_EXPAT
  rb_define_const(cXMLParser, "COMMENT", INT2FIX(XML_COMMENT));
  rb_define_const(cXMLParser, "START_CDATA", INT2FIX(XML_START_CDATA));
  rb_define_const(cXMLParser, "END_CDATA", INT2FIX(XML_END_CDATA));
  rb_define_const(cXMLParser, "START_NAMESPACE_DECL",
		  INT2FIX(XML_START_NAMESPACE_DECL));
  rb_define_const(cXMLParser, "END_NAMESPACE_DECL",
		  INT2FIX(XML_END_NAMESPACE_DECL));
#endif
#ifdef XML_DTD
  rb_define_const(cXMLParser, "PARAM_ENTITY_PARSING_NEVER",
		  INT2FIX(XML_PARAM_ENTITY_PARSING_NEVER));
  rb_define_const(cXMLParser, "PARAM_ENTITY_PARSING_UNLESS_STANDALONE",
		  INT2FIX(XML_PARAM_ENTITY_PARSING_UNLESS_STANDALONE));
  rb_define_const(cXMLParser, "PARAM_ENTITY_PARSING_ALWAYS",
		  INT2FIX(XML_PARAM_ENTITY_PARSING_ALWAYS));
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
  rb_define_const(cXMLParser, "START_DOCTYPE_DECL",
		  INT2FIX(XML_START_DOCTYPE_DECL));
  rb_define_const(cXMLParser, "END_DOCTYPE_DECL",
		  INT2FIX(XML_END_DOCTYPE_DECL));
#endif
#if 0
  rb_define_const(cXMLParser, "UNKNOWN_ENCODING",
		  INT2FIX(XML_UNKNOWN_ENCODING));
#endif

  id_map = rb_intern("_map");
  id_startElementHandler = rb_intern("startElement");
  id_endElementHandler = rb_intern("endElement");
  id_characterDataHandler = rb_intern("character");
  id_processingInstructionHandler = rb_intern("processingInstruction");
  id_defaultHandler = rb_intern("default");
  id_unparsedEntityDeclHandler = rb_intern("unparsedEntityDecl");
  id_notationDeclHandler = rb_intern("notationDecl");
  id_externalEntityRefHandler = rb_intern("externalEntityRef");
#ifdef NEW_EXPAT
  id_defaultExpandHandler = rb_intern("defaultExpand");
  id_commentHandler = rb_intern("comment");
  id_startCdataSectionHandler = rb_intern("startCdata");
  id_endCdataSectionHandler = rb_intern("endCdata");
  id_startNamespaceDeclHandler = rb_intern("startNamespaceDecl");
  id_endNamespaceDeclHandler = rb_intern("endNamespaceDecl");
  id_notStandaloneHandler = rb_intern("notStandalone");
#endif
#ifdef HAVE_XML_SETDOCTYPEDECLHANDLER
  id_startDoctypeDeclHandler = rb_intern("startDoctypeDecl");
  id_endDoctypeDeclHandler = rb_intern("endDoctypeDecl");
#endif
  id_unknownEncoding = rb_intern("unknownEncoding");
  id_convert = rb_intern("convert");
}
