/* $Id: visual.c,v 1.39 1999/03/24 17:53:12 marcus Exp $
******************************************************************************

   XF86DGA display target.

   Copyright (C) 1997-1998 Steve Cheng		[steve@ggi-project.org]
   Copyright (C) 1999      Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/xf86dga.h>
#include "xf86dga.h"

#include <stdlib.h>

static int GGI_xf86dga_idleaccel(ggi_visual *vis)
{
	XSync(DGA_PRIV(vis)->display, 0);

	vis->accelactive = 0;
	
	return 0;
}

static int GGI_xf86dga_flush(ggi_visual *vis, int x, int y, int w, int h,
			     int tryflag)
{
	XSync(DGA_PRIV(vis)->display, 0);

	vis->accelactive = 0;
	
	return 0;
}

void _GGI_xf86dga_freedbs(ggi_visual *vis)
{
	int i;
	int first = LIBGGI_APPLIST(vis)->first_targetbuf;
	int last = LIBGGI_APPLIST(vis)->last_targetbuf;
	
	if (first < 0) {
		return;
	}
	for (i = (last - first); i >= 0; i--) {
		if (LIBGGI_APPBUFS(vis)[i+first]->resource) {
			while (LIBGGI_APPBUFS(vis)[i+first]->resource->count
			       > 0) {
				ggiResourceRelease(LIBGGI_APPBUFS(vis)[i+first]
						   ->resource);
			}
			free(LIBGGI_APPBUFS(vis)[i+first]->resource);
		}
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i+first]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i+first);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf = -1;
}


static int _GGI_xf86dga_getbpp(dga_priv *priv)
{
	XImage *bppcheckimage;
	unsigned int bits_per_pixel = 0;
	
	if ((bppcheckimage = XGetImage(priv->display,
				       RootWindow(priv->display, priv->screen),
				       0, 0, 1, 1, AllPlanes, ZPixmap))
	    != NULL) {
		bits_per_pixel = bppcheckimage->bits_per_pixel;
		XDestroyImage(bppcheckimage);
	}
	return bits_per_pixel;
}


static int do_cleanup(ggi_visual *vis)
{
	dga_priv *priv = LIBGGI_PRIVATE(vis);
	int i;

	/* We may be called more than once due to the LibGG cleanup stuff */
	if (priv == NULL) return 0;

	_GGI_xf86dga_freedbs(vis);

	_ggi_XF86DGADirectVideo(priv->display, priv->screen, 0);
	XSync(priv->display, False);

	XUngrabPointer(priv->display, CurrentTime);
	XUngrabKeyboard(priv->display, CurrentTime);

	if (priv->cmap)	XFreeColormap(priv->display,priv->cmap);
	if (priv->cmap2) XFreeColormap(priv->display,priv->cmap2);

	if (priv->dgamodes) {
		_ggi_XF86VidModeSwitchToMode(priv->display, priv->screen,
					     priv->dgamodes[0]);
		/* Free the modelines */
		for (i = 0; i < priv->num_modes; i++) {
			if (priv->dgamodes[i]->privsize > 0)
				XFree(priv->dgamodes[i]->private);
		}
		XFree(priv->dgamodes);
	}
	if (priv->modes) free(priv->modes);
	
	_ggi_XF86DGAUnmap();

	XSync(priv->display, False);
	XCloseDisplay(priv->display);

	ggLockDestroy(priv->XLibLock);

	free(LIBGGI_PRIVATE(vis));

	if (vis->palette) free(vis->palette);
	
	free(LIBGGI_GC(vis));
	
	LIBGGI_PRIVATE(vis) = NULL;

	ggUnregisterCleanup((ggcleanup_func *)do_cleanup, vis);
	
	return 0;
}


int GGIdlinit(ggi_visual *vis, const char *args, void *argptr)
{
	dga_priv *priv;
	Display *disp;
	int dgafeat;
	/* Just dummy/temp variables */
	int x, y;
	unsigned z;
	Window root;

	GGIDPRINT_MISC("display-DGA starting.\n");

	GGIDPRINT_MISC("display-DGA wants display %s.\n", args);
	disp = XOpenDisplay(args);
	if (disp == NULL) {
		return GGI_DL_ERROR;
	}
	GGIDPRINT("display-DGA has display %s.\n",args);

	_ggi_XF86DGAQueryVersion(disp, &x, &y);
	GGIDPRINT("display-DGA version %d.%d\n", x, y);
	if (x < 1) {
		fprintf(stderr, "Your XF86DGA is too old (%d.%d).\n", x, y);
		XCloseDisplay(disp);
		return GGI_DL_ERROR;
	}

	_ggi_XF86VidModeQueryVersion(disp, &x, &y);
	GGIDPRINT("XF86VidMode version %d.%d\n", x, y);
	
	_ggi_XF86DGAQueryDirectVideo(disp, DefaultScreen(disp), &dgafeat);
	if (! (dgafeat & XF86DGADirectPresent)) {
		fprintf(stderr, "display-DGA: No direct video capability available!\n");
		XCloseDisplay(disp);
		return GGI_DL_ERROR;
	}

	priv = _ggi_malloc(sizeof(dga_priv));

	priv->display	= disp;
	priv->screen	= DefaultScreen(priv->display);
	priv->dgafeat	= dgafeat;
	priv->gc	= 0;
	priv->cmap	= 0;
	priv->cmap2	= 0;
	priv->nocols	= 0;

	if (!_ggi_XF86DGAGetVideo(priv->display, priv->screen,
				  (char**) &priv->fb, &priv->stride,
				  &priv->bank_size, &priv->mem_size)) {
		fprintf(stderr, "display-DGA: Unable to map video memory!\n");
		free(priv);
		XCloseDisplay(disp);
		return GGI_DL_ERROR;
	}

	GGIDPRINT("fb: %p, stride: %d, bank_size: %d, mem_size: %d\n",
	       priv->fb, priv->stride, priv->bank_size, priv->mem_size);

	if (priv->bank_size != priv->mem_size*1024) {
		fprintf(stderr, "display-DGA: Sorry, banked framebuffer layout not supported.\n");
		_ggi_XF86DGAUnmap();
		free(priv);
		XCloseDisplay(disp);
		return GGI_DL_ERROR;
	}

	if ((priv->XLibLock = ggLockCreate()) == NULL) {
		_ggi_XF86DGAUnmap();
		XCloseDisplay(disp);
		free(priv);
		return GGI_DL_ERROR;
	}

	/* Get virtual dimensions */
	XGetGeometry(priv->display,
		RootWindow(priv->display, priv->screen),
		&root, &x, &y,
		&priv->width, &priv->height,
		(unsigned int*)&z,
		&priv->depth);
	
	priv->size = _GGI_xf86dga_getbpp(priv);

	GGIDPRINT_MISC("Virtwidth: %d, depth: %d, size: %d\n", 
		    priv->width, priv->depth, priv->size);

	/* Get XF86VidMode modelines */
	_ggi_XF86VidModeGetAllModeLines(priv->display, priv->screen,
				   &priv->num_modes, &priv->dgamodes);

	priv->modes = _ggi_malloc((priv->num_modes+1)*sizeof(ggi_modelistmode));
	
	for(x = 0; x<priv->num_modes; x++) {
		priv->modes[x].x = priv->dgamodes[x]->hdisplay;
		priv->modes[x].y = priv->dgamodes[x]->vdisplay;
		priv->modes[x].bpp = priv->depth;
		priv->modes[x].gt = GT_CONSTRUCT(priv->depth,
			(priv->depth <= 8) ? GT_PALETTE : GT_TRUECOLOR,
			priv->size);
		GGIDPRINT_MISC("Found mode: %dx%d\n",
			    priv->modes[x].x,
			    priv->modes[x].y);
	}
	priv->modes[priv->num_modes].bpp = 0;

	LIBGGI_PRIVATE(vis) = priv;

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));

	/* Get all the events */
	XGrabKeyboard(priv->display, DefaultRootWindow(priv->display), 
		      True, GrabModeAsync, GrabModeAsync, CurrentTime);

	XGrabPointer(priv->display, DefaultRootWindow(priv->display), True,
		     ButtonPressMask | ButtonReleaseMask | PointerMotionMask,
		     GrabModeAsync, GrabModeAsync,
		     None, None, CurrentTime);

	/* Register cleanup handler */
	ggRegisterCleanup((ggcleanup_func *)do_cleanup, vis);

	{
		gii_inputxwin_arg args;
		gii_input *inp;
		
		args.disp = priv->display;
		args.win = DefaultRootWindow(priv->display);
		args.ptralwaysrel = 1;
		args.wait = 0;
		args.exposefunc = args.exposearg = NULL;
		args.gglock = priv->XLibLock;

		if ((inp = giiOpen("xwin", &args, NULL)) == NULL) {
			GGIDPRINT_MISC("Unable to open xwin inputlib\n");
			GGIdlcleanup(vis);
			return GGI_DL_ERROR;
		}

		priv->inp = inp;
		/* Now join the new event source in. */
		vis->input = giiJoinInputs(vis->input, inp);
        }

	/* Has mode management */
	vis->opdisplay->getmode   = GGI_xf86dga_getmode;
	vis->opdisplay->setmode   = GGI_xf86dga_setmode;
	vis->opdisplay->checkmode = GGI_xf86dga_checkmode;
	vis->opdisplay->getapi    = GGI_xf86dga_getapi;
	vis->opdisplay->setflags  = GGI_xf86dga_setflags;
	vis->opdisplay->flush     = GGI_xf86dga_flush;
	vis->opdisplay->idleaccel = GGI_xf86dga_idleaccel;

	return GGI_DL_OPDISPLAY;
}


int GGIdlcleanup(ggi_visual *vis)
{
	return do_cleanup(vis);
}

#include <ggi/internal/ggidlinit.h>
