/*
 * cubedemo.c - written in 1998 by Andreas Beck   becka@ggi-project.org
 *
 * This is just a pretty cool demo which runs up to 6 other GGI programs
 * on the sides of a spinning 3D cube.
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 */

/* Include the necessary headers used for e.g. error-reporting.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
              
/* A little math for the 3d stuff not much ...
 */
#include <math.h>

/* Include the LibGGI declarations.
 */
#include <ggi/gii.h>
#include <ggi/ggi.h>

/* We do shm here !
 */
#include <sys/ipc.h>
#include <sys/shm.h>
#include <signal.h>

ggi_visual_t vis;
int vissizex,vissizey;

/* Pixel value for white, red and black. See main() on how to get at it.
 */
ggi_pixel white;
ggi_pixel red;
ggi_pixel black;

struct rgb { unsigned char r,g,b; };

struct texture {
	int filex,filey;
	ggi_visual_t	vis;
	pid_t pid;
};

/* In case we were called with wrong parameters, give an explanation.
 */
void usage(const char *prog)
{
	fprintf(stderr, "Usage:\n\n"
		        "%s [[-slavex,slavey] [program]]* \n\n"
		        "Example: %s -320,200 nixterm -160,100 ./demo"
			,prog,prog);
	exit(1);
}

/* Palette helper function
 */
static void Setup_Palette(ggi_visual_t vis, ggi_color *palette)
{
	ggi_mode mode;

	int depth, numcols;

	ggi_color black  = { 0x0000, 0x0000, 0x0000 };
	ggi_color white  = { 0xffff, 0xffff, 0xffff };
	ggi_color blue   = { 0x0000, 0x0000, 0xffff };
	ggi_color yellow = { 0xffff, 0xffff, 0x0000 };


	ggiGetMode(vis, &mode);

	depth = GT_DEPTH(mode.graphtype);
	numcols = 1 << depth;

	if (depth < 3) {

		palette[0] = black;
		palette[1] = white;
		palette[2] = blue;
		palette[3] = yellow;

	} else {
		int i;

		int rnum = (depth+1) / 3;
		int gnum = (depth+2) / 3;
		int bnum = (depth)   / 3;

		int rmask = (1 << rnum) - 1;
		int gmask = (1 << gnum) - 1;
		int bmask = (1 << bnum) - 1;

		for (i=0; i < numcols; i++) {

			int j=i, r, g, b;
			
			b = j & bmask;  j >>= bnum;
			g = j & gmask;  j >>= gnum;
			r = j & rmask;

			palette[i].r = r * 0xffff / rmask;
			palette[i].g = g * 0xffff / gmask;
			palette[i].b = b * 0xffff / bmask;
		}
        }

        ggiSetPalette(vis, 0, numcols, palette);
}

typedef double Matrix3D[3][3];
typedef double Vector3D[3];
typedef struct {
	Vector3D	origin;
	Vector3D	xsize;
	Vector3D	ysize;
	struct texture  texture;
	double		relfacing;
} Plane3D;

void Matrix_times_Vector(double *mat, double *vec, double *result)
{
	int x,y;
	for(x=0;x<3;x++) {
		result[x]=0.0;
		for(y=0;y<3;y++) {
			result[x]+=mat[x*3+y]*vec[y];
		}
	}
}

void Matrix_times_Matrix(double *mat, double *mat2)
{
	int x,y,z;
	double result[9];
	for(x=0;x<3;x++) {
		for(y=0;y<3;y++) {
			result[x+3*y]=0.0;
			for(z=0;z<3;z++)
				result[x+3*y]+=mat2[z+3*y]*mat[x+3*z];
		}
	}
	memcpy(mat,result,sizeof(result));
}

double Matrix_Deter(double *mat)
{
	return  mat[0+3*0]*mat[1+3*1]*mat[2+3*2]
	+	mat[1+3*0]*mat[2+3*1]*mat[0+3*2]
	+	mat[2+3*0]*mat[0+3*1]*mat[1+3*2]
	-	mat[2+3*0]*mat[1+3*1]*mat[0+3*2]
	-	mat[1+3*0]*mat[0+3*1]*mat[2+3*2]
	-	mat[0+3*0]*mat[2+3*1]*mat[1+3*2]
	;
}

double Vector_Length(double *vec)
{
	return  sqrt(vec[0]*vec[0]+vec[1]*vec[1]+vec[2]*vec[2]);
}

void transform_plane(double *mat,Plane3D *in,Plane3D *out)
{
	Matrix_times_Vector(mat,in->origin,out->origin);
	Matrix_times_Vector(mat,in->xsize ,out->xsize );
	Matrix_times_Vector(mat,in->ysize ,out->ysize );
	
	out->texture=in->texture;	/* Copy as well ... */
	
	out->relfacing=  out->xsize[0]*out->ysize[1]
			-out->xsize[1]*out->ysize[0] ;
}

Plane3D cubesides[6]={
	{	{-100, -75,-130}, { 200, 0.0, 0.0}, { 0.0, 150, 0.0},{0,0,NULL,-1}	},
	{	{-130, -75, 100}, { 0.0, 0.0,-200}, { 0.0, 150, 0.0},{0,0,NULL,-1}	},
	{	{ 100, -75, 130}, {-200, 0.0, 0.0}, { 0.0, 150, 0.0},{0,0,NULL,-1}	},
	{	{ 130, -75,-100}, { 0.0, 0.0, 200}, { 0.0, 150, 0.0},{0,0,NULL,-1}	},
	{	{-100,-105,  75}, { 200, 0.0, 0.0}, { 0.0, 0.0,-150},{0,0,NULL,-1}	},
	{	{-100, 105, -75}, { 200, 0.0, 0.0}, { 0.0, 0.0, 150},{0,0,NULL,-1}	}
};

Plane3D currcube[6];

Matrix3D	current={
	{1.0,0.0,0.0},
	{0.0,1.0,0.0},
	{0.0,0.0,1.0}
};

Matrix3D	speed={
	{1.0,0.0,0.0},
	{0.0,1.0,0.0},
	{0.0,0.0,1.0}
};

ggi_pixel inline SetColPat(int posx,int posy,struct texture *file)
{
	ggi_color hlpmap;
	ggi_pixel pixel;
	
#if 1
	ggiGetPixel(file->vis,posx,posy,&pixel);
	ggiUnmapPixel(file->vis,pixel,&hlpmap);
	return ggiMapColor(vis, &hlpmap);
#else
	ggiGetPixel(file->vis,posx,posy,&pixel);
	return pixel;
#endif
}

#define FP_SHIFT	(16)
#define FP_ZERO		(0x00000000)
#define FP_HALF		(0x00008000)

/* Fourth try. Use scanline technique
 */
#if 1
void doblit(int x1,int y1,int xstepx,int xstepy,int ystepx,int ystepy,struct texture *file,int transp)
{
	int miny,maxy,xx,patxadd,patyadd;
#else
void doblit(double x1,double y1,double xstepx,double xstepy,double ystepx,double ystepy,struct texture *file,int transp)
{
	double miny,maxy,xx,patxadd,patyadd;
#endif
	ggi_pixel pixel;
	
	struct {
		int xpos,patx,paty;
	} min,max,hlp;
	
	miny=0;maxy=0;
	
	/* Calc min/max scanline */
	if (      +xstepy<miny) miny=      +xstepy;
	if (ystepy       <miny) miny=ystepy       ;
	if (ystepy+xstepy<miny) miny=ystepy+xstepy;
	if (      +xstepy>maxy) maxy=      +xstepy;
	if (ystepy       >maxy) maxy=ystepy       ;
	if (ystepy+xstepy>maxy) maxy=ystepy+xstepy;

       	ggiSetGCForeground(vis, white);
	for(y1+=miny;miny<=maxy;miny++,y1++)
	{
		min.xpos= 1000000000;
		max.xpos=-1000000000;
		
		if (xstepy && ( ( 0<=miny && miny<=xstepy) ||
				( 0>=miny && miny>=xstepy) ) ) {
			hlp.patx=((int)((file->filex-1)*miny/xstepy))<<FP_SHIFT;
			hlp.paty=FP_ZERO;
			hlp.xpos=x1 + xstepx*miny/xstepy;
			if (hlp.xpos<min.xpos) min=hlp;
			if (hlp.xpos>max.xpos) max=hlp;
		}

		if (ystepy && ( ( 0<=miny && miny<=ystepy) ||
				( 0>=miny && miny>=ystepy) ) ) {
			hlp.patx=FP_ZERO;
			hlp.paty=((int)((file->filey-1)*miny/ystepy))<<FP_SHIFT;
			hlp.xpos=x1 + ystepx*miny/ystepy;
			if (hlp.xpos<min.xpos) min=hlp;
			if (hlp.xpos>max.xpos) max=hlp;
		}
		if (xstepy && ( ( 0<=miny-ystepy && miny-ystepy<=xstepy) ||
				( 0>=miny-ystepy && miny-ystepy>=xstepy) ) ) {
			hlp.patx=((int)((file->filex-1)*(miny-ystepy)/xstepy))<<FP_SHIFT;
			hlp.paty=(file->filey-1)<<FP_SHIFT;
			hlp.xpos=x1 + ystepx + xstepx*(miny-ystepy)/xstepy;
			if (hlp.xpos<min.xpos) min=hlp;
			if (hlp.xpos>max.xpos) max=hlp;
		}

		if (ystepy && ( ( 0<=miny-xstepy && miny-xstepy<=ystepy) ||
				( 0>=miny-xstepy && miny-xstepy>=ystepy) ) ) {
			hlp.patx=(file->filex-1)<<FP_SHIFT;
			hlp.paty=((int)((file->filey-1)*(miny-xstepy)/ystepy))<<FP_SHIFT;
			hlp.xpos=x1 + xstepx + ystepx*(miny-xstepy)/ystepy;
			if (hlp.xpos<min.xpos) min=hlp;
			if (hlp.xpos>max.xpos) max=hlp;
		}

		if (max.xpos< 0       ) continue;
		if (min.xpos>=vissizex) continue;

		if (min.xpos==max.xpos) {
			if ((pixel=SetColPat((min.patx>>FP_SHIFT),(min.paty>>FP_SHIFT),file)) || !transp)
				ggiPutPixel(vis,min.xpos,y1,pixel);
		} else {
			patxadd=(max.patx-min.patx)/(xx=(max.xpos-min.xpos+1));
			patyadd=(max.paty-min.paty)/xx;
			min.patx+=FP_HALF;
			min.paty+=FP_HALF;
			for(xx=min.xpos;xx<=max.xpos;xx++,
						min.patx+=patxadd,
						min.paty+=patyadd) {
				if (xx<0) continue;
				if (xx>vissizex) break;
				if ((pixel=SetColPat((min.patx>>FP_SHIFT),(min.paty>>FP_SHIFT),file))||!transp)
					ggiPutPixel(vis,xx,y1,pixel);
			}
		}
	}
}

void turn_add(int ax1,int ax2,double degree)
{
	Matrix3D turnit;
	memset(turnit,0,sizeof(turnit));

	turnit[0]  [0]  =  turnit[1]  [1]  =turnit[2][2]=1.0;
	turnit[ax1][ax1]=  turnit[ax2][ax2]=cos(degree);
	turnit[ax1][ax2]=-(turnit[ax2][ax1]=sin(degree));
	
	Matrix_times_Matrix((double *)speed,(double *)turnit);
}

void scale(double fac)
{
	int x;
	for(x=0;x<9;x++) ((double *)speed)[x]*=fac;
}

void stop_speed(void)
{
	memset(speed,0,sizeof(speed));
	speed[0][0]=speed[1][1]=speed[2][2]=1.0;
}

int align(int face)
{
	double angle1,angle2,angle3;
	static int major,x;
	Vector3D help;
	
	stop_speed();
	
	for(x=0;x<3;x++)
		help[x]=currcube[face].xsize[x];
	help[1]=0.0;
	angle1=currcube[face].xsize[0]/Vector_Length(help);
	if (angle1<=1.0) angle1=acos(angle1);else angle1=0.0;

	for(x=0;x<3;x++)
		help[x]=currcube[face].ysize[x];
	help[0]=0.0;
	angle2=currcube[face].ysize[1]/Vector_Length(help);
	if (angle2<=1.0) angle2=acos(angle2);else angle2=0.0;

	for(x=0;x<3;x++)
		help[x]=currcube[face].xsize[x];
	help[2]=0.0;
	angle3=currcube[face].xsize[0]/Vector_Length(help);
	if (angle3<=1.0) angle3=acos(angle3);else angle3=0.0;

	major=!major;

	{
		if (angle1>3.0/180.0*M_PI) angle1=3.0/180.0*M_PI;
		if (currcube[face].xsize[2]>0) angle1=-angle1;
		turn_add(0,2,angle1);
	}
	{
		if (angle2>3.0/180.0*M_PI) angle2=3.0/180.0*M_PI;
		if (currcube[face].ysize[2]>0) angle2=-angle2;
		turn_add(1,2,angle2);
	}
	{
		if (angle3>3.0/180.0*M_PI) angle3=3.0/180.0*M_PI;
		if (currcube[face].xsize[1]>0) angle3=-angle3;
		turn_add(0,1,angle3);
	}

	return (fabs(angle1)+fabs(angle2)+fabs(angle3) <=1e-3);
}

void highlight_face(int active_face)
{
	double upscale=1.10;


	ggiDrawLine(vis,vissizex/2+upscale*(currcube[active_face].origin[0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]),
			vissizex/2+upscale*(currcube[active_face].origin[0]+currcube[active_face].xsize [0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]+currcube[active_face].xsize [1])
	);
	ggiDrawLine(vis,vissizex/2+upscale*(currcube[active_face].origin[0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]),
			vissizex/2+upscale*(currcube[active_face].origin[0]+currcube[active_face].ysize [0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]+currcube[active_face].ysize [1])
	);
	ggiDrawLine(vis,vissizex/2+upscale*(currcube[active_face].origin[0]+currcube[active_face].ysize [0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]+currcube[active_face].ysize [1]),
			vissizex/2+upscale*(currcube[active_face].origin[0]+currcube[active_face].xsize [0]+currcube[active_face].ysize [0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]+currcube[active_face].xsize [1]+currcube[active_face].ysize [1])
	);
	ggiDrawLine(vis,vissizex/2+upscale*(currcube[active_face].origin[0]+currcube[active_face].xsize [0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]+currcube[active_face].xsize [1]),
			vissizex/2+upscale*(currcube[active_face].origin[0]+currcube[active_face].ysize [0]+currcube[active_face].xsize [0]),
			vissizey/2+upscale*(currcube[active_face].origin[1]+currcube[active_face].ysize [1]+currcube[active_face].xsize [1])
	);
}

int spawn_bg(char *what)
{
	int pid;

	pid = fork();
	if (pid == -1)
		return -1;
	if (pid == 0) {
		execlp("/bin/sh","/bin/sh","-c",what, NULL);
		exit(127);
	} else return pid;
}

int main(int argc, char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */

	const char *prog;	/* Make an alias for the program name */
	ggi_color map;		/* Helper for color mapping */

	ggi_graphtype type;	/* graphics type */
	int depth;		/* bit depth */
	
	int err,x;		/* counters and other helpers. */

	int frame=0;		/* The frame we are drawing on. */
	int frcnt=0;		/* framecounter for profiling performace */
	clock_t clk;		/* clock helper for profiling */

	int noquit;		/* main loop termination variable */
	int is_escape;		/* are we in escaped state ? */
	struct timeval tv;	/* timeout for polling the keyboard */

	int active_face=0;	/* program state. Active face of the cube */
	int autoactive=0;	/* automatic activation of "frontmost" face */
	int doalign=0;		/* autoalignment in progress */
	int backfaces=1;	/* show back faces */
	int blink=0;		/* make the active frame blink */
	int transparency=1;	/* make color 0 transparent */
	int progarg=0;
	int slavesizex,slavesizey;
	
	/* The mode we want. 2 frames for doublebuffering */
	ggi_mode mode = { /* This will cause the default mode to be set */
		2,                      /* 2 frames */
		{GGI_AUTO,GGI_AUTO},    /* Default size */
		{GGI_AUTO,GGI_AUTO},    /* Virtual */
		{0,0},                  /* size in mm don't care */
		GT_AUTO,               /* Mode */
		{GGI_AUTO,GGI_AUTO}     /* Font size */
	};

	/* The memvisuals for the other programs ... */
 	ggi_visual_t memvis[6];
 	ggi_mode     submode[6];
 	int shmid[6];

	/* Get the arguments from the command line. 
	 * Set defaults for optional arguments.
	 */

	prog=*argv; argv++; argc--;
	if (strrchr(prog, '/') != NULL) prog=strrchr(prog, '/')+1;

	if ((argc > 0) &&
	    ((strcmp(*argv, "-h")==0) || 
	     (strcmp(*argv, "--help")==0))) {

		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}


	/* Open up GGI and a visual.
	 */
	if (ggiInit() != 0) {
		fprintf(stderr, "unable to initialize libggi, exiting.\n");
		exit(1);
	}

	vis=ggiOpen(NULL);

	if (vis == NULL) {
		fprintf(stderr,
			"unable to open default visual, exiting.\n");
		ggiExit();
		exit(1);
	}

	/* Go to async-mode.
	 */
        ggiSetFlags(vis, GGIFLAG_ASYNC);

	/* Check and set the mode ...
	 */
	ggiCheckMode(vis,&mode);
	fprintf(stderr,"Suggested mode ");
	ggiFPrintMode(stderr,&mode);
	fprintf(stderr,"\n");
	err=ggiSetMode(vis,&mode);   /* now try it. it *should* work! */

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		ggiClose(vis);
		ggiExit();
		return 2;
	}


	type=mode.graphtype;
	vissizex=mode.visible.x;
	vissizey=mode.visible.y;
	slavesizex=mode.visible.x/2;
	slavesizey=mode.visible.y/2;
	depth=GT_DEPTH(mode.graphtype);

	if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
		ggiSetColorfulPalette(vis);
	}
	
	map.r= map.g= map.b= 0xFFFF;
	white= ggiMapColor(vis, &map);
	map.r= map.g= map.b= 0x0;
	black= ggiMapColor(vis, &map);
	map.g= map.b= 0x0; map.r= 0xFFFF;
	red  = ggiMapColor(vis, &map);

	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);
	ggiSetGCForeground(vis, white);
	ggiSetGCBackground(vis, black);
	ggiPuts(vis,0,0,"Hold on - creating faces.");
	ggiFlush(vis);

	printf("Server is running - start another GGI demo (e.g. flying_ggis)\n");
	printf("with the following variable settings:\n");

	for(x=0;x<6;x++)
	{
		char text[1024];
		int memlen;

		/* First check for arguments.
		 */
		if (progarg<argc && argv[progarg][0]=='-') {
			/* We have an option */
			sscanf(argv[progarg]+1,"%dx%d",&slavesizex,&slavesizey);
			progarg++;
		}
		submode[x]=mode;
		submode[x].frames=1;
		submode[x].visible.x=slavesizex;
		submode[x].visible.y=slavesizey;
		submode[x].virt.x   =slavesizex;
		submode[x].virt.y   =slavesizey;
		
		memlen=submode[x].virt.x*submode[x].virt.y*((depth+7)/8)+64*1024;
		/* Add some slack for the -input queues */

		/* Allocate space for the shmem visuals. */
		shmid[x]=shmget(ftok("/dev/null",'0'+x), memlen, IPC_CREAT|0666);

		/* Open a shared "memory-visual" which is simply a simulated 
		 * display in shared memory.
		 */
		sprintf(text,"display-memory:-input:shmid:%d",shmid[x]);

		if ((memvis[x]=ggiOpen(text, NULL)) == NULL) {
			ggiPanic("Ouch - can't open the shmem target %d !",x);
		}

		err=ggiSetMode(memvis[x],&submode[x]);
		
		if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
			ggiSetColorfulPalette(memvis[x]);
		}

		/* Check for errors */
		if (err) ggiPanic("Ouch - can't setmode on shmem target %d !",x);

		cubesides[x].texture.vis=memvis[x];
		cubesides[x].texture.filex=submode[x].visible.x;
		cubesides[x].texture.filey=submode[x].visible.y;

		sprintf(text,"display-memory:-input:keyfile:%d:%d:%s",memlen,x,"/dev/null");
		setenv("GGI_DISPLAY",text,1);
		ggiSPrintMode(text,&submode[x]);
		setenv("GGI_DEFMODE",text,1);
		if (progarg<argc) {
			printf("face %d execing: %s\n",x,argv[progarg]);
			cubesides[x].texture.pid=spawn_bg(argv[progarg++]);
		} else {
			printf("GGI_DEFMODE=\"%s\"\n",text);
			sprintf(text,"display-memory:-input:keyfile:%d:%d:%s",memlen,x,"/dev/null");
			printf("GGI_DISPLAY=%s\n",text);
		}
	}

	{
		ggi_color black  = { 0x0000, 0x0000, 0x0000 };
		ggi_color white  = { 0xffff, 0xffff, 0xffff };
		ggiSetGCForeground(memvis[0],ggiMapColor(memvis[0],&white));
		ggiSetGCBackground(memvis[0],ggiMapColor(memvis[0],&black));
		ggiPuts(memvis[0],0,  0,"Keyboard:");
		ggiPuts(memvis[0],0, 10,"#: Got to Cube control");
		ggiPuts(memvis[0],0, 20,"Cursor,Home,End: Rotate");
		ggiPuts(memvis[0],0, 30,"PgUp/PgDown    : Resize");
		ggiPuts(memvis[0],0, 40,"q: quit s:stop b:backfaces");
	}

	ggiFlush(vis);
	noquit=1;
	is_escape=0;
	clk=clock();
	stop_speed();
	scale(vissizey/150.0/1.10*0.5);	/* Scale to 50% of screen */
	Matrix_times_Matrix((double *)current,(double *)speed);
	stop_speed();
	turn_add(0,2,-M_PI/180.0);
	turn_add(0,1,-M_PI/180.0);
	turn_add(2,1,-M_PI/180.0);

	while(noquit) {
		
		
		blink=!blink;
		Matrix_times_Matrix((double *)current,(double *)speed);

		{
			double best=0.0;
			for(x=0;x<6;x++) {
				ggiGetMode(memvis[x],&submode[x]);
				if (cubesides[x].texture.filex!=
					submode[x].visible.x ||
					cubesides[x].texture.filey!=
					submode[x].visible.y) {
						ggiSetMode(memvis[x],&submode[x]);
						if (GT_SCHEME(submode[x].graphtype) == GT_PALETTE) {
							ggiSetColorfulPalette(memvis[x]);
						}
						cubesides[x].texture.filex=submode[x].visible.x;
						cubesides[x].texture.filey=submode[x].visible.y;
						printf("Mode change on face %d to %dx%d\n",x,
						cubesides[x].texture.filex,
						cubesides[x].texture.filey);
				}
			}
			for(x=0;x<6;x++) {
				transform_plane((double *)current,&cubesides[x],&currcube[x]);
				if (autoactive && currcube[x].relfacing > best ) {
					best=currcube[x].relfacing;
					active_face=x;
				}
			}
		}

		if (backfaces) {
			for(x=0;x<6;x++) {
				if (0.0>currcube[x].relfacing) {
					doblit(	vissizex/2+currcube[x].origin[0],
						vissizey/2+currcube[x].origin[1],
						currcube[x].xsize [0],
						currcube[x].xsize [1],
						currcube[x].ysize [0],
						currcube[x].ysize [1],
						&currcube[x].texture,
						transparency
						);
					if (x==active_face) {
						ggiSetGCForeground(vis, blink ? black : (is_escape ? red : white));
						highlight_face(x);
					}
				}
			}
		} else {
			for(x=0;x<6;x++) {
				if (0.0>currcube[x].relfacing && x==active_face) {
					ggiSetGCForeground(vis, blink ? black : (is_escape ? red : white));
					highlight_face(x);
				}
			}
		}

		for(x=0;x<6;x++)
		{
			if (0.0<=currcube[x].relfacing)
			{
				doblit( vissizex/2+currcube[x].origin[0],
					vissizey/2+currcube[x].origin[1],
					currcube[x].xsize [0],
					currcube[x].xsize [1],
					currcube[x].ysize [0],
					currcube[x].ysize [1],
					&currcube[x].texture,
					transparency
					);
				if (x==active_face) {
					ggiSetGCForeground(vis, blink ? black : (is_escape ? red : white));
					highlight_face(x);
				}
			}
		}

		if (doalign) 
			if (align(active_face)) 
				doalign=0;

		tv.tv_sec=tv.tv_usec=0;

		while(ggiEventPoll(vis,emAll,&tv))
		{
			ggi_event event;
			ggiEventRead(vis,&event,emAll);

			if (event.any.type==evKeyPress)
			{
				if (!is_escape) {
					if (event.key.sym=='#') is_escape=1;
					else {
						sendit:
						ggiEventSend(memvis[active_face], &event);
						is_escape=0;continue;
					}
				} else {
					switch(event.key.sym)
					{
						case GIIK_Left:
							turn_add(0,2,-M_PI/180.0);
							break;
						case GIIK_Right:
							turn_add(0,2, M_PI/180.0);
							break;
						case GIIK_Up:
							turn_add(1,2,-M_PI/180.0);
							break;
						case GIIK_Down:
							turn_add(1,2,M_PI/180.0);
							break;
						case GIIK_Home:
							turn_add(0,1,-M_PI/180.0);
							break;
						case GIIK_End:
							turn_add(0,1,M_PI/180.0);
							break;
						case GIIK_PageUp:
							scale(1.01);break;
						case GIIK_PageDown:
							scale(1.0/1.01);break;
						case 's':
						case 'S':
							stop_speed();
							break;
						case 'c':
						case 'C':
							doalign=!doalign;
							break;
						case 'b':
						case 'B':
							backfaces=!backfaces;
							break;
						case 'a':
						case 'A': 
							autoactive=!autoactive;
							break;
						case 't':
						case 'T': 
							transparency=!transparency;
							break;
						case '0':
							autoactive=0;
							active_face=0;
							break;
						case '1':
							autoactive=0;
							active_face=1;
							break;
						case '2':
							autoactive=0;
							active_face=2;
							break;
						case '3':
							autoactive=0;
							active_face=3;
							break;
						case '4':
							autoactive=0;
							active_face=4;
							break;
						case '5':
							autoactive=0;
							active_face=5;
							break;
						case 'q':
						case 'Q':
							noquit=0;
							break;
						case '#':
							goto sendit;
						case 'f':
						case 'F':
							printf("FPS:%f\n",(double)frcnt/(clock()-clk+1.0)*CLOCKS_PER_SEC);
							clk=clock();frcnt=0;
							break;
						case 'd':
						case 'D':
							printf("%f,%f,%f %f,%f,%f %f,%f,%f\n",
								currcube[0].origin[0],
								currcube[0].origin[1],
								currcube[0].origin[1],
								currcube[0].xsize[0],
								currcube[0].xsize[1],
								currcube[0].xsize[2],
								currcube[0].ysize[0],
								currcube[0].ysize[1],
								currcube[0].ysize[2]
								);
							break;
						default:
							printf("Unknown command key %d\n",event.key.sym);
						case GIIK_Enter:
							is_escape=0;
							break;
					}
				}
			} else {
				ggiEventSend(memvis[active_face], &event);
			}
		}
		ggiSetDisplayFrame(vis,frame);
		ggiFlush(vis);
		frcnt++;
		ggiSetWriteFrame(vis,frame=!frame);
        	ggiSetGCForeground(vis, black);
        	ggiFillscreen(vis);
	}

	for(x=0;x<6;x++)
	{
		if (cubesides[x].texture.pid>0)
			kill(cubesides[x].texture.pid,SIGHUP);
		ggiClose(memvis[x]);

		/* this makes the memory detach when all programs using this memory
		 * complete (according to SVR4 standard).
		 * note - this may not work on all platforms....
		 * note2: This may _not be called at an earlier place.
		 * Calling before our own shmat() will destroy _instantly_,
		 * calling before the other programs are running will deny
		 * the shmem to them.
		 */
		shmctl(shmid[x],IPC_RMID, NULL);
	}
	ggiClose(vis);
	ggiExit();	
	return 0;
}
