///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  heap.h
  ------------------------------------------------------------------------
  hoardHeap, the base class for threadHeap and processHeap.
  ------------------------------------------------------------------------
  @(#) $Id: heap.h,v 1.46 2000/02/12 21:45:48 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _HEAP_H_
#define _HEAP_H_

#include "config.h"

#include <assert.h>
#include <math.h>

#include "arch-specific.h"
#include "superblock.h"
#include "heapstats.h"

class processHeap; // forward declaration


class hoardHeap {

public:

  hoardHeap (void);

  // A superblock that holds more than one object must hold at least
  // this many bytes.
  enum { SUPERBLOCK_SIZE = 8192 };

  // The number of empty superblocks that we allow any thread heap to hold.
  enum { MAX_EMPTY_SUPERBLOCKS = 2 };

  // A thread heap must be at least 1/EMPTY_FRACTION empty before we
  // start returning superblocks to the process heap.
  enum { EMPTY_FRACTION = 2 };

  // The maximum number of thread heaps we allow.  (NOT the maximum
  // number of threads -- Hoard imposes no such limit.)  This must be
  // a power of two! NB: This number is twice the maximum number of
  // PROCESSORS supported by Hoard.
  enum { MAX_HEAPS = 128 };

  // ANDing with this rounds to MAX_HEAPS.
  enum { MAX_HEAPS_MASK = MAX_HEAPS - 1 };

  // The number of size classes.  This combined with the
  // SIZE_CLASS_BASE determine the maximum size of an object.
  enum { SIZE_CLASSES = 115 };

  // Every object is aligned so that it can hold a double.
  enum { ALIGNMENT = sizeof(double) };

  // ANDing with this rounds to ALIGNMENT.
  enum { ALIGNMENT_MASK = ALIGNMENT - 1};

  // Used for sanity checking.
  enum { HEAP_MAGIC = 0x0badcafe };

  // Update memory in-use and allocated statistics.
  // (*UStats = just update U.)
  inline void incStats (int sizeclass, int updateU, int updateA);
  inline void incUStats (int sizeclass);

  inline void decStats (int sizeclass, int updateU, int updateA);
  inline void decUStats (int sizeclass);
  inline void decUStats (int sizeclass, int& U, int& A);

  inline void getStats (int sizeclass, int& U, int& A);


#if HEAP_STATS
  // How much is the maximum ever in use for this size class?
  inline int maxInUse (int sizeclass);

  // How much is the maximum memory allocated for this size class?
  inline int maxAllocated (int sizeclass);
#endif

  // Insert a superblock into our list.
  void insertSuperblock (int sizeclass,
			 superblock * sb);

  // Remove the superblock with the most free space.
  superblock * removeMaxSuperblock (int sizeclass);

  // Move a particular superblock from one bin to another.
  void moveSuperblock (superblock *,
		       int sizeclass,
		       int fromBin,
		       int toBin);

  // Remove a particular superblock.
  void removeSuperblock (superblock *, int sizeclass);

  // Find an available superblock (i.e., with some space in it).
  inline superblock * findAvailableSuperblock (int sizeclass,
					       block *& b);

  // Return the size class for a given size.
  inline static int sizeClass (const size_t sz);

  // Return the size corresponding to a given size class.
  inline static int sizeFromClass (const int sizeclass);

  // Return the release threshold corresponding to a given size class.
  inline static int getReleaseThreshold (const int sizeclass);

  // Align a value.
  inline static size_t align (const size_t sz);

  // Lock this heap.
  inline void lock (void);

  // Unlock this heap.
  inline void unlock (void);

  // Set our index number (which heap we are).
  inline void setIndex (int i);

  // Get our index number (which heap we are).
  inline int getIndex (void);

  // Free a block into a superblock.
  // This is used by processHeap::free().
  void freeBlock (block *& b,
		  superblock *& sb,
		  int sizeclass,
		  processHeap * processHeap);

protected:

#if HEAP_DEBUG
  // For sanity checking.
  const unsigned long _magic;
#else
  #define _magic HEAP_MAGIC
#endif

  // Heap statistics.
  heapStats	_stats[SIZE_CLASSES];

  // The per-heap lock.
  hoardLockType _lock;

  // Which heap this is (0 = process heap).
  int _index;

private:

  // Disable copying and assignment.

  hoardHeap (const hoardHeap&);
  const hoardHeap& operator= (const hoardHeap&);

  // Lists of superblocks.
  superblock *	_superblocks[SUPERBLOCK_FULLNESS_GROUP][SIZE_CLASSES];

  // The current least-empty superblock bin.
  int	_leastEmptyBin[SIZE_CLASSES];

  // The lookup table for size classes.
  static size_t	_sizeTable[SIZE_CLASSES];

  // The lookup table for release thresholds.
  static size_t	_threshold[SIZE_CLASSES];
};


void hoardHeap::incStats (int sizeclass, int updateU, int updateA) {
  assert (_magic == HEAP_MAGIC);
  assert (updateU >= 0);
  assert (updateA >= 0);
  _stats[sizeclass].incStats (updateU, updateA);
}



void hoardHeap::incUStats (int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].incUStats ();
}


void hoardHeap::decStats (int sizeclass, int updateU, int updateA) {
  assert (_magic == HEAP_MAGIC);
  assert (updateU >= 0);
  assert (updateA >= 0);
  _stats[sizeclass].decStats (updateU, updateA);
}


void hoardHeap::decUStats (int sizeclass)
{
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].decUStats();
}


void hoardHeap::decUStats (int sizeclass,
		      int& U,
		      int& A)
{
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].decUStats (U, A);
}


void hoardHeap::getStats (int sizeclass, int& U, int& A) {
  assert (_magic == HEAP_MAGIC);
  _stats[sizeclass].getStats (U, A);
}


#if HEAP_STATS
int hoardHeap::maxInUse (int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  return _stats[sizeclass].getUmax();
}


int hoardHeap::maxAllocated (int sizeclass) {
  assert (_magic == HEAP_MAGIC);
  return _stats[sizeclass].getAmax(); 
}
#endif


superblock * hoardHeap::findAvailableSuperblock (int sizeclass,
						 block *& b)
{
  assert (this);
  assert (_magic == HEAP_MAGIC);
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);

  superblock * sb = NULL;

  // Look through the superblocks, starting with the
  // almost-full ones and going to the emptiest ones.
  // (starting value for i was SUPERBLOCK_FULLNESS_GROUP - 2)
  for (int i = _leastEmptyBin[sizeclass]; i >= 0; i--) {
    sb = _superblocks[i][sizeclass];
    if (!sb) {
      if (i == _leastEmptyBin[sizeclass]) {
	_leastEmptyBin[sizeclass]--;
      }
    } else {
      assert (sb->getFullness() == i);
      assert (sb->isValid());
      assert (sb->getOwner() == this);
      b = sb->getBlock();
      assert (b != NULL);

      // If we've crossed a fullness group,
      // move the superblock.
      int fullness = sb->getFullness();
      // printf ("fullness = %d, numblocks = %d, numavail = %d\n", fullness, sb->getNumBlocks(), sb->getNumAvailable());
      if (fullness != i) {
	// Move the superblock.
	moveSuperblock (sb, sizeclass, i, fullness);
	// removeSuperblock (sb, sizeclass);
	// insertSuperblock (sizeclass, sb);
      }
      break;
    }
  }

  // Either we didn't find a superblock or we did and got a block.
  assert ((sb == NULL) || (b != NULL));
  // Either we didn't get a block or we did and we also got a superblock.
  assert ((b == NULL) || (sb != NULL));

  return sb;
}


int hoardHeap::sizeClass (const size_t sz) {
  // Find the size class for a given object size
  // (the smallest i such that _sizeTable[i] >= sz).
  int sizeclass = 0;
  while (_sizeTable[sizeclass] < sz) 
    {
      sizeclass++;
      assert (sizeclass < SIZE_CLASSES);
    }
  return sizeclass;
}


int hoardHeap::sizeFromClass (const int sizeclass) {
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);
  return _sizeTable[sizeclass];
}


int hoardHeap::getReleaseThreshold (const int sizeclass) {
  assert (sizeclass >= 0);
  assert (sizeclass < SIZE_CLASSES);
  return _threshold[sizeclass];
}


void hoardHeap::lock (void) 
{
  assert (_magic == HEAP_MAGIC);
  hoardLock (_lock);
}


void hoardHeap::unlock (void) {
  assert (_magic == HEAP_MAGIC);
  hoardUnlock (_lock);
}


size_t hoardHeap::align (const size_t sz)
{
  // Align sz up to the nearest multiple of ALIGNMENT.
  // This is much faster than using multiplication
  // and division.
  return (sz + ALIGNMENT_MASK) & ~ALIGNMENT_MASK;
}


void hoardHeap::setIndex (int i) 
{
  _index = i; 
}


int hoardHeap::getIndex (void)
{
  return _index;
}


#endif // _HEAP_H_
