/* File "libmalaga.c":
 * A Malaga library to analyse words and sentences. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <setjmp.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "files.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rules.h"
#include "tries.h"
#include "analysis.h"
#include "input.h"
#include "commands.h"
#include "options.h"
#include "malaga_lib.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL
#include "libmalaga.h"

/* variables ================================================================*/

LOCAL jmp_buf error_exit; /* marks place where to jump after an error. */
LOCAL bool_t exit_valid;  /* TRUE iff "error_exit" is valid. */
LOCAL char error_buffer[300];

/* functions ================================================================*/

GLOBAL void error (string_t format, ...)
/* error handling */
{
  va_list arg;

  if (executing_rule) 
  {
    long_t line;
    string_t file_name, rule_name;
    
    /* Get source file position.
     * Subtract 1 from <pc> because when an instruction is executed,
     * <pc> usually already points to next instruction */
    source_of_instr (executed_rule_sys, pc - 1, &line, NULL, &file_name,
		     &rule_name);
    
    if (line != -1)
      sprintf (error_buffer, "file \"%s\", line %ld, rule \"%s\": ", 
	       name_in_path (file_name), line, rule_name);
  }
  else 
    error_buffer[0] = EOS; 

  va_start (arg, format);
  vsprintf (error_buffer + strlen (error_buffer), format, arg);
  va_end (arg);
  
  if (exit_valid)
  {
    malaga_error = error_buffer;
    longjmp (error_exit, 1);
  }
  else
  {
    fprintf (stderr, "error: %s", error_buffer);
    exit (1);
  }
}

/*---------------------------------------------------------------------------*/

LOCAL bool_t check_error (void)
{
  malaga_error = NULL;
  if (setjmp (error_exit) != 0)
  {
    exit_valid = FALSE;
    return TRUE;
  }
  exit_valid = TRUE;
  return FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t analyse_item (string_t item, grammar_t grammar, 
			    bool_t build_tree)
/* Analyse <item> according to <grammar>.
 * <grammar> must be MORPHOLOGY or SYNTAX.
 * If <build_tree> == TRUE, create an analysis tree.
 * Return TRUE if there is at least one analysis result. */
{
  if (check_error ())
    return FALSE;

  copy_string (analysis_input, ENCODED_STRING (item), 
	       analysis_input + ANALYSIS_INPUT_LENGTH);
  preprocess_input (analysis_input);
  analyse (grammar, analysis_input, build_tree, TRUE);
  exit_valid = FALSE;

  reset_analysis_nodes ();
  return reset_analysis_results ();
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t get_info (grammar_t grammar)
/* Return information about <grammar>. */
{
  malaga_error = NULL;
  return info[grammar];
}

/*---------------------------------------------------------------------------*/

GLOBAL void set_option (string_t option)
/* Set a libmalaga option. */
{
  if (check_error ())
    return;

  set_command.command (option);
  exit_valid = FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL void init_libmalaga (string_t project_file)
/* Initialise this module. */
{
  if (check_error ())
    return;

  init_malaga (project_file);
  exit_valid = FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_libmalaga (void)
/* Terminate this module. */
{
  if (check_error ())
    return;
  terminate_malaga ();
  exit_valid = FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t next_analysis_result (void)
/* Get the next result of the last call of "analyse_item".
 * Return NULL if there is no more result. */
{
  malaga_error = NULL;
  return get_next_analysis_result ();
}

/*---------------------------------------------------------------------------*/

GLOBAL analysis_node_t *next_analysis_node (void)
/* Get the next analysis tree node of the last call of "analyse_item".
 * Return NULL if there is no more node. 
 * The analysis node is valid until this function is called again. */
{
  analysis_node_t *node;

  if (check_error ())
    return NULL;
  node = get_next_analysis_node ();
  exit_valid = FALSE;
  return node;
}

/*---------------------------------------------------------------------------*/
