;;;
;;; elmo-msgdb.el -- Message Database for Elmo.
;;;   Copyright 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'elmo-vars)
(require 'elmo-util)
(require 'emu)
(require 'std11)
(require 'elmo-cache)

(defun elmo-msgdb-expand-path (folder &optional spec)
  (elmo-convert-standard-filename
   (let* ((spec (or spec (elmo-folder-get-spec folder)))
	  (type    (nth 0 spec))
	  (fld-name (nth 1 spec))
	  (server  (nth 2 spec))
	  (user    (nth 3 spec)))
     (cond 
      ((eq type 'imap4)
       (expand-file-name 
	fld-name
	(expand-file-name (or user "nobody")
			  (expand-file-name (or server "nowhere")
					    (expand-file-name 
					     "imap"
					     elmo-msgdb-dir)))))
      ((eq type 'nntp)
       (expand-file-name 
	fld-name
	(expand-file-name (or server "nowhere")
			  (expand-file-name "nntp"
					    elmo-msgdb-dir))))
      ((eq type 'maildir) ; qmail Maildir... ???
       (expand-file-name (elmo-safe-filename fld-name)
			 (expand-file-name "maildir"
					   elmo-msgdb-dir)))
      ((eq type 'folder)
       (expand-file-name (elmo-safe-filename fld-name)
			 (expand-file-name "folder"
					   elmo-msgdb-dir)))
      ((eq type 'multi)
       (expand-file-name (elmo-safe-filename folder)
			 (expand-file-name "multi"
					   elmo-msgdb-dir)))
      ((eq type 'filter)
       (expand-file-name
	(elmo-safe-filename folder)
	(expand-file-name "filter"
			  elmo-msgdb-dir)))
      ((eq type 'archive)
       (expand-file-name 
	(directory-file-name
	 (concat
	  (elmo-replace-in-string 
	   (elmo-replace-in-string
	    (elmo-replace-in-string 
	     fld-name
	     "/" "_")
	    ":" "__")
	   "~" "___")
	  "/" user))
	(expand-file-name (concat (symbol-name type) "/" (symbol-name server))
			  elmo-msgdb-dir)))
      ((eq type 'pop3)
       (expand-file-name 
	(elmo-safe-filename fld-name)
	(expand-file-name server
			  (expand-file-name
			   "pop"
			   elmo-msgdb-dir))))
      ((eq type 'localnews)
       (expand-file-name
	(elmo-replace-in-string fld-name "/" ".")
	(expand-file-name "localnews"
			  elmo-msgdb-dir)))
      ((eq type 'internal)
       (expand-file-name (elmo-safe-filename (concat (symbol-name (nth 1 spec))
						     (nth 2 spec)))
			 (expand-file-name "internal"
					   elmo-msgdb-dir)))
      (t ; local dir or undefined type
       (expand-file-name fld-name
			 (expand-file-name (symbol-name type)
					   elmo-msgdb-dir)))))))

(defsubst elmo-msgdb-append-element (list element)
  (if list
      ;(append list (list element))
      (nconc list (list element))
    ;; list is nil
    (list element)))

(defsubst elmo-msgdb-get-overview (msgdb)
  (car msgdb))
(defsubst elmo-msgdb-get-number-alist (msgdb)
  (cadr msgdb))
(defsubst elmo-msgdb-get-mark-alist (msgdb)
  (caddr msgdb))
(defsubst elmo-msgdb-get-location (msgdb)
  (cadddr msgdb))

;;
;; number <-> Message-ID handling
;;
(defsubst elmo-msgdb-number-add (alist number id)
  (let ((ret-val alist))
    (setq ret-val 
	  (elmo-msgdb-append-element ret-val (cons number id)))
    ret-val))

;;;
;;; parsistent mark handling
;;; (for global!)

(defvar elmo-msgdb-global-mark-alist nil)

(defun elmo-msgdb-global-mark-delete (msgid)
  (let* ((path (expand-file-name
		elmo-msgdb-global-mark-filename
		elmo-msgdb-dir))
	 (malist (or elmo-msgdb-global-mark-alist 
		     (setq elmo-msgdb-global-mark-alist 
			   (elmo-object-load path))))
	 match)
    (when (setq match (assoc msgid malist))
      (setq elmo-msgdb-global-mark-alist
	    (delete match elmo-msgdb-global-mark-alist))
      (elmo-object-save path elmo-msgdb-global-mark-alist))))

(defun elmo-msgdb-global-mark-set (msgid mark)
  (let* ((path (expand-file-name
		elmo-msgdb-global-mark-filename
		elmo-msgdb-dir))
	 (malist (or elmo-msgdb-global-mark-alist 
		     (setq elmo-msgdb-global-mark-alist 
			   (elmo-object-load path))))
	 match)
    (if (setq match (assoc msgid malist))
	(setcdr match mark)
      (setq elmo-msgdb-global-mark-alist
	    (append elmo-msgdb-global-mark-alist
		    (list (cons msgid mark)))))
    (elmo-object-save path elmo-msgdb-global-mark-alist)))

(defun elmo-msgdb-global-mark-get (msgid)
  (cdr (assoc msgid (or elmo-msgdb-global-mark-alist 
			(setq elmo-msgdb-global-mark-alist 
			      (elmo-object-load 
			       (expand-file-name
				elmo-msgdb-global-mark-filename
				elmo-msgdb-dir)))))))

;;
;; number <-> location handling
;;
(defsubst elmo-msgdb-location-load (dir)
  (elmo-object-load
   (expand-file-name
    elmo-msgdb-location-filename
    dir)))
			  
(defsubst elmo-msgdb-location-add (alist number location)
  (let ((ret-val alist))
    (setq ret-val 
	  (elmo-msgdb-append-element ret-val (cons number location)))
    ret-val))

(defsubst elmo-msgdb-location-save (dir alist)
  (elmo-object-save 
   (expand-file-name
    elmo-msgdb-location-filename
    dir) alist))

(defun elmo-list-folder-by-location (spec location &optional msgdb)
  (let* ((path (elmo-msgdb-expand-path nil spec))
	 (location-alist 
	  (if msgdb
	      (elmo-msgdb-get-location msgdb)
	    (elmo-msgdb-location-load path)))
	 (i 0)
	 result pair
	 location-max modified)
    (setq location-max
	  (or (elmo-max-of-list (mapcar 'car location-alist)) 0))
    (when location-max
      (while location
	(setq i (1+ i))
	(if (setq pair (rassoc (car location) location-alist))
	    (setq result
		  (append result
			  (list (cons (car pair) (car location)))))
	  (setq result (append result
			       (list 
				(cons (+ location-max i) (car location))))))
	(setq location (cdr location))))
    (setq result (sort result (lambda (x y)
				(< (car x)(car y)))))
    (if (not (equal result location-alist))
	(setq modified t))
    (if modified
	(elmo-msgdb-location-save path result))
    (mapcar 'car result)))

;;;
;;; persistent mark handling
;;; (for each folder)
(defun elmo-msgdb-mark-set (alist id mark)
  (let ((ret-val alist)
	entity)
    (setq entity (assq id alist))
    (if entity
	(if (eq mark nil)
	    ;; delete this entity
	    (setq ret-val (delq entity alist))
	  ;; set mark
	  (setcar (cdr entity) mark))
      (if mark 
	  (setq ret-val (elmo-msgdb-append-element ret-val
						   (list id mark)))))
    ret-val))

(defun elmo-msgdb-mark-append (alist id mark)
  "Append mark"
  (setq alist (elmo-msgdb-append-element alist
					 (list id mark))))

(defun elmo-msgdb-mark-alist-to-seen-list (number-alist mark-alist seen-marks)
  "Make seen-list from mark-alist"
  (let ((seen-mark-list (string-to-char-list seen-marks))
	ret-val ent)
    (while number-alist
      (if (setq ent (assq (car (car number-alist)) mark-alist))
	  (if (and (cadr ent)
		   (memq (string-to-char (cadr ent)) seen-mark-list))
	      (setq ret-val (cons (cdr (car number-alist)) ret-val)))
	(setq ret-val (cons (cdr (car number-alist)) ret-val)))
      (setq number-alist (cdr number-alist)))
    ret-val))

;;
;; overview handling
;;
(if (fboundp 'std11-fetch-field)
    (defalias 'elmo-field-body 'std11-fetch-field) ;;no narrow-to-region
  (defalias 'elmo-field-body 'std11-field-body))

(defsubst elmo-msgdb-get-field-value (field-name beg end buffer)
  (save-excursion
    (save-restriction
      (set-buffer buffer)
      (narrow-to-region beg end)
      (elmo-field-body field-name))))

(defsubst elmo-msgdb-get-last-message-id (string)
  (if string
      (save-match-data
	(let (beg)
	  (elmo-set-work-buf
	   (insert string)
	   (goto-char (point-max))
	   (when (search-backward "<" nil t)
	     (setq beg (point))
	     (if (search-forward ">" nil t)
		 (buffer-substring beg (point)))))))))

(defun elmo-msgdb-number-load (dir)
  (elmo-object-load 
   (expand-file-name elmo-msgdb-number-filename dir)))

(defun elmo-msgdb-overview-load (dir)
  (elmo-object-load 
   (expand-file-name elmo-msgdb-overview-filename dir)))

(defun elmo-msgdb-mark-load (dir)
  (elmo-object-load 
   (expand-file-name elmo-msgdb-mark-filename dir)))

(defsubst elmo-msgdb-seen-load (dir)
  (elmo-object-load (expand-file-name
		     elmo-msgdb-seen-filename
		     dir)))

(defun elmo-msgdb-number-save (dir obj)
  (elmo-object-save 
   (expand-file-name elmo-msgdb-number-filename dir)
   obj))

(defun elmo-msgdb-mark-save (dir obj)
  (elmo-object-save 
   (expand-file-name elmo-msgdb-mark-filename dir)
   obj))

(defsubst elmo-msgdb-seen-save (dir obj)
  (elmo-object-save 
   (expand-file-name elmo-msgdb-seen-filename dir)
   obj))

(defun elmo-msgdb-overview-save (dir database)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *Elmo Overview TMP*"))
	  (ret-val t)
	  write-file curp)
      (set-buffer tmp-buffer)
      (erase-buffer)
      (setq curp database)
      (prin1 database tmp-buffer)
      (princ "\n" tmp-buffer)
      (setq write-file (expand-file-name elmo-msgdb-overview-filename
					 dir))
      (if (file-directory-p dir)
	  (); ok.
	(if (file-exists-p dir)
	    (error "File %s already exists." dir)
	  (elmo-make-directory dir)))
      (if (file-writable-p write-file)
	  (as-binary-output-file (write-region (point-min) (point-max) 
					       write-file nil 'no-msg))
	(message (format "%s is not writable." write-file))
	(setq ret-val nil))
      (kill-buffer tmp-buffer)
      ret-val)))

(defun elmo-msgdb-delete-msgs (folder msgs msgdb &optional reserve-cache)
  "Delete MSGS from FOLDER in MSGDB. 
content of MSGDB is changed."
  (save-excursion
    (let* ((msg-list msgs)
	   (dir (elmo-msgdb-expand-path folder))
	   (overview (or (car msgdb)
			 (elmo-msgdb-overview-load dir)))
	   (number-alist (or (cadr msgdb)
			     (elmo-msgdb-number-load dir)))
	   (mark-alist (or (caddr msgdb)
			   (elmo-msgdb-mark-load dir)))
	   message-id)
      ;; remove from current database.
      (while msg-list
	(setq message-id (cdr (assq (car msg-list) number-alist)))
	(if (and (not reserve-cache) message-id) 
	    (elmo-cache-delete message-id
			       folder (car msg-list)))
	;; This is no good!!!!
	;(setq overview (delete (assoc message-id overview) overview))
	(setq overview 
	      (delq
	       (elmo-msgdb-overview-get-entity-by-number overview 
							 (car msg-list))
	       overview))
	(setq number-alist
	      (delq (assq (car msg-list) number-alist) number-alist))
	(setq mark-alist (delq (assq (car msg-list) mark-alist) mark-alist))
	(setq msg-list (cdr msg-list)))
      (setcar msgdb overview)
      (setcar (cdr msgdb) number-alist)
      (setcar (cddr msgdb) mark-alist))
    t)) ;return value

(defsubst elmo-msgdb-set-overview (msgdb overview)
  (setcar msgdb overview))

(defsubst elmo-msgdb-set-number-alist (msgdb number-alist)
  (setcar (cdr msgdb) number-alist))

(defsubst elmo-msgdb-set-mark-alist (msgdb mark-alist)
  (setcar (cddr msgdb) mark-alist))

(defsubst elmo-msgdb-overview-entity-get-references (entity)
  (and entity (aref (cdr entity) 1)))

;; id -> parent-entity
(defsubst elmo-msgdb-overview-get-parent-entity (id database)
  (let (reference)
    (if (setq reference (elmo-msgdb-overview-entity-get-references 
			 (assoc id database)))
	(assoc reference database))))
  
(defsubst elmo-msgdb-overview-entity-get-number (entity)
  (and entity (aref (cdr entity) 0)))

(defsubst elmo-msgdb-overview-entity-get-from (entity)
  (and entity 
       (let ((from (aref (cdr entity) 2)))
	 (and from
	      (decode-mime-charset-string from elmo-mime-charset)))))

(defsubst elmo-msgdb-overview-entity-set-number (entity number)
  (and entity (aset (cdr entity) 0 number))
  entity)
  ;(setcar (cadr entity) number) entity)

(defsubst elmo-msgdb-overview-entity-set-from (entity from)
  (and entity (aset (cdr entity) 2 from))
  entity)
  ;(setcar (car (cddadr entity)) from))

(defsubst elmo-msgdb-overview-entity-get-subject (entity)
  (and entity
       (let ((sub (aref (cdr entity) 3)))
	 (and sub
	      (decode-mime-charset-string sub elmo-mime-charset)))))

(defsubst elmo-msgdb-overview-entity-set-subject (entity subject)
  (and entity (aset (cdr entity) 3 subject))
  entity)

(defsubst elmo-msgdb-overview-entity-get-date (entity)
  (and entity (aref (cdr entity) 4)))

(defsubst elmo-msgdb-overview-entity-get-to (entity)
  (and entity (aref (cdr entity) 5)))

(defsubst elmo-msgdb-overview-entity-get-cc (entity)
  (and entity (aref (cdr entity) 6)))

(defsubst elmo-msgdb-overview-entity-get-size (entity)
  (and entity (aref (cdr entity) 7)))

(defsubst elmo-msgdb-overview-entity-get-id (entity)
  (and entity (car entity)))

(defsubst elmo-msgdb-overview-entity-get-extra-field (entity field-name)
  (let ((extra (and entity (aref (cdr entity) 8))))
    (and extra
	 (cdr (assoc field-name extra)))))

(defun elmo-msgdb-overview-get-entity-by-number (database number)
  (let ((db database)
	entity)
    (catch 'loop
      (while db
	(if (eq (elmo-msgdb-overview-entity-get-number (car db)) number)
	    (progn
	      (setq entity (car db))
	      (throw 'loop nil))
	  (setq db (cdr db)))))
    entity))

;;
;; deleted message handling
;;
(defun elmo-msgdb-killed-list-load (dir)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *ELMO Killed TMP*"))
	  (killed-list-file (expand-file-name elmo-msgdb-killed-filename dir))
	  insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
	  insert-file-contents-post-hook 
	  ret-val)
      (if (not (file-readable-p killed-list-file))
	  ()
	(set-buffer tmp-buffer)
	(insert-file-contents killed-list-file)
	(setq ret-val
	      (condition-case nil
		  (read (current-buffer)) 
		(error nil nil))))
      (kill-buffer tmp-buffer)
      ret-val)))
;;
;; currently not used...
;;
(defun elmo-msgdb-killed-list-save (dir killed-list)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *Elmo Killed TMP*"))
	  (ret-val t)
	  write-file)
      (set-buffer tmp-buffer)
      (erase-buffer)
      (prin1 killed-list tmp-buffer)
      (princ "\n" tmp-buffer)
      (setq write-file (expand-file-name elmo-msgdb-killed-filename
					 dir))
      (if (file-writable-p write-file)
	  (write-region (point-min) (point-max) 
			write-file nil 'no-msg)
	(message (format "%s is not writable." write-file))
	(setq ret-val nil))
      (kill-buffer tmp-buffer)
      ret-val
      )))

(defun elmo-msgdb-killed-message-p (killed-list msg)
  (and killed-list
       (not (listp
	     (catch 'found
	       (mapcar 
		(function 
		 (lambda (entity)
		   (cond 
		    ((integerp entity)
		     (if (eq entity msg)
			 (throw 'found t)))
		    ((consp entity)
		     (if (and (<= (car entity) msg)
			      (<= msg (cdr entity)))
			 (throw 'found t)))))
		 killed-list)))))))

(defun elmo-msgdb-set-as-killed (killed-list msg)
  "if cons cell, set car-cdr messages as killed.
if integer, set number th message as killed."
  (let ((dlist killed-list)
	(ret-val killed-list)
	entity found)
    (cond
     ((integerp msg)
      (while (and dlist (not found))
	(setq entity (car dlist))
	(if (or (and (integerp entity) (eq entity msg))
		(and (consp entity) 
		     (<= (car entity) msg)
		     (<= msg (cdr entity))))
	    (setq found t))
	(setq dlist (cdr dlist))
	)
      (if (not found)
	  (setq ret-val (elmo-msgdb-append-element killed-list msg)))
      )
     ((consp msg)
      (while (and dlist (not found))
	(setq entity (car dlist))
	(if (integerp entity)
	    (cond 
	     ((and (<= (car msg) entity)(<= entity (cdr msg)))
	      (setcar dlist msg)
	      (setq found t)
	      )
	     ((= (1- (car msg)) entity)
	      (setcar dlist (cons entity (cdr msg)))
	      (setq found t)
	      )
	     ((= (1+ (cdr msg)) entity)
	      (setcar dlist (cons (car msg) entity))
	      (setq found t)
	      ))
	  ;; entity is consp
	  (cond  ; there are four patterns
	   ((and (<= (car msg) (car entity))
		 (<= (cdr entity) (cdr msg)))
	    (setcar dlist msg)
	    (setq found t))
	   ((and (< (car entity)(car msg))
		 (< (cdr msg) (cdr entity)))
	    (setq found t))
	   ((and (<= (car msg) (car entity))
		 (<= (cdr msg) (cdr entity)))
	    (setcar dlist (cons (car msg) (cdr entity)))
	    (setq found t))
	   ((and (<= (car entity) (car msg))
		 (<= (cdr entity) (cdr msg)))
	    (setcar dlist (cons (car entity) (cdr msg)))
	    (setq found t))))
	(setq dlist (cdr dlist)))
      (if (not found)
	  (setq ret-val (elmo-msgdb-append-element killed-list msg)))))
    ret-val))

(defun elmo-msgdb-finfo-load (&optional filename)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *ELMO Finfo TMP*"))
	  (finfo-file (expand-file-name 
		       (or filename elmo-msgdb-finfo-filename)
		       elmo-msgdb-dir))
	  insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
	  insert-file-contents-post-hook 
	  ret-val)
      (if (not (file-readable-p finfo-file))
	  ()
	(set-buffer tmp-buffer)
	(insert-file-contents finfo-file)
	(setq ret-val
	      (condition-case nil
		  (read (current-buffer)) 
		(error nil nil))))
      (kill-buffer tmp-buffer)
      ret-val)))

(defun elmo-msgdb-finfo-save (finfo &optional filename)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *ELMO Finfo TMP*"))
	  (finfo-file (expand-file-name
		       (or filename elmo-msgdb-finfo-filename )
		       elmo-msgdb-dir)))
      (set-buffer tmp-buffer)
      (erase-buffer)
      (prin1 finfo tmp-buffer)
      (princ "\n" tmp-buffer)
      (if (file-directory-p elmo-msgdb-dir)
	  (); ok.
	(if (file-exists-p elmo-msgdb-dir)
	    (error "File %s already exists." elmo-msgdb-dir)
	  (elmo-make-directory elmo-msgdb-dir)))
      (if (file-writable-p finfo-file)
	  (write-region (point-min) (point-max) 
			finfo-file nil 'no-msg)
	(message (format "%s is not writable." finfo-file)))
      (kill-buffer tmp-buffer))))

(defun elmo-msgdb-flist-load (folder)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *ELMO Flist TMP*"))
	  (flist-file (expand-file-name 
		       elmo-msgdb-flist-filename 
		       (elmo-msgdb-expand-path folder (list 'folder folder))
		       ))
	  insert-file-contents-pre-hook   ; To avoid autoconv-xmas...
	  insert-file-contents-post-hook
	  ret-val)
      (if (not (file-readable-p flist-file))
	  ()
	(set-buffer tmp-buffer)
	(insert-file-contents flist-file)
	(setq ret-val
	      (condition-case nil
		  (read (current-buffer)) 
		(error nil nil))))
      (kill-buffer tmp-buffer)
      ret-val)))

(defsubst elmo-msgdb-create-overview-from-buffer (number &optional size)
  "Create overview entity from current buffer. 
Header region is supposed to be narrowed."
  (save-excursion
    (let ((extras elmo-msgdb-extra-fields)
	  message-id references from subject to cc date
	  extra field-body)
      (elmo-set-buffer-multibyte default-enable-multibyte-characters)
      (setq message-id (elmo-field-body "message-id"))
      (setq references
	    (or (elmo-msgdb-get-last-message-id
		 (elmo-field-body "in-reply-to"))
		(elmo-msgdb-get-last-message-id
		 (elmo-field-body "references"))))
      (setq from (encode-mime-charset-string
		  (elmo-eword-decode-string
		   (decode-mime-charset-string
		    (elmo-delete-char
		     ?\"
		     (or 
		      (elmo-field-body "from")
		      elmo-no-from))
		    elmo-mime-charset))
		  elmo-mime-charset))
      (setq subject (encode-mime-charset-string
		     (elmo-eword-decode-string
		      (decode-mime-charset-string
		       (or (elmo-field-body "subject")
			   elmo-no-subject)
		       elmo-mime-charset))
		     elmo-mime-charset))
      (setq date (elmo-field-body "date"))
      (setq to   (elmo-field-body "to"))
      (setq cc   (elmo-field-body "cc"))
      (or size
	  (if (setq size (elmo-field-body "content-length"))
	      (setq size (string-to-int size))
	    (setq size 0)));; No mean...
      (while extras
	(if (setq field-body (elmo-field-body (car extras)))
	    (setq extra (cons (cons (downcase (car extras))
				    field-body) extra)))
	(setq extras (cdr extras)))
      (cons message-id (vector number references 
			       from subject date to cc
			       size extra))
      )))
  
(defun elmo-msgdb-flist-save (folder flist)
  (save-excursion
    (let* ((tmp-buffer (get-buffer-create " *ELMO Flist TMP*"))
	   (dir (elmo-msgdb-expand-path folder
					(list 'folder folder)))
	   (flist-file (expand-file-name elmo-msgdb-flist-filename 
					 dir)))
      (set-buffer tmp-buffer)
      (erase-buffer)
      (prin1 flist tmp-buffer)
      (princ "\n" tmp-buffer)
      (if (file-directory-p dir)
	  (); ok.
	(if (file-exists-p dir)
	    (error "File %s already exists." dir)
	  (elmo-make-directory dir)))
      (if (file-writable-p flist-file)
	  (write-region (point-min) (point-max) 
			flist-file nil 'no-msg)
	(message (format "%s is not writable." flist-file)))
      (kill-buffer tmp-buffer))))

(defun elmo-msgdb-overview-sort-by-date (overview)
  (sort overview
	(function
	 (lambda (x y) 
	   (condition-case nil
	       (string<
		(timezone-make-date-sortable 
		 (elmo-msgdb-overview-entity-get-date x))
		(timezone-make-date-sortable 
		 (elmo-msgdb-overview-entity-get-date y)))
	     (error))))))

(defun elmo-msgdb-sort-by-date (msgdb)
  (message "sorting...")
  (let ((overview (elmo-msgdb-get-overview msgdb)))
    (setq overview (elmo-msgdb-overview-sort-by-date overview))
    (message "sorting...done.")
    (list overview (nth 1 msgdb)(nth 2 msgdb))))

(defsubst elmo-msgdb-search-overview-entity (number number-alist overview)
  (let ((message-id (cdr (assq number number-alist)))
	ovs)
    (if message-id
	(assoc message-id overview)
      (elmo-msgdb-overview-get-entity-by-number overview number))))

(defun elmo-msgdb-upgrade-folder (folder &optional dir)
  "A subroutine for msgdb-upgrade."
  (let* ((path (or dir (elmo-msgdb-expand-path folder)))
	 (number-alist (elmo-msgdb-number-load path))
	 (overview (elmo-msgdb-overview-load path))
	 (mark-alist (elmo-msgdb-mark-load path)))
    (when (and number-alist
	       overview)
      (setq number-alist
	    (mapcar
	     (lambda (x)
	       (if (stringp (car x))
		   (cons (string-to-int (car x))
			 (cdr x))
		 (cons (car x) (cdr x))))
	     number-alist))
      (setq overview
	    (mapcar (lambda (x)
		      (if (vectorp (cdr x)) ; already 0.8
			  x
			(let ((env (nth 1 x)))
			  (if (stringp (car env))
			      ;; not converted yet.
			      (cons (car x)
				    (vector
				     (string-to-int (car env)) ;number 0
				     (nth 1 env)         ;reference 1 
				     (car (nth 2 env))   ;from 2
				     (nth 1 (nth 2 env)) ;subject 3
				     (nth 2 (nth 2 env)) ;date 4
				     (nth 3 (nth 2 env)) ;to 5
				     (nth 4 (nth 2 env)) ;cc 6
				     (string-to-int (or 
						     (nth 5 (nth 2 env)) 
						     "0"));size 7
				     nil))
			    ;; already integered
			    (cons (car x)
				  (vector
				   (car env) ;number 0
				   (nth 1 env) ;reference 1
				   (car (nth 2 env)) ; from 2
				   (nth 1 (nth 2 env)) ;subject 3
				   (nth 2 (nth 2 env)) ;date 4
				   (nth 3 (nth 2 env)) ;to 5
				   (nth 4 (nth 2 env)) ;cc 6
				   (nth 5 (nth 2 env)) ;size 7
				   nil)))))) 
		    overview))
      (setq mark-alist
	    (mapcar
	     (lambda (x)
	       (if (stringp (car x))
		   (list (string-to-int (car x)) (nth 1 x))
		 (list (car x)(nth 1 x))))
	     mark-alist))
      (elmo-msgdb-overview-save path overview)
      (elmo-msgdb-number-save path number-alist)
      (elmo-msgdb-mark-save path mark-alist))))

(defun elmo-msgdb-append (msgdb msgdb-append)
  (list 
   (append (car msgdb) (car msgdb-append))
   (append (cadr msgdb) (cadr msgdb-append))
   (append (caddr msgdb) (caddr msgdb-append))
   (append (cadddr msgdb) (cadddr msgdb-append))))

(provide 'elmo-msgdb)
