;;;
;;; Wanderlust -- Yet Another Message Interface on Emacsen.
;;;
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; Time-stamp: <99/06/28 16:53:13 teranisi>

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'sendmail)
(require 'wl-util)
(require 'wl-template)
(require 'wl-vars)
(require 'emu)
(if (module-installed-p 'timezone)
    (require 'timezone))
(require 'std11)

(eval-when-compile
  (require 'smtp)
  (mapcar
   (function
    (lambda (symbol)
      (unless (boundp symbol)
	(make-local-variable symbol)
	(eval (list 'setq symbol nil)))))
   '(x-face-add-x-face-version-header 
     mail-reply-buffer
     mail-from-style))
  (or (fboundp 'x-face-insert)
      (defun x-face-insert (a)))
  (or (fboundp 'x-face-insert-version-header)
      (defun x-face-insert-version-header ())))

(defvar wl-draft-buf-name "Draft")
(defvar wl-caesar-region-func nil)
(defvar wl-draft-cite-func 'wl-default-draft-cite)
(defvar wl-draft-buffer-file-name nil)
(defvar wl-draft-field-completion-list nil)
(defvar wl-draft-verbose-send t)
(defvar wl-draft-queue-flushing nil)
(defvar wl-draft-config-variables nil)
(defvar wl-draft-config-exec-flag t)
(defvar wl-draft-buffer-cur-summary-buffer nil)
(defvar wl-draft-clone-local-variable-regexp "^mime")

(defvar wl-draft-config-sub-func-alist
  '((body        . wl-draft-config-sub-body)
    (top         . wl-draft-config-sub-top)
    (bottom      . wl-draft-config-sub-bottom)
    (body-file   . wl-draft-config-sub-body-file)
    (top-file    . wl-draft-config-sub-top-file)
    (bottom-file . wl-draft-config-sub-bottom-file)
    (template    . wl-draft-config-sub-template)
    (x-face      . wl-draft-config-sub-x-face)))

(make-variable-buffer-local 'wl-draft-buffer-file-name)
(make-variable-buffer-local 'wl-draft-buffer-cur-summary-buffer)
(make-variable-buffer-local 'wl-draft-config-variables)
(make-variable-buffer-local 'wl-draft-config-exec-flag)

(defun wl-draft-insert-date-field ()
  (insert "Date: " (wl-make-date-string) "\n"))

(defun wl-draft-insert-x-face-field ()
  "Insert x-face header."
  (interactive)
  (if (not (file-exists-p wl-x-face-file))
      (error "File %s does not exist." wl-x-face-file)
    (beginning-of-buffer)
    (search-forward mail-header-separator nil t)
    (beginning-of-line)
    (wl-draft-insert-x-face-field-here)
    (run-hooks 'wl-draft-insert-x-face-field-hook) ; highlight it if you want.
    ))

(defun wl-draft-insert-x-face-field-here ()
  "insert x-face field at point."
  (insert "X-Face: " 
	  (elmo-get-file-string wl-x-face-file))
  (if (not (= (preceding-char) ?\n))
      (insert ?\n))
  (and (fboundp 'x-face-insert-version-header) ; x-face.el...
       (boundp 'x-face-add-x-face-version-header)
       x-face-add-x-face-version-header
       (x-face-insert-version-header)))

(defun wl-draft-setup ()
  (let ((field wl-draft-fields)
	ret-val)
    (while field
      (setq ret-val (append ret-val
			    (list (cons (concat (car field) " ")
					(concat (car field) " ")))))
      (setq field (cdr field)))
    (setq wl-draft-field-completion-list ret-val)))

(defun wl-draft-make-mail-followup-to (recipients)
  (let ((rlist (delete (wl-address-header-extract-address wl-from)
		       (copy-sequence recipients)))
	(mlists (mapcar 'downcase wl-subscribed-mailing-list)))
    (if (catch 'loop
	  (while rlist
	    (if (member (downcase (car rlist)) mlists)
		(throw 'loop t))
	    (setq rlist (cdr rlist))))
	rlist
      (append recipients (list (wl-address-header-extract-address 
				wl-from))))))

(defun wl-draft-delete-myself-from-cc (to cc)
  (let ((myself (wl-address-header-extract-address wl-from))
	(rlist (append to cc))
	(mlists (mapcar 'downcase wl-subscribed-mailing-list)))
    (if wl-draft-always-delete-myself
	(delete myself cc)
      (if (member myself cc)
	  (if (catch 'loop
		(while rlist
		  (if (member (downcase (car rlist)) mlists)
		      (throw 'loop t))
		  (setq rlist (cdr rlist)))) 
	      ;; member list is contained in recipients.
	      (delete myself cc)
	    cc
	    )
	cc))))

(defun wl-draft-forward (original-subject summary-buf)
  (wl-draft "" (concat "Forward: " original-subject) 
	    nil nil nil nil nil nil nil nil summary-buf)
  (goto-char (point-max))
  (funcall wl-insert-mail-func)
  (mail-position-on-field "To"))

(defun wl-draft-reply (buf no-arg)
;  (save-excursion 
  (set-buffer buf)
  (let ((r-list (if no-arg wl-draft-reply-without-argument-list
		  wl-draft-reply-with-argument-list))
	to mail-followup-to cc subject in-reply-to references newsgroups
	(summary-buf (save-excursion
		       (set-buffer (get-buffer wl-message-buf-name))
		       wl-message-buffer-cur-summary-buffer))
	)
    (if (string= (wl-address-header-extract-address wl-from)
		 (wl-address-header-extract-address (std11-field-body "From")))
	(setq to (std11-field-body "To")
	      cc (std11-field-body "Cc")
	      newsgroups (or (std11-field-body "Newsgroups") ""))
    (catch 'done
      (while r-list
	(when (let ((condition (car (car r-list))))
		(cond ((stringp condition)
		       (std11-field-body condition))
		      ((listp condition)
		       (catch 'done
			 (while condition
			   (if (not (std11-field-body (car condition)))
			       (throw 'done nil))
			   (setq condition (cdr condition)))
			 t))
		      ((symbolp condition)
		       (funcall condition))))
	  (setq to (delq nil (cons to (std11-field-bodies 
				       (nth 0 (cdr (car r-list))))))
		to (mapconcat 'identity to ","))
	  (setq cc (delq nil (cons cc (std11-field-bodies 
				       (nth 1 (cdr (car r-list))))))
		cc (mapconcat 'identity cc ","))
	  (setq newsgroups (delq nil (cons newsgroups 
					   (std11-field-bodies 
					    (nth 2 (cdr (car r-list))))))
		newsgroups (mapconcat 'identity newsgroups ","))
	  (throw 'done nil))
	(setq r-list (cdr r-list)))
      (error "No match field. Check your `wl-draft-reply-without-argument-list'.")))
    (setq subject (std11-field-body "Subject"))
    (elmo-set-buffer-multibyte default-enable-multibyte-characters)    
    (setq subject (or (and subject
			   (elmo-eword-decode-string
			    (decode-mime-charset-string
			     subject
			     wl-mime-charset)))))
    (and subject wl-reply-subject-prefix
	 (let ((case-fold-search t))
	   (not
	    (equal
	     (string-match (regexp-quote wl-reply-subject-prefix)
			   subject)
	     0)))
	 (setq subject (concat wl-reply-subject-prefix subject)))
    (and (setq in-reply-to (std11-field-body "Message-Id"))
	 (setq in-reply-to
	       (format "In your message of \"%s\"\n\t%s"
		       (or (std11-field-body "Date") "some time ago")
		       in-reply-to)))
    (setq references (nconc
		      (std11-field-bodies '("References" "In-Reply-To"))
		      (list in-reply-to)))
    (setq to (wl-parse-addresses to)
	  cc (wl-parse-addresses cc)
	  )
    (setq to (mapcar '(lambda (addr)
			(wl-address-header-extract-address
			 addr)) to))
    (setq cc (mapcar '(lambda (addr)
			(wl-address-header-extract-address
			 addr)) cc))
    ;; if subscribed mailing list is contained in cc or to
    ;; and myself is contained in cc,
    ;; delete myself from cc.
    (setq cc (wl-draft-delete-myself-from-cc to cc))
    (if wl-insert-mail-followup-to
	(progn
	  (setq mail-followup-to 
		(wl-draft-make-mail-followup-to (append to cc)))
	  (setq mail-followup-to (wl-delete-duplicates mail-followup-to 
						       nil t))))
    (setq newsgroups (wl-parse newsgroups 
			       "[ \t\f\r\n,]*\\([^ \t\f\r\n,]+\\)")
	  newsgroups (wl-delete-duplicates newsgroups)
	  newsgroups (if newsgroups (mapconcat 'identity newsgroups ",")))
    (setq to (wl-delete-duplicates to nil t))
    (setq cc (wl-delete-duplicates 
	      (append (wl-delete-duplicates cc nil t)
		      to (copy-sequence to))
	      t t))
    (and to (setq to (mapconcat 'identity to ",\n\t")))
    (and cc (setq cc (mapconcat 'identity cc ",\n\t")))
    (and mail-followup-to (setq mail-followup-to 
				(mapconcat 'identity 
					   mail-followup-to ",\n\t")))
    (and (null to) (setq to cc cc nil))
    (setq references (delq nil references)
	  references (mapconcat 'identity references " ")
	  references (wl-parse references "[^<]*\\(<[^>]+>\\)")
	  references (wl-delete-duplicates references)
	  references (if references 
			 (mapconcat 'identity references "\n\t")))
    (let (wl-draft-prepared-config-alist)
      (wl-draft
       to subject in-reply-to cc references newsgroups mail-followup-to
       nil nil nil summary-buf
       ))
    (if wl-draft-prepared-config-alist
	(wl-draft-prepared-config-exec buf))
    (setq mail-reply-buffer buf))
  (run-hooks 'wl-reply-hook))

(defun wl-yank-content (decode-it)
  (interactive)
  (save-restriction
    (narrow-to-region (point)(point))
    (insert 
     (save-excursion
       (set-buffer mail-reply-buffer)
       (if decode-it
	   (decode-mime-charset-region (point-min) (point-max)
				       wl-mime-charset))       
       (buffer-substring-no-properties 
	(point-min) (point-max))))
    (goto-char (point-max))
    (push-mark)
    (goto-char (point-min)))
  (let ((beg (point)))
    (cond (mail-citation-hook (run-hooks 'mail-citation-hook))
	  (mail-yank-hooks (run-hooks 'mail-yank-hooks))
	  (t (and wl-draft-cite-func
		  (funcall wl-draft-cite-func)))) ; default cite
    (run-hooks 'wl-draft-cited-hook)
    (if wl-highlight-body-too
	(wl-highlight-body-region beg (point-max))))
  )

(defun wl-draft-confirm ()
  (interactive)
  (y-or-n-p (format "Send current draft as %s? "
		    (if (wl-message-mail-p)
			(if (wl-message-news-p) "Mail and News" "Mail")
		      "News"))))

(defun wl-message-news-p ()
  (std11-field-body "Newsgroups")
  )

(defun wl-message-field-exists-p (field)
  (let ((value (std11-field-body field)))
    (and value
	 (not (string= value "")))))

(defun wl-message-mail-p ()
  (or (wl-message-field-exists-p "To")
      (wl-message-field-exists-p "Cc")
      (wl-message-field-exists-p "Bcc")
      (wl-message-field-exists-p "Fcc")		; This may be needed..
      ))

(defun wl-draft-open-file (&optional file)
  (interactive)				; "*fFile to edit: ")
  (wl-draft-edit-string (elmo-get-file-string 
			 (or file
			     (read-file-name "File to edit: " 
					     (or wl-tmp-dir "~/"))))))

(defun wl-draft-edit-string (string)
  (let ((cur-buf (current-buffer))
	(tmp-buf (get-buffer-create " *wl-draft-edit-string*"))
	to subject in-reply-to cc references newsgroups mail-followup-to
	content-type 
	body-beg buffer-read-only
	)
    (set-buffer tmp-buf)
    (erase-buffer)
    (insert string)
    (setq to (std11-field-body "To"))
    (setq to (and to
		  (elmo-eword-decode-string
		   (decode-mime-charset-string
		    to
		    wl-mime-charset))))
    (setq subject (std11-field-body "Subject"))
    (setq subject (and subject
		       (elmo-eword-decode-string
			(decode-mime-charset-string
			 subject
			 wl-mime-charset))))
    (setq in-reply-to (std11-field-body "In-Reply-To"))
    (setq cc (std11-field-body "Cc"))
    (setq cc (and cc
		  (elmo-eword-decode-string
		   (decode-mime-charset-string
		    cc
		    wl-mime-charset))))
    (setq references (std11-field-body "References"))
    (setq newsgroups (std11-field-body "Newsgroups"))
    (setq mail-followup-to (std11-field-body "Mail-Followup-To"))
    (setq content-type  (std11-field-body "Content-Type"))
    (goto-char (point-min))
    (or (re-search-forward "\n\n" nil t)
	(search-forward (concat mail-header-separator "\n") nil t))
    (unwind-protect
	(set-buffer
	 (wl-draft to subject in-reply-to cc references newsgroups 
		   mail-followup-to
		   content-type
		   (buffer-substring (point) (point-max))
		   'edit-again
		   ))
      (and to (mail-position-on-field "To"))
      (delete-other-windows)
      (kill-buffer tmp-buf))
    )
  (setq buffer-read-only nil) ;;??
  (run-hooks 'wl-mail-setup-hook)
  )

(defun wl-draft-insert-message (dummy)
  (interactive)
  (let ((mail-reply-buffer (wl-message-get-original-buffer))
	mail-citation-hook mail-yank-hooks
	wl-draft-cite-func)
    (wl-yank-content nil)))

;;
;; default body citation func
;;
(defun wl-default-draft-cite ()
  (let ((mail-yank-ignored-headers "[^:]+:")
	(mail-yank-prefix "> ")
	(summary-buf wl-draft-buffer-cur-summary-buffer)
	(message-buf (get-buffer wl-message-buf-name))
	from date cite-title num entity
	)
    (if (and summary-buf
	     message-buf)
	(progn
	  (save-excursion 
	    (set-buffer summary-buf)
	    (setq num 
		  (save-excursion
		    (set-buffer message-buf)
		    wl-message-buffer-cur-number))
	    (setq entity (assoc (cdr (assq num 
					   (elmo-msgdb-get-number-alist 
					    wl-summary-buffer-msgdb)))
				(elmo-msgdb-get-overview 
				 wl-summary-buffer-msgdb)))
	    (setq from (elmo-msgdb-overview-entity-get-from entity))
	    (setq date (elmo-msgdb-overview-entity-get-date entity)))
	  (setq cite-title (format "At %s,\n%s wrote:" 
				   (or date "some time ago")
				   (wl-summary-from-func-internal
					    (or from "you"))))))
    (and cite-title 
	 (insert cite-title "\n"))
    (mail-indent-citation))
  )

(defun wl-caesar-region (beg end)
  (interactive "r")
  (funcall wl-caesar-region-func beg end)
  )

(defun wl-draft-yank-original ()
  (interactive)
  (funcall wl-yank-original-func))

(defun wl-draft-hide (editing-buffer)
  "Hide the editing draft buffer if possible."
  (interactive)
  (when editing-buffer
    (set-buffer editing-buffer)
    (let ((sum-buf wl-draft-buffer-cur-summary-buffer)
	  next-buf next-win)
      ;; hide draft window
      (or (one-window-p)
	  (delete-window))
      ;; stay folder window if required
      (when wl-stay-folder-window
	(if (setq next-buf (get-buffer wl-folder-buffer-name))
	    (if (setq next-win (get-buffer-window next-buf))
		(select-window next-win)
	      (switch-to-buffer next-buf))))
      (if (buffer-live-p sum-buf)
	  (if (setq next-win (get-buffer-window sum-buf))
	      ;; if Summary is on the frame, select it.
	      (select-window next-win)
	    ;; if summary is not on the frame, switch to it.
	    (if wl-stay-folder-window
	      (wl-folder-select-buffer sum-buf)
		(switch-to-buffer sum-buf))))
	)))

(defun wl-draft-delete (editing-buffer)
  "kill the editing draft buffer and delete the file corresponds to it."
  (save-excursion
    (when editing-buffer
      (set-buffer editing-buffer)
      (if (and wl-draft-buffer-file-name
	       (file-exists-p wl-draft-buffer-file-name))
	  (delete-file wl-draft-buffer-file-name))
      (set-buffer-modified-p nil)		; force kill
      (kill-buffer editing-buffer))))

(defun wl-draft-kill (&optional force-kill)
  "Kill current draft buffer and quit editing."
  (interactive)
  (save-excursion
    (when (and (or (eq major-mode 'wl-draft-mode)
				   (eq major-mode 'mail-mode))
			   (or force-kill
				   (y-or-n-p "Kill Current Draft?")))
      (let ((cur-buf (current-buffer)))
	(wl-draft-hide cur-buf)
	(wl-draft-delete cur-buf)))
    (message "")))

(defun wl-draft-fcc ()
  "Add a new FCC field, with file name completion."
  (interactive)
  (or (mail-position-on-field "fcc" t)  ;Put new field after exiting FCC.
      (mail-position-on-field "to"))
  (insert "\nFCC: "))

(defun wl-draft-send-mail-with-qmail ()
  "Pass the prepared message buffer to qmail-inject.
Refer to the documentation for the variable `send-mail-function'
to find out how to use this."
  ;; replace the header delimiter with a blank line
  (let ((case-fold-search t)
	delimline)
    (goto-char (point-min))
    (re-search-forward
     (concat "^" (regexp-quote mail-header-separator) "\n"))
    (replace-match "\n")
    (forward-char -1)
    (setq delimline (point-marker))  
    ;(run-hooks 'wl-mail-send-pre-hook)
    (goto-char (point-min))
    (if (re-search-forward "^FCC:" delimline t)
	(wl-draft-do-fcc delimline))
    ;; send the message
    (case
	(as-binary-process
	  (apply
	   'call-process-region 1 (point-max) wl-qmail-inject-program
	   nil nil nil
	   wl-qmail-inject-args))
      ;; qmail-inject doesn't say anything on it's stdout/stderr,
      ;; we have to look at the retval instead
      (0 nil)
      (1   (error "qmail-inject reported permanent failure"))
      (111 (error "qmail-inject reported transient failure"))
      ;; should never happen
      (t   (error "qmail-inject reported unknown failure")))))

;;
;; from Semi-gnus
;;
(defun wl-draft-send-mail-with-smtp ()
  "Send the prepared message buffer with SMTP."
  (require 'smtp)
  (let ((errbuf (if mail-interactive
		    (generate-new-buffer " smtp errors")
		  0))
	(smtp-server (or wl-smtp-posting-server
			 smtp-server
			 "localhost"))
	(user-mail-address (or user-mail-address
			       (wl-address-header-extract-address
				wl-from)))
	(case-fold-search t)
	(default-case-fold-search t)
	resend-to-addresses
	delimline err)
    (unwind-protect
	(save-excursion
	  ;; Change header-delimiter to be what sendmail expects.
	  (goto-char (point-min))
	  (re-search-forward
	   (concat "^" (regexp-quote mail-header-separator) "\n"))
	  (replace-match "\n")
	  (forward-char -1)
	  (setq delimline (point-marker))
	  ;; ignore any blank lines in the header
	  (goto-char (point-min))
	  (while (and (re-search-forward "\n\n\n*" delimline t)
		      (< (point) delimline))
	    (replace-match "\n"))
	  (let ((case-fold-search t))
	    (goto-char (point-min))
	    (while (re-search-forward "^Resent-to:" delimline t)
	      (setq resend-to-addresses
		    (save-restriction
		      (narrow-to-region (point)
					(save-excursion
					  (end-of-line)
					  (point)))
		      (append (mail-parse-comma-list)
			      resend-to-addresses))))
	    ;; Don't send out a blank subject line
	    (goto-char (point-min))
	    (if (re-search-forward "^Subject:[ \t]*\n" delimline t)
		(replace-match ""))
	    ;; Put the "From:" field in unless for some odd reason
	    ;; they put one in themselves.
	    (goto-char (point-min))
	    (if (not (re-search-forward "^From:" delimline t))
		(let* ((login user-mail-address)
		       (fullname (user-full-name)))
		  (cond ((eq mail-from-style 'angles)
			 (insert "From: " fullname)
			 (let ((fullname-start (+ (point-min) 6))
			       (fullname-end (point-marker)))
			   (goto-char fullname-start)
			   ;; Look for a character that cannot appear unquoted
			   ;; according to RFC 822.
			   (if (re-search-forward "[^- !#-'*+/-9=?A-Z^-~]"
						  fullname-end 1)
			       (progn
				 ;; Quote fullname, escaping specials.
				 (goto-char fullname-start)
				 (insert "\"")
				 (while (re-search-forward "[\"\\]"
							   fullname-end 1)
				   (replace-match "\\\\\\&" t))
				 (insert "\""))))
			 (insert " <" login ">\n"))
			((eq mail-from-style 'parens)
			 (insert "From: " login " (")
			 (let ((fullname-start (point)))
			   (insert fullname)
			   (let ((fullname-end (point-marker)))
			     (goto-char fullname-start)
			     ;; RFC 822 says \ and nonmatching parentheses
			     ;; must be escaped in comments.
			     ;; Escape every instance of ()\ ...
			     (while (re-search-forward "[()\\]" fullname-end 1)
			       (replace-match "\\\\\\&" t))
			     ;; ... then undo escaping of matching parentheses,
			     ;; including matching nested parentheses.
			     (goto-char fullname-start)
			     (while (re-search-forward 
				     "\\(\\=\\|[^\\]\\(\\\\\\\\\\)*\\)\\\\(\\(\\([^\\]\\|\\\\\\\\\\)*\\)\\\\)"
				     fullname-end 1)
			       (replace-match "\\1(\\3)" t)
			       (goto-char fullname-start))))
			 (insert ")\n"))
			((null mail-from-style)
			 (insert "From: " login "\n")))))
	    ;; Insert an extra newline if we need it to work around
	    ;; Sun's bug that swallows newlines.
	    (goto-char (1+ delimline))
	    (if (eval mail-mailer-swallows-blank-line)
		(newline))
	    ;(run-hooks 'wl-mail-send-pre-hook) 
	    ;; Find and handle any FCC fields.
	    (goto-char (point-min))
	    (if (re-search-forward "^FCC:" delimline t)
		(wl-draft-do-fcc delimline))
	    (if mail-interactive
		(save-excursion
		  (set-buffer errbuf)
		  (erase-buffer))))
	  (let ((recipient-address-list
		 (or resend-to-addresses
		     (smtp-deduce-address-list (current-buffer)
					       (point-min) delimline))))
	    (wl-draft-delete-field "bcc" delimline)
	    (if wl-draft-verbose-send
		(message "Sending...")) ; for x-pgp-sig ;-)
	    (as-binary-process
	     (if recipient-address-list
		 (let ((err (smtp-via-smtp 
			     user-mail-address
			     recipient-address-list
			     (current-buffer))))
		   (if (not (eq err t))
		       (error "Sending failed; SMTP protocol error:%s" err)))
	       )
	     )))
      (if (bufferp errbuf)
	  (kill-buffer errbuf)))))

(defun wl-draft-send-mail-with-pop-before-smtp ()
  "Send the prepared message buffer with POP-before-SMTP."
  (require 'elmo-pop3)
  (elmo-pop3-get-connection 
   (or wl-smtp-posting-user
       (getenv "USER") (getenv "LOGNAME") (user-login-name))
   wl-smtp-posting-server elmo-pop-before-smtp-port
   elmo-pop-before-smtp-authenticate-type
   elmo-default-pop3-ssl)
  (wl-draft-send-mail-with-smtp))

(defun wl-draft-normal-send-func (editing-buffer kill-when-done)
  "Send the message in the current buffer. "
  (save-restriction
    (std11-narrow-to-header mail-header-separator)
    ;; Insert Message-Id field...
    (goto-char (point-min))
    (when (and wl-insert-message-id
	       (not (re-search-forward "^Message-ID[ \t]*:" nil t)))
      (insert (concat "Message-ID: "
		      (wl-draft-make-message-id-string)
		      "\n")))
    ;; Insert date field.
    (goto-char (point-min))
    (or (re-search-forward "^Date[ \t]*:" nil t)
	(wl-draft-insert-date-field))
    ;; Delete null fields.
    (goto-char (point-min))
    (while (re-search-forward "^[^ \t\n:]+:[ \t]*\n" nil t)
      (replace-match ""))
    )
  (run-hooks 'wl-mail-send-pre-hook) ;; X-PGP-Sig, Cancel-Lock
  (if wl-plugged 
      (progn (if (wl-message-mail-p)
		 (funcall wl-draft-send-mail-func))
	     (if (wl-message-news-p)
		 (funcall wl-draft-send-news-func)))
    (if wl-draft-enable-queuing    
	(wl-draft-queue-append)
      (error "Unplugged.")))
  (when kill-when-done
    ;; hide editing-buffer.
    (wl-draft-hide editing-buffer)
    ;; delete editing-buffer and its file.
    (wl-draft-delete editing-buffer)
    ))

(defun wl-draft-raw-send (&optional kill-when-done force-pre-hook)
  "Force send current buffer as raw message."
  (interactive)
  (save-excursion
    (let (wl-interactive-send
	  wl-draft-verbose-send
	  (wl-mail-send-pre-hook (and force-pre-hook wl-mail-send-pre-hook))
;	  wl-news-send-pre-hook
	  mail-send-hook
	  mail-send-actions)
      (wl-draft-send kill-when-done))))

(defun wl-draft-clone-local-variables ()
  (let ((locals (buffer-local-variables))
	result)
    (mapcar
     (function
      (lambda (local)
	(when (and (consp local)
		   (car local)
		   (string-match 
		    wl-draft-clone-local-variable-regexp
		    (symbol-name (car local))))
	  (setq result (wl-append result (list (car local)))))))
     locals)
    result))

(defun wl-draft-send (&optional kill-when-done)
  "Send current draft message. 
If optional argument is non-nil, current draft buffer is killed"
  (interactive)
  (when (and wl-draft-config-alist
	     wl-draft-config-exec-flag)
    (setq wl-draft-config-variables
	  (wl-draft-config-exec))
    (goto-char (point-min)))
  (when (or (not wl-interactive-send)
	    (y-or-n-p "Send current draft. OK?"))
    (let ((send-mail-function 'wl-draft-raw-send)
	  (editing-buffer (current-buffer))
	  (sending-buffer (wl-draft-generate-clone-buffer 
			   " *wl-draft-sending-buffer*"
			   (wl-append wl-draft-config-variables
				      (wl-draft-clone-local-variables)
				      )))
	  err)
      (if wl-draft-verbose-send
	  (message "Sending..."))
      (save-excursion
	(set-buffer sending-buffer)
	(condition-case err
	    (progn
	      (if (and (not (wl-message-mail-p))
		       (not (wl-message-news-p)))
		  (error "No recipient is specified."))
	      (expand-abbrev) ; for mail-abbrevs
	      (run-hooks 'mail-send-hook) ; translate buffer
	      (if wl-draft-verbose-send
		  (message "Sending..."))
	      (funcall wl-draft-send-func editing-buffer kill-when-done)
	      ;; Now perform actions on successful sending.
	      (while mail-send-actions
		(condition-case ()
		    (apply (car (car mail-send-actions))
			   (cdr (car mail-send-actions)))
		  (error))
		(setq mail-send-actions (cdr mail-send-actions)))
	      (if wl-draft-verbose-send
		  (message "Sending...done."))
	      (kill-buffer sending-buffer))
	  (error
	   (kill-buffer sending-buffer)
	   (wl-display-error err t)
	   (ding)
	   )
	  (quit 
	   (kill-buffer sending-buffer)
	   (error "Sending aborted.")))))))
  
(defun wl-draft-send-and-exit ()
  "Send current draft message and kill it."
  (interactive)
  (wl-draft-send t)
  )

(defun wl-draft-send-from-toolbar ()
  (interactive)
  (let ((wl-interactive-send t))
    (wl-draft-send-and-exit)
    ))

(defun wl-draft-delete-field (field delimline)
  (save-restriction
    (narrow-to-region (point-min) delimline)
    (goto-char (point-min))
    (let ((regexp (concat "^" (regexp-quote field) ":"))
	  (case-fold-search t)
	  last)
      (while (not (eobp))
	(if (looking-at regexp)
	    (progn
	      (delete-region
	       (point)
	       (progn
		 (forward-line 1)
		 (if (re-search-forward "^[^ \t]" nil t)
		     (goto-char (match-beginning 0))
		   (point-max)))))
	  (forward-line 1)
	  (if (re-search-forward "^[^ \t]" nil t)
	      (goto-char (match-beginning 0))
	    (point-max)))))))

(defun wl-draft-do-fcc (header-end)
  (let (fcc-list
	(send-mail-buffer (current-buffer))
	(tembuf (generate-new-buffer " fcc output"))
	(case-fold-search t)
	beg end)
    (or (markerp header-end) (error "header-end must be a marker"))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward "^FCC:[ \t]*" header-end t)
	(setq fcc-list (cons (buffer-substring (point)
					       (progn
						 (end-of-line)
						 (skip-chars-backward " \t")
						 (point)))
			     fcc-list))
	(delete-region (match-beginning 0)
		       (progn (forward-line 1) (point))))
      (set-buffer tembuf)
      (erase-buffer)
      ;; insert just the headers to avoid moving the gap more than
      ;; necessary (the message body could be arbitrarily huge.)
      (insert-buffer-substring send-mail-buffer 1 header-end)
      (goto-char (point-min))
      (or (re-search-forward "^From[ \t]*:" header-end t)
	  (insert "From: " (user-login-name) " (" (user-full-name) ")\n"))
      (goto-char (point-min))
      (or (re-search-forward "^Date[ \t]*:" header-end t)
	  (wl-draft-insert-date-field))
      (goto-char (point-max))
;      (insert "\n\n")
      (insert-buffer-substring send-mail-buffer header-end)
      (while fcc-list
	(elmo-append-msg (car fcc-list) (buffer-substring 
					 (point-min) (point-max))
			 (std11-field-body "Message-ID"))
	(setq fcc-list (cdr fcc-list))))
    (kill-buffer tembuf)))

(defun wl-draft-complete-value-if ()
  (save-excursion
    (beginning-of-line)
    (while (and (< (point-min) (point))	(looking-at "^[ \t]"))
      (forward-line -1)
      )
    (if (looking-at "\\([^:]*:\\)")
	(let ((field (wl-match-buffer 1)))
	  (cond 
	   ((string-match wl-address-complete-header-regexp field)
	    (wl-address-complete-address wl-address-completion-list ?@))
	   (t
	    (wl-address-complete-address wl-folder-entity-hashtb)))
	  t)
      nil)
    ))

(defun wl-draft-on-field-p ()
  (if (< (point)
	 (save-excursion
	   (goto-char (point-min))
	   (search-forward (concat "\n" mail-header-separator "\n") nil 0)
	   (point)))
      (if (bolp)
	  (if (bobp) 
	      t
	    (save-excursion
	      (forward-line -1)
	      (if (looking-at ".*,[ \t]?$") nil t)
	      )
	    )
	(let ((pos (point)))
	  (save-excursion
	    (beginning-of-line)
	    (if (looking-at "^[ \t]")
		nil
	      (if (re-search-forward ":" pos t) nil t))
	    ))
	)))

(defun wl-draft-random-alphabet ()
  (let ((alphabet '(?A ?B ?C ?D ?E ?F ?G ?H ?I ?J ?K ?L ?M ?N ?O ?P ?Q ?R ?S ?T ?U ?V ?W ?X ?Y ?Z)))
    (nth (abs (% (random) 26)) alphabet)))

(defun wl-draft-insert-signature ()
  "insert signature file."
  (interactive)
  (if (fboundp 'insert-signature)
      (funcall 'insert-signature)
    (insert-file-contents "~/.signature")))

(defun wl-draft (&optional to subject in-reply-to cc references newsgroups
			   mail-followup-to
			   content-type
			   body edit-again summary-buf)
  "Write and send mail/news message with Wanderlust."
  (interactive)
  (if (interactive-p)
      (setq summary-buf (wl-summary-get-buffer wl-summary-buffer-folder-name)))
  (let ((draft-folder-spec (elmo-folder-get-spec wl-draft-folder))
	buf-name file-name num wl-demo change-major-mode-hook)
    (wl-init) ;; returns immediately if already initialized.
    (if (not (eq (car draft-folder-spec) 'localdir))
	(error "%s folder cannot be used for draft folder." wl-draft-folder))
    (setq buf-name (find-file-noselect 
		       (setq file-name
			     (expand-file-name
			      (setq num (int-to-string 
					 (+ 1 (or 
					       (car 
						(elmo-localdir-max-of-folder 
						 draft-folder-spec)
						)0))))
			      (expand-file-name
			       (nth 1 draft-folder-spec)
			       elmo-localdir-folder-path)))))
    (switch-to-buffer buf-name)
    (set-buffer buf-name)
    (if (not (string-match (regexp-quote wl-draft-folder)
			   (buffer-name)))
	(rename-buffer (concat wl-draft-folder "/" (buffer-name))))
    (if (or (eq wl-draft-reply-buffer-style 'full)
	    (eq this-command 'wl-draft)
	    (eq this-command 'wl-summary-write-current-newsgroup))
	(delete-other-windows))
    (auto-save-mode -1)
    (wl-draft-mode)
    (if (stringp wl-from)
	(insert "From: " wl-from "\n"))
    (and (or (interactive-p)
	     to)
	 (insert "To: " (or to "") "\n"))
    (and cc (insert "Cc: " (or cc "") "\n"))
    (insert "Subject: " (or subject "") "\n")
    (and newsgroups (insert "Newsgroups: " newsgroups "\n"))
    (and mail-followup-to (insert "Mail-Followup-To: " mail-followup-to "\n"))
    (and wl-insert-mail-reply-to 
	 (insert "Mail-Reply-To: " 
		 (wl-address-header-extract-address
		  wl-from) "\n"))
    (and in-reply-to (insert "In-Reply-To: " in-reply-to "\n"))
    (and references (insert "References: " references "\n"))
    (insert (funcall wl-generate-mailer-string-func) 
	    "\n")
    (setq wl-draft-buffer-file-name file-name)
    (if mail-default-reply-to
	(insert "Reply-To: " mail-default-reply-to "\n"))
    (if (or wl-bcc mail-self-blind)
	(insert "Bcc: " (or wl-bcc (user-login-name)) "\n"))
    (if wl-fcc
	(insert "FCC: " wl-fcc "\n"))
    (if wl-organization
	(insert "Organization: " wl-organization "\n"))
    (and wl-auto-insert-x-face
	 (file-exists-p wl-x-face-file)
	 (wl-draft-insert-x-face-field-here))
    (if mail-default-headers
	(insert mail-default-headers))
    (if (not (= (preceding-char) ?\n))
	(insert ?\n))
    (if edit-again
	(let (start)
	  (setq start (point))
	  (when content-type 
	    (insert "Content-type: " content-type "\n")
	    (insert "\n"))
	  (and body (insert body))
	  (save-restriction
	    (narrow-to-region start (point))
	    (if edit-again
		(and wl-edit-again-func
		     (funcall wl-edit-again-func (not wl-use-semi) t)))
	    (widen)
	    (goto-char start)
	    (insert mail-header-separator "\n")
	    ))
      (insert  mail-header-separator "\n")
      (and body (insert body)))
    (if wl-on-nemacs
	(push-mark (point) t)
      (push-mark (point) t t))
    (as-binary-output-file
     (write-region (point-min)(point-max) wl-draft-buffer-file-name
		   nil t))
    (and (not edit-again)
	 wl-editor-mode-func
	 (funcall wl-editor-mode-func))    
    (wl-draft-overload-functions)
    (let (wl-highlight-x-face-func)
      (wl-highlight-headers))
    (goto-char (point-min))
    (if wl-draft-prepared-config-alist
	(wl-draft-prepared-config-exec))
    (if (interactive-p)
	(run-hooks 'wl-mail-setup-hook))
    (cond ((eq this-command 'wl-summary-write-current-newsgroup)
 	   (mail-position-on-field "Subject"))
 	  ((and (interactive-p)(null to))
 	   (mail-position-on-field "To"))
 	  (t
 	   (goto-char (point-max))))    
    (setq wl-draft-config-exec-flag t)
    (setq wl-draft-buffer-cur-summary-buffer (or summary-buf
						 (get-buffer
						  wl-summary-buffer-name)))    
    buf-name
    ))

(defun wl-draft-elmo-nntp-send ()
  (let ((elmo-nntp-post-pre-hook wl-news-send-pre-hook))
    (goto-char (point-min))
    (if (re-search-forward
	 (concat "^" (regexp-quote mail-header-separator) "\n") nil t)
	(replace-match "\n"))
    (elmo-nntp-post wl-nntp-posting-server (current-buffer))))

(defun wl-draft-generate-clone-buffer (name &optional local-variables)
  "generate clone of current buffer named NAME."
  (let ((editing-buffer (current-buffer)))
    (save-excursion
      (set-buffer (generate-new-buffer name))
      (erase-buffer)
      (wl-draft-mode)
      (and wl-editor-mode-func
	   (funcall wl-editor-mode-func))
      (message "")
      (insert-buffer editing-buffer)
      (when local-variables
	(mapcar
	 (function
	  (lambda (var)
	    (make-local-variable var)
	    (set var (save-excursion
		       (set-buffer editing-buffer)
		       (symbol-value var)))))
	 local-variables))
      (current-buffer))))

(defun wl-draft-reedit (number)
  (let ((draft-folder-spec (elmo-folder-get-spec wl-draft-folder))
	buf-name file-name change-major-mode-hook)
    (setq buf-name (find-file-noselect 
		    (setq file-name
			  (expand-file-name
			   (int-to-string number)
			   (expand-file-name
			    (nth 1 draft-folder-spec)
			    elmo-localdir-folder-path)))))
    (switch-to-buffer buf-name)
    (set-buffer buf-name)
    (if (not (string-match (regexp-quote wl-draft-folder)
			   (buffer-name)))
	(rename-buffer (concat wl-draft-folder "/" (buffer-name))))    
    (auto-save-mode -1)
    (wl-draft-mode)
    (setq wl-draft-buffer-file-name file-name)
    (goto-char (point-min))
    (or (re-search-forward "\n\n" nil t)
	(search-forward (concat mail-header-separator "\n") nil t))
    (if wl-on-nemacs
	(push-mark (point) t)
      (push-mark (point) t t))
    (write-region (point-min)(point-max) wl-draft-buffer-file-name
		  nil t)
    (wl-draft-overload-functions)
    (and wl-editor-mode-func
	 (funcall wl-editor-mode-func))
    (let (wl-highlight-x-face-func)
      (wl-highlight-headers))
    (run-hooks 'wl-mail-setup-hook)
    (goto-char (point-max))
    buf-name
    ))

(defmacro wl-draft-config-body-goto-top ()
  (` (progn
       (goto-char (point-min))
       (if (re-search-forward mail-header-separator nil t)
	   (forward-char 1)
	 (goto-char (point-max))))))

(defmacro wl-draft-config-body-goto-bottom ()
  (` (goto-char (point-max))))

(defun wl-draft-config-sub-body (content)
  (wl-draft-config-body-goto-top)
  (delete-region (point) (point-max))
  (if content (insert (eval content))))

(defun wl-draft-config-sub-top (content)
  (wl-draft-config-body-goto-top)
  (if content (insert (eval content))))

(defun wl-draft-config-sub-bottom (content)
  (wl-draft-config-body-goto-bottom)
  (if content (insert (eval content))))

(defsubst wl-draft-config-sub-file (content)
  (let ((coding-system-for-read wl-cs-autoconv)
	(file (expand-file-name (eval content))))
    (if (file-exists-p file)
	(insert-file-contents file)
      (error "%s: no exists file." file))))

(defun wl-draft-config-sub-body-file (content)
  (wl-draft-config-body-goto-top)
  (delete-region (point) (point-max))
  (wl-draft-config-sub-file content))

(defun wl-draft-config-sub-top-file (content)
  (wl-draft-config-body-goto-top)
  (wl-draft-config-sub-file content))

(defun wl-draft-config-sub-bottom-file (content)
  (wl-draft-config-body-goto-bottom)
  (wl-draft-config-sub-file content))

(defun wl-draft-config-sub-template (content)
  (wl-template-insert (eval content)))

(defun wl-draft-config-sub-x-face (content)
  (if (and (string-match "\\.xbm\\(\\.gz\\)?$" content) 
	   (fboundp 'x-face-insert)) ; x-face.el is installed.
      (x-face-insert content)
    (wl-draft-replace-field "X-Face" (elmo-get-file-string content t) t)))

(defsubst wl-draft-config-sub-func (field content)
  (let (func)
    (if (setq func (assq field wl-draft-config-sub-func-alist))
	(progn
	  (funcall (cdr func) content)
	  t))))

(defsubst wl-draft-config-exec-sub (clist)
  (let (config local-variables)
    (while clist
      (setq config (car clist))
      (cond
       ((consp config)
	(let ((field (car config))
	      (content (cdr config)))
 	  (cond
 	   ((stringp field)
 	    (wl-draft-replace-field field (eval content) t))
 	   ((wl-draft-config-sub-func field content))
 	   ((boundp field) ;; variable
 	    (make-local-variable field)
 	    (set field (eval content))
 	    (wl-append local-variables (list field)))
 	   (t
 	    (error "%s: not variable." field)))))
       ((or (functionp config)
	    (and (symbolp config)
		 (fboundp config)))
	(funcall config))
       (t
	(error "%s: not supported type." config)))
      (setq clist (cdr clist)))
    local-variables))

(defun wl-draft-prepared-config-exec (&optional reply-buf)
  (interactive)
  (let ((case-fold-search t)
 	(alist wl-draft-prepared-config-alist)
 	key clist found)
    (save-excursion
      (catch 'done
 	(while alist
 	  (setq key (caar alist)
 		clist (cdar alist))
 	  (cond
 	   ((eq key 'reply)
 	    (when (and
 		   reply-buf
 		   (save-excursion
 		     (set-buffer reply-buf)
 		     (save-restriction
 		       (std11-narrow-to-header)
 		       (goto-char (point-min))
 		       (re-search-forward (car clist) nil t))))
 	      (wl-draft-config-exec-sub (cdr clist))
 	      (setq found t)))
 	   ((stringp key)
 	    (when (save-restriction
 		    (std11-narrow-to-header mail-header-separator)
 		    (goto-char (point-min))
 		    (re-search-forward key nil t))
 	      (wl-draft-config-exec-sub clist)
 	      (setq found t)))
 	   ((eval key)
 	    (wl-draft-config-exec-sub clist)
 	    (setq found t)))
 	  (if (and found wl-draft-prepared-config-matchone)
 	      (throw 'done t))
 	  (setq alist (cdr alist))))
      (run-hooks 'wl-draft-prepared-config-exec-hook)
      )))

(defun wl-draft-config-exec ()
  (interactive)
  (let ((case-fold-search t)
	(alist wl-draft-config-alist)
	local-variables key clist found)
    (setq wl-draft-config-variables nil)
    (save-excursion
      (catch 'done
	(while alist
	  (setq key (caar alist)
		clist (cdar alist))
	  (cond
	   ((stringp key)
	    (when (save-restriction
		    (std11-narrow-to-header mail-header-separator)
		    (goto-char (point-min))
		    (re-search-forward key nil t))
	      (wl-append local-variables
			 (wl-draft-config-exec-sub clist))
	      (setq found t)))
	   ((eval key)
	    (wl-append local-variables
		       (wl-draft-config-exec-sub clist))
	    (setq found t)))
	  (if (and found wl-draft-config-matchone)
	      (throw 'done t))
	  (setq alist (cdr alist))))
      (setq wl-draft-config-exec-flag nil)
      (run-hooks 'wl-draft-config-exec-hook)
      (wl-uniq-list local-variables)
      )))

(defun wl-draft-replace-field (field content &optional add)
  (save-excursion
    (save-restriction
      (let ((case-fold-search t)
	    (inhibit-read-only t) ;; added by teranisi.
	    beg)
	(std11-narrow-to-header mail-header-separator)
	(goto-char (point-min))
	(if (re-search-forward (concat "^" (regexp-quote field) ":") nil t)
	    (if content
		;; replace field
		(progn
		  (setq beg (point))
		  (re-search-forward "^[^ \t]" nil 'move)
		  (beginning-of-line)
		  (skip-chars-backward "\n")
		  (delete-region beg (point))
		  (insert " " content))
	      ;; delete field
	      (save-excursion
		(beginning-of-line)
		(setq beg (point)))
	      (re-search-forward "^[^ \t]" nil 'move)
	      (beginning-of-line)
	      (delete-region beg (point)))
	  (when (and add content)
	    ;; add field
	    (goto-char (point-max))
	    (insert (concat field ": " content "\n"))))
	))))

(defun wl-draft-queue-append ()
  (if wl-draft-verbose-send
      (message "Queuing..."))
  (let ((send-buffer (current-buffer))
	(message-id (std11-field-body "Message-ID")))
    (elmo-append-msg wl-queue-folder
		     (buffer-substring (point-min) (point-max))
		     message-id)
    (elmo-dop-lock-message message-id)
    (if wl-draft-verbose-send
	(message "Queuing...done."))))

(defun wl-draft-queue-flush ()
  "Flush draft queue."
  (interactive)
  (let ((msgs (elmo-list-folder wl-queue-folder))
	(i 0)
	(performed 0)
	(wl-draft-queue-flushing t)
	failure	len buffer msgid)
    (when (> (setq len (length msgs)) 0)
      (if (y-or-n-p (format
		     "%d message(s) are in the sending queue. Send now?"
		     len))
	  (progn
	    (save-excursion
	      (setq buffer (get-buffer-create " *wl-draft-queue-flush*"))
	      (set-buffer buffer)
	      (while msgs
		(erase-buffer)
		(setq i (+ 1 i)
		      failure nil)
		(if wl-draft-verbose-send
		    (message "Sending (%d/%d)..." i len))
		(elmo-read-msg-no-cache wl-queue-folder (car msgs) 
					(current-buffer))
		;;(when (re-search-forward "^Date[ \t]*:" nil t)
		;; (delete-region (match-beginning 0) (std11-field-end))
		;; (wl-draft-insert-date-field))???
		(condition-case err
		    (wl-draft-raw-send)
		  (error
		   (wl-display-error err t)
		   (setq failure t))
		  (quit 
		   (setq failure t)))
		(unless failure
		  (elmo-delete-msgs wl-queue-folder (cons (car msgs) nil))
		  (elmo-dop-unlock-message (std11-field-body "Message-ID"))
		  (setq performed (+ 1 performed)))
		(setq msgs (cdr msgs))
		)
	      (message "%d message(s) are sent." performed)
	      (kill-buffer buffer)))
	(message "%d message(s) are remained to be sent." len)))))

(defun wl-jump-to-draft-buffer (&optional arg)
  "Jump to the draft if exists."
  (interactive "P")
  (if arg
      (wl-jump-to-draft-folder)
    (let ((bufs (buffer-list))
	  (draft-regexp (concat
			 "^" (regexp-quote
			      (expand-file-name
			       (nth 1 (elmo-folder-get-spec wl-draft-folder))
			       (expand-file-name
				elmo-localdir-folder-path)))))
	  buf draft-bufs)
      (while bufs
	(if (and
	     (setq buf (buffer-file-name (car bufs)))
	     (string-match draft-regexp buf))
	    (setq draft-bufs (cons (buffer-name (car bufs)) draft-bufs)))
	(setq bufs (cdr bufs)))
      (cond
       ((null draft-bufs)
	(message "No draft buffer exist."))
       (t
	(setq draft-bufs
	      (sort draft-bufs (function (lambda (a b) (not (string< a b))))))
	(if (setq buf (cdr (member (buffer-name) draft-bufs)))
	    (setq buf (car buf))
	  (setq buf (car draft-bufs)))
	(switch-to-buffer buf)
	)))))

(defun wl-jump-to-draft-folder ()
  (let ((msgs (reverse (elmo-list-folder wl-draft-folder)))
	(mybuf (buffer-name))
	msg buf)
    (if (not msgs)
	(message "No draft message exist.")
      (if (string-match (concat "^" wl-draft-folder "/") mybuf)
	  (setq msg (cadr (memq
			   (string-to-int (substring mybuf (match-end 0)))
			   msgs))))
      (or msg (setq msg (car msgs)))
      (if (setq buf (get-buffer (format "%s/%d" wl-draft-folder msg)))
	  (switch-to-buffer buf)
	(wl-draft-reedit msg)))))

(defun wl-draft-highlight-and-recenter ()
  (interactive)
  (if wl-highlight-body-too
      (let ((beg (point-min))
	    (end (point-max)))
	(put-text-property beg end 'face nil)
	(wl-highlight-message beg end t)))
  (recenter)
)

(provide 'wl-draft)
