/*
   Egon Animator
   Copyright (C) 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/*
 * fileio.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "egon.h"
#include "types.h"
#include "../common/cmalloc.h"
#include "user_interface.h"

#define MAX_FORMATS (20)

static int format_count = 0;
static int nloader = 0;
static int nsaver = 0;

struct {
	int (*load)(char *, buffer *);
	int (*save)(char *, buffer *);
	int (*myformat)(char *);
	char *pattern;
} fileformats[MAX_FORMATS];

static int save_all(char *fn, buffer *buf)
{
	int n = select_from_list("Which format?", saver_patterns, nsaver);
	if (n < 0) return -1;
	return savematrix(fn, buf, fileformats[n].pattern);
}

static int load_all(char *fn, buffer *buf)
{
	int n = select_from_list("Which format?", loader_patterns, nloader);
	if (n < 0) return -1;
	return loadmatrix(fn, buf, fileformats[n].pattern);
}

static int myformat(char *fn)
{
	return 0;	/* this doesn't handle any format at all */
}

/* Need this in a separate structure for fsel_input */
char *loader_patterns[MAX_FORMATS];
char *saver_patterns[MAX_FORMATS];

/* ---
Register a file format handler.
*/

void register_format(int (*load)(char *, buffer *),
		int (*save)(char *, buffer *),
		int (*myformat)(char *), char *p)
{
	char *pattern = cstrdup(p);
	fileformats[format_count].load = load;
	fileformats[format_count].save = save;
	fileformats[format_count].myformat = myformat;
	fileformats[format_count].pattern = pattern;
	format_count++;
	if (load) {
		loader_patterns[nloader++] = pattern;
		loader_patterns[nloader] = NULL;
	}
	if (save) {
		saver_patterns[nsaver++] = pattern;
		saver_patterns[nsaver] = NULL;
	}
}

/* These are not declared in calc.h because they will only be called here */
extern void fileio_egon_init(void);
extern void fileio_c_init(void);
extern void fileio_gif_init(void);
extern void fileio_ps_init(void);
extern void fileio_scm_init(void);

static int siag_fmt;

/* ---
*/
void fileio_init(void)
{
	/* Register Egon format first */
	siag_fmt = format_count;
	fileio_egon_init();
	fileio_c_init();
	fileio_gif_init();
	fileio_ps_init();
	fileio_scm_init();

	/* Put this catch-all last */
	register_format(load_all, save_all, myformat, "All files (*)");
}

static int fmt_index(char *fmt)
{
	int i;

	if (fmt == NULL) return siag_fmt;

	for (i = 0; i < format_count; i++)
		if (!strcmp(fileformats[i].pattern, fmt)) return i;
	return -1;
}

/* ---
Save a file of unknown format.
*/

int savematrix(char *fn, buffer *buf, char *format)
{
	int (*fun)(char *, buffer *);
	int i;

	if (format) {
		i = fmt_index(format);
		if (i < 0) return 1;
	} else i = format_count-1;	/* Ask for format */

	fun = fileformats[i].save;
	if (fun) return fun(fn, buf);
	else return 1;
}

/* ---
Load a file of unknown format.
*/

int loadmatrix(char *fn, buffer *buf, char *format)
{
	int (*fun)(char *, buffer *);
	int i;

	if (format) {
		i = fmt_index(format);
		if (i < 0) return 1;
	} else i = format_count-1;	/* Ask for format */

	fun = fileformats[i].load;
	if (fun) return fun(fn, buf);
	else return 1;
}

/* ---
Guess the format of a file.
*/

char *guess_file_format(char *fn)
{
        int i;
        int (*my_format)(char *);

        for (i = 0; i < format_count; i++) {
                my_format = fileformats[i].myformat;
                if (my_format && my_format(fn))
                        return fileformats[i].pattern;
        }
        return NULL;
}

