/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module supports reading and parsing input. */

/* includes =================================================================*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include "basic.h"

#undef GLOBAL
#define GLOBAL

#include "input.h"

/* functions ================================================================*/

GLOBAL void parse_whitespace (string_t *input)
/* Read whitespace in *<input> and update *<input>. */
{
  while (isspace (**input))
    (*input)++;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t parse_word (string_t *input)
/* If there is a word in *<input>, parse it up to the next space
 * and update *<input>. Return the word. It must be freed with "free".
 * If there's no word, report an error. */
{
  text_t text = new_text ();

  if (**input == EOS)
    error ("argument expected");
  
  if (**input == '\"') /* Read quoted word that contains spaces and quotes. */
  {
    (*input)++;
    while (**input != '\"')
    {
      if (**input == EOS)
	error ("missing closing '\"'");
      
      if ((*input)[0] == '\\' && (*input)[1] != EOS)
	(*input)++;
      add_char_to_text (text, *(*input)++);
    }
    (*input)++;
  } 
  else
  {
    while (**input != EOS && ! isspace (**input))
      add_char_to_text (text, *(*input)++);
  }
  
  parse_whitespace (input);
  return text_to_string (&text);
}

/*---------------------------------------------------------------------------*/

GLOBAL void parse_end (string_t input)
/* Test if there are no more arguments in <input>. */
{
  if (*input != EOS)
    error ("unexpected argument: \"%s\"", input);
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t parse_int (string_t *input)
/* Parse the next integer number from *<input> and update *<input>.
 * If there is no integer, an error is reported. */
{
  int_t number;
  string_t string; 

  string = parse_word (input);
  if (sscanf (string, "%ld", &number) != 1)
    error ("illegal integer value");
  free_mem (&string);
  return number;
}

/*---------------------------------------------------------------------------*/
  
GLOBAL double parse_double (string_t *input)
/* Parse the next double from *<input> and update *<input>.
 * If there is no double, an error is reported. */
{
  double number;
  string_t string;
  
  string = parse_word (input);
  if (sscanf (string, "%lf", &number) != 1)
    error ("illegal double value");
  free_mem (&string);
  return number;
}
  
/*---------------------------------------------------------------------------*/

GLOBAL bool_t parse_yes_no (string_t *input)
/* Parse next word in <input>. It must be "yes", "no", "on" or "off".
 * Return TRUE iff next word is "yes" or "on". */
{
  string_t argument = parse_word (input);
  bool_t return_value;

  if (strcmp_no_case (argument, "yes") == 0 
      || strcmp_no_case (argument, "on") == 0)
    return_value = TRUE;
  else if (strcmp_no_case (argument, "no") == 0
	   || strcmp_no_case (argument, "off") == 0)
    return_value = FALSE;
  else
    error ("\"yes\"/\"on\" or \"no\"/\"off\" expected, not \"%s\"", argument);

  free_mem (&argument);
  return return_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t read_line (FILE *stream)
/* Read user input from <stream> until eof or newline is met.
 * Return the result string (without final newline or eof). 
 * The string must be freed with "free" after use. */
{ 
  text_t text = new_text ();

  while (TRUE) 
  {
    int_t c;

    c = fgetc (stream);
    if (c == '\n' || c == EOF)
      return text_to_string (&text);
    else
      add_char_to_text (text, c);
  }
}

/* end of file ==============================================================*/
