/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module manages the emission of instructions and keeps track of the 
 * stack index. It also holds buffers for the compiled code. */

/* constants ================================================================*/

/* possible values of file_type */
#define ALLO_RULE_FILE 0
#define MORPHO_RULE_FILE 1
#define SYNTAX_RULE_FILE 2

/* variables ================================================================*/

GLOBAL struct /* internal structure that contains compiled Malaga rule code. */
{
  int_t file_type; /* ALLO_RULE_FILE, MORPHO_RULE_FILE or SYNTAX_RULE_FILE */
  int_t rule_number; /* the current rule number */
  int_t stack_index; /* the current stack index */
  int_t next_instr_index; /* the index of the instruction to be emitted */

  /* the following values must be copied to the rule file */
  int_t initial_rule_set; /* index into <rule_set_pool> */
  int_t initial_cat; /* index into <value_pool> */
  int_t robust_rule; /* index into <rule_pool> */
  int_t pruning_rule; /* index into <rule_pool> */
  int_t allo_rule; /* index into <rule_pool> */
  int_t input_filter; /* index into <rule_pool> */
  int_t output_filter; /* index into <rule_pool> */
  pool_t rule_pool; /* the pool of all rules */
  pool_t rule_set_pool; /* the pool of rule sets */
  pool_t instr_pool; /* the pool of all instructions */
  pool_t value_pool; /* the pool of all constant Malaga values */
  pool_t src_line_pool; /* the pool of all correspondences
			 * between source lines and rule code */
  pool_t var_pool; /* the pool of all variables */
  pool_t var_scope_pool; /* the pool of all variable scopes */
  pool_t string_pool; /* the pool of all strings
		       * (variable and rule names, patterns) */
} code;

/* functions ================================================================*/

extern void init_rule_code (int_t file_type);
/* Initialise this module. 
 * <code> will contain compilation data for a file of <file_type>.
 * <file_type> may be ALLO_RULE_FILE, MORPHO_RULE_FILE, or SYNTAX_RULE_FILE. */

extern void term_rule_code (void);
/* Terminate this module. */

extern void write_code (string_t file_name);
/* Write <code> to <file_name>. */

/* functions for constant folding ===========================================*/

extern void buffer_instr (int_t opcode, int_t info);
/* Buffer the instructions BUILD_LIST, BUILD_RECORD, PUSH_SYMBOL,
 * and PUSH_CONST for constant folding. */

extern void buffer_push_number_instr (double number);
/* Buffer the instruction PUSH_CONST with <number> converted to a value. */

extern void buffer_push_string_instr (string_t string, string_t string_end);
/* Buffer the instruction PUSH_CONST with the given value. */

extern void flush_buffer (void);
/* Emit the instructions that are still in the buffer. */

extern value_t get_buffer_top_value (void);
/* Test if the buffer contains a value and return the top value. */

extern value_t pop_buffer_top_value (void);
/* Pop the top value in the buffer. */

extern instr_t *emit_instr (int_t opcode, int_t info);
/* Emit an instruction to the instruction pool (flushes buffer before)
 * and return the address of the instruction in the pool. */

/* end of file ==============================================================*/
