#!@PERL@

use strict;
use Getopt::Long;

use File::Basename;
use File::Path;
use Cwd qw(cwd abs_path);
use DBI;

=head1 NAME

mysqlhotcopy - fast on-line hot-backup utility for local MySQL databases

=head1 SYNOPSIS

  mysqlhotcopy db_name
  mysqlhotcopy db_name new_db_name
  mysqlhotcopy db_name /path/to/new_directory

WARNING: THIS IS VERY MUCH A FIRST-CUT ALPHA. Comments/patched welcome.

=cut

# Documentation continued at end of file

my $VERSION = sprintf("%d.%02d", q$Revision: 1.1 $ =~ /(\d+)\.(\d+)/o);

my $OPTIONS = <<"_OPTIONS";

Usage: $0 db_name new_db_name

  -?, --help           display this helpscreen and exit
  -u, --user=#         user for database login if not current user
  -p, --password=#     password to use when connecting to server
  -P, --port=#         port to use when connecting to local server

  --allowold           don't abort if target already exists (rename it _old)
  --keepold            don't delete previous (now renamed) target when done
  --noindices          don't copy index files
  --method=#           method for copy (only "cp" currently supported)

  --quiet              be silent except for errors
  --debug=N            set debug level to N (0..3)
_OPTIONS

sub usage {
    die @_, $OPTIONS;
}

my %opt = (
    user	=> getpwuid($>),
    indices	=> 1,	# for safety
    allowold	=> 0,	# for safety
    keepold	=> 0,
    method	=> "cp",
);
Getopt::Long::Configure(qw(no_ignore_case)); # disambuguate -p and -P
GetOptions( \%opt,
    "help",
    "user|u=s",
    "password|p=s",
    "port|P=s",
    "allowold!",
    "keepold!",
    "indices!",
    "datadir=s",
    "method=s",
    "debug",
    "quiet",
    "mv!",
) or usage("Invalid option");

my $src_dbname = $ARGV[0] or usage("Database name to hotcopy not specified");
my $tgt_name   = $ARGV[1] || "${src_dbname}_copy";

my $mysqld_help;
my %mysqld_vars;
my $start_time = time;
$0 = $1 if $0 =~ m:/([^/]+)$:;
$opt{quiet} = 0 if $opt{debug};
$opt{allowold} = 1 if $opt{keepold};

# --- connect to the database ---
my $dsn = "database=$src_dbname;host=localhost";
$dsn .= ";port=$opt{port}" if $opt{port};
my $dbh = DBI->connect("dbi:mysql:$dsn", $opt{user}, $opt{password}, {
    RaiseError => 1,
    PrintError => 0,
    AutoCommit => 1,
});


# --- get variables from database ---
my $sth_vars = $dbh->prepare("show variables");
$sth_vars->execute;
while ( my ($var,$value) = $sth_vars->fetchrow_array ) {
    $mysqld_vars{ $var } = $value;
}
my $datadir = $mysqld_vars{datadir}
    || die "datadir not in mysqld variables";
$datadir =~ s:/$::;


# --- get target path ---
my $tgt_dirname;
if ($tgt_name =~ m:^\w+$:) {
$tgt_dirname = "$datadir/$tgt_name";
}
elsif ($tgt_name =~ m:/:) {
    $tgt_dirname = $tgt_name;
}
else {
    die "Target '$tgt_name' doesn't look like a database name or directory path.\n";
}


# --- get src db directory and list of potential files to copy ---
my $src_dbdata_path = "$datadir/$src_dbname";
opendir(DBDIR, $src_dbdata_path)
    or die "Can't open datadir '$src_dbdata_path': $!\n";
my @mysqld_dbfiles = grep { /.+\.\w+$/ } readdir(DBDIR)
    or die "Can't find any tables in '$src_dbdata_path'\n";
closedir(DBDIR);
unless ($opt{indices}) {
    @mysqld_dbfiles = grep { not /\.(ISM|MYI)$/ } @mysqld_dbfiles;
}
printf "Found ".@mysqld_dbfiles." potential files to copy from $src_dbdata_path.\n"
	unless $opt{quiet};


# --- reprocess flat file list into a per-prefix (table) list ---
my %mysqld_dbfiles; # "foo" => [ "foo.frm", "foo.MYD", "foo.MYI" ]
foreach my $file (@mysqld_dbfiles) {
    next unless $file =~ m/^(.+)\.\w+$/;
    push @{ $mysqld_dbfiles{$1} ||= [] }, $file;
}


# --- get list of tables to hotcopy ---
my @dbh_tables = $dbh->func( '_ListTables' );
my @hc_tables = @dbh_tables;


# --- convert list of tables to list of files ---
my @hc_files;
foreach my $table (@hc_tables) {
    my $table_files = $mysqld_dbfiles{$table}
	    or die "Table '$table' doesn't seem to exist in $src_dbdata_path.\n";
    push @hc_files, @$table_files;
}


# --- get tgt db directory and check validity ---
my $tgt_dirpath = $tgt_dirname;
my $tgt_oldpath;	# undef unless it exists
if (-d $tgt_dirpath) {
    die "Can't hotcopy to '$tgt_dirpath' because it already exists and --allowold option not given.\n"
	    unless $opt{allowold};
    $tgt_oldpath = "${tgt_dirpath}_old";
    if (-d $tgt_oldpath) {
	print "Deleting previous 'old' hotcopy directory ('$tgt_oldpath')\n" unless $opt{quiet};
	rmtree([$tgt_oldpath])
    }
    rename($tgt_dirpath, $tgt_oldpath)
	or die "Can't rename $tgt_dirpath=>$tgt_oldpath: $!\n";
    print "Existing hotcopy directory renamed to '$tgt_oldpath'\n" unless $opt{quiet};
}
mkdir($tgt_dirpath, 0750)
	or die "Can't create '$tgt_dirpath': $!\n";
# convert to absolute path because we chdir() later
$tgt_dirpath = abs_path($tgt_dirpath);
print "Hotcopy destination '$tgt_dirpath'\n" unless $opt{quiet};


##############################
# --- PERFORM THE HOT-COPY ---
#
# Note that we try to keep the time between the LOCK and the UNLOCK
# as short as possible, and only start when we know that we should
# be able to complete without error.

chdir($src_dbdata_path)
	or die "Can't chdir($src_dbdata_path): $!\n";

# read lock all the tables we'll be copying
# in order to get a consistent snapshot of the database
my $hc_locks = join ", ", map { "$_ READ" } @hc_tables;
my $start = time;
$dbh->do("LOCK TABLES $hc_locks");
printf "Locked tables in %d seconds.\n", time-$start unless $opt{quiet};
my $hc_started = time;	# count from time lock is granted

# flush tables to make on-disk copy uptodate
$start = time;
$dbh->do("FLUSH TABLES");
printf "Flushed tables in %d seconds.\n", time-$start unless $opt{quiet};

eval {
    copy_files($opt{method}, \@hc_files, $tgt_dirpath);
};
my $failed = $@;

$dbh->do("UNLOCK TABLES");
my $hc_dur = time - $hc_started;
printf "Unlocked tables.\n" unless $opt{quiet};

#
# --- HOT-COPY COMPLETE ---
###########################

$dbh->disconnect;

if ( $failed ) {
    # hotcopy failed - cleanup
    # delete anything left lying around in $tgt_dirpath
    # rename _old copy back to original
    print "Deleting $tgt_dirpath\n" if $opt{debug};
    rmtree([$tgt_dirpath]);
    if ($tgt_oldpath && -e $tgt_oldpath) {
	print "Renaming $tgt_oldpath back to $tgt_dirpath\n" if $opt{debug};
	rename($tgt_oldpath, $tgt_dirpath)
	    or die "Can't rename $tgt_oldpath to $tgt_dirpath: $!\n";
    }

    die $failed;
}
else {
    # hotcopy worked
    # delete _old unless $opt{keepold}
    if ( !$opt{keepold} && $tgt_oldpath && -e $tgt_oldpath ) {
	print "Deleting previous copy in $tgt_oldpath\n" if $opt{debug};
	rmtree([$tgt_oldpath]);
    }

    printf "$0 copied %d tables (%d files) in %d second%s (%d seconds overall).\n",
	    scalar(@hc_tables), scalar(@hc_files),
	    $hc_dur, ($hc_dur==1)?"":"s", time - $start_time
	unless $opt{quiet};
}


exit 0;


# ---

sub copy_files {
    my ($method, $files, $target) = @_;
    my @cmd;
    print "Copying ".@$files." files...\n" unless $opt{quiet};
    if ($method =~ /^s?cp\b/) { # cp or scp with optional flags
	@cmd = ($method);
	# add option to preserve mod time etc of copied files
	# not critical, but nice to have
	push @cmd, "-p" if $^O =~ m/^(solaris)$/;
	# add files to copy and the destination directory
	push @cmd, @$files, $target;
    }
    else {
	die "Can't use unsupported method '$method'\n";
    }
    print "Executing '@cmd'\n" if $opt{debug};
    my $cp_status = system @cmd;
    if ($cp_status != 0) {
	die "Error: @cmd failed ($cp_status) while copying files from $src_dbdata_path.\n";
    }
}


__END__

=head1 DESCRIPTION

mysqlhotcopy is designed to make stable copies of live MySQL databases.

Here "live" means that the database server is running and the database
may be in active use. And "stable" means that the copy will not have
any corruptions that could occur if the table files were simply copied
without first being locked and flushed from within the server.

=head1 WARRANTY

This software is free and comes without warranty of any kind. You
should never trust backup software without studying the code yourself.
Study the code inside this script and only rely on it if I<you> believe
that it does the right thing for you.

Patches adding bug fixes, documentation and new features are welcome.

=head1 TO DO

Allow a list of tables (or regex) to be given on the command line to
enable a logically-related subset of the tables to be hot-copied
rather than force the whole db to be copied in one go.

Extend the above to allow multiple subsets of tables to be specified
on the command line:

  mysqlhotcopy db newdb  t1 t2 /^foo_/ : t3 /^bar_/ : +

where ":" delimits the subsets, the /^foo_/ indicates all tables
with names begining with "foo_" and the "+" indicates all tables
not copied by the previous subsets.

Add option to lock each table in turn for people who don't need
cross-table integrity.

Add option to FLUSH STATUS and option to FLUSH LOGS just before
UNLOCK TABLES. Combining FLUSH LOGS with a hotcopy of an
entire database may be useful for some backup+rollforward schemes.

Add support for other copy methods (eg tar to single file?).

Add support for forthcoming MySQL ``RAID'' table subdirectory layouts.

=head1 AUTHOR

Tim Bunce

=cut
