;;; ldap.el --- Client interface to LDAP for Emacs

;; Copyright (C) 1998 Free Software Foundation, Inc.

;; Author: Oscar Figueiredo <Oscar.Figueiredo@di.epfl.ch>
;; Maintainer: Oscar Figueiredo <Oscar.Figueiredo@di.epfl.ch>
;; Created: April 1998
;; Version: $Revision: 1.6 $
;; Keywords: comm

;; This file is not part of GNU Emacs

;; GNU Emacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to 
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:
;;    This package provides basic functionality to perform searches on LDAP
;;    servers.  It uses the `ldapsearch' utility that can be found either in:
;;      - UMich LDAP 3.3 Library (http://www.umich.edu/~dirsvcs/ldap/)
;;      - OpenLDAP 1.0.3 (http://www.openldap.org/)
;;      - Netscape's LDAP SDK (http://developer.netscape.com)

(require 'custom)

(defgroup ldap nil
  "Lightweight Directory Access Protocol"
  :group 'comm)

(defcustom ldap-default-host nil
  "*Default LDAP server."
  :type '(choice (string :tag "Host name")
		 (const :tag "Use library default" nil))
  :group 'ldap)

(defcustom ldap-default-port nil
  "*Default TCP port for LDAP connections.
Initialized from the LDAP library at build time. Default value is 389."
  :type '(choice (const :tag "Use library default" nil)
		 (integer :tag "Port number"))
  :group 'ldap)

(defcustom ldap-default-base nil
  "*Default base for LDAP searches.
This is a string using the syntax of RFC 1779.
For instance, \"o=ACME, c=US\" limits the search to the
Acme organization in the United States."
  :type '(choice (const :tag "Use library default" nil)
		 (string :tag "Search base"))
  :group 'ldap)


(defcustom ldap-host-parameters-alist nil
  "*Alist of host-specific options for LDAP transactions.
The format of each list element is:
\(HOST PROP1 VAL1 PROP2 VAL2 ...)
HOST is the name of an LDAP server. PROPn and VALn are property/value 
pairs describing parameters for the server.  Valid properties include: 
  `binddn' is the distinguished name of the user to bind as 
    (in RFC 1779 syntax).
  `passwd' is the password to use for simple authentication.
  `auth' is the authentication method to use. 
    Possible values are: `simple', `krbv41' and `krbv42'.
  `base' is the base for the search as described in RFC 1779.
  `scope' is one of the three symbols `subtree', `base' or `onelevel'.
  `deref' is one of the symbols `never', `always', `search' or `find'.
  `timelimit' is the timeout limit for the connection in seconds.
  `sizelimit' is the maximum number of matches to return."
  :type '(repeat :menu-tag "Host parameters"
		 :tag "Host parameters"
		 (list :menu-tag "Host parameters"
		       :tag "Host parameters"
		       :value nil
		       (string :tag "Host name")
		       (checklist :inline t
				  :greedy t
				  (list
				   :tag "Search Base" 
				   :inline t
				   (const :tag "Search Base" base)
				   string)
				  (list
				   :tag "Binding DN"
				   :inline t
				   (const :tag "Binding DN" binddn)
				   string)
				  (list
				   :tag "Password"
				   :inline t
				   (const :tag "Password" passwd)
				   string)
				  (list
				   :tag "Authentication Method"
				   :inline t
				   (const :tag "Authentication Method" auth)
				   (choice
				    (const :menu-tag "None" :tag "None" nil)
				    (const :menu-tag "Simple" :tag "Simple" simple)
				    (const :menu-tag "Kerberos 4.1" :tag "Kerberos 4.1" krbv41)
				    (const :menu-tag "Kerberos 4.2" :tag "Kerberos 4.2" krbv42)))
				  (list
				   :tag "Search Base" 
				   :inline t
				   (const :tag "Search Base" base)
				   string)
				  (list
				   :tag "Search Scope" 
				   :inline t
				   (const :tag "Search Scope" scope)
				   (choice
				    (const :menu-tag "Default" :tag "Default" nil)
				    (const :menu-tag "Subtree" :tag "Subtree" subtree)
				    (const :menu-tag "Base" :tag "Base" base)
				    (const :menu-tag "One Level" :tag "One Level" onelevel)))
				  (list
				   :tag "Dereferencing"
				   :inline t
				   (const :tag "Dereferencing" deref)
				   (choice
				    (const :menu-tag "Default" :tag "Default" nil)
				    (const :menu-tag "Never" :tag "Never" never)
				    (const :menu-tag "Always" :tag "Always" always)
				    (const :menu-tag "When searching" :tag "When searching" search)
				    (const :menu-tag "When locating base" :tag "When locating base" find)))
				  (list
				   :tag "Time Limit"
				   :inline t
				   (const :tag "Time Limit" timelimit)
				   (integer :tag "(in seconds)"))
				  (list
				   :tag "Size Limit"
				   :inline t
				   (const :tag "Size Limit" sizelimit)
				   (integer :tag "(number of records)")))))
:group 'ldap)

(defcustom ldap-ldapsearch-prog "ldapsearch"
  "*The name of the ldapsearch command line program"
  :type '(string :tag "`ldapsearch' Program")
  :group 'ldap)

(defcustom ldap-ldapsearch-args nil
  "*A list of additional arguments to pass to `ldapsearch'.
It is recommended to use the `-T' switch with Nescape's
implementation to avoid line wrapping."
  :type '(repeat :tag "`ldapsearch' Arguments"
		 (string :tag "Argument"))
  :group 'ldap)

(defun ldap-search (filter &optional host attributes attrsonly)
  "Perform an LDAP search.
FILTER is the search filter in RFC1558 syntax
HOST is the LDAP host on which to perform the search
ATTRIBUTES is the specific attributes to retrieve, nil means 
retrieve all
ATTRSONLY if non nil retrieves the attributes only without 
the associated values.
Additional search parameters can be specified through 
`ldap-host-parameters-alist' which see."
  (interactive "sFilter:")
  (let (host-alist)
    (if (null host)
	(setq host ldap-default-host))
    (if (null host)
	(error "No LDAP host specified"))
    (setq host-alist
	  (assoc host ldap-host-parameters-alist))
    (ldap-search-internal (append
			   (list 'host host
				 'filter filter
				 'attributes attributes 
				 'attrsonly attrsonly)
			   (cdr host-alist)))))


(defun ldap-search-internal (search-plist)
  "Perform a search on a LDAP server.
SEARCH-PLIST is a property list describing the search request.
Valid keys in that list are:
  `host' is a string naming one or more (blank separated) LDAP servers to
to try to connect to. Each host name may optionally be of the form host:port.
  `filter' is a filter string for the search as described in RFC 1558
  `attributes' is a list of strings indicating which attributes to retrieve
for each matching entry. If nil return all available attributes.
  `attrsonly' if non-nil indicates that only the attributes are retrieved, not
the associated values.
  `base' is the base for the search as described in RFC 1779.
  `scope' is one of the three symbols `sub', `base' or `one'.
  `binddn' is the distinguished name of the user to bind as (in RFC 1779 syntax).
  `passwd' is the password to use for simple authentication.
  `deref' is one of the symbols `never', `always', `search' or `find'.
  `timelimit' is the timeout limit for the connection in seconds.
  `sizelimit' is the maximum number of matches to return.
The function returns a list of matching entries.  Each entry is itself
an alist of attribute/values."

  (let ((buf (get-buffer-create "*ldap-search*"))
	(host (or (plist-get search-plist 'host)
		  ldap-default-host))
	(filter (plist-get search-plist 'filter))
	(attributes (plist-get search-plist 'attributes))
	(attrsonly (plist-get search-plist 'attrsonly))
	(base (or (plist-get search-plist 'base)
		  ldap-default-base))
	(scope (plist-get search-plist 'scope))
	(binddn (plist-get search-plist 'binddn))
	(passwd (plist-get search-plist 'passwd))
	(deref (plist-get search-plist 'deref))
	(timelimit (plist-get search-plist 'timelimit))
	(sizelimit (plist-get search-plist 'sizelimit))
	(numres 0)
	arglist record result)
    (if (or (null filter)
	    (equal "" filter))
	(error "No search filter"))
    (setq filter (cons filter attributes))
    (save-excursion
      (set-buffer buf)
      (erase-buffer)
      (if (and host
	       (not (equal "" host)))
	  (setq arglist (nconc arglist (list (format "-h%s" host)))))
      (if (and attrsonly
	       (not (equal "" attrsonly)))
	  (setq arglist (nconc arglist (list "-A"))))
      (if (and base
	       (not (equal "" base)))
	  (setq arglist (nconc arglist (list (format "-b%s" base)))))
      (if (and scope
	       (not (equal "" scope)))
	  (setq arglist (nconc arglist (list (format "-s%s" scope)))))
      (if (and binddn
	       (not (equal "" binddn)))
	  (setq arglist (nconc arglist (list (format "-D%s" binddn)))))
      (if (and passwd
	       (not (equal "" passwd)))
	  (setq arglist (nconc arglist (list (format "-w%s" passwd)))))
      (if (and deref
	       (not (equal "" deref)))
	  (setq arglist (nconc arglist (list (format "-a%s" deref)))))
      (if (and timelimit
	       (not (equal "" timelimit)))
	  (setq arglist (nconc arglist (list (format "-l%s" timelimit)))))
      (if (and sizelimit
	       (not (equal "" sizelimit)))
	  (setq arglist (nconc arglist (list (format "-z%s" sizelimit)))))
      (eval `(call-process ldap-ldapsearch-prog
			   nil
			   buf
			   nil	  
			   ,@arglist
			   ,@ldap-ldapsearch-args
			   ,@filter))
      (insert "\n")
      (goto-char (point-min))
      (skip-chars-forward " \t\n")
      (if (looking-at "usage")
	  (error "Incorrect ldapsearch invocation")
	(message "Parsing results... ")
	(while (equal 0 (forward-line 1))
	  (while (looking-at "^\\(\\w*\\)[=:\t ]+\\(.*\\)$")
	    (setq record (cons (list (match-string 1)
				     (match-string 2))
			       record))
	    (forward-line 1))
	  (setq result (cons (nreverse record) result))
	  (setq record nil)
	  (skip-chars-forward " \t\n")      
	  (message "Parsing results... %d" numres)
	  (1+ numres))
	(message "Parsing results... done")
	(nreverse result)))))


(provide 'ldap)

;;; ldap.el ends here
