/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                      
 *		  	  Centro SERRA, University of Pisa
 *		 	  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/*
 * Copyright (c) 1994, 1996
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include "ntop.h"


#define DONT_USE_FORK 1
/* #define DEBUG */


/* Extern */
extern int inet_aton (const char*, struct in_addr*);
extern char* intoa(struct in_addr addr);
extern void loadPlugins();
extern void unloadPlugins();
extern void addDefaultAdminUser();
extern void init_counters();
extern void cleanup_curses();
extern int  checkKeyPressed();
extern void returnHTTPaccessDenied();
extern int  checkHTTPpassword(char *pw);
extern void printTCPSessions();
extern void printUDPSessions();
extern void printIPpkt();
extern void printTCPpkt();
extern void handleHTTPrequest();
extern void clrscr();
extern void scanTimedoutTCPSessions();
extern void initIPServices();
extern RETSIGTYPE _printHostsTraffic(int signumber_ignored);
extern RETSIGTYPE printHostsTraffic(int signumber_ignored, int reportType, 
				    int sortedColumn, int revertOrder);
extern time_t nextSessionTimeoutScan;
extern char *optarg, version[];
extern ServiceEntry *udpSvc[SERVICE_HASH_SIZE], *tcpSvc[SERVICE_HASH_SIZE];
extern void updateThpt();
extern void LogStatsToFile();
extern void printLogHeader();
extern char* formatPkts(TrafficCounter pktNr);
extern void handleLocalAddresses(char* addresses);
extern void resetStats();
#ifdef MULTITHREADED
void* updateDBHostsTrafficLoop(void* notUsed);
#endif

/* *** SQL Engine *** */
extern void openSQLsocket(char* dstHost, int dstPort);
extern void closeSQLsocket();
extern void updateDbHostsTraffic();
/* ****************** */

#ifdef SUN_OS_4_1_3
extern int optind;
#endif

extern void handleFlowsSpecs(char* flows);
extern void init_curses();
#ifdef MULTITHREADED
extern void* dequeuePacket(void*);
#ifdef ASYNC_ADDRESS_RESOLUTION
extern void* dequeueAddress(void*);
#endif
extern void queuePacket(u_char *unused, const struct pcap_pkthdr *h,
			const u_char *p);
extern void* handleCursesRefresh(void* notUsed);
#endif
#ifdef HAVE_LSOF 
extern void readLsofInfo();
#endif
extern void readNepedInfo();
extern void getLocalHostAddress(struct in_addr *hostAddress, char* device);
#ifdef WIN32
extern short isWinNT();
extern int SIZE_BUF;
#endif
extern char domainName[];

/* Globals */
char *program_name, *device, deviceName[64];
u_int maxNumLines;
time_t actTime, initialSniffTime, lastRefreshTime;
int datalink, logFlag=0, showProtoWarnings=0, enableDBsupport=0;
#ifdef DEBUG
int numChildren;
#endif
short printingUsage, useLsof;
pcap_t *pcapPtr;
char *protoIPTrafficInfos[MAX_NUM_HANDLED_IP_PROTOCOLS]; /* array 0-numIpProtosToMonitor */
u_short numIpProtosToMonitor=0, numIpPortsToHandle=0, updateLsof;
int* ipPortMapper;
char* rFileName=NULL;
#ifdef MULTITHREADED
pthread_mutex_t     packetQueueMutex, hostsHashMutex; 
#ifdef HAVE_LSOF
pthread_mutex_t     lsofMutex;
#endif
pthread_t           dequeueThreadId, handleWebConnectionsThreadId, 
  cursesRefreshThreadId, thptUpdateThreadId, 
  scanIdleThreadId, logFileLoopThreadId,
  dbUpdateThreadId;

#ifdef HAVE_LSOF
pthread_t lsofThreadId;
#endif
extern PacketInformation packetQueue[PACKET_QUEUE_LENGTH];
#ifdef ASYNC_ADDRESS_RESOLUTION
extern struct hnamemem   *addressQueue[ADDRESS_QUEUE_LENGTH];
#endif
extern unsigned int packetQueueLen, maxPacketQueueLen, packetQueueHead, packetQueueTail;
TrafficCounter droppedPackets;
#ifdef USE_SEMAPHORES
sem_t queueSem;
#ifdef ASYNC_ADDRESS_RESOLUTION
sem_t queueAddressSem;
#endif
#else
ConditionalVariable queueCondvar;
#ifdef ASYNC_ADDRESS_RESOLUTION
ConditionalVariable queueAddressCondvar;
#endif
#endif
#ifdef ASYNC_ADDRESS_RESOLUTION
pthread_t dequeueAddressThreadId;
extern unsigned int addressQueueLen, maxAddressQueueLen, addressQueueHead, addressQueueTail;
TrafficCounter droppedAddresses;
pthread_mutex_t addressQueueMutex;
#endif
#endif

/* Libwrap support courtesy of 
   Georg Schwarz
   <schwarz@itp4.physik.TU-Berlin.DE> */
#ifdef HAVE_LIBWRAP
int allow_severity = LOG_INFO;
int deny_severity = LOG_WARNING;
#endif /* HAVE_LIBWRAP */

#ifdef HAVE_GDBM_H
GDBM_FILE gdbm_file, pwFile;
#ifdef MULTITHREADED
pthread_mutex_t gdbmMutex;
#endif
#endif

/* Locals */
static int sock;

/* Forwards */
void handleDbSupport(char* addr);
RETSIGTYPE cleanup(int);
RETSIGTYPE handleDiedChild(int);
RETSIGTYPE dontFreeze(int);
void detachFromTerminal();
void daemonize();
void usage(int);
void handleProtocols(char*);
void addDefaultProtocols();
short handleProtocol(char* protoName, char *protocol);
void handleProtocolList(char* protoName, char *protocolList);
void* handleWebConnections(void*);
#ifdef MULTITHREADED
void* updateThptLoop(void* notUsed);
void* scanIdleLoop(void* notUsed);
void* logFileLoop(void* notUsed);
#ifdef HAVE_LSOF
void* periodicLsofLoop(void* notUsed);
#endif
#endif

/* Global */
unsigned int localnet, netmask;
struct in_addr localHostAddress;
int numericFlag, refreshRate, localAddrFlag, percentMode;
int idleFlag, webMode, newSock, logTimeout;
time_t nextLogTime=0;
FILE *logd;
FlowFilterList *flowsList;

/* *************************** */

#ifndef WIN32
void handleSigHup(int signalId) {
  if(webMode)
    printf("Caught sighup: statistics have been reset.\n");
  resetStats();
  (void)setsignal(SIGHUP,  handleSigHup);
}
 
/* *************************** */

void ignoreSignal(int signalId) {
  close(newSock);
#ifndef DONT_USE_FORK
  exit(0);
#endif
}
 
#endif /* WIN32 */

/* *************************** */


int main(int argc, char *argv[])
{
  int pflag, rc, webPort, daemonMode, i;
#ifdef WIN32
  int optind=0;
#endif
  int op;
  char *cp, *expression, *localAddresses=NULL, *webAddr=NULL;
#ifndef WIN32
  RETSIGTYPE (*oldhandler)(int); 
#endif
  char ebuf[PCAP_ERRBUF_SIZE], flowSpecs[2048];
  struct sockaddr_in sin;
  time_t lastTime;
  struct bpf_program fcode;

#ifdef HAVE_LSOF
  useLsof = 1;
#endif

  flowSpecs[0] = '\0';
  flowsList = NULL;

  localAddrFlag = 1;
  logTimeout = 0;

  device = NULL;

  daemonMode = 0, pflag = 0, numericFlag=0, percentMode = 0;
  refreshRate = 0, idleFlag = 1, webMode = 0, logd = NULL;
  printingUsage = 0;
  rFileName=NULL;

  domainName[0] = '\0';

  actTime = time(NULL);

  if ((cp = strrchr(argv[0], '/')) != NULL)
    program_name = cp + 1;
  else
    program_name = argv[0];

  memset(protoIPTrafficInfos, 0, sizeof(protoIPTrafficInfos));
  ipPortMapper = (int*)malloc(sizeof(int)*TOP_IP_PORT);

#ifdef WIN32
  initWinsock32();
#endif

  for(i=0; i<TOP_IP_PORT; i++) ipPortMapper[i] = -1;

  initIPServices();

  maxNumLines = MAX_NUM_TABLE_ROWS;

#ifdef WIN32
  while ((op = getopt(argc, argv, "e:F:hr:p:l:nw:m:b:B:D:")) != EOF)
#else
  while ((op = getopt(argc, argv, "de:f:F:hr:i:p:l:nw:m:b:D:")) != EOF)
#endif
	  switch (op) {

#ifndef WIN32
    case 'd':
	daemonMode=1;
      break;
#endif

#ifdef WIN32
    case 'B':
	SIZE_BUF=atoi(optarg)*1024;
    break;
#endif

    case 'b': /* host:port */
      handleDbSupport(optarg);
      break;

    case 'D': /* domain */
      strcpy(domainName, optarg);
      break;

    case 'f':
#ifdef HAVE_LSOF
      useLsof = 0; /* Don't make debugging too complex */
#endif
      rFileName = optarg;
      break;

    case 'r':
      if(!isdigit(optarg[0])) {
		printf("FATAL ERROR: flag -r expects a numeric argument.\n");
		exit(-1);
      }
      refreshRate = atoi(optarg);
      break;

    case 'e':
      maxNumLines = atoi(optarg);
      break;

#ifndef WIN32
    case 'i':
      device = optarg;
      break;
#endif

    case 'p':
      showProtoWarnings = 1;
      handleProtocols(optarg);
      break;

    case 'F':
      strcpy(flowSpecs, optarg); 
      break;

    case 'm':
      localAddresses = strdup(optarg);
      break;

    case 'l':
      if(!isdigit(optarg[0])) {
		printf("FATAL ERROR: flag -l expects a numeric argument.\n");
		exit(-1);
      }

      logTimeout = atoi(optarg);

      if(logTimeout < 0) {
		printf("Log period out of range: set to default (30 mins)\n");
		logTimeout = 1800; /* seconds */
      }

      logd = fopen("ntop.log", "w+");
      if(logd == NULL) {
		fprintf(stderr, "Logging disabled: unable to log onto file ntop.log.\n");
		logTimeout = 0;
      }
      break;
      
    case 'n':
      numericFlag++;
      break;

    case 'w':
      if(!isdigit(optarg[0])) {
	printf("FATAL ERROR: flag -w expects a numeric argument.\n");
	exit(-1);
      }
      webMode++;

      /* Courtesy of Daniel Savard <daniel.savard@gespro.com> */
      if ((webAddr = strchr(optarg,':'))) {
	/* DS: Search for : to find xxx.xxx.xxx.xxx:port */
	*webAddr = '\0';  /* This code is to be able to bind to a particular interface */
	webPort = atoi(webAddr+1);
	webAddr = optarg;
      } else
         webPort = atoi(optarg);
      break;

    default:
      usage(0);
      exit(-1);
      /* NOTREACHED */
    }

#ifndef HAVE_CURSES
  if(webMode == 0) {
    webMode++;
    webPort = 3000; /* default port */
  }
#endif

  if(maxNumLines == 0)
    maxNumLines = MAX_NUM_TABLE_ROWS;

#ifndef WIN32
  if((rFileName == NULL) 
     && ((getuid() && geteuid()) || (setuid(0)))) {
    printf("Sorry, you must be superuser in order to run ntop.\n");
    return(-1);
  }
  
  if(daemonMode && (!webMode)) {
    printf("WARNING: -d is incompatible with interactive mode.\n");
  } else if(daemonMode)
    daemonize();
#endif

  if(webMode) {
    /* 
       Make sure that the other flags are't set. They have
       no effect in web mode 
    */
    pflag = 0;
    percentMode = 0;

    if(refreshRate == 0)
      refreshRate = REFRESH_TIME;
    else if(refreshRate < MIN_REFRESH_TIME)
      refreshRate = MIN_REFRESH_TIME;
  } else {
    useLsof = 0; /* Don't use lsof for the moment */
    if(refreshRate == 0)
      refreshRate = ALARM_TIME;
    else if(refreshRate < MIN_ALARM_TIME)
      refreshRate = MIN_ALARM_TIME;
  }

  if(numIpProtosToMonitor == 0)
    addDefaultProtocols();

  if(logd != NULL)
    printLogHeader();

  lastTime = time(NULL) + refreshRate;

#ifndef MULTITHREADED
  if(logTimeout != 0)
    nextLogTime = time(NULL) + logTimeout;
#endif

#ifdef HAVE_GDBM_H
#ifdef MULTITHREADED
  createMutex(&gdbmMutex);
#endif
  gdbm_file = gdbm_open ("ntop.db", 0, GDBM_WRCREAT, 00664, NULL);

  if(gdbm_file == NULL) {
    printf("Database open failed: %s\n", gdbm_strerror(gdbm_errno));
    exit(-1);    
  } else {
    /* Let's remove from the database entries that were not
       yet resolved in (i.e. those such as "*132.23.45.2*")
    */
    datum data_data, key_data, return_data = gdbm_firstkey (gdbm_file);
    u_int numDbEntries = 0;
    
    while (return_data.dptr != NULL) {
      numDbEntries++;
      key_data = return_data;
      return_data = gdbm_nextkey(gdbm_file, key_data);
      data_data = gdbm_fetch(gdbm_file, key_data);
      if((data_data.dptr != NULL) && (data_data.dptr[0] == '*')) {
		gdbm_delete(gdbm_file, key_data);
		printf("Deleted '%s' entry.\n", data_data.dptr);
		numDbEntries--;
      }	

      if(data_data.dptr != NULL) free(data_data.dptr);
      free(key_data.dptr);
    }

    pwFile = gdbm_open ("ntop_pw.db", 0, GDBM_WRCREAT, 00664, NULL);
    
    if(pwFile == NULL) {
      printf("FATAL ERROR: Database 'ntop_pw.db' cannot be opened.");
      exit(-1);
    } else
      /* ******************************************* */
      addDefaultAdminUser();

#ifdef DEBUG
    printf("The ntop.db database contains %d entries.\n", numDbEntries);
#endif
  }
#endif

#ifdef MULTITHREADED
  packetQueueLen = maxPacketQueueLen = packetQueueHead = packetQueueTail = 0;
  droppedPackets = 0;
#ifdef USE_SEMAPHORES
  createSem(&queueSem, 0);
#ifdef ASYNC_ADDRESS_RESOLUTION
  createSem(&queueAddressSem, 0);
#endif
#else
  createCondvar(&queueCondvar);
#ifdef ASYNC_ADDRESS_RESOLUTION
  createCondvar(&queueAddressCondvar);
#endif
#endif
  createMutex(&packetQueueMutex);
#ifdef HAVE_LSOF
  if(useLsof)
    createMutex(&lsofMutex);
#endif
  createMutex(&hostsHashMutex);
  createThread(&dequeueThreadId, dequeuePacket);
  createThread(&thptUpdateThreadId, updateThptLoop);
  createThread(&scanIdleThreadId, scanIdleLoop);
  if(enableDBsupport)
    createThread(&dbUpdateThreadId, updateDBHostsTrafficLoop);

  if(logTimeout != 0)
    createThread(&logFileLoopThreadId, logFileLoop);

#ifdef ASYNC_ADDRESS_RESOLUTION
  addressQueueLen = maxAddressQueueLen = addressQueueHead = addressQueueTail = 0;
  droppedAddresses = 0;
  memset(addressQueue, 0, sizeof(addressQueue));
  createMutex(&addressQueueMutex);
  createThread(&dequeueAddressThreadId, dequeueAddress);
#endif
#endif

  if(!webMode) {
#ifdef MULTITHREADED
#ifdef HAVE_CURSES
    createThread(&cursesRefreshThreadId, handleCursesRefresh);
#endif
#else
#ifndef WIN32
    (void)setsignal(SIGALRM, _printHostsTraffic); 
#endif
#endif
  } else {
#ifndef WIN32
    (void)setsignal(SIGALRM, dontFreeze);
#endif
    /* 
       The handler below has been restored due to compatibility
       problems with om: 
       Courtesy of Martin Lucina <mato@kotelna.sk>
    */
#ifndef WIN32
    (void)setsignal(SIGCHLD, handleDiedChild);
    /* signal(SIGCHLD, SIG_IGN); */
#endif
  }  

#ifdef HAVE_LSOF 
  if(useLsof) {
#ifdef MULTITHREADED
    updateLsof = 1;
    createThread(&lsofThreadId, periodicLsofLoop);
#else
    readLsofInfo(); 
    readNepedInfo();
#endif
  }
#endif


/* ******************************************* */

  /* Determine the device name if not specified */
  ebuf[0] = '\0';
  if ((device == NULL)
	  && (device = pcap_lookupdev(ebuf)) == NULL) {
    printf(ebuf);
    return(-1);
  }


#ifdef WIN32
	if(isWinNT())
	{
		static char tmpString[128];
		int i, j;

		for(j=0, i=0; !((device[i] == 0) && (device[i+1] == 0)); i++) {
			if(device[i] != 0)
				tmpString[j++] = device[i];
		}

		tmpString[j++] = 0;
		device = tmpString;
	}
#endif

/* ******************************************* */
	
  sprintf(deviceName, device);
  getLocalHostAddress(&localHostAddress, device);

  if(rFileName == NULL) {
    if(webMode) {
      (void)fprintf(stderr, "ntop v.%s %s [%s]"
		    " listening on %s"
		    ".\n", 
		    version, THREAD_MODE, osName, 
		    deviceName);
      (void)fprintf(stderr, "Copyright 1998-99 by %s\n", author);
      (void)fprintf(stderr, "Initializing...\n");
      (void)fflush(stderr);
    }
#ifdef HAVE_CURSES
    else {
      init_curses();
    }
#endif

    /* Fire up libpcap */
    pcapPtr = pcap_open_live(device, DEFAULT_SNAPLEN, !pflag, 1000, ebuf);
  } else {
    pcapPtr = pcap_open_offline(rFileName, ebuf);
	deviceName[0] = '\0';
  }

  if (pcapPtr == NULL) {
    printf(ebuf);
    return(-1);
  }

  if (pcap_lookupnet(device, &localnet, &netmask, ebuf) < 0) {
    printf(ebuf);
    return(-1);
  }

#ifndef WIN32
  localnet = htonl(localnet);
  netmask = htonl(netmask);
#else
  /* This looks like a wIN32 libpcap open issue... */
  {
	struct hostent* h;
	char szBuff[80];

    gethostname(szBuff, 79);
	h=gethostbyname(szBuff);
	localnet = ((h->h_addr_list[0][0]<<24))+
			   ((h->h_addr_list[0][1]<<16))+
			   ((h->h_addr_list[0][2]<<8));
  }
#endif

  /* Sanity check... */
  if((localHostAddress.s_addr & netmask) != localnet) {
   struct in_addr addr1;
 
	printf("WARNING: your IP address (%s), ",
	       intoa(localHostAddress));
    addr1.s_addr = netmask;
    printf("netmask %s", intoa(addr1));
    addr1.s_addr = localnet;
    printf(", network %s\ndo not match.\n", intoa(addr1));
    localnet = localHostAddress.s_addr & netmask;
    printf("ntop will use: IP address (%s), ",
	   intoa(localHostAddress));
    addr1.s_addr = netmask;
	printf("netmask %s", intoa(addr1));
    addr1.s_addr = localnet;
 	printf(", network %s.\n", intoa(addr1));
	localnet = localHostAddress.s_addr & netmask;
  }

  if(localAddresses != NULL) {
    handleLocalAddresses(localAddresses);
    free(localAddresses);
    localAddresses = NULL;
  }

#ifndef WIN32
  /* Now that the socket is open, throw away potential setuid/setgid */
  (void)setgid(getgid());
  (void)setuid(getuid());
#endif

  /* get datalink type */
#ifdef AIX
  /* This is a bug of libpcap on AIX */
  switch(deviceName[0]) {
  case 't': /* TokenRing */
    datalink = DLT_IEEE802;
    break;
  case 'l': /* Loopback */
    datalink = DLT_NULL;
    break;
  default:
    datalink = DLT_EN10MB; /* Ethernet */
  }
#else

#if defined(__FreeBSD__)
  if(strncmp(deviceName, "tun", 3) == 0)
    datalink = DLT_PPP;
  else
    datalink = pcap_datalink(pcapPtr);
#else
  datalink = pcap_datalink(pcapPtr);
#endif

  /*
    datalink = DLT_PPP; 
    printf("type=%d\n", datalink);
  */

  /* Construct, compile and set filter */
  if(optind > 0) {
    expression = copy_argv(argv + optind);
    if(expression != NULL) {
      if((pcap_compile(pcapPtr, &fcode, expression, 1, netmask) < 0)
	 || (pcap_setfilter(pcapPtr, &fcode) < 0)) {
	printf("FATAL ERROR: wrong filter '%s' (%s)\n", expression, pcap_geterr(pcapPtr));
	return(-1);
      }
    }
  }
#endif

  /* Handle flows (if any) */
  if(flowSpecs[0] != '\0')
    handleFlowsSpecs(flowSpecs);

#ifndef WIN32
  /* Setup signal handlers */
  (void)setsignal(SIGTERM, cleanup);
  (void)setsignal(SIGINT,  cleanup);
  (void)setsignal(SIGHUP,  handleSigHup);

  if(webMode) {
    (void)setsignal(SIGPIPE, ignoreSignal); 
  }

  /* Cooperate with nohup(1) */
  if ((oldhandler = setsignal(SIGHUP, cleanup)) != SIG_DFL)
    (void)setsignal(SIGHUP, oldhandler);
#endif


  loadPlugins();

  init_counters();

  if(!webMode)
    printHostsTraffic(-1, 0, 0, 0);
  else {
    int sockopt = 1;

#ifdef DEBUG
    numChildren = 0;
#endif    
    sin.sin_family      = AF_INET;
    sin.sin_port        = (int)htons(webPort);
    sin.sin_addr.s_addr = INADDR_ANY;

    if (webAddr) {      /* Code added to be able to bind to a particular interface */
      if (!inet_aton(webAddr,&sin.sin_addr))
	printf("Unable to convert address '%s'...\n"
	       "Not binding to a particular interface!\n",  webAddr);
     }

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if(sock < 0)
      {
	printf("unable to create a new socket");
	return(-1);
      }

    setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&sockopt, sizeof(sockopt));

    if(bind(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
      printf("bind: port %d already in use.", webPort);
#ifndef WIN32
      close(sock);
#else
      closesocket(sock);
#endif     
      return(-1);
    } 
    
    if(listen(sock, 5) < 0) {
      printf("listen error.\n");
#ifndef WIN32
      close(sock);
#else
      closesocket(sock);
#endif     
      return(-1);
    } else {
      /* Courtesy of Daniel Savard <daniel.savard@gespro.com> */
      if (webAddr) {
	fprintf(stderr, "Waiting for HTTP connections on %s port %d...\n", 
		webAddr, webPort);
      } else {
	fprintf(stderr, "Waiting for HTTP connections on port %d...\n", 
		webPort);
      }
    }
  }

  if(webMode)
    printf("Sniffying...\n");

#ifdef MULTITHREADED
  if(webMode) 
    createThread(&handleWebConnectionsThreadId, handleWebConnections);
#endif

  initialSniffTime = lastRefreshTime = time(NULL);

  for(;;) {    
#ifndef MULTITHREADED
    short justRefreshed;
#endif

#ifdef MULTITHREADED
    rc = pcap_dispatch(pcapPtr, -1, queuePacket, NULL);   
#else
    rc = pcap_dispatch(pcapPtr, -1, pbuf_process, NULL);   
#endif

	if(rc == -1) {
		printf("Error while reading packets: %s.\n", pcap_geterr(pcapPtr));
		continue;
	}
    actTime = time(NULL);

#ifndef MULTITHREADED
    if(actTime > lastTime) {      
      if(!webMode) 
	printHostsTraffic(0, 0, 0, 0);
      else if(nextSessionTimeoutScan < actTime) {
	/* It's time to check for timeout sessions */
	scanTimedoutTCPSessions();
	nextSessionTimeoutScan = actTime+SESSION_SCAN_DELAY;
      }
      
      updateThpt(); /* Update Throughput */
      lastTime = actTime + THROUGHPUT_REFRESH_TIME;
      justRefreshed=1;
    } else
      justRefreshed=0;
#endif

    if(!webMode) {
#ifndef MULTITHREADED
      if(checkKeyPressed()) {
	if(!justRefreshed) {
	  printHostsTraffic(0, 0, 0, 0);
	  lastTime = actTime + refreshRate; /* refresh in 'refreshRate' seconds */
	}
      }
#endif
    } else {
      /* webMode */
#ifndef MULTITHREADED
      handleWebConnections(NULL);
#endif
    }

#ifndef MULTITHREADED
    if((logTimeout != 0) && (actTime > nextLogTime)) {
       LogStatsToFile();
      nextLogTime = actTime + logTimeout;
    }
#endif
  }/* for(;;) */ 
    
  printf("pcap_loop exited");
}

/* **************************************** */

void handleSingleWebConnection() {
  struct sockaddr_in from;
  int from_len = sizeof(from);
  newSock  = accept(sock, (struct sockaddr*)&from, &from_len);
	  
  if(newSock > 0) {
    int childpid;

#ifdef DONT_USE_FORK
    childpid = 0;
#else
    if((childpid=fork()) < 0) {
      printf("ntop: fork error [errno=%d]\n", errno);
      exit(0);
    } else
#endif
      {	    
	if(!childpid) {
	  /* webMode - child process */
#ifndef DONT_USE_FORK
	  /* There's no need to sniff anymore because 
	     ntop will quit after having served 
	     the HTTP client 
	  */
	  detachFromTerminal();
#endif /* DONT_USE_FORK */


#ifdef HAVE_LIBWRAP
	  {
	    struct request_info req;
	    request_init(&req, RQ_DAEMON, DAEMONNAME, RQ_FILE, newSock, NULL);
	    fromhost(&req);
	    if (!hosts_access(&req)) {
	      closelog(); /* just in case */
	      openlog(DAEMONNAME,LOG_PID,SYSLOG_FACILITY);
#ifdef DONT_USE_FORK
	      syslog(deny_severity, "refused connect from %s", eval_client(&req));
	    }
	    else
	      handleHTTPrequest();
#else
	    refuse(&req);
	  }
	  handleHTTPrequest();
#endif /* DONT_USE_FORK */
	}
#else
	handleHTTPrequest(); 
#endif /* HAVE_LIBWRAP */
	
#ifndef DONT_USE_FORK
	  pcap_close(pcapPtr);
#endif
#ifndef WIN32
	  close(newSock);
#else
	  closesocket(newSock);
#endif     
#ifndef DONT_USE_FORK
	  exit(0);
#endif
	} else {
	  /* webMode - parent process */
#ifdef DEBUG
	  int pid;
	  numChildren++;
	  printf("Forked a new child (pid=%d): %d\n", 
		 childpid, numChildren);
#endif
#ifndef WIN32
	  close(newSock);
#else
	  closesocket(newSock);
#endif     
	}
      }
  }
}


/* **************************************** */

void* handleWebConnections(void* notUsed) {
#ifndef MULTITHREADED  
  struct timeval wait_time;
#endif
  fd_set mask;

  FD_ZERO(&mask);
  FD_SET(sock, &mask);    

#ifndef MULTITHREADED  
  /* select returns immediately */
  wait_time.tv_sec = 0, wait_time.tv_usec = 0; 
  if(select(sock+1, &mask, 0, 0, &wait_time) == 1)
    handleSingleWebConnection();
#else
  while(1) {
    if(select(sock+1, &mask, 0, 0, NULL /* Infinite */) == 1)
      handleSingleWebConnection();
  }

#endif

  return(NULL); /* NOTREACHED */
}


/* **************************************** */
#ifndef WIN32
RETSIGTYPE handleDiedChild(int signal) {
  int status;
  pid_t pidId;

  pidId = waitpid(-1, &status, WNOHANG);

#ifdef DEBUG
  if(status == 0) {
    numChildren--;
    printf("A child has terminated [pid=%d status=%d children=%d]\n",
	   pidId, status, numChildren);
  }
#endif


  (void)setsignal(SIGCHLD, handleDiedChild);
}
#endif


/* **************************************** */

RETSIGTYPE dontFreeze(int signo)
{
#ifdef DEBUG
  printf("Caught a SIGALRM...\n");
#endif
  (void)setsignal(SIGALRM, dontFreeze);
}

/* **************************************** */

/* Report statistics and write out the raw packet file */
RETSIGTYPE cleanup(int signo)
{
  static int unloaded = 0;
  struct pcap_stat stat;

  if(unloaded)
    return;
  else
    unloaded = 1;

#ifdef MULTITHREADED
  /* Courtesy of Felipe Tonioli <tonioli@mtec.com.br> */
  if(signo != -1) /* the user pressed the 'q' key */
    {
      killThread(&dequeueThreadId);
      killThread(&thptUpdateThreadId);
      killThread(&scanIdleThreadId);
      if(enableDBsupport)
	killThread(&dbUpdateThreadId);

#ifdef HAVE_LSOF
      if(useLsof)
	killThread(&lsofThreadId);
#endif
#ifdef ASYNC_ADDRESS_RESOLUTION
      killThread(&dequeueAddressThreadId);
#endif

      if(!webMode) {
#ifdef HAVE_CURSES
	killThread(&cursesRefreshThreadId);
#endif
      } else
	killThread(&handleWebConnectionsThreadId);

      if(logTimeout != 0)
	killThread(&logFileLoopThreadId);

      deleteMutex(&packetQueueMutex);
      deleteMutex(&hostsHashMutex);
#ifdef HAVE_LSOF
      if(useLsof)
	deleteMutex(&lsofMutex);
#endif
#ifdef USE_SEMAPHORES
      deleteSem(&queueSem);
#ifdef ASYNC_ADDRESS_RESOLUTION
      deleteSem(&queueAddressSem);    
#endif
#else
      deleteCondvar(&queueCondvar);
#ifdef ASYNC_ADDRESS_RESOLUTION
      deleteCondvar(&queueAddressCondvar);    
#endif
#endif
    }
#endif

#ifdef HAVE_CURSES
  if(!webMode)
    cleanup_curses();
#endif

  (void)fflush(stdout);

  unloadPlugins();

  if (pcap_stats(pcapPtr, &stat) < 0)
    (void)fprintf(stderr, "\npcap_stats: %s\n", pcap_geterr(pcapPtr));
  else {
    (void)fprintf(stderr, "%s packets received by filter\n",
		  formatPkts((TrafficCounter)stat.ps_recv));
    (void)fprintf(stderr, "%s packets dropped by kernel\n",
		  formatPkts((TrafficCounter)(stat.ps_drop)));
#ifdef MULTITHREADED
    (void)fprintf(stderr, "%s packets dropped by ntop\n",
		  formatPkts(droppedPackets));
#endif
    if (logTimeout) {
      fclose(logd);
    }
  }
  
#ifndef WIN32
  endservent();
#endif

#ifdef HAVE_GDBM_H
  gdbm_close(gdbm_file);
  gdbm_close(pwFile);
#ifdef MULTITHREADED
  deleteMutex(&gdbmMutex);
#endif
#endif

  if(enableDBsupport)
    closeSQLsocket(); /* *** SQL Engine *** */

#ifdef WIN32
  termWinsock32();
#endif

  exit(0);
}

/* **************************************** */

void usage(int cursesMode)
{
  char buf[80];
  int row=0;

  printingUsage = 1;

#ifdef HAVE_CURSES
  if(cursesMode) {
    alarm(0); /* Avoid to be interrupted */
    clrscr(); /* clear the screen */  
  }
#endif

  sprintf(buf, "%s v.%s for %s", program_name, version, osName);
#ifdef HAVE_CURSES
  if(cursesMode) { 
    mvprintw (row++, 0, buf); 
  } else
#endif
    fprintf(stderr, "%s\n", buf);

  sprintf(buf, "Written by %s.", author);

#ifdef HAVE_CURSES
  if(cursesMode) { 
    mvprintw (row, 0, buf); 
    row+=2; 
  } else
#endif
    fprintf(stderr, "%s\n\n", buf);

  sprintf(buf, "Usage: %s", program_name);

  if(!cursesMode) fprintf(stderr, "%s\n", buf);

  if(!cursesMode) {
#ifdef WIN32
	  fprintf(stderr,"    [-r <refresh time (web = %d sec)>]\n", REFRESH_TIME);
#else
	  fprintf(stderr,"    [-r <refresh time (interactive = %d sec/web = %d sec)>]\n",
	    ALARM_TIME, REFRESH_TIME);
#endif
	fprintf(stderr,"    %s\n",   "[-f <traffic dump file (see tcpdump)>]");
    fprintf(stderr,"    %s\n",   "[-n (numeric IP addresses)]");
    fprintf(stderr,"    %s\n",   "[-p <IP protocols to monitor> (see man page)]");
#ifdef WIN32
    fprintf(stderr,"    %s%d Kb)>]\n", "[-B <NDIS buffer in Kbytes (default ", (int)(SIZE_BUF/1024));
#endif
#ifndef WIN32
    fprintf(stderr,"    %s\n",   "[-i <interface>]");
#endif
    fprintf(stderr,"    %s\n",   "[-w <HTTP port>]");  
    fprintf(stderr,"    %s\n",   "[-B <Internet domain name>]");  
    fprintf(stderr,"    %s\n",   "[-e <max # table rows"
#ifndef WIN32
		" (use only with -w)>"
#endif
		);  
#ifndef WIN32
    fprintf(stderr,"    %s\n",   "[-d (daemon mode (use only with -w))]");  
#endif
    fprintf(stderr,"    %s\n",   "[-m <local addresses (see man page)>]");  
    fprintf(stderr,"    %s\n",   "[-l <log period (seconds)>]");  
    fprintf(stderr,"    %s\n",   "[-F <flow specs (see man page)>]");  
    fprintf(stderr,"    %s\n",   "[-b <client:port (ntop DB client)>]");  
    fprintf(stderr,"    %s\n\n", "[ <filter expression (like tcpdump)>]");
  }

#ifdef HAVE_CURSES
  sprintf(buf, "While %s is running interactively, the following keys are active:", 
	  program_name);
  if(cursesMode) { 
    mvprintw (row++, 0, buf); 
  } else
    fprintf(stderr, "%s\n", buf);

  sprintf(buf, "'q' - quit ntop");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'r' - reset statistics");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'n' - toggle address format (num <-> sym <-> MAC <-> Nw Board Manufact.)");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'p' - toggle traffic values (bytes <-> %% <-> thpt)");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'l' - toggle hosts display (local subnet <-> all)");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'d' - toggle idle (idle <-> send/receive)");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'t' - toggle sort (sent <-> received)");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'y' - toggle columns sort");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "'h' - show this help");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  sprintf(buf, "' ' - toggle protocol");

  if(cursesMode) { 
    mvprintw (row++, 7, buf); 
  } else
    fprintf(stderr, "    %s\n", buf);

  if(cursesMode) { 
    mvprintw(++row, 0, "Strike a key to continue..."); 
    refresh();

    while(getchar() == EOF)
      ;

#ifndef MULTITHREADED
    alarm(refreshRate);
#endif
  }
#endif /* HAVE_CURSES */

  printingUsage = 0;
}

/* **************************************** */

#ifndef WIN32

void daemonize() {
  int childpid;

  (void)signal(SIGHUP, SIG_IGN);
  (void)signal(SIGCHLD, SIG_IGN);
  (void)signal(SIGQUIT, SIG_IGN);

  if((childpid=fork()) < 0)
    printf("An error occurred while daemonizing ntop...\n");
  else {
    if(!childpid) { /* child */
      printf("Bye bye: I'm becoming a daemon...\n");
      detachFromTerminal();
    } else { /* father */
      exit(0);
    }
  }
}


/* **************************************** */

void detachFromTerminal() {
  int fd;

#ifndef MULTITHREADED
  alarm(120); /* Don't freeze */
#else
  alarm(0);
#endif

  setsid();  /* detach from the terminal */

#ifdef ORIGINAL_DETACH
  if (freopen("/dev/null", "r", stdin) == NULL) {
    fprintf(stderr, "ntop: unable to replace stdin with /dev/null: %s\n",
	    strerror(errno));
    }

  if (freopen("/dev/null", "w", stdout) == NULL) {
    fprintf(stderr, "ntop: unable to replace stdout with /dev/null: %s\n",
	    strerror(errno));
  }

  if (freopen("/dev/null", "w", stderr) == NULL) {
    fprintf(stderr, "ntop: unable to replace stderr with /dev/null: %s\n",
	    strerror(errno));
  }
#else /* ORIGINAL_DETACH */
  for (fd = 3; fd < NOFILE; fd ++)
    close (fd);
  
  /*
   * clear any inherited file mode creation mask
   */
  umask (0);
  
  /*
   * Use linebuffered stdout
   */
  setlinebuf (stdout);
#endif /* ORIGINAL_DETACH */
}
#endif /* WIN32 */

/* **************************************** */

void handleDbSupport(char* addr /* host:port */) {
  char *hostName;
  int  portNumber;

  if((addr == NULL) || (addr[0] == '\0'))
    return;
  
  hostName = strtok(addr, ":");
  portNumber = atoi(strtok(NULL, ":"));

  if((hostName == NULL) || (portNumber == 0)) {
    printf("WARNING: invalid value specified for '-b' parameter. \n"
	   "         It should be host:port.\n");
    return;
  } else {
    enableDBsupport = 1;
    openSQLsocket(hostName, portNumber); /* *** SQL Engine *** */
  }
}

/* **************************************** */

void handleProtocols(char *protos) {
  char *proto = strtok(protos, ",");

  while(proto != NULL) {
    char* protoName = strchr(proto, '=');

    if(protoName == NULL)
      printf("Unknown protocol '%s'. It has been ignored.\n", proto);
    else {
      char tmpStr[255];
      int len;

      protoName[0] = '\0';
      memset(tmpStr, 0, sizeof(tmpStr));
      strcpy(tmpStr, &protoName[1]);
      len = strlen(tmpStr);

      if(tmpStr[len-1] != '|') {
	/* Make sure that the string ends with '|' */
	tmpStr[len] = '|';
	tmpStr[len+1] = '\0';
      }

      handleProtocolList(proto, tmpStr);
     
    }
    proto = strtok(NULL, ",");
  }
}

/* **************************************** */

void addDefaultProtocols() {
  handleProtocolList("FTP", "ftp|ftp-data|");
  handleProtocolList("HTTP", "http|www|https|");
  handleProtocolList("DNS", "name|domain|");
  handleProtocolList("Telnet", "telnet|login|");
  handleProtocolList("NBios-IP", "netbios-ns|netbios-dgm|netbios-ssn|");
  handleProtocolList("Mail", "pop-2|pop-3|pop3|kpop|smtp|imap|imap2|");
  handleProtocolList("SNMP", "snmp|snmp-trap|");
  handleProtocolList("NEWS", "nntp|");
  handleProtocolList("NFS", "mount|pcnfs|bwnfs|nfsd|nfsd-status|");
  handleProtocolList("X11", "6000-6010|");
  handleProtocolList("SSH", "22|"); /* 22 == ssh (just to make sure the port is defined) */
}

/* **************************************** */

short handleProtocol(char* protoName, char *protocol) {
  int i, idx, lowProtoPort, highProtoPort;
  
  if(protocol[0] == '\0')
    return(-1);
  else if(isdigit(protocol[0])) {
    /* numeric protocol port handling */
    lowProtoPort = highProtoPort = 0;
    sscanf(protocol, "%d-%d", &lowProtoPort, &highProtoPort);
    if(highProtoPort < lowProtoPort)
      highProtoPort = lowProtoPort;

    if(lowProtoPort < 0) lowProtoPort = 0;
    if(highProtoPort >= TOP_IP_PORT) highProtoPort = TOP_IP_PORT-1;

    for(idx=lowProtoPort; idx<= highProtoPort; idx++) {
      if(ipPortMapper[idx] == -1) {
	numIpPortsToHandle++;
	
#ifdef DEBUG
	printf("[%d] '%s' [port=%d]\n", numIpProtosToMonitor, protoName, idx);
#endif
	ipPortMapper[idx] = numIpProtosToMonitor;
      } else if(showProtoWarnings)
	printf("WARNING: IP port %d (%s) has been discarded (multiple instances).\n", 
	       idx, protoName);
    }

    return(1);
  } 

  for(i=1; i<SERVICE_HASH_SIZE; i++) {
    idx = -1;

    if((udpSvc[i] != NULL) && (strcmp(udpSvc[i]->name, protocol) == 0))
      idx = udpSvc[i]->port;
    else if((tcpSvc[i] != NULL) && (strcmp(tcpSvc[i]->name, protocol) == 0))
      idx = tcpSvc[i]->port;

    if(idx != -1) {
      if(ipPortMapper[idx] == -1) {
	numIpPortsToHandle++;

#ifdef DEBUG
	printf("[%d] '%s' [%s:%d]\n", numIpProtosToMonitor, protoName, protocol, idx);
#endif
	ipPortMapper[idx] = numIpProtosToMonitor;
      } else if(showProtoWarnings)
	printf("WARNING: protocol '%s' has been discarded (multiple instances).\n", protocol);
      
      return(1);
    }
  }

  if(showProtoWarnings)
    printf("WARNING: unknown protocol '%s'. It has been ignored.\n", protocol);

  return(-1);
}

/* **************************************** */

void handleProtocolList(char* protoName, char *protocolList) {
  char tmpStr[255];
  char* lastEntry, *protoEntry;
  int increment=0, rc;

#ifdef DEBUG
  printf("%s - %s\n", protoName, protocolList);
#endif

  if(numIpProtosToMonitor == MAX_NUM_HANDLED_IP_PROTOCOLS) {
    /* MAX_NUM_HANDLED_IP_PROTOCOLS is defined in ntop.h: increase it! */
    printf("WARNING: Unable to handle '%s'. You've reached the max number\n"
	   "of handled IP protocols.\n", protoName);
    return;
  }

  /* The trick below is used to avoid to modify static
     memory like in the case where this function is
     called by addDefaultProtocols()
  */
  lastEntry = strcpy(tmpStr, protocolList);

  while((protoEntry  = strchr(lastEntry, '|')) != NULL) {
    protoEntry[0] = '\0';
    rc = handleProtocol(protoName, lastEntry);

    if(rc != -1)
      increment=1;

    lastEntry = &protoEntry[1];
  }

  if(increment == 1) {
    protoIPTrafficInfos[numIpProtosToMonitor] = strdup(protoName);    
    numIpProtosToMonitor++;
#ifdef DEBUG
    printf("%d) %s - %s\n", numIpProtosToMonitor, protoName, protocolList); 
#endif
  }
}

/* **************************************** */

int mapGlobalToLocalIdx(int port) {
  if((port < 0) || (port >= TOP_IP_PORT))
    return -1;
  else {
#ifdef DEBUG
    printf("IP port %d -> %d\n", port, ipPortMapper[port]);
#endif
    return(ipPortMapper[port]);
  }
}

/* **************************************** */

#ifdef MULTITHREADED
void* updateThptLoop(void* notUsed) {
  for(;;) {
#ifdef DEBUG
    printf("Sleeping for %d seconds\n", THROUGHPUT_REFRESH_TIME);
#endif

    sleep(THROUGHPUT_REFRESH_TIME);
#ifdef DEBUG
    printf("Trying to update throughput\n");
#endif

    /* Don't update Thpt if the traffic is high */
    /* if(packetQueueLen < (PACKET_QUEUE_LENGTH/3)) */ {
      actTime = time(NULL);
#ifdef MULTITHREADED
      accessMutex(&hostsHashMutex);
#endif
#ifdef DEBUG
      printf("Updating throughput\n");
#endif
      updateThpt(); /* Update Throughput */
#ifdef MULTITHREADED
      releaseMutex(&hostsHashMutex);
#endif
    }
  }
}

/* **************************************** */

#ifdef MULTITHREADED
void* updateDBHostsTrafficLoop(void* notUsed) {
  u_short updateTime = DEFAULT_DB_UPDATE_TIME; /* This should be user configurable */
  
  for(;;) {
#ifdef DEBUG
    printf("Sleeping for %d seconds\n", updateTime);
#endif

    sleep(updateTime);

    /* accessMutex(&hostsHashMutex); */
    updateDbHostsTraffic();
    /* releaseMutex(&hostsHashMutex); */
  }
}
#endif

/* **************************************** */

void* scanIdleLoop(void* notUsed) {
  for(;;) {
    sleep(SESSION_SCAN_DELAY);   
    actTime = time(NULL);

    /* Don't purge hosts if the traffic is high */
    if(packetQueueLen < (PACKET_QUEUE_LENGTH/3)) {
#ifdef MULTITHREADED
      accessMutex(&hostsHashMutex);
#endif      
      scanTimedoutTCPSessions();
#ifdef MULTITHREADED
      releaseMutex(&hostsHashMutex);
#endif
    }
  }
}

/* **************************************** */

void* logFileLoop(void* notUsed) {
  for(;;) {
    sleep(logTimeout);
    actTime = time(NULL);

    /* Don't purge hosts if the traffic is high */
    if(packetQueueLen < (PACKET_QUEUE_LENGTH/3)) {
#ifdef MULTITHREADED
      accessMutex(&hostsHashMutex);
#endif      
      /* printf("Called LogStatsToFile()"); */
      LogStatsToFile();
#ifdef MULTITHREADED
      releaseMutex(&hostsHashMutex);
#endif
    }
  }
}

/* **************************************** */

#ifdef MULTITHREADED
#ifdef HAVE_LSOF
void* periodicLsofLoop(void* notUsed) {
  for(;;) {
    /*
      refresh process list each minute 
      if needed
    */
    if(updateLsof) {
#ifdef DEBUG
      printf("Wait please: reading lsof information...\n");
#endif
      readLsofInfo();
      readNepedInfo();
#ifdef DEBUG
      printf("Done with lsof.\n");
#endif
    }
    sleep(60);
  }
}
#endif
#endif

#endif

