/*
nntpd -- the NNTP server

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <Randolf.Skerka@gmx.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Kazushi (Jam) Marukawa <jam@pobox.com>.
Copyright of the modifications 1998, 1999.

See file COPYING for restrictions on the use of this software.
*/

#include "leafnode.h"

#ifdef SOCKS
#include <socks.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <netdb.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <syslog.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <utime.h>

#define MAXLINELENGTH 1000

#ifdef HAVE_IPV6
/* 
 *  * local union struct
 *   */
union sockaddr_union
{
    struct sockaddr         sa;
    struct sockaddr_in      sin;
    struct sockaddr_in6     sin6;
};
#endif

int write2(int fd, const char *msg);
int hash (const char *);
FILE * fopenart(const char *);
FILE * fopenpseudoart(const char * arg, const unsigned long article_num);
FILE * buildpseudoart(const char * grp);
void list(struct newsgroup * ng , int what, const char * pattern );
void rereadactive(void);

void parser(void);
void error(const char *msg);
void doarticle(const char *arg, int what);
void dogroup(const char *);
void dohelp(void);
void domove(int);
void dolist(char *p);
void donewgroups(const char *);
/* void donewnews(const char*); */
void dopost( void );
void doxhdr(char *);
void doxover(const char *);
void dolistgroup( const char * );
void markinterest( void );

struct newsgroup * group;	/* current group, initially none */
struct newsgroup * xovergroup = NULL;
int pseudogroup;		/* indicates whether we are in a "real"
				   group or pretend only */
unsigned long artno;		/* current article number */
char *cmd; /* current command line */
time_t activetime;
static int justaftergroup;

int debug = 0;
int verbose = 0;		/* verbose doesn't count here */

static jmp_buf timeout;

static void timer(int sig) {
    longjmp(timeout, 1);
    exit(sig);			/* not reached */
}

/*
 * call getaline with 15 min timeout (not configurable)
 */
static char * mgetaline( FILE *f ) {
    char * l;

    if ( setjmp(timeout) ) {
	 return NULL;
    }
    ( void ) signal( SIGALRM, timer );
    ( void ) alarm( 15*60 );
    l = getaline( f );
    ( void ) alarm( 0U );
    return l;
}

void rereadactive(void) {
    struct stat st;

    strcpy(s, spooldir);
    strcat(s, "/leaf.node/groupinfo");

    if ( (!stat(s, &st) && (st.st_mtime > activetime)) ||
	 (active==NULL) ) {
	if ( debugmode )
	    syslog( LOG_DEBUG, "rereading %s", s );
	readactive();
	activetime = st.st_mtime;
    }
}



void error(const char *msg) {
    printf("%s: %s\r\n", msg, strerror(errno) );
}


void parser(void) {
    char *arg;
    int n;

    while ( ( cmd=mgetaline( stdin )) ) {
	n = strlen(cmd);
	if ( n == 0 )
	    continue;		/* ignore */
	if ( n > MAXLINELENGTH ) {
	    /* ignore attempts at buffer overflow */
	    if ( debugmode )
		syslog( LOG_DEBUG, ">500 Dazed and confused" );
	    printf( "500 Dazed and confused\r\n" );
	    continue;
	}

	/* parse command line */
	n = 0;
	while( isalpha((unsigned char)cmd[n]) )
	    n++;
	while( isspace((unsigned char)cmd[n]) )
	    cmd[n++] = '\0';

	arg = cmd+n;

	while( cmd[n] )
	    n++;
	n--;
	while(isspace((unsigned char)cmd[n]))
	    cmd[n--] = '\0';

	if (!strcasecmp(cmd, "quit")) {
	    if ( debugmode )
		syslog( LOG_DEBUG, ">205 Always happy to serve!" );
	    printf("205 Always happy to serve!\r\n");
	    return;
	}
	rereadactive();
	if (!strcasecmp(cmd, "article")) {
	    doarticle(arg, 3);
	} else if (!strcasecmp(cmd, "head")) {
	    doarticle(arg, 2);
	} else if (!strcasecmp(cmd, "body")) {
	    doarticle(arg, 1);
	} else if (!strcasecmp(cmd, "stat")) {
	    doarticle(arg, 0);
	} else if (!strcasecmp(cmd, "help")) {
	    dohelp();
	} else if (!strcasecmp(cmd, "ihave")) {
	    if ( debugmode )
		syslog( LOG_DEBUG, ">500 IHAVE is for big news servers" );
	    printf("500 IHAVE is for big news servers\r\n");
	} else if (!strcasecmp(cmd, "last")) {
	    domove(-1);
	} else if (!strcasecmp(cmd, "next")) {
	    domove(1);
	} else if (!strcasecmp(cmd, "list")) {
	    dolist(arg);
	} else if (!strcasecmp(cmd, "mode")) {
	    if ( debugmode )
		syslog( LOG_DEBUG, ">200 Leafnode %s, pleased to meet you!",
			version);
	    printf("200 Leafnode %s, pleased to meet you!\r\n", version);
	} else if (!strcasecmp(cmd, "newgroups")) {
	    donewgroups(arg);
	} else if (!strcasecmp(cmd, "newnews")) {
	    if ( debugmode )
		syslog( LOG_DEBUG,
			">500 NEWNEWS is meaningless for this server" );
	    printf( "500 NEWNEWS is meaningless for this server\r\n" );
	} else if (!strcasecmp(cmd, "post")) {
	    dopost();
	} else if (!strcasecmp(cmd, "slave")) {
	    if ( debugmode )
		syslog( LOG_DEBUG, ">202 Cool - I always wanted a slave" );
	    printf("202 Cool - I always wanted a slave\r\n");
	} else if (!strcasecmp(cmd, "xhdr")) {
	    doxhdr(arg);
	} else if (!strcasecmp(cmd, "xover")) {
	    doxover(arg);
	} else if (!strcasecmp(cmd, "over")) {
	    doxover(arg);
	} else if (!strcasecmp(cmd, "listgroup")) {
	    dolistgroup(arg);
	} else if (!strcasecmp(cmd, "group")) {
	    dogroup(arg);
	} else {
	    if ( debugmode )
		syslog( LOG_DEBUG, ">500 Unknown command" );
	    printf("500 Unknown command\r\n");
	}
	fflush(stdout);
    }
    if ( debugmode )
	syslog( LOG_DEBUG, ">421 Network error" );
    error("421 Network error");
}

/*
 * pseudo article stuff
 */

/* build and return an open fd to pseudoart in group */
FILE * buildpseudoart(const char * grp) {
    FILE * f;

    f = tmpfile();
    if (!f) {
	syslog( LOG_ERR, "Could not create pseudoarticle" );
	return f;
    }

    fprintf(f, "Path: %s\n", fqdn);
    fprintf(f, "Newsgroups: %s\n", grp);
    fprintf(f, "From: Leafnode <nobody@%s>\n", fqdn);
    fprintf(f, "Subject: Leafnode placeholder for group %s\n", grp);
    fprintf(f, "Date: %s\n", rfctime() );
    fprintf(f, "Message-ID: <leafnode:placeholder:%s@%s>\n", grp, fqdn);
    fprintf(f, "\n");
    fprintf(f,
	  "This server is running leafnode, which is a dynamic NNTP proxy.\n"
	  "This means that it does not retrieve newsgroups unless someone is\n"
	  "actively reading them.\n"
	  "\n"
	  "If you do an operation on a group - such as reading an article,\n"
	  "looking at the group table of contents or similar, then leafnode\n"
	  "will go and fetch articles from that group when it next updates.\n"
	  "\n"
	  "Since you have read this dummy article, leafnode will retrieve\n"
	  "the newsgroup %s when fetch is run\n"
	  "the next time. If you'll look into this group a little later, you\n"
	  "will see real articles.\n"
	  "\n"
	  "If you see articles in groups you do not read, that is almost\n"
	  "always because of cross-posting.  These articles do not occupy any\n"
	  "more space - they are hard-linked into each newsgroup directory.\n"
	  "\n"
	  "If you do not understand this, please talk to your newsmaster.\n"
	  "\n"
	  "Leafnode can be found at\n"
	  "\thttp://wpxx02.toxi.uni-wuerzburg.de/~krasel/leafnode.html\n\n", grp );

    rewind(f);
    return f;
}

/* open a pseudo art */
FILE * fopenpseudoart(const char * arg, const unsigned long article_num) {
    FILE *f = NULL;
    char msgidbuf[128];
    char * c;
    struct newsgroup * g;    

    if ( debugmode )
	syslog( LOG_DEBUG, "%s: first %lu last %lu artno %lu",
    		group->name, group->first, group->last, article_num );
    if ( article_num && article_num == group->first &&
	  group->first >= group->last ) {
	f = buildpseudoart(group->name);
    } else if ( !article_num ) {
	if (!strncmp(arg, "<leafnode:placeholder:", 22)) {
	    strncpy(msgidbuf, arg+22, 128);
	    msgidbuf[127] = '\0';
	    if ((c = strchr(msgidbuf, '@')) != NULL) {
		*c = '\0';
		g = findgroup(msgidbuf);
		if (g)
		    f = buildpseudoart(g->name);
	    }
	}
    }
    return f;
}


/* open an article by number or message-id */
FILE * fopenart(const char * arg) {
    unsigned long int a;
    FILE *f;
    char *t;

    t = NULL;
    a = strtoul(arg, &t, 10);
    if (a && t && !*t && group) {
	if ( pseudogroup )
	    f = fopenpseudoart( arg, a );
	else {
	    f = fopen(arg, "r");
	    if (!f)
		f = fopenpseudoart( arg, a );
	}
	if ( f )
	    artno = a;
	markinterest();
	/* else try message-id lookup */
    } else if ( arg && *arg=='<' ) {
	f = fopen(lookup(arg), "r");
    } else if (group && artno) {
	sprintf(s, "%lu", artno);
	f = fopen(s, "r");
	if (!f)
	    f = fopenpseudoart(s, a);
	markinterest();
    } else {
	f = NULL;
    }
    return f;
}


/*
 * Mark an article for download by appending its number to the
 * corresponding file in interesting.groups
 */
static int markdownload( long id )
{
    int i;
    long n;
    FILE *f;

    sprintf( s, "%s/interesting.groups/%s", spooldir, group->name );
    if ( ( f = fopen( s, "r+" ) ) ) {
	 i = 0;
	 while ( fgets( s, 1024, f ) ) {
	     if ( sscanf( s, "%ld", &n ) == 1 && n == id )
	         return 0; /* already marked */
	     ++i;
	 }
	 if ( i < BODY_DOWNLOAD_LIMIT ) {
	     fprintf( f, "%ld\n", id );
	    if ( debugmode )
	 	syslog( LOG_DEBUG, "Marking %s %ld for download",
	 		group->name, id );
	 }
	 else {
	     syslog( LOG_ERR, "Too many bodies marked in %s",
	             group->name );
	 }
	 fclose( f );
    }
    return 1;
}

/* display an article or somesuch */
/* DOARTICLE */
void doarticle(const char *arg, int what) {
    FILE *f;
    char *p = NULL;
    char *q = NULL;
    unsigned long localartno;
    char *localmsgid = NULL;

    f = fopenart(arg);
    if (!f) {
	if ( arg && *arg != '<' && !group ) {
	    printf( "412 No newsgroup selected\r\n" );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">412 No newsgroup selected" );
	} else if ( strlen( arg ) ) {
	    printf("430 No such article: %s\r\n", arg );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">430 No such article: %s", arg );
	} else {
	    printf( "423 No such article: %lu\r\n", artno );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">423 No such article: %lu", artno );
	}
	return;
    }

    if ( !arg ) {
	localartno = artno;
	localmsgid = fgetheader( f, "Message-ID:" );
    }
    else if ( *arg == '<' ) {
	localartno = 0;
	localmsgid = strdup(arg);
    }
    else {
	localartno = strtoul( arg, NULL, 10 );
	localmsgid = fgetheader( f, "Message-ID:" );
    }

    if ( !localartno ) {
	/* we have to extract the article number from the Xref: header */
	p = fgetheader( f, "Xref:" );
	if ( p ) {
	    /* search article number of 1st group in Xref: */
	    q = strchr( p, ':' );
	    if ( q ) {
	        q++;
		localartno = strtoul( q, NULL, 10 );
	    }
	    if ( group ) {
		/* search article number of group in Xref: */
		q = strstr( p, group->name );
		if ( q ) {
		    q += strlen( group->name );
		    if ( *q++ == ':' )
		        localartno = strtoul( q, NULL, 10 );
		}
	    }
	    free( p );
	}
    }

    sprintf( s, "%3d %lu %s article retrieved - ", 223-what,
		localartno, localmsgid );
    free( localmsgid );

    if ( what == 0 )
	strcat( s, "request text separately" );
    else if ( what == 1 )
	strcat( s, "body follows" );
    else if ( what == 2 )
	strcat( s, "head follows" );
    else
	strcat( s, "text follows" );
    printf("%s\r\n", s );
    if ( debugmode )
	syslog( LOG_DEBUG, ">%s", s );

    while (fgets(s, 1024, f) && *s && (*s!='\n')) {
	if (what & 2) {
	    p = s;
	    if ( (p = strchr(p, '\n')) )
		*p = '\0';

	    printf("%s%s\r\n", *s=='.' ? "." : "", s); /* . on headers :( */
	}
    }

    if (what==3)
	printf("\r\n"); /* empty separator line */

    if ( what & 1 ) {
	if ( delaybody && *s != '\n' ) {
	    if ( ! markdownload( localartno ) ) {
		printf( "\r\n\r\n"
			"\t[ Leafnode: ]\r\n"
			"\t[ This message has already been "
			"marked for download. ]\r\n" );
	    } else {
		printf( "\r\n\r\n"
			"\t[ Leafnode: ]\r\n"
			"\t[ Message %ld of %s ]\r\n"
			"\t[ has been marked for download. ]\r\n",
			localartno, group->name );
	    }
	} else {
	    while ( fgets(s, 1024, f) && *s) {
		p = s;
		if ( (p = strchr(p, '\n')) )
		    *p = '\0';
		printf("%s%s\r\n", *s=='.' ? "." : "", s);
	    }
	}
    }
    if ( what )
	printf(".\r\n");
    fclose(f);

    return; /* OF COURSE there were no errors */
}


/* note bug.. need not be _immediately after_ GROUP */
void markinterest( void ) {
    struct stat st;
    struct utimbuf buf;
    int error;
    time_t now;
    FILE * f;

    if ( !justaftergroup )
	return;
    error = 0;
    sprintf( s, "%s/interesting.groups/%s", spooldir, group->name );
    if ( stat( s, &st ) == 0 ) {
	now = time(0);
	buf.actime = ( now < st.st_atime ) ? st.st_atime : now;
			/* now < update may happen through HW failures */
	buf.modtime = st.st_mtime;
	if ( utime( s, &buf ) )
	    error = 1;
    }
    else
	error = 1;
    if ( error ) {
	f = fopen( s, "w" );
	if ( f )
	    fclose( f );
	else
	    syslog( LOG_ERR, "Could not create %s: %m", s );
    }

    justaftergroup = FALSE;
}
    


/* change to group - note no checks on group name */
void dogroup(const char *arg) {
    struct newsgroup * g;

    g = findgroup(arg);
    if (g) {
	group = g;
	if ( chdirgroup( g->name, FALSE ) ) {
	    if ( debugmode )
		syslog( LOG_DEBUG, ">211 %lu %lu %lu %s group selected",
			g->last >= g->first ? g->last-g->first+1 : 0,
			g->first, g->last, g->name);
	    printf("211 %lu %lu %lu %s group selected\r\n", 
		   g->last >= g->first ? g->last-g->first+1 : 0, 
		   g->first, g->last, g->name);
	    pseudogroup = FALSE;
	} else {
	    if ( debugmode )
		syslog( LOG_DEBUG,
			">211 1 1 1 %s group selected (pseudo article)",
			g->name );
	    printf("211 1 1 1 %s group selected (pseudo article)\r\n",
		    g->name );
	    g->first = 1;
	    g->last  = 1;
	    artno = 1;
	    pseudogroup = TRUE;
	}
	artno = g->first;
	fflush( stdout );
	justaftergroup = TRUE;
    } else {
	if ( debugmode )
    	    syslog( LOG_DEBUG, ">411 No such group");
	printf("411 No such group\r\n");
    }
}

void dohelp(void) {
    printf("100 Legal commands\r\n");
/*  printf("  authinfo user Name|pass Password|generic <prog> <args>\r\n"); */
    printf("  article [MessageID|Number]\r\n");
    printf("  body [MessageID|Number]\r\n");
/*  printf("  date\r\n"); */
    printf("  group newsgroup\r\n");
    printf("  head [MessageID|Number]\r\n");
    printf("  help\r\n");
    printf("  ihave\r\n");
    printf("  last\r\n");
/*  printf("  list [active|newsgroups|distributions|schema] [group_pattern]\r\n"); */
    printf("  list [active|newsgroups] [group_pattern]\r\n");
    printf("  listgroup newsgroup\r\n");
    printf("  mode reader\r\n");
    printf("  newgroups yymmdd hhmmss [\"GMT\"] [<distributions>]\r\n");
/*  printf("  newnews newsgroups yymmdd hhmmss [\"GMT\"] [<distributions>]\r\n"); */
    printf("  next\r\n");
    printf("  post\r\n");
    printf("  slave\r\n");
    printf("  stat [MessageID|Number]\r\n");
/*  printf("  xgtitle [group_pattern]\r\n"); */
    printf("  xhdr header [range|MessageID]\r\n");
    printf("  xover [range]\r\n");
/*  printf("  xpat header range|MessageID pat [morepat...]\r\n"); */
/*  printf("  xpath MessageID\r\n"); */
    printf(".\r\n");
}

void domove( int by ) {
    char * msgid;

    by = ( by < 0 ) ? -1 : 1 ;
    if ( group ) {
	if (artno) {
	    artno += by ;
	    do {
		sprintf( s, "%lu", artno );
		msgid = getheader( s, "Message-ID:" );
		if ( !msgid )
		    artno += by;
	    } while ( !msgid && artno >= group->first && artno <= group->last );
	    if ( artno > group->last ) {
		artno = group->last;
		printf("421 There is no next article\r\n" );
		if ( debugmode )
		    syslog( LOG_DEBUG, ">421 There is no next article" );
	    }
	    else if ( artno < group->first ) {
		artno = group->first;
		printf("422 There is no previous article\r\n" );
		if ( debugmode )
		    syslog( LOG_DEBUG, ">422 There is no previous article" );
	    }
	    else {
	        printf( "223 %lu %s article retrieved\r\n", artno, msgid);
		if ( debugmode )
		    syslog( LOG_DEBUG, ">223 %lu %s article retrieved",
			    artno, msgid );
	    }
	    free( msgid );
	} else {
	    printf( "420 There is no current article\r\n" );
	    if ( debugmode )
	        syslog( LOG_DEBUG, ">420 There is no current article" );
	}
    } else {
	if ( debugmode )
	    syslog( LOG_DEBUG, ">412 No newsgroup selected" );
	printf( "412 No newsgroup selected\r\n" );
    }
}



/* LIST ACTIVE if what==0, else LIST NEWSGROUPS */
void list(struct newsgroup *g, int what, const char * pattern) {
    if (g) {
	list (g->right, what, pattern);
	if (what) {
	    if ( !pattern )
		printf( "%s\t%s\r\n", g->name, g->desc ? g->desc : "" );
	    else if ( ngmatch( pattern, g->name ) == 0 )
		printf( "%s\t%s\r\n", g->name, g->desc ? g->desc : "" );
	} else {
	    if ( !pattern )
		printf( "%s %010lu %010lu y\r\n", g->name, g->last, g->first);
	    else if ( ngmatch( pattern, g->name ) == 0 )
		printf( "%s %010lu %010lu y\r\n", g->name, g->last, g->first);
	}
	list (g->left, what, pattern);
    }
}
	
void dolist( char *arg ) {
    char * p = NULL;

    if ( !strcasecmp(arg, "extensions") ) {
	printf("202 extensions supported follow\r\n"
	       " OVER\r\n"
	       " LISTGROUP\r\n"
	       ".\r\n" );
	if ( debugmode )
	     syslog( LOG_DEBUG, ">202 extensions supported follow" );
    } else if ( !strcasecmp(arg, "overview.fmt") ) {
	 printf("215 information follows\r\n"
	        "Subject:\r\n"
	        "From:\r\n"
	        "Date:\r\n"
	        "Message-ID:\r\n"
	        "References:\r\n"
	        "Bytes:\r\n"
	        "Lines:\r\n"
	        "Xref:full\r\n"
	        ".\r\n");
	if ( debugmode )
	     syslog( LOG_DEBUG, ">215 information follows" );
    } else if ( !strcasecmp(arg, "active.times") ) {
	 printf("215 Placeholder - Leafnode will fetch groups on demand\r\n"
	        "news.announce.newusers 42 tale@uunet.uu.net\r\n"
	        "news.answers 42 tale@uunet.uu.net\r\n"
	        ".\r\n");
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    ">215 Placeholder - Leafnode will fetch groups on demand" );
    } else {
	if (!active) {
	     printf("503 Group information file does not exist!\r\n");
	     syslog( LOG_ERR, ">503 Group information file does not exist!" );
	 } else if ( !*arg || !strncasecmp(arg, "active", 6) ) {
	    printf("215 Newsgroups in form \"group high low flags\".\r\n");
	    if ( debugmode )
		syslog( LOG_DEBUG,
			">215 Newsgroups in form \"group high low flags\"." );
	    if ( active ) {
		if ( !*arg || strlen(arg) == 6 )
		    list( active, 0, NULL );
		else {
		    while ( *arg && (!isspace((unsigned char)*arg)) )
			arg++;
		    while ( *arg && isspace((unsigned char)*arg) )
			arg++;
		    p = arg;
	 	    list ( active, 0, arg );
		}
	    }
	     printf(".\r\n");
	 } else if ( !strncasecmp(arg, "newsgroups", 10) ) {
	    printf("215 Descriptions in form \"group description\".\r\n");
	    if ( debugmode )
		syslog( LOG_DEBUG,
			">215 Descriptions in form \"group description\"." );
	    if ( active ) {
		if ( strlen(arg) == 10 )
		    list( active, 1, NULL );
		else {
		    while ( *arg && (!isspace((unsigned char)*arg)) )
			arg++;
		    while ( *arg && isspace((unsigned char)*arg) )
			arg++;
	 	    list ( active, 1, arg );
		}
	    }
	     printf(".\r\n");
	  } else {
	     printf( "503 Syntax error\r\n" );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">503 Syntax error" );
	 }
    }
}

static void helpdonewgroups( struct newsgroup *g, time_t age ) {
    if ( g ) {
	helpdonewgroups( g->right, age );
	if ( g->age >= age )
	    printf( "%s %lu %lu y\r\n", g->name, g->first, g->last );
	helpdonewgroups( g->left, age );
    }
}

void donewgroups(const char *arg) {
    struct tm timearray;
    struct tm * ltime;
    time_t age;
    time_t now;
    int year, century;
    char * l;
    int a, b;

    now = time( 0 );
    ltime = localtime( &now );
    year    = ltime->tm_year % 100;
    century = ltime->tm_year / 100;

    memset( &timearray, 0, sizeof(timearray) );
    l = NULL;
    a = strtol( arg, &l, 10 );
    b = a / 10000;
    if ( b < 100 ) {
	/* YYMMDD */
	if ( b <= year )
	    timearray.tm_year = b + (century*100) ;
	else
	    timearray.tm_year = b + (century-1)*100 ;
    } else if ( b < 1000 ) {
    	/* YYYMMDD - this should not happen, but leafnode will parse it */
	syslog( LOG_NOTICE,
		"NEWGROUPS year is %d: please update your newsreader", b );
	timearray.tm_year = b;
    } else {
	/* YYYYMMDD */
	timearray.tm_year = b - 1900 ;
    }
    timearray.tm_mon  = (a % 10000 / 100) - 1 ;
    timearray.tm_mday = a % 100 ;
    while ( *l && isspace((unsigned char)*l) )
	l++;
    a = strtol( l, NULL, 10 );	/* we don't care about the rest of the line */
    timearray.tm_hour = a / 10000;
    timearray.tm_min  = a % 10000 / 100;
    timearray.tm_sec  = a % 100;
    age = mktime( &timearray ) ;

    printf( "231 List of new newsgroups follows\r\n" );
    if ( debugmode )
	syslog( LOG_DEBUG, ">231 List of new newsgroups follows" );
    helpdonewgroups( active, age );
    printf( ".\r\n" );
}

/* next bit is copied from INN 1.4 and modified ("broken") by agulbra

   mail to Rich $alz <rsalz@uunet.uu.net> bounced */

/* Scale time back a bit, for shorter Message-ID's. */
#define OFFSET	673416000L

static char ALPHABET[] = "0123456789abcdefghijklmnopqrstuv";

char * generateMessageID( void );

char * generateMessageID( void ) {
    static char buff[80];
    static time_t then;
    static unsigned int fudge;
    time_t now;
    char * p;
    int n;

    now = time(0); /* might be 0, in which case fudge will almost fix it */
    if ( now != then )
	fudge = 0;
    else
	fudge++;
    then = now;

    p = buff;
    *p++ = '<';
    n = (unsigned int)now - OFFSET;
    while( n ) {
	*p++ = ALPHABET[(int)(n & 31)];
	n >>= 5;
    }
    *p++ = '.';
    n = fudge * 32768 + (int)getpid();
    while( n ) {
	*p++ = ALPHABET[(int)(n & 31)];
	n >>= 5;
    }
    if ( owndn )
	sprintf( p, ".ln@%s>", owndn );
    else
	sprintf( p, ".ln@%s>", fqdn );
    return buff;
}

/* the end of what I stole from rsalz and then mangled */


void dopost( void ) {
    char * line;
    int havefrom = 0;
    int havepath = 0;
    int havedate = 0;
    int havenewsgroups = 0;
    int havemessageid = 0;
    int havesubject = 0;
    int err = 0;
    int i, len;
    int out;
    char outname[80];
    static int postingno; /* starts as 0 */

    sprintf( outname, "%s/out.going/%d-%d-%d", 
	     spooldir, (int)getpid(), (int)time(NULL), ++postingno );

    out = open( outname, O_WRONLY|O_EXCL|O_CREAT, 0444 );
    if ( out < 0 ) {
	char *error = strerror(errno);
	printf( "441 Unable to open spool file %s: %s\r\n", outname, error );
	syslog( LOG_ERR, ">441 Unable to open spool file %s: %s",
		outname, error );
	return;
    }

    printf( "340 Go ahead.\r\n" );
    if ( debugmode )
	syslog( LOG_DEBUG, ">340 Go ahead." );
    fflush( stdout );

    /* get headers */
    do {
	int len;
	debug = 0;
	line = getaline( stdin );
	debug = debugmode;
	if ( !line ) {
	    unlink( outname );
	    exit( 0 );
	}
	if ( !strncasecmp( line, "From: ", 6 ) ) {
	    if ( havefrom )
		err = TRUE;
	    else
		havefrom = TRUE;
	}
	if ( !strncasecmp( line, "Path: ", 6 ) ) {
	    if ( havepath )
		err = TRUE;
	    else
		havepath = TRUE;
	}
	if ( !strncasecmp( line, "Message-ID: ", 12 ) ) {
	    if ( havemessageid )
		err = TRUE;
	    else
		havemessageid = TRUE;
	}
	if ( !strncasecmp( line, "Subject: ", 9 ) ) {
	    if ( havesubject )
		err = TRUE;
	    else
		havesubject = TRUE;
	}
	if ( !strncasecmp( line, "Newsgroups: ", 12 ) ) {
	    if ( havenewsgroups )
		err = TRUE;
	    else
		havenewsgroups = TRUE;
	}
	if ( !strncasecmp( line, "Date: ", 6 ) ) {
	    if ( havedate )
		err = TRUE;
	    else
		havedate = TRUE;
	}
	len = strlen( line );
	/* replace tabs with spaces */
	for ( i = 0; i < len; i++ )
	    if ( isspace((unsigned char)line[i]) )
		line[i] = ' ';
	if ( len && line[len-1]=='\n' )
	    line[--len] = '\0';
	if ( len && line[len-1]=='\r' )
	    line[--len] = '\0';
	if ( len && line[len-1]=='\r' )
	    line[--len] = '\0';
	if ( len )
	    write( out, line, len );
	else {
	    if ( !havepath ) {
		write( out, "Path: ", 6 );
		write( out, fqdn, strlen( fqdn ) );
		write( out, "!nobody\r\n", 9 );
	    }
	    if ( !havedate ) {
		const char * l = rfctime();
		write( out, "Date: ", 6 );
		write( out, l, strlen( l ) );
		write( out, "\r\n", 2 );
	    }
	    if ( !havemessageid ) {
		char tmp[80];
		sprintf( tmp, "Message-ID: %s\r\n",
			 generateMessageID() );
		write( out, tmp, strlen( tmp ) );
	    }
	}
	write( out, "\r\n", 2 );
    } while ( *line );

    /* get bodies */
    do {
	debug = 0;
	line = getaline( stdin );
	debug = debugmode;
	if ( !line ) {
	    unlink( outname );
	    exit( 1 );
	}

	len = strlen( line );
	if ( len && line[len-1]=='\n' )
	    line[--len] = '\0';
	if ( len && line[len-1]=='\r' )
	    line[--len] = '\0';
	if ( len && line[len-1]=='\r' )
	    line[--len] = '\0';
	if ( line[0] == '.' ) {
	    if ( len > 1 ) {
		write( out, line+1, len-1 );
		write( out, "\r\n", 2 );
	    }
	}
	else {
	    write( out, line, len );
	    write( out, "\r\n", 2 );
	}
    } while ( line[0] != '.' || line[1] != '\0' );
    close( out );

    if ( havefrom && havesubject && havenewsgroups && !err ) {
	printf( "240 Article posted, now be patient\r\n" );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">240 Article posted, now be patient" );
	return;
    }

    unlink( outname );
    printf( "441 Formatting error, article not posted\r\n" );
    syslog( LOG_INFO, ">441 Formatting error, article not posted" );
}



void doxhdr(char *arg) {
    char * h[] = { "Subject", "From", "Date", "Message-ID", "References",
		   "Bytes", "Lines"
    }; /* ugly emacs workaround */

    int n = 7;
    int dash;
    char * l;
    int a,b=0,c;
    FILE * f;

    if ( !arg || !*arg ) {
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    ">502 Usage: XHDR header first[-last] or "
		    "XHDR header message-id" );
	printf( "502 Usage: XHDR header first[-last] or "
		"XHDR header message-id\r\n" );
	return;
    }

    l = arg;
    while( l && *l && !isspace((unsigned char)*l) )
	l++;
    if (l && *l)
	*l++ = '\0';
    while( l && *l && isspace((unsigned char)*l) )
	l++;

    if (l && *l=='<') { /* handle message-id form (well) */
	FILE * f;

	f = fopenart(l);
	if (!f) {
	    printf("430 No such article\r\n");
	    if ( debugmode )
		syslog( LOG_DEBUG, ">430 No such article" );
	    return;
	}
	do {
	    l = getaline( f );
	} while (l && *l && 
		 strncasecmp(arg, l, strlen(arg)) && 
		 l[strlen(arg)]!=':'); /* uh, space before : allowed? */
	while( l && *l && !isspace((unsigned char)*l) )
	    l++;
	while( l && *l && isspace((unsigned char)*l) )
	    l++;
	if ( debugmode ) {
	    syslog( LOG_DEBUG, ">221 %s header follows:", arg );
	    syslog( LOG_DEBUG, ">%s", l ? l : "" );
	    syslog( LOG_DEBUG, "." );
	}
	printf( "221 %s header follows:\r\n%s\r\n.\r\n",
		arg, l ? l : "" ); /* dubious - if not found, "" */
	fclose( f );
	return;
    }

    if (!group) {
	if ( debugmode )
	    syslog( LOG_DEBUG, ">412 Use the GROUP command first" );
	printf( "412 Use the GROUP command first\r\n" );
	return;
    }

    markinterest();

    if ( !chdirgroup( group->name, FALSE ) )
	pseudogroup = TRUE;

    if ( !pseudogroup && ( xovergroup != group ) ) {
	if ( !getxover() )
	    pseudogroup = TRUE;
	else
	    xovergroup = group;
    }

    if ( pseudogroup ) {
	do {
	    n--;
	} while ( n >= 0 && strncasecmp( h[n], arg, strlen(h[n]) ) != 0 ) ;
	if ( n < 0 ) {
	    printf( "430 No such header: %s\r\n", arg );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">430 No such header: %s", arg );
	    return;
	}
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    ">221 First line of %s pseudo-header follows:", arg );
	printf( "221 First line of %s pseudo-header follows:\r\n", arg );
	if ( n == 0 )		/* Subject: */
	    printf( "1 Leafnode placeholder for group %s\r\n", group->name );
	else if ( n == 1 )	/* From: */
	    printf( "1 Leafnode <nobody@%s>\r\n", fqdn );
	else if ( n == 2 )	/* Date: */
	    printf( "1 %s\r\n", rfctime() );
	else if ( n == 3 )	/* Message-ID: */
	    printf( "1 <leafnode:placeholder:%s@%s>\r\n", group->name, fqdn );
	else if ( n == 4 )	/* References */
	    printf( "1\r\n" );
	else if ( n == 5 )	/* Bytes */
	    printf( "1 %d\r\n", 1024 );	/* just a guess */
	else if ( n == 6 )	/* Lines */
	    printf( "1 %d\r\n", 22 );		/* from buildpseudoart() */
	 printf( ".\r\n" );
	 return;
    }

    dash = 0;
    if (l && isdigit((unsigned char)*l) ) {	/* handle range form */
	a = strtol( l, &l, 10 );
	if ( a < xfirst )
	    a = xfirst;
	if (a && l && *l) {
	    while (l && isspace((unsigned char)*l))
		l++;
	    if ( *l == '-' ) {
	        dash = 1;
		b = xlast;
		l++;
		if ( l && *l )
		    b = strtol( l, &l, 10 );
	    }
	    while (l && isspace((unsigned char)*l))
		l++;
	    if ( l && *l ) {
		if ( debugmode )
		    syslog( LOG_DEBUG, ">502 Usage: XHDR header first[-last] "
			    "or XHDR header message-id" );
		printf( "502 Usage: XHDR header first[-last] "
			"or XHDR header message-id" );
		return;
	    }
	    if ( dash && b > xlast )
		b = xlast;
	}
    } else {
	a = b = artno;
    }

    do {
	n--;
    } while( n>-1 && strncasecmp(arg, h[n], strlen(h[n])) );

    if ( a < group->first )
	a = group->first;
    else if ( a > group->last )
	a = group->last;

    if ( b < a )
	b = a;
    else if ( b > group->last )
	b = group->last;

    if ( n >= 0 ) {
	if ( debugmode )
	    syslog( LOG_DEBUG, "221 %s header (from overview) "
		    "for postings %d-%d:", h[n],a,b );
	printf( "221 %s header (from overview) for postings %d-%d:\r\n",
		h[n], a, b );

	s[1023] = '\0';
	for( c=a; c<=b; c++ ) {
	    if ( xoverinfo &&
		 c >= xfirst && c <= xlast &&
		 xoverinfo[c-xfirst].text ) {
		char * l = xoverinfo[c-xfirst].text;
		int d;
		for( d=0; l && d<=n; d++ )
		    l = strchr(l+1, '\t');
		if (l) {
		    char * p;
		    strncpy( s, ++l, 1023 );
		    p = strchr(s, '\t');
		    if (p)
			*p = '\0';
		}
		printf("%d %s\r\n", c, l && *l ? s : "");
	    }
	}
    } else {
	if ( debugmode )
	    syslog( LOG_DEBUG, ">221 %s header (from overview) "
		    "for postings %d-%d:",arg,a,b );
	printf( "221 %s header (from article files) for postings %d-%d:\r\n",
		arg, a, b );
	n = strlen( arg );
	for( c=a; c<=b; c++ ) {
	    sprintf( s, "%d", c );
	    f = fopen( s, "r");
	    while( f ) {
		l = getaline( f );
		if ( !l || !*l || *l == '\r' || *l == '\n' ) {
		    fclose( f );
		    f = 0;
		} else if ( l[n] == ':' &&
			    strncasecmp( arg, l, n ) == 0 ) {
		    printf("%d %s\r\n", c, l + n + 2 );
		}
	    }
	}
    }

    printf( ".\r\n" );
    return;
}


void doxover(const char *arg) {
    char * l;
    int a,b, art;

    if (!group) {
	printf( "412 Use the GROUP command first\r\n" );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">412 Use the GROUP command first" );
	return;
    }

    markinterest();

    l = NULL;
    b = a = strtol( arg, &l, 10 );
    if (a && l && *l) {
	while (l && isspace((unsigned char)*l))
	    l++;
	if ( *l=='-' )
	    b = strtol( ++l, &l, 10 );
	while (l && isspace((unsigned char)*l))
	    l++;
	if ( l && *l ) {
	    printf( "502 Usage: XOVER first[-last]\r\n" );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">502 Usage: XOVER first[-last]" );
	    return;
	}
    }

    if ( !chdirgroup( group->name, FALSE ) )
	pseudogroup = TRUE;

    if ( !pseudogroup ) {
	 if ( xovergroup != group && getxover() )
	    xovergroup = group;

	if ( b == 0 )
	    b = xlast;
	if ( ( a > xlast ) || ( b < xfirst ) ) {
	    printf( "420 No articles in specified range.\r\n" );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">420 No articles in specified range." );
	    return;
	}
	if ( b > xlast )
	    b = xlast;
	if ( a < xfirst )
	    a = xfirst;

	printf( "224 Overview information for postings %d-%d:\r\n", a, b );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">224 Overview information for postings %d-%d:",
		    a, b );
	for( art=a; art<=b; art++ ) {
	    if ( xoverinfo[art-xfirst].text )
		printf( "%s\r\n", xoverinfo[art-xfirst].text );
	}
	printf( ".\r\n" );
    } else {
	if ( ( b < 1 ) || ( a > 1 ) || ( a > b ) ) {
	    printf( "420 No articles in specified range.\r\n" );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">420 No articles in specified range." );
	    return;
	}

	printf( "224 Overview information (pseudo) for postings 1-1:\r\n" );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">224 Overview information (pseudo) for "
		    "postings 1-1:" );
	printf( "%d\t"
	        "Leafnode placeholder for group %s\t"
		"nobody@%s (Leafnode)\t%s\t"
		"<leafnode.%s@%s>\t\t1000\t40\r\n",
		b, group->name, fqdn, rfctime(), group->name, fqdn );
	printf( ".\r\n" );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">%d\tLeafnode placeholder for group %s\t"
		    "nobody@%s (Leafnode)\t%s\t<leafnode.%s@%s>\t\t1000\t40",
		    b, group->name, fqdn, rfctime(), group->name, fqdn );
    }
}



void dolistgroup( const char * arg ) {
    struct newsgroup * g;
    int art;

    if ( arg && strlen(arg) ) {
	g = findgroup( arg );
	if ( !g ) {
	    printf( "411 No such group: %s\r\n", arg );
	    if ( debugmode )
		syslog( LOG_DEBUG, ">411 No such group: %s", arg );
	    return;
	} else {
	    group = g;
	    artno = g->first;
	    justaftergroup = TRUE; 
	}   
    } else if ( group ) {
	g = group;
    } else {
	printf( "412 No group specified\r\n" );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">412 No group specified" );
	return;
    }

    markinterest();
    group = g;
    if ( !chdirgroup( g->name, FALSE ) )
	pseudogroup = TRUE;
    else if ( ( xovergroup != group ) && !getxover() )
	pseudogroup = TRUE;
    else {
	pseudogroup = FALSE;
	xovergroup = group;
    }

    if ( pseudogroup ) {
	printf( "211 Article list for %s follows (pseudo)\r\n", g->name );
	if ( debugmode )
	    syslog( LOG_DEBUG,
		    ">211 Article list for %s follows (pseudo)", g->name );
	printf( "%lu \r\n", g->last ? g->last : 1 );
    } else {
	printf( "211 Article list for %s follows\r\n", g->name );
	if ( debugmode )
	    syslog( LOG_DEBUG, ">211 Article list for %s follows", g->name );
	for( art=xfirst; art<=xlast; art++ ) {
	    if ( xoverinfo[art-xfirst].text )
		printf( "%d \r\n", art );
	}
    }
    printf( ".\r\n" );
}



int main( int argc, char ** argv ) {
    int fodder;
    struct hostent *he;
#ifdef HAVE_IPV6
    union sockaddr_union su;
#else
    struct sockaddr_in sa;
#endif

    if ( !initvars( NULL ) )
	exit( 1 );

#ifdef HAVE_IPV6
    fodder = sizeof(union sockaddr_union);
    if (getsockname(0, (struct sockaddr *)&su, &fodder)) {
	strcpy( fqdn, "localhost" );
    } else {
	he = gethostbyaddr((char *)&su.sin6.sin6_addr,
			   sizeof(su.sin6.sin6_addr),
			   su.sin.sin_family);
	if ( he && he->h_name) {
	    strncpy(fqdn, he->h_name, 63);
	} else {
	     if (su.sa.sa_family == AF_INET6)
	        inet_ntop(AF_INET6, &su.sin6.sin6_addr, fqdn, 63);
	    else
		inet_ntop(AF_INET,  &su.sin.sin_addr,   fqdn, 63);
	}
    }
#else
    fodder = sizeof(struct sockaddr_in);
    if (getsockname(0, (struct sockaddr *)&sa, &fodder)) {
	strcpy( fqdn, "localhost" );
    } else {
	he = gethostbyaddr((char *)&sa.sin_addr.s_addr,
			   sizeof(sa.sin_addr.s_addr),
			   AF_INET);
	strncpy( fqdn, 
		 he && he->h_name ? he->h_name : inet_ntoa(sa.sin_addr),
		 63 );
    }
#endif
    if ( strncasecmp( fqdn, "localhost", 9 ) == 0 )
	whoami();

    artno = 0;
    verbose = 0;
    umask(2);

    openlog( "leafnode", LOG_PID|LOG_CONS, LOG_NEWS );

    printf("200 Leafnode NNTP Daemon, version %s running at %s\r\n", 
	   version, fqdn);
    fflush(stdout);
 
    if ( !readconfig() ) {
	 printf( "503 Unable to read configuration file, exiting\r\n" );
	 syslog( LOG_NOTICE, "Reading configuration failed, exiting "
	 	 "(see syslog for more information).\n");
	 exit(1);
    }

    pseudogroup = FALSE;

    rereadactive();

    parser();
    fflush(stdout);
/*
    if ( nntpout )
	fprintf( nntpout, "QUIT\r\n" );
*/

    exit(0);
}
