/******************************************************************************\
 gnofin/account.c   $Revision: 1.15 $
 Copyright (C) 1999 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <glib.h>
#include "gnofin.h"
#include "account.h"
#include "record.h"

FinAccount *
fin_account_new (const gchar * name,
		 const gchar * info)
{
  FinAccount * account;

  fin_trace("name=%s, info=%s", name, info);

  g_return_val_if_fail(name != NULL, NULL);
  g_return_val_if_fail(info != NULL, NULL);
  
  account = g_new0(FinAccount, 1);
  account->ref_count = 1;
  account->name = g_strdup(name);
  account->info = g_strdup(info);

  return account;
}

static void
fin_account_free (FinAccount * account)
{
  GList * it;

  fin_trace("");

  g_return_if_fail(account);
  g_return_if_fail(account->ref_count == 0);

  if (account->name)
    g_free(account->name);

  if (account->info)
    g_free(account->info);

  if (account->records)
  {
    for (it=account->records; it; it=it->next)
      fin_record_unref(LIST_GET(FinRecord, it));
    g_list_free(account->records);
  }
}

void
fin_account_ref (FinAccount * account)
{
  g_return_if_fail(account);

  account->ref_count++;
}

void
fin_account_unref (FinAccount * account)
{
  g_return_if_fail(account);

  if (--account->ref_count == 0)
    fin_account_free(account);
}

FinAccount *
fin_account_dup (FinAccount * account)
{
  GList * it;
  FinAccount * dup = g_new0(FinAccount, 1);

  dup->name = g_strdup(account->name);
  dup->info = g_strdup(account->info);
  dup->records = account->records;

  for (it=dup->records; it; it=it->next)
    fin_record_ref(LIST_GET(FinRecord, it));

  return dup;
}

FinAccountSet *
fin_account_set_new ()
{
  FinAccountSet * set;

  fin_trace("");

  set = g_new0(FinAccountSet, 1);
  set->ref_count = 1;

  return set;
}

static void
fin_account_set_free (FinAccountSet * set)
{
  GList * it;

  fin_trace("");

  g_return_if_fail(set);

  if (set->accounts)
  {
    GList * ac;
    for (ac=set->accounts; ac; ac=ac->next)
      fin_account_unref(LIST_GET(FinAccount, ac));
    g_list_free(set->accounts);
  }
  if (set->info_cache)
  {
    for (it=set->info_cache; it; it=it->next)
    {
      FinRecordInfo * info = LIST_GET(FinRecordInfo, it);
      if (info->string)
	g_free(info->string);
      g_free(info);
    }
    g_list_free(set->info_cache);
  }

  if (set->header_line)
    g_free(set->header_line);
}

void
fin_account_set_ref (FinAccountSet * set)
{
  g_return_if_fail(set);

  fin_trace("");
  
  set->ref_count++;
}

void
fin_account_set_unref (FinAccountSet * set)
{
  g_return_if_fail(set);

  fin_trace("");

  if (--set->ref_count == 0)
    fin_account_set_free(set);
}

gint
fin_account_sort_fcn (FinAccount * a, FinAccount * b)
{
  fin_trace("");
  return strcasecmp(a->name, b->name);
}

void
fin_account_refresh_cleared_bal (FinAccount * account)
{
  GList * it;
  money_t cleared_bal = 0;

  fin_trace("");

  g_return_if_fail(account);

  for (it=account->records; it; it=it->next)
  {
    FinRecord * record = LIST_GET(FinRecord, it);

    if (FIN_RECORD_IS_CLEARED(record))
      cleared_bal += record->amount;
  }
  account->cleared_bal = cleared_bal;
}

void
fin_account_store_record (FinAccount * account, 
			  FinRecord  * record,
			  gint	       sort)
{
  fin_trace("");

  g_return_if_fail(account);
  g_return_if_fail(record);

  if (sort)
    account->records = g_list_insert_sorted(account->records, record, (GCompareFunc) fin_record_sort_fcn);
  else
    account->records = g_list_append(account->records, record);
}

void
fin_account_dump (FinAccount * account,
		  FILE       * stream)
{
  GList * it;

  fprintf(stream,"  account at %p\n", account);
  fprintf(stream,"    name   \t\"%s\"\n", account->name);
  fprintf(stream,"    info   \t\"%s\"\n", account->info);
  fprintf(stream,"    records\t(length=%d)\n", g_list_length(account->records));

  for (it=account->records; it; it=it->next)
    fin_record_dump(LIST_GET(FinRecord,it),stream);
}

/*
FinAccountSet *
fin_account_set_copy (FinAccountSet * set,
		      gint	      prune)
{
  FinAccountSet * copy;

  if (prune)
  {
    copy = fin_account_set_new();
  }
  else
  {
    GList * ac;

    copy = fin_account_set_new();

    if (set->header_line)
      copy->header_line = g_strdup(set->header_line);

    for (ac=set->accounts; ac; ac=ac->next)
    {
      GList * it;
      FinAccount * acct1, * acct0 = LIST_GET(FinAccount, ac);

      acct1 = fin_account_new(acct->name, acct->info);

      for (it=acct0->records; it; it=it->next)
      {
        FinRecord * rec1, * rec0 = LIST_GET(FinRecord, it);

	rec1 = fin_record_new(copy->info_cache);
	memcpy(rec1, rec0, sizeof(*rec0));
	rec1->ref_count = 1;

	fin_account_store_record(acct1, rec1, FALSE);
      }

      fin_account_set_store_account(copy, acct1, FALSE);
    }
  }

  return copy;
}
*/

void
fin_account_set_dump (FinAccountSet * set,
		      FILE	    * stream)
{
  GList * it;

  fprintf(stream,"\nDUMP\n");

  fprintf(stream,"account_set at %p\n", set);

  if (set)
  {
    fprintf(stream,"  info_cache\t(length=%d, address=%p)\n", g_list_length(set->info_cache), set->info_cache);
    for (it=set->info_cache; it; it=it->next)
      fin_record_info_dump(LIST_GET(FinRecordInfo,it),stream);

    fprintf(stream,"  accounts\t(length=%d)\n", g_list_length(set->info_cache));
    for (it=set->accounts; it; it=it->next)
      fin_account_dump(LIST_GET(FinAccount,it),stream);
    fprintf(stream,"\n");
  }
}

FinAccount *
fin_account_set_lookup_account (FinAccountSet * set,
				const gchar   * name)
{
  GList * ac;

  fin_trace("");

  g_return_val_if_fail(set, NULL);
  g_return_val_if_fail(name, NULL);

  for (ac=set->accounts; ac; ac=ac->next)
  {
    FinAccount * account = LIST_GET(FinAccount, ac);

    if (strcasecmp(account->name,name) == 0)
      return account;
  }
  return NULL;
}

FinRecord *
fin_account_set_transfer_funds (FinAccountSet * set,
				FinAccount    * account,
				FinRecord     * record,
				FinAccount    * d_acc)
{
  FinRecord * d_rec;

  fin_trace("");

  g_return_val_if_fail(set, NULL);
  g_return_val_if_fail(account, NULL);
  g_return_val_if_fail(record, NULL);
  g_return_val_if_fail(d_acc, NULL);

  /* create the new record */
  d_rec = fin_record_dup(record);
  d_rec->amount = -1 * d_rec->amount; /* amount is of course reversed */
  d_rec->cleared = 0;
  d_rec->type_d.transfer_link = g_new0(FinTransferLink, 1);
  d_rec->type_d.transfer_link->account = account;
  d_rec->type_d.transfer_link->record = record;
  d_rec->have_linked_XFR = 1;

  /* attach to corresponding account */
  d_acc->records = g_list_prepend(d_acc->records, d_rec);
  
  /* link the other */
  record->type_d.transfer_link = g_new0(FinTransferLink, 1);
  record->type_d.transfer_link->account = d_acc;
  record->type_d.transfer_link->record = d_rec;
  record->have_linked_XFR = 1;
  record->ref_count++;

  return d_rec;
}

void
fin_account_set_store_account (FinAccountSet * set,
			       FinAccount    * account,
			       gint	       sort)
{
  fin_trace("");

  g_return_if_fail(set);
  g_return_if_fail(account);

  if (sort)
    set->info_cache = g_list_insert_sorted(set->accounts, account, (GCompareFunc) fin_account_sort_fcn);
  else
    set->accounts = g_list_append(set->accounts, account);
}

void
fin_account_set_store_info_string (FinAccountSet * set,
				   const gchar   * string,
				   gint		   sort)
{
  FinRecordInfo * info;

  fin_trace("");

  g_return_if_fail(set);
  g_return_if_fail(string);

  info = fin_record_info_new(string);
  if (sort)
    set->info_cache = g_list_insert_sorted(set->info_cache, info, (GCompareFunc) fin_record_info_sort_fcn);
  else
    set->info_cache = g_list_append(set->info_cache, info);
}
