/*  Screem:  plugin.c,
 *  Handles loading of plugins
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <gmodule.h>
#include <gnome.h>

#include <sys/types.h>
#include <sys/wait.h>

#include "editor.h"
#include "plugin.h"
#include "site.h"
#include "page.h"

extern Site *current_site;
extern Page *current_page;
extern GtkWidget *app;


static GList *scripts = NULL;

typedef void(*Init)();

static void load_plugins( GtkWidget *output, gchar *directory );

static void load_script_plugin( gchar *file );
static void exec_plugin_script( Script *script );
static void* plugin_exec( Script *script );


void scan_plugins( GtkWidget *output )
{
	const gchar *search_paths[] = {
		PLUGIN_PATH,
		gnome_libdir_file( "" ),
                "/usr/local/lib/",
                "/usr/lib/",
                "/usr/X11R6/lib",
                g_getenv( "SCREEM_LIB_PATH" ),
                NULL
	};
	gint path = 0;
	gchar *directory = NULL;

	/* global paths to look for the plugins are searched in order,
	   if plugins are found in one of them no more are checked */

	do {
		g_free( directory );
                directory = g_strdup_printf( "%sscreem%cplugins%c",
                                             search_paths[ path ],
                                             G_DIR_SEPARATOR,
                                             G_DIR_SEPARATOR );
                path ++;
        } while( ( search_paths[ path ] ) &&
                 ( ! g_file_test( directory, G_FILE_TEST_ISDIR ) ) );

	/* now load the global plugins */
	load_plugins( output, directory );

	/* now load from the users $HOME/.screem directory */
	directory = g_strdup_printf( "%s%c.screem", g_get_home_dir(),
				     G_DIR_SEPARATOR );

	if( g_file_test( directory, G_FILE_TEST_ISDIR ) )
		load_plugins( output, directory );

	g_free( directory );
}

/* load the shared library plugins */
static void load_plugins( GtkWidget *output, gchar *directory )
{
        struct dirent **namelist;
       	gint number;
	gint position;
	gint length;
	gchar *file;
	gchar *module;

	GModule *self;
	GModule *mod;

	Init init;

	g_return_if_fail( directory != NULL );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	number = scandir( directory, &namelist, 0, alphasort );

	for( position = 0; position < number; position ++ ) {  
		file = namelist[ position ]->d_name;
                length = strlen( file );
                if( length > 3 && ! strcmp( ".so", ( file + length - 3 ) ) ) {
                        /* we've got a plugin */
                        module = g_strdup_printf( "%s%s", directory, file );

			gtk_text_insert(GTK_TEXT( output ), 0, 0, 0, "\t", 1 );
			gtk_text_insert(GTK_TEXT( output ), 0, 0, 0, file,-1 );
			gtk_text_insert(GTK_TEXT( output ), 0, 0, 0, "\n", 1 );
			while( gtk_events_pending() )
				gtk_main_iteration();

                        mod = g_module_open( module, G_MODULE_BIND_LAZY );
                        if( mod ) {
                                if( ! g_module_symbol( mod, "init", 
                                                       (gpointer*)&init ) ) {
                                        g_print( _("Module has no init() function\n" ) );
                                        g_module_close( mod );
                                } else {
                                        init();
                                }
                        } else {
                                g_print( _( "Error loading module %s\n%s" ),
                                         module, g_module_error() );
                        }
                        g_free( module );
                } else if( length > 3 && ! strcmp( ".pl", 
						   ( file + length - 3 ) ) ) {
			/* we have a perl script plugin */
                        module = g_strdup_printf( "%s%s", 
						  directory, file );
			load_script_plugin( module );

			g_free( module );
		}

        }

}


/*** The following code is all experimental ***/

static void load_script_plugin( gchar *file )
{
	ScriptFeatures features = 0;
	gchar path[ 1024 ];
	Script *script;
	gchar buffer[ 1024 ];
	gint size;
	FILE *script_pipe;

	gchar *command;

	g_return_if_fail( file != NULL );

	command = g_strconcat( file, " --init", NULL );

	script_pipe = popen( command, "r" );

	g_free( command );

	g_return_if_fail( script_pipe != NULL );

	/* read the result of executing the script with the --init arg */
	size = fread( buffer, 1, 1024, script_pipe );
	buffer[ size ] = '\0';

	sscanf( buffer, "%i %[^]]", (int*)&features, path );
  
	g_print( "%s supports:\n", file );

	if( ! ( features & MODAL ) )
		g_print( "\trunning non-modal\n" );

	g_print( "\tmenu path: %s\n", path );

	g_print( "\n");

	pclose( script_pipe );

	script = g_new( Script, 1 );
	script->exec = g_strdup( file );
	script->modal = ( features && MODAL );
	script->features = features;
	script->path = g_strdup( path );
	scripts = g_list_append( scripts, script );
}

GtkWidget* screem_build_scripts_menu()
{
	GtkWidget *menu_item;
	GtkWidget *item;
	GtkWidget *menu;
	GList *list;
	Script *script;

	g_return_val_if_fail( scripts != NULL, NULL );

	menu_item = gtk_menu_item_new_with_label( _( "Scripts" ) );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM( menu_item ), menu );

	for( list = scripts; list; list = list->next ) {
		script = list->data;
		item = gtk_menu_item_new_with_label( script->path );
		gtk_widget_show( item );
		gtk_signal_connect_object( GTK_OBJECT( item ),
					   "activate",
					   GTK_SIGNAL_FUNC(exec_plugin_script),
					   (gpointer)script );
		gtk_menu_append( GTK_MENU( menu ), item );
	}

	return menu_item;
}


static void exec_plugin_script( Script *script )
{
#ifdef HAVE_PTHREAD
	pthread_t thread;

	if( ( script->features & MODAL ) ) {
		plugin_exec( script );
	} else {
		/* create a new thread */
		pthread_create( &thread, NULL, &plugin_exec, script );
		pthread_detach( &thread );
	}
#else
	/* no pthreads so execute it modally */
	plugin_exec( script );
#endif

}

static void* plugin_exec( Script *script )
{
#define BUFFER_SIZE 1024

#ifndef STDIN
#define STDIN 0
#endif

#ifndef STDOUT
#define STDOUT 1
#endif
	int p[ 2 ]; /* parent write, child read */
	int q[ 2 ]; /* parent read, child write */

	pid_t pid;

	FILE *in;
	FILE *out;

	gint size;
	gint status;
	gchar buffer[ BUFFER_SIZE + 1 ];
	gchar insert[ BUFFER_SIZE + 1 ];

	gchar *temp;
	gint len;
	gint pos;
	gint num;

	if( pipe( p ) < 0 )
		return NULL;
	if( pipe( q ) < 0 )
		return NULL;

	switch( ( pid = fork() ) ) {
	case -1:
		/* error */
		break;
	case 0:
		close( STDIN );
                dup( p[ 0 ] );
                close( p[ 1 ] );
                close( STDOUT );
                dup( q[ 1 ] );
                close( q[ 0 ] );
                execl( "/bin/sh", "/bin/sh", "-c", script->exec, NULL );
                _exit( 0 );
		break;
	default:
		close( p[ 0 ] );
                close( q[ 1 ] );
		out = fdopen( p[ 1 ], "w" );
		in = fdopen( q[ 0 ], "r" );

		while( ( size = fread( buffer, 1, BUFFER_SIZE, in ) ) ) {
			buffer[ size ] = '\0';
			/* interpret what the plugin is asking for */
			temp = strchr( buffer, ' ' );
			if( ! temp )
				temp = buffer + strlen( buffer ) - 1;

			len = temp - buffer;

			if( ! strncmp( "INSERT", buffer, len )  ){
				/* INSERT command */
				sscanf( buffer, "INSERT %i", &pos );

				if( temp )
					temp = strchr( ++temp, ' ' );
				if( temp )
					temp ++;
				if( pos == -1 )
					pos = screem_editor_get_pos();
				screem_editor_insert( pos, temp );
			} else if( ! strncmp( "BUFFER", buffer, len ) ) {
				/* BUFFER command */
				screem_editor_buffer_text();
			} else if(! strncmp("CURRENT", buffer, len ) ) {
				/* CURRENT command */
				temp = screem_editor_get_text( 0, -1 );
				fwrite( temp, 1, strlen( temp ), out );
				g_free( temp );
			} else if( ! strncmp( "DELETE", buffer, len ) ) {
				/* DELETE command */
				sscanf( buffer, "DELETE %i %i", &pos, &num );

				screem_editor_delete_forward( pos, num );
			}
		}
		
		fclose( out );
		fclose( in );

		waitpid( pid, &status, 0 );

		break;
	}

	return NULL;
}
