/*  Screem:  siteUI.c,
 *  a rewrite of Site.c to make it neater and overall better code,
 *  this file contains GUI code that deals with the site
 * 
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <glade/glade.h>

#include "editor.h"
#include "fileops.h"
#include "preferences.h"
#include "preview.h"
#include "site.h"
#include "siteTree.h"
#include "siteTreeUI.h"
#include "siteUI.h"
#include "support.h"

extern Site *current_site;
extern GtkWidget *app;
extern Preferences *cfg;

GladeXML *xml;
static GtkWidget *dialog1 = NULL;

extern gint open_pages;

extern GList *loaded_sites;

static gboolean ignore_change = FALSE;

void screem_site_settings_dialog_destroy_cb( GtkWidget *widget, 
					     GdkEvent *event );
void screem_site_settings_dialog_cb( GtkWidget *widget, gint button );

static void recent_proxy( GtkWidget *widget, gchar *filename );

/**
 * screem_site_save_confirmation:
 * @site:  the site which we want confirmation for
 *
 * inquires if the site is to be saved or not
 *
 * return values: none
 */
void screem_site_save_confirmation( Site *site )
{
	GtkWidget *box;
	gint button;

	box = gnome_message_box_new( _( "Save current site?" ),
				     GNOME_MESSAGE_BOX_QUESTION,
				     GNOME_STOCK_BUTTON_YES,
				     GNOME_STOCK_BUTTON_NO, NULL );
	gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );

	button = gnome_dialog_run_and_close( GNOME_DIALOG( box ) );

	if( button == 0 )
		screem_site_save( site );
}

/**
 * screem_site_close_site:
 *
 * closes the site
 *
 * return values: none
 */
void screem_site_close_site()
{
	Site *site;
	GtkWidget *tree;
	GList *list;
	Page *page;
	gint num;
	GtkWidget *pagebook = gtk_object_get_data( GTK_OBJECT( app ),
						   "pagebook" );

	site = current_site;

	g_return_if_fail( site != NULL );

	/* remove any open page tabs */
	list = screem_site_get_pages( site );
	for( ; list; list = list->next ) {
		page = list->data;
		if( ( num = screem_page_get_tab( page ) ) != -1 ) {
			/* we have an open page */
			if( page == screem_site_get_current_page( site ) )
				screem_editor_buffer_text();
			gtk_notebook_remove_page( GTK_NOTEBOOK( pagebook ), 
						  num );
			screem_page_update_tab_nums( list, num );
		}
	}

	/* remove from the list of loaded sites */
	loaded_sites = g_list_remove( loaded_sites, site );
	/* update the site combo box */
	screem_build_sites_combo_list();

	screem_editor_clear();
	screem_editor_disable();
	screem_preview_display( "" );

	screem_site_save_confirmation( site );

	screem_site_destroy( site );
	current_site = NULL;

	tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );
	gtk_clist_clear( GTK_CLIST( tree ) );

	open_pages = -1;
}

/**
 * screem_site_close_all:
 *
 * closes all the loaded sites
 *
 * return values: none
 */
void screem_site_close_all()
{
	GList *list;

	while( ( list = loaded_sites ) ) {
		if( ! ( current_site = list->data ) )
			break;
		screem_site_close_site();
	}
}

/**
 * screem_site_create_site:
 *
 * creates a new site
 *
 * return values: none
 */
void screem_site_create_site()
{
	Site *site;

	site = screem_site_new();
	screem_site_wizard( site );
}

/**
 * screem_site_open:
 *
 * displays a file selecter to choose which site to open
 *
 * return values: none
 */
void screem_site_open()
{
	Site *site;
	gchar *filename;

	const gchar *name;

	filename = file_select( _( "Select directory to open as a site" ) );

	/* we didn't select one */
	if( ! filename )
		return;

	screem_site_open_with_filename( filename );
}

/**
 * screem_site_open_with_filename:
 * @filename:  the path name of the site
 *
 * opens the site with the given filename
 *
 * return values: gboolean
 */
gboolean screem_site_open_with_filename( gchar *filename )
{
	Site *site;
	gchar *path = NULL;
	const gchar *page_path;
	gchar cwd[ 16384 ];

	GtkWidget *combo = gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	g_return_val_if_fail( filename != NULL, FALSE );

	/* is the filename absolute? */
	if( filename[ 0 ] != G_DIR_SEPARATOR ) {
		/* no */
		getcwd( cwd, 16384 );
		path = g_strconcat( cwd, G_DIR_SEPARATOR_S, filename, NULL );
	}

	if( ! g_file_test( filename, G_FILE_TEST_ISDIR ) ) {
		if( ! path )
			path = g_dirname( filename );
		else {
			filename = g_dirname( path );
			g_free( path );
			path = filename;
		}
	} else if( ! path ) {
		path = g_strdup( filename );
	}

	site = screem_site_new();
	screem_site_set_pathname( site, path );

	g_free( path );

	if( ! screem_site_load( site ) ) {
		/* we failed to load the site */
		screem_site_destroy( site );
		return FALSE;
	}

	/* add this site to the list of loaded sites */
	loaded_sites = g_list_append( loaded_sites, site );
     	/* build the combo options */
	screem_build_sites_combo_list();
	/* set the combo entry to be the name of the current site */
	gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ),
			    screem_site_get_name( site ) );

	current_site = site;

	/* create the site file tree */
	refresh_file_tree();
	
	/* add to recent sites list */
	page_path = screem_site_get_pathname( site );
	cfg->recent_sites = g_list_append( cfg->recent_sites, 
					   g_strdup( page_path  ) );
	build_recent_sites_menu();

	/* if we are importing a site open the site settings dialog */
	if( screem_site_get_is_import( site ) ) {
		screem_site_settings_dialog();
		screem_site_set_is_import( site, FALSE );
	}

	return TRUE;
}

/**
 * screem_site_save_as:
 *
 * save the current site in a directory other than site->pathname
 *
 * return values: none
 */
void screem_site_save_as()
{
	GList *list;
	gchar *filename;
	gchar *new;
	Page *page;
	Site *site;

	const gchar *pathname;
	const gchar *src;

	site = current_site;

	g_return_if_fail( site != NULL );

	filename = file_select( _( "Select directory to save site as" ) );

	/* directory was not selected */
	if( ! filename )
		return;

	src = screem_site_get_pathname( site );

	if( ! copy_dir( src, filename, FALSE ) ) {
		/* failed to copy the site to the new location */

	}

	/* modify the pathnames of all the pages */
	for( list = screem_site_get_pages( site ); list; list = list->next ) {
		page = (Page*)list->data;
		pathname = screem_page_get_pathname( page );
		new = g_strdup_printf( "%s%s", filename, 
				       pathname + strlen( src ) );
		screem_page_set_pathname( page, new );
		g_free( new );
	}

	/* save the site */
	screem_site_set_pathname( site, filename );
	screem_site_save( site );

	refresh_file_tree();
}

/**
 * screem_site_save_proxy
 *
 * callback for toolbar/menu items to save the current site
 *
 * return values: none
 */
void screem_site_save_proxy()
{
	g_return_if_fail( current_site != NULL );

	screem_editor_buffer_text();
	screem_site_save( current_site );
}

/**
 * screem_site_settings_dialog:
 * opens the site settings dialog, this is the current 0.1 releases
 * site wizard
 *
 * return values: none
 */
void screem_site_settings_dialog()
{
     	GtkWidget *entry;
	GtkWidget *toggle;
	Site *site;

	GtkWidget *object;

	site = current_site;

	g_return_if_fail( site );
	
	if( dialog1 ) {
		gdk_window_raise( dialog1->window );
                gdk_window_show( dialog1->window );
                return;
	}

	xml = glade_xml_new( GLADE_PATH"/site_settings.glade",
			     "site_settings_dialog" );

	/* kludge time boys and girls */
	toggle = glade_xml_get_widget( xml, "cvs_check" );
	object = glade_xml_get_widget( xml, "cvs_file_entry" );
	gtk_signal_connect_object( GTK_OBJECT( toggle ), "toggled",
				   GTK_SIGNAL_FUNC( change_state ),
				   (gpointer)object );

	toggle = glade_xml_get_widget( xml, "template_check" );
	object = glade_xml_get_widget( xml, "template_file_entry" );
	gtk_signal_connect_object( GTK_OBJECT( toggle ), "toggled",
				   GTK_SIGNAL_FUNC( change_state ),
				   (gpointer)object );

	glade_xml_signal_autoconnect( xml );

	dialog1 = glade_xml_get_widget( xml, "site_settings_dialog" );

	/* fill in the details */
	/* name */
	entry = glade_xml_get_widget( xml, "site_name_entry" );
	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_name( site ) );
	
	/* path */
	entry = glade_xml_get_widget( xml, "site_path_entry" );
	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_pathname( site ) );

	/* cvs root */
	entry = glade_xml_get_widget( xml, "cvs_rep_path" );
	if( screem_site_get_cvs_root( site ) ) {
		toggle = glade_xml_get_widget( xml, "cvs_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		gtk_entry_set_text( GTK_ENTRY( entry ),
				    screem_site_get_cvs_root( site ) );
	}

	/* template path */
	entry = glade_xml_get_widget( xml, "template_entry" );
	if( screem_site_get_template_path( site ) ) {
		toggle = glade_xml_get_widget( xml, "template_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		gtk_entry_set_text(GTK_ENTRY( entry ),
				   screem_site_get_template_path(site));
	}

	/* remote url */
	entry = glade_xml_get_widget( xml, "remote_url_entry" );
       	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_remote_url( site ) );

	/* http url */
	entry = glade_xml_get_widget( xml, "http_url_entry" );
       	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_http_url( site ) );

	/* remote method */
	entry = glade_xml_get_widget( xml, "upload_method_entry" );
       	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_remote_method( site ) );

	/* remote path */
	entry = glade_xml_get_widget( xml, "remote_path_entry" );
	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_remote_path( site ) );

	/* remote user */
	entry = glade_xml_get_widget( xml, "remote_user_entry" );
	gtk_entry_set_text( GTK_ENTRY( entry ), 
			    screem_site_get_remote_user( site ) );
	
	/* remote password */
	entry = glade_xml_get_widget( xml, "remote_password_entry" );
	gtk_entry_set_text( GTK_ENTRY( entry ),
			    screem_site_get_remote_pass( site ) );
	
	/* attach the site to the dialog */
	gtk_object_set_data( GTK_OBJECT( dialog1 ), "site", site );

}

void screem_site_settings_dialog_destroy_cb( GtkWidget *widget, 
					     GdkEvent *event )
{
	dialog1 = NULL;
}

void screem_site_settings_dialog_cb( GtkWidget *widget, gint button ) 
{
	GtkWidget *entry;
	GtkWidget *toggle;
	Site *site;

	if( button < 2 ) { /* ok or apply clicked */
		site = gtk_object_get_data(GTK_OBJECT( widget ), "site");

		/* name */
		entry = glade_xml_get_widget( xml, "site_name_entry" );
		screem_site_set_name( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		/* path */
		entry = glade_xml_get_widget( xml, "site_path_entry" );
		screem_site_set_pathname( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		/* cvs root */
		toggle = glade_xml_get_widget( xml, "cvs_check" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
			entry = glade_xml_get_widget(xml,"cvs_rep_path");
			screem_site_set_cvs_root( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		} else {
			screem_site_set_use_cvs( site, FALSE );
		}

		/* template */
		toggle = glade_xml_get_widget( xml, "template_check" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
			entry = glade_xml_get_widget( xml, "template_entry" );
			screem_site_set_template_path( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		} else {
			screem_site_set_use_template( site, FALSE );
		}

		/* remote url */
		entry = glade_xml_get_widget( xml, "remote_url_entry" );
		screem_site_set_remote_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
		/* http url */
		entry = glade_xml_get_widget( xml, "http_url_entry" );
		screem_site_set_http_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote method */
		entry = glade_xml_get_widget( xml,"upload_method_entry");
		screem_site_set_remote_method( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote path */
		entry = glade_xml_get_widget( xml, "remote_path_entry" );
		screem_site_set_remote_path( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote user */
		entry = glade_xml_get_widget( xml, "remote_user_entry" );
		screem_site_set_remote_user( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

		/* remote password */
		entry = glade_xml_get_widget( xml, 
					      "remote_password_entry");
		screem_site_set_remote_pass( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	}

	if( ( button == 0 ) || ( button == 2 ) ) {
		/* close the dialog */
		gtk_widget_destroy( widget );
		dialog1 = NULL;
	}
}

void build_recent_sites_menu()
{

	GnomeUIInfo *menu;
	gchar *path = N_( "_File/_Recent Sites/" );
        int i;
     
	GList *list = cfg->recent_sites;
	GList *prev = NULL;

	if( ( i = g_list_length( list ) ) ) {
                gnome_app_remove_menu_range( GNOME_APP( app ), path, 0, i );
        } else {
		return;
	}

	list = g_list_last( list );
	for( i = 3; list && i > 0; i --, list = list->prev )
		prev = list;
 
	if( ! list )
		list = prev;

	cfg->recent_sites = list;
	list->prev = NULL;

	for( list = cfg->recent_sites; list; list = list->next ) {
		menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
		menu->label = escape_path( list->data );
		
		menu->type = GNOME_APP_UI_ITEM;
		menu->hint = NULL;
		
		menu->moreinfo = recent_proxy;
		
		menu->user_data = g_strdup( list->data );
		menu->unused_data = NULL;
		menu->pixmap_type = 0;
		menu->pixmap_info = NULL;
		menu->accelerator_key = 0;
		
		(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
		
		gnome_app_insert_menus( GNOME_APP(app), path, menu );
	}

	save_preferences();
}

static void recent_proxy( GtkWidget *widget, gchar *filename )
{
	/*screem_site_close_site();*/
	screem_site_open_with_filename( filename );
}

void screem_build_sites_combo_list()
{
	GtkWidget *combo = gtk_object_get_data(GTK_OBJECT( app ), "sitecombo");

	GList *li;
	GList *li2;
	Site *site;
	const gchar *name;

	GtkWidget *list;

	ignore_change = TRUE;

	li = NULL;
	for( li2 = loaded_sites; li2; li2 = li2->next ) {
		site = li2->data;
		name = screem_site_get_name( site );
		/* cast to gchar* from const gchar* to avoid compiler
		   warning - name will not be freed so its ok, if a little
		   dirty */
		li = g_list_append( li, (gchar*)name );
	}
     	/* make li the combo's list entries */
	if( ! li )
		li = g_list_append( li, "" );
	gtk_combo_set_popdown_strings( GTK_COMBO( combo ), li );
	g_list_free( li );
}

void screem_site_change( GtkWidget *widget, gpointer data )
{
	gchar *name;
	GList *list;
	Site *site = NULL;

	const gchar *n;

	const gchar *page_path;
	gchar *pname;
	GtkWidget *label;
	GtkWidget *junk;

	Page *page;

	gint num;
	gint c;
	GtkWidget *book = gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );

	if( ignore_change ) {
		ignore_change = FALSE;
		return;
	}

	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	/* find name in the list of loaded sites */
	for( list = loaded_sites; list && ! site; list = list->next ) {
		n = screem_site_get_name( list->data );
		if( ! strcmp( n, name ) )
			site = list->data;
	}

	if( site ) {
		gtk_object_set_data( GTK_OBJECT( book ), "hack", (gpointer)1 );
		/* remove old file tabs */
		if( current_site ) {
			num = screem_site_get_open_pages( current_site );
			while( num-- >= 0 ) {
				gtk_notebook_remove_page( GTK_NOTEBOOK( book ),
							  num );
			}
		}

		/* create new ones, FIXME: do this better */
		num = screem_site_get_open_pages( site );
		for( c = 0; num >= 0; num --, c++ ) {
			list = screem_site_get_pages( site );
			while( list ) {
				page_path = 
					screem_page_get_pathname( list->data );
				if( screem_page_get_tab( list->data ) == c ) {
					pname = g_dirname( page_path );
					label = gtk_label_new(page_path +
							      strlen( pname ) +
							      1 );
					g_free( pname );
					junk = gtk_hseparator_new();
					gtk_widget_show( junk );
					gtk_widget_set_usize( junk, 0, 0 );
					gtk_widget_show( label );
					gtk_notebook_append_page( GTK_NOTEBOOK( book ), junk, label );
					break;
				}
				list = list->next;
			}
		}
		
		/* set current site to be this site */
		current_site = site;
       
		/* show the file tree */
		refresh_file_tree();

		gtk_object_set_data( GTK_OBJECT( book ), "hack", (gpointer)0 );

		page = screem_site_get_current_page( site );
		screem_editor_clear();
		if( page ) {
			screem_editor_insert( 0, page->data );
			gtk_notebook_set_page( GTK_NOTEBOOK( book ),
					       screem_page_get_tab( page ) );
		} else {
			screem_editor_disable();
		}

	}
}

void screem_site_rename_pages_with_path( Site *site, gchar *path, 
					 gchar *new_path )
{
	GList *pages;
	Page *page;
	gint num;
	GtkWidget *book;
	GtkWidget *note;
	GtkWidget *label;
	const gchar *page_path;
	gchar *new_page_path;
	gchar *pname;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );
	g_return_if_fail( new_path != NULL );

	book = gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );
	pages = screem_site_get_pages( site );

	while( pages ) {
		page = pages->data;
		page_path = screem_page_get_pathname( page );
		if( ! strncmp( page_path, path, strlen( path ) ) ) {
			/* match found, rename page path */
			new_page_path = g_strconcat( new_path,
						     page_path + strlen(path),
						     NULL );
			screem_page_set_pathname( page, new_page_path );
			if( ( num = screem_page_get_tab( page ) ) > -1 ) {
				/* page is open, rename the tab */
				note = gtk_notebook_get_nth_page( GTK_NOTEBOOK( book ), num );
				pname = g_dirname( new_page_path );
				label = gtk_label_new( new_page_path + 
						       strlen( pname ) + 1 );
				g_free( pname );
				gtk_notebook_set_tab_label( GTK_NOTEBOOK(book),
							    note, label );
			}
			g_free( new_page_path );
		}
		pages = pages->next;
	}
}

void screem_site_purge_pages_with_path( Site *site, gchar *path )
{
	GList *pages;
	Page *page;
	gint num;
	GtkWidget *book;
	const gchar *page_path;

	g_return_if_fail( site != NULL );
	g_return_if_fail( path != NULL );

	book = gtk_object_get_data( GTK_OBJECT( app ), "pagebook" );
	pages = screem_site_get_pages( site );

	while( pages ) {
		page = pages->data;
		page_path = screem_page_get_pathname( page );
		if( ! strncmp( page_path, path, strlen( path ) ) ) {
			/* match found, remove the page */
			if( ( num = screem_page_get_tab( page ) ) > -1 ) {
				/* page is open */
				gtk_notebook_remove_page( GTK_NOTEBOOK( book ),
							  num );
			}
			/* remove page from site */
			page = screem_site_remove_page( site, page_path );
			screem_page_destroy( page );
			pages = screem_site_get_pages( site );
			if( num > -1 )
				screem_page_update_tab_nums( pages, num );

		} else {
			pages = pages->next;
		}
	}
}
