/* 
   HTTP authentication routines
   Copyright (C) 1999, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef HTTPAUTH_H
#define HTTPAUTH_H

#include <sys/types.h>

#include "md5.h"
#include "common.h"

/* HTTP Authentication - a full client implementation of RFC2617.
 * 
 * TODO:
 *  - Bring the external 'can authenticate' state in?
 *  - Add handling for domains and realms
 *  - Allow on-the-fly request for password
 */

/* To use:
 * 
 * 1. Declare a http_auth_session_t variable, to hold the details for
 * the server. If you support Proxy authentication, declare another
 * one for that.
 *
 * 2. To begin a new session, call http_auth_init() with this variable.
 *
 * 3. Before sending a request, pass http_auth_new_request it's details.
 * 
 * 4. If the server has previously demanded authentication credentials, 
 * call http_auth_request() and add your 'Authentication:' header to the
 * request.
 * 
 * 5. Send the request.
 *
 * 6. Read the response:
 *  - Pass the value of the '(Proxy|WWW)-Authenticate' header to 
 *    http_auth_challenge.
 *  - If there is a '(Proxy-)Authentication-Info', save its value.
 *  - Pass each block of the response entity-body to http_auth_response_body.
 *
 * 7. After reading the response, call http_auth_verify_response to
 * check whether the server was authenticated okay, passing the previously
 * saved '(Proxy-)Authentication-Info' value.
 */

/* The authentication scheme we are using */
typedef enum {
    http_auth_scheme_basic,
    http_auth_scheme_digest,
    http_auth_scheme_none
} http_auth_scheme_t;

typedef enum { 
    http_auth_alg_md5,
    http_auth_alg_md5_sess,
    http_auth_alg_unknown
} http_auth_algorithm_t;

/* Selected method of qop which the client is using */
typedef enum {
    http_auth_qop_none,
    http_auth_qop_auth,
    http_auth_qop_auth_int
} http_auth_qop_t;

/* Authentication session state. */
typedef struct {
    /* The scheme used for this authentication session */
    http_auth_scheme_t scheme;
    /* The username and password we are using to authenticate with */
    const char *username;
    const char *password;
    /* The method and URI we are using for the current request */
    const char *uri;
    const char *method;
    /* Whether we have a request body for the current request */
    bool got_body;
    /* And what the body is - stream or buffer */
    FILE *body_stream;
    const char *body_buffer;
    /* This used for Basic auth */
    char *basic; 
    /* These all used for Digest auth */
    char *unq_realm;
    char *unq_nonce;
    char *unq_cnonce;
    char *opaque;
    http_auth_qop_t qop;
    http_auth_algorithm_t alg;
    int nonce_count;
    /* The ASCII representation of the session's H(A1) value */
    char h_a1[33];
    /* Used for calculation of H(entity-body) of the response */
    struct md5_ctx response_body;
    /* Temporary store for half of the Request-Digest
     * (an optimisation - used in the response-digest calculation) */
    struct md5_ctx stored_rdig;
} http_auth_session_t;

/* Initializes the authentication state for the given session,
 * which will use the given username and password. */
void http_auth_init( http_auth_session_t *sess, 
		     const char *username, const char *password );

/* Finishes off the given authentication session, freeing
 * any memory used. */
void http_auth_finish( http_auth_session_t *sess );

/* Call this before sending a new request.
 * Pass ONE OF body_buffer as non-NULL if the request will 
 * include an entity-body. If body_buffer is non-NULL, it MUST
 * be \0-terminated. */
void http_auth_new_request( http_auth_session_t *sess,
			    const char *method, const char *uri,
			    const char *body_buffer, FILE *body_stream );

/* Returns the value of the authentication field if one is to be sent,
 * else NULL. The return value will be taken from malloc()'ed memory,
 * so should be free()'ed after use. */
char *http_auth_request( http_auth_session_t *sess );

/* Pass this the value of the "(Proxy,WWW)-Authenticate: " header field.
 * Returns true if we can now authenticate ourselves with the server.
 */
bool http_auth_challenge( http_auth_session_t *sess, const char *value );

/* As you receive sections of the response entity-body, pass them to 
 * this function. */
void http_auth_response_body( http_auth_session_t *sess, 
			      const char *buffer, size_t buffer_len );

/* If you receive a "(Proxy-)Authentication-Info:" header, pass its value to
 * this function. Returns true if this successfully authenticates
 * the response as coming from the server, and false if it hasn't. */
bool http_auth_verify_response( http_auth_session_t *sess, const char *value );

#endif /* HTTPAUTH_H */
