
/*-
# X-BASED PYRAMINX(tm)
#
#  xpyraminx.c
#
###
#
#  Copyright (c) 1994 - 96	David Albert Bagley, bagleyd@hertz.njit.edu
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 4: 94/05/31 Xt
  Version 3: 93/04/01 Motif
  Version 2: 92/01/29 XView
  Version 1: 91/03/19 SunView
*/

#include <stdlib.h>
#include <stdio.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin cuserid
#else
#ifndef apollo
#include <unistd.h>
#endif
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#include "Pyraminx.h"
#include "pyraminx.xbm"

#ifndef SCOREFILE
#define SCOREFILE "/usr/games/lib/pyraminx.scores"
#endif

/* The following are in PyraminxP.h also */
#define MINTETRAS 1
#define PERIOD2 2
#define PERIOD3 3
#define BOTH 4
#define MAXMODES 3

#define MAXTETRAS 7
#define MAXRECORD 32767
#define MAXPROGNAME 80
#define MAXNAME 256

static void Initialize(Widget w);
static void CallbackPyraminx(Widget w, caddr_t clientData, pyraminxCallbackStruct * callData);

static void PrintRecord(int size, int mode, Boolean orient, Boolean sticky, Boolean practice, char *record);
static int  HandleSolved(int counter, int size, int mode, Boolean orient, Boolean sticky);
static void PrintState(Widget w, char *prog, int mode, int size, Boolean sticky, int moves, char *record, char *message);
static void ReadRecords(void);
static void WriteRecords(void);

static Arg  arg[5];
static int  pyraminxRecord[MAXMODES][2][MAXTETRAS - MINTETRAS + 2];
static int  movesDsp = 0;
static char progDsp[64] = "xpyraminx";
static char recordDsp[16] = "INF";
static char messageDsp[128] = "Welcome";
static char titleDsp[256] = "";
static int  oldSize;

static void
Usage(void)
{
	(void) fprintf(stderr, "usage: xpyraminx\n");
	(void) fprintf(stderr,
	     "\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
	       "\t[-display [{host}]:[{vs}]] [-fg {color}] [-bg {color}]\n");
	(void) fprintf(stderr,
	      "\t[-{border|bd} {color}] [-mono] [-{size {int} | sticky}]\n");
	(void) fprintf(stderr,
		 "\t[-{mode {int} | both}] [-[no]orient] [-[no]practice]\n");
	(void) fprintf(stderr,
		       "\t[-face{0|1|2|3} {color}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{"-fg", "*pyraminx.Foreground", XrmoptionSepArg, NULL},
	{"-bg", "*Background", XrmoptionSepArg, NULL},
	{"-foreground", "*pyraminx.Foreground", XrmoptionSepArg, NULL},
	{"-background", "*Background", XrmoptionSepArg, NULL},
	{"-border", "*pyraminx.pieceBorder", XrmoptionSepArg, NULL},
	{"-bd", "*pyraminx.pieceBorder", XrmoptionSepArg, NULL},
	{"-mono", "*pyraminx.mono", XrmoptionNoArg, "TRUE"},
	{"-size", "*pyraminx.size", XrmoptionSepArg, NULL},
	{"-sticky", "*pyraminx.sticky", XrmoptionNoArg, "FALSE"},
	{"-mode", "*pyraminx.mode", XrmoptionSepArg, NULL},
	{"-both", "*pyraminx.mode", XrmoptionNoArg, "4"},
	{"-orient", "*pyraminx.orient", XrmoptionNoArg, "TRUE"},
	{"-noorient", "*pyraminx.orient", XrmoptionNoArg, "FALSE"},
	{"-practice", "*pyraminx.practice", XrmoptionNoArg, "TRUE"},
	{"-nopractice", "*pyraminx.practice", XrmoptionNoArg, "FALSE"},
	{"-face0", "*pyraminx.faceColor0", XrmoptionSepArg, NULL},
	{"-face1", "*pyraminx.faceColor1", XrmoptionSepArg, NULL},
	{"-face2", "*pyraminx.faceColor2", XrmoptionSepArg, NULL},
	{"-face3", "*pyraminx.faceColor3", XrmoptionSepArg, NULL}
};

int
main(int argc, char **argv)
{
	Widget      toplevel, pyraminx;

	toplevel = XtInitialize(argv[0], "Pyraminx",
				options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage();

	XtSetArg(arg[0], XtNiconPixmap,
		 XCreateBitmapFromData(XtDisplay(toplevel),
				       RootWindowOfScreen(XtScreen(toplevel)),
		   (char *) pyraminx_bits, pyraminx_width, pyraminx_height));
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(toplevel, arg, 2);
	pyraminx = XtCreateManagedWidget("pyraminx", pyraminxWidgetClass, toplevel,
					 NULL, 0);
	XtAddCallback(pyraminx, XtNselectCallback, (XtCallbackProc) CallbackPyraminx,
		      (XtPointer) NULL);
	Initialize(pyraminx);
	XtRealizeWidget(toplevel);
	XGrabButton(XtDisplay(pyraminx), AnyButton, AnyModifier, XtWindow(pyraminx),
		TRUE, ButtonPressMask | ButtonMotionMask | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, XtWindow(pyraminx),
		    XCreateFontCursor(XtDisplay(pyraminx), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}

static void
Initialize(Widget w)
{
	int         size, mode;
	Boolean     orient, sticky, practice;

	XtVaSetValues(w,
		      XtNstart, FALSE,
		      NULL);
	XtVaGetValues(w,
		      XtNsize, &size,
		      XtNmode, &mode,
		      XtNorient, &orient,
		      XtNsticky, &sticky,
		      XtNpractice, &practice,
		      NULL);
	ReadRecords();
	PrintRecord(size, mode, orient, sticky, practice, recordDsp);
	oldSize = size;
	PrintState(XtParent(w), progDsp, mode, size, sticky, movesDsp,
		   recordDsp, messageDsp);
}

static void
CallbackPyraminx(Widget w, caddr_t clientData, pyraminxCallbackStruct * callData)
{
	int         size, mode;
	Boolean     orient, sticky, practice, start;

	XtVaGetValues(w,
		      XtNsize, &size,
		      XtNorient, &orient,
		      XtNmode, &mode,
		      XtNorient, &orient,
		      XtNsticky, &sticky,
		      XtNpractice, &practice,
		      XtNstart, &start,
		      NULL);
	(void) strcpy(messageDsp, "");
	switch (callData->reason) {
		case PYRAMINX_RESTORE:
			if (practice)
				(void) strcpy(recordDsp, "practice");
			movesDsp = 0;
			break;
		case PYRAMINX_RESET:
			movesDsp = 0;
			break;
		case PYRAMINX_AMBIGUOUS:
			(void) strcpy(messageDsp, "Ambiguous move");
			break;
		case PYRAMINX_ILLEGAL:
			if (practice || start)
				(void) strcpy(messageDsp, "Illegal move");
			else
				(void) strcpy(messageDsp, "Randomize to start");
			break;
		case PYRAMINX_MOVED:
			movesDsp++;
#ifdef DEBUG
			if (movesDsp > 256)
				exit(1);
#endif
			XtSetArg(arg[0], XtNstart, TRUE);
			XtSetValues(w, arg, 1);
			break;
		case PYRAMINX_CONTROL:
			return;
		case PYRAMINX_SOLVED:
			if (practice)
				movesDsp = 0;
			else {
				if (HandleSolved(movesDsp, size, mode, orient, sticky))
					(void) sprintf(messageDsp, "Congratulations %s!!", getlogin());
				else
					(void) strcpy(messageDsp, "Solved!");
			}
			XtSetArg(arg[0], XtNstart, FALSE);
			XtSetValues(w, arg, 1);
			break;
		case PYRAMINX_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strcpy(messageDsp, "Randomize to start");
			PrintRecord(size, mode, orient, sticky, practice, recordDsp);
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, FALSE);
			XtSetValues(w, arg, 2);
			break;
		case PYRAMINX_RANDOMIZE:
			movesDsp = 0;
			XtSetArg(arg[0], XtNpractice, FALSE);
			XtSetArg(arg[1], XtNstart, FALSE);
			XtSetValues(w, arg, 2);
			break;
		case PYRAMINX_DEC:
			if (!sticky) {
				movesDsp = 0;
				size--;
				oldSize = size;
				PrintRecord(size, mode, orient, sticky, practice, recordDsp);
				XtSetArg(arg[0], XtNsize, size);
				XtSetValues(w, arg, 1);
			}
			break;
		case PYRAMINX_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(size, mode, orient, sticky, practice, recordDsp);
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
			break;
		case PYRAMINX_INC:
			if (!sticky) {
				movesDsp = 0;
				size++;
				oldSize = size;
				PrintRecord(size, mode, orient, sticky, practice, recordDsp);
				XtSetArg(arg[0], XtNsize, size);
				XtSetValues(w, arg, 1);
			}
			break;
		case PYRAMINX_PERIOD2:
		case PYRAMINX_PERIOD3:
		case PYRAMINX_BOTH:
			movesDsp = 0;
			mode = callData->reason - PYRAMINX_PERIOD2 + PERIOD2;
			PrintRecord(size, mode, orient, sticky, practice, recordDsp);
			XtSetArg(arg[0], XtNmode, mode);
			XtSetValues(w, arg, 1);
			break;
		case PYRAMINX_STICKY:
			movesDsp = 0;
			sticky = !sticky;
			if (sticky)
				size = 4;
			else
				size = oldSize;
			PrintRecord(size, mode, orient, sticky, practice, recordDsp);
			XtSetArg(arg[0], XtNsticky, sticky);
			XtSetArg(arg[1], XtNsize, size);
			XtSetValues(w, arg, 2);
			break;
		case PYRAMINX_COMPUTED:
			XtSetArg(arg[0], XtNstart, FALSE);
			XtSetValues(w, arg, 1);
			break;
		case PYRAMINX_UNDO:
			movesDsp--;
			XtSetArg(arg[0], XtNstart, TRUE);
			XtSetValues(w, arg, 1);
			break;
	}
	PrintState(XtParent(w), progDsp, mode, size, sticky, movesDsp,
		   recordDsp, messageDsp);
}

static void
PrintRecord(int size, int mode, Boolean orient, Boolean sticky, Boolean practice, char *record)
{
	int         i = mode - PERIOD2;
	int         j = (orient) ? 1 : 0;
	int         k = (sticky) ? MAXTETRAS - MINTETRAS + 1 : size - MINTETRAS;

	if (practice)
		(void) strcpy(record, "practice");
	else if (!sticky && size > MAXTETRAS)
		(void) strcpy(record, "NOT RECORDED");
	else if (pyraminxRecord[i][j][k] >= MAXRECORD)
		(void) strcpy(record, "NEVER");
	else
		(void) sprintf(record, "%d", pyraminxRecord[i][j][k]);
}

static int
HandleSolved(int counter, int size, int mode, Boolean orient, Boolean sticky)
{
	int         i = mode - PERIOD2;
	int         j = (orient) ? 1 : 0;
	int         k = (sticky) ? MAXTETRAS - MINTETRAS + 1 : size - MINTETRAS;

	if ((sticky || size <= MAXTETRAS) && counter < pyraminxRecord[i][j][k]) {
		pyraminxRecord[i][j][k] = counter;
		if (size < 4 || mode == PERIOD2 || (orient &&
		       (pyraminxRecord[i][j][k] < pyraminxRecord[i][!j][k])))
			pyraminxRecord[i][!j][k] = counter;
		WriteRecords();
		(void) sprintf(recordDsp, "%d", counter);
		return TRUE;
	}
	return FALSE;
}

static void
PrintState(Widget w, char *prog, int mode, int size, Boolean sticky, int moves, char *record, char *message)
{
	char        ss[10], mb[10];

	if (sticky)
		(void) strcpy(ss, "sticky");
	else
		(void) sprintf(ss, "%d", size);
	if (mode == BOTH)
		(void) strcpy(mb, "both");
	else
		(void) sprintf(mb, "%d", mode);
	(void) sprintf(titleDsp, "%s.%s: %s@ (%d/%s) - %s", prog, mb, ss, moves,
		       record, message);
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
}

static void
ReadRecords(void)
{
	FILE       *fp;
	int         i, n, mode, orient;

	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++)
			for (i = 0; i <= MAXTETRAS - MINTETRAS + 1; i++)
				pyraminxRecord[mode][orient][i] = MAXRECORD;
	if ((fp = fopen(SCOREFILE, "r")) == NULL)
		(void) sprintf(messageDsp, "Can not open %s, taking defaults.", SCOREFILE);
	else {
		for (mode = 0; mode < MAXMODES; mode++)
			for (orient = 0; orient < 2; orient++)
				for (i = 0; i <= MAXTETRAS - MINTETRAS + 1; i++) {
					(void) fscanf(fp, "%d", &n);
					pyraminxRecord[mode][orient][i] = n;
				}
		(void) fclose(fp);
	}
}

static void
WriteRecords(void)
{
	FILE       *fp;
	int         i, mode, orient;

	if ((fp = fopen(SCOREFILE, "w")) == NULL)
		(void) sprintf(messageDsp, "Can not write to %s.", SCOREFILE);
	else {
		for (mode = 0; mode < MAXMODES; mode++) {
			for (orient = 0; orient < 2; orient++) {
				for (i = 0; i <= MAXTETRAS - MINTETRAS + 1; i++)
					(void) fprintf(fp, "%d ", pyraminxRecord[mode][orient][i]);
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
		(void) fclose(fp);
	}
}
