/********************************************************************\
**                             _________________________________    **
**   A n t h o n y            |________    __    __    _________|   **
**                                     |  |o_|  |o_|  |             **
**           T h y s s e n           __|   __    __   |__           **
**                                __|   __|  |  |  |__   |__        **
**  `` Dragon Computing! ''    __|   __|     |  |     |__   |__     **
**                            |_____|        |__|        |_____|    **
**                                                                  **
\********************************************************************/
/* 
** bitmap.c
**
**   This module represents a complete re-write of the data structures used
** by xbmbrowser to orginise the bitmaps and widgets. These structures
** although initially created in get_files(), are static to this module.
** Previously the data structure did not allow for expandsion of bitmap
** types, storage of pixel colors used, or for checks on modification
** times.
**                         Anthony Thyssen <anthony@dragon.cit.gu.edu.au>
*/
#include "xbmbrowser.h"

static Item   *file_list = NULL;     /* current list of all files */
static Item   *new_file_list = NULL; /* new file list just scanned (to merge) */
static Item   *old_file_list = NULL; /* old file list already loaded (merge) */
static Item  **last_link = NULL;     /* point to the last next ptr in merge */

static Widget *widget_array = NULL;  /* array of widgets available to use */
static int     allocated = 0;        /* number of widgets in above array */
static int     managed = 0;          /* number of widgets being managed */

static int items_allocated = 0;      /* current number of allocated items */
static int num_files;                /* number files in file_list */
static int num_visible;              /* number items which are visible */
static int file_counts[(int)NumFileTypes]; /* count of the various file types */

static char *dir_list[256];          /* directories in directory menu */
static int   num_dirs;               /* number of directories in this list */

/*--------------------------------------------------------------------------*/

Item *
alloc_item()
/* allocate and default a file item */
{
  Item *item = (Item *) XtMalloc( sizeof(Item) );

  item->fname[0] = '\0';       /* set defaults */
  item->info[0]  = '\0';   
  item->type     = Unknown;
  item->bitmap   = (Pixmap)None;
  item->index    = -1;
  item->next     = NULL;

  items_allocated++;
  return item;
}


static void
free_bitmap(item)
/* Free any bitmap that may be currently accociated with this item.
** This is usally done either as a prelude to freeing the items
** memory, or hiding the item from the user (why bother?).
*/
  Item *item;
{
  if( item->bitmap != (Pixmap)None &&
      item->bitmap != filesyms[item->type] ) {
    XFreePixmap(display, item->bitmap );
#ifdef DO_XPMS
    if( item->type == Xpm ) {
      /* free colors and attributes that were used by this pixmap only */
      XFreeColors(display, colormap, item->attr.pixels, item->attr.npixels, 0);
      XpmFreeAttributes((XpmAttributes *)&item->attr);
    }
#endif
  }
  item->bitmap = (Pixmap)None;
}


Item *
free_item(item)
/* free the item and any aspect about that item. This is the ONLY place
** where bitmaps are to be freed. Return the next item in this list!
*/
  Item *item;
{
  Item *next = item->next;

  if( item->index != -1 ) { /* do we need to remove bitmap from widget */
    /* remove from bitmap from widget
    ** NOTE: for some reason if the bitmap used by the widget is
    ** set to `None', the XFreePixmap following will coredump!
    */
    XtVaSetValues(widget_array[item->index],
                  XtNbitmap, (XtArgVal)None,  /* None */
                  XtNlabel,  (XtArgVal)NULL,  NULL);
  }
  free_bitmap(item);
  XtFree( (char *) item );

  items_allocated--;
  return next;
}


void
free_list(list)
/* completely free all items in the given item list */
  Item **list;   /* pointer to a list pointer */
{
  while( *list != NULL )
    *list = free_item( *list );
}

/*--------------------------------------------------------------------------*/

static void
allocate_widgets( n )
/* create enough widgets to display n bitmaps */
  int n;
{
  char name[8];
  Widget w;

  if( allocated == n ) return;   /* just right -- no need to do anthing */

  if( allocated < n ) {   /* ok allocate more widgets */

    if(widget_array == NULL) {
      widget_array = (Widget *)XtMalloc( n * sizeof(Widget) );
      if( widget_array == NULL ) {
        perror("xbmbrowser: XtMalloc widget_array");
        abort();
      }
    } else {
      widget_array = (Widget *)realloc(widget_array, n * sizeof(Widget) );
      if( widget_array == NULL ) {
        perror("xbmbrowser: realloc larger widget_array");
        abort();
      }
    }

    for( ; allocated < n; allocated++ ) {
   
      sprintf(name, "%d", allocated);
      w = XtVaCreateWidget(name, labelWidgetClass, bitmaps, NULL);
      XtOverrideTranslations(w, XtParseTranslationTable(Translations));
      widget_array[allocated] = w;
    }

  }
#if 0
/* This code will eventually cause a core dump when lots of directory
** changes are performed by the user. The problem seems to occur when
** widgets are really destroyed (they are just flaged for destruction).
** I can't seem to find any problem with the code below however.
** It just doesn't seem to work.       Anthony
*/
  else {  /* ok we have too many widgets -- remove about a 1/3 of them */
    n = ( allocated + allocated + n ) / 3; /* the final level we want */
    
    for( allocated--; allocated >= n ; allocated-- )
      XtDestroyWidget( widget_array[allocated] );
    allocated++;

    widget_array = (Widget *)realloc(widget_array, n * sizeof(Widget) );
    if( widget_array == NULL ) {
      perror("xbmbrowser: realloc smaller widget_array");
      abort();
    }
  }

  /* insure that allocated is now equal to n (which may be calculated above */
  assert( allocated == n );
#endif
}


static void
load_bitmap(item)
/* Attempt to determine the type of the file given and load bitmap
** stored in that file.
*/
  Item *item;
{ 
  int i, status;        /* junk integer, size of icon, return status */
  int c;                /* junk char -- integer for EOF test */
  unsigned int x, y;    /* size of bitmap/pixmap */
#ifdef DO_XPMS
  Pixmap m;             /* junk pixmap mask */
#endif
  FILE *fp;             /* file pointer */

  /* insure the bitmap is not allocated already */
  assert( item->bitmap == (Pixmap)None || item->bitmap == filesyms[item->type]);

  /* if the type is a bad X pixmap -- skip direct to load pixmap */
  if( item->type == XpmBad )
    goto load_xpm;

  /* Test if the file is not a binary file
   * This should not be necessary, however the XReadBitmapFile and
   * XpmReadPixmapFile functions have a nasty habit of crashing with
   * some binary files.    Ashley
   */
   /* open the file and look to see if the first few chars are binary */
   if((fp = fopen(item->fname,"r")) == (FILE *) NULL) {
     /* can't open file -- declare it as such */
     item->type = Unknown;
     item->bitmap = (Pixmap)None;
     (void) sprintf(item->info, "%c\"%s\", unreadable",
                     item->type+'A', item->fname );
     return;
   } else {
     for(i = 0; i < 100; i++) {  /* check 100 chars */
       c = getc(fp);
       if(c == EOF)
	 break;
       if( c > 127 || c < 7 ) {  /* a char with non ascii values */
	 fclose(fp);
	 item->type = Binary;
         item->bitmap = (Pixmap)None;
         /* (void) sprintf(item->info, "%c\"%s\", binary (0x%02x at %d)", 
         **            item->type+'A', item->fname, c, i ); /**/
	 return;
       }
     }
     fclose(fp);
     if( i == 0 ) {  /* hey empty file -- declare it as such */
       item->type = File;   /* leave it as a plain file */
       item->bitmap = (Pixmap)None;
       (void) sprintf(item->info, "%c\"%s\", empty file",
                     item->type+'A', item->fname );
       return;
     }
       
   } /* probably a non binary file so continue */

  /* first try for a X Bitmap file format */
  item->bitmap = None; /* Precaution */
  status = XReadBitmapFile(display, DefaultRootWindow(display),
                            item->fname, &x, &y, &item->bitmap, &i, &i);

  if( status == BitmapSuccess && item->bitmap != None ) {
    item->type = Xbm;
    (void) sprintf(item->info, "%c\"%s\", %dx%d bitmap",
                     item->type+'A', item->fname, x, y );
    return;
  }

  /* Now try the X Pixmap format */
load_xpm:
#ifdef DO_XPMS
  item->bitmap = m = None;   /* Precaution */
  item->attr.valuemask = XpmReturnPixels | XpmCloseness;
  status = XpmReadPixmapFile(display, DefaultRootWindow(display),
                            item->fname, &item->bitmap,&m,&item->attr);

  /* ignore any masks at this time -- maybe a future feature */
  if( m != None )
    XFreePixmap(display, m);

  switch( status ) {
    case XpmColorError:
    case XpmSuccess:
      item->type = Xpm;
      (void) sprintf(item->info, "%c\"%s\", %dx%d pixmap (%d colors)",
               item->type+'A', item->fname, item->attr.width,
               item->attr.height, item->attr.npixels);
#ifdef TRIAL
      if ( item->attr.width == 0 || item->attr.height == 0 ) {
        /* this pixmap has NO PIXELS, and is a color table only! */
        /* remove the pixmap and let the default symbol be used */
        free_bitmap(item);
      }
#endif /* TRIAL */
      if ( item->bitmap != None )
        return;  /* return if bitmap was assigned other wise fall through */
    case XpmColorFailed:
    case XpmNoMemory:
      item->type = XpmBad;
      /* if a pixmap was assigned -- remove it */
      /* this actually should never be needed but just in case */
      free_bitmap(item);   
      return;
    case XpmOpenFailed:
    case XpmFileInvalid:
      /* OK so it isn't a pixmap */
      break;
   }
#endif /* DO_XPMS */
  /* Non Bitmap file -- it must be a plain text file */
  item->type = Text;
  item->bitmap = None;
  return;
}

/*------------------------------------------------------------------------*/

static void
merge_init()
/* initialize the appropiate variables for to merge `new_file_list'
** into the current `file_list'.
*/
{
  int i;  /* looping variable */

  /* first un-manage all the widgets and free the current file_list
  ** This saves on multiple geometry requests while we do the work
  */
  if( managed > 0 ) {
    XtUnmanageChildren(widget_array, managed);
    managed = 0;
  }

  old_file_list = file_list;  /* make the current list old */
  file_list = NULL;           /* empty destination of merged file_lists */
  last_link = &file_list;     /* last next ptr is the list ptr itself */

  /* zero all the counts required */
  num_files   = 0;            /* reset number of files in current list */
  num_dirs    = 0;            /* number of directoires in new dirmenu list */
  num_visible = 0;            /* number of visible items requiring widgets */
  for( i = 0 ; i < NumFileTypes ; i++ )
     file_counts[i] = 0;

  /* initialize the directory list */
  dir_list[num_dirs++] = "/";
  dir_list[num_dirs++] = "~/";
}


static void
merge_item(item)
/* Add this item to the current (merging) `file_list'.
** Assigning the appropiate information line, bitmap.
** And adjusting all file counts appropiately.
*/
  Item *item;
{
  Boolean visible = True;           /* is this item to be visible? */

  /* add item to the file_list */
  *last_link = item;            /* append new item to merged list */
  last_link = &(item->next);    /* move last_link to new last next ptr */
  /* *last_ptr = NULL;          /* terminate current list (not needed) */
  num_files++;                  /* increment number of file in list */

  /* if this item is a directory add it to the directory menu list */
  if( item->type >= Dir && item->type < File  ) {
    dir_list[num_dirs++] = item->fname;
  }

  /* Try to load a bitmap or pixmap from this file and determine file type
  ** more closly. This may also set the info line for the file if a load
  ** was successful.  BadXpm's are NOT attempted again otherwise rescan's
  ** become very very slow on large directories of X pixmaps.
  */
  if( visible  && item->type == File )
    load_bitmap( item );

  /* the item type should now be correct so count it */
  file_counts[item->type]++;

  /* Determine item visiblity as appropiate for current options */
  switch( item->type ) {
  case Dir:
  case DirUp:
  case DirLink:
  case DirBad: /* Directory Items */
    visible = visible && !app_data.icons_only && app_data.show_dir;
    break;
  case Unknown: /* Special File -- pipe, device */
  case Text:    /* Plain Text file */
  case Binary:  /* Binary file */
  case File:    /* New or Empty file */
    visible = visible && !app_data.icons_only && app_data.show_other;
    break;
  case XpmBad:  /* x pixmap which failed this load attempt */
    visible = visible && !app_data.icons_only && app_data.show_xpmbad;
    break;
  case Xbm: 
  case Xpm:
    /* visible = visible;  /* this should be visible */
    break;
  default:      /* Unknown file type -- error */
    visible = False;
    assert( False ); /* assertion failure */
  }

  /* if it is not visible -- finished */
  if( !visible ) {
    free_bitmap(item);  /* insure no bitmap is set */
    return;
  }

  /* assign a default info line if none present */
  /* Format :-  <type_char>"<filename>", <description_info> */
  if( item->type+'A' != item->info[0] ) {
    char *desc = "Description Error"; /* default description -- just in case */

    switch ( item->type ) {
      case Unknown: desc = "unknown";                    break;
      case Dir:     desc = "directory";                  break;
      case DirUp:   desc = "parent directory";           break;
      case DirLink: desc = "directory sym-link";         break;
      case DirBad:  desc = "directory (no access)";      break;
      case Text:    desc = "text";                       break;
      case Binary:  desc = "binary";                     break;
      case XpmBad:  desc = "pixmap (unable to display)"; break;
    }
    (void) sprintf(item->info, "%c\"%s\", %s",
                         item->type+'A', item->fname, desc);
  }

  /* assign default bitmap is none present */
  if( visible && item->bitmap == (Pixmap)None ) {
    item->bitmap = filesyms[item->type];
  }

  /* This item is visible give it a widget */
  num_visible++;
}


static void
merge_finish()
/* Finialise the merger of the new and old file_lists, assign the
** widgets, and set the default information line.
*/
{
  Item *item;  /* looping variable */
  int   i;     /* junk integer */

  /* insure the last next pointer in the newly merged list is NULL */
  *last_link = NULL;

  /* finialise the directory menu list and install */
  dir_list[num_dirs] = NULL;
  XawListChange(dirlist, dir_list, 0, 0, True);
  XtVaSetValues(dirlist, XtNdefaultColumns, (XtArgVal)(num_dirs/25+1), NULL);

  /* allocate/deallocate widgets as needed */
  allocate_widgets( num_visible );

  /* assign the bitmaps to the widgets in order */
  managed = 0;   /* this should be zero but make sure */
  for( item = file_list;  item != NULL;  item = item->next )
    if( item->bitmap != None ) {
      item->index = managed;
      XtVaSetValues( widget_array[item->index],
                       XtNbitmap, (XtArgVal)item->bitmap,
                       XtNlabel,  (XtArgVal)item->info,
                       XtNforeground, 
                         (XtArgVal)(item->bitmap == filesyms[item->type] ?
                                    app_data.sym_fore : app_data.icon_fore ),
                       XtNbackground, 
                         (XtArgVal)(item->bitmap == filesyms[item->type] ?
                                    app_data.sym_back : app_data.icon_back ),
                       NULL); 
      managed++;
    }

  /* manage the widgets */
  if( managed > 0 )
    XtManageChildren(widget_array, managed);

  /* Set the default information label */
# define strend  label_info+strlen(label_info)
  label_info[0] = '\0';  /* empty the string */
  if( num_files <= 1 ) /* only the ".." directory? */
    strcpy(label_info, "Empty Directory" );
  if( (i = file_counts[Xbm]) > 0 )
    sprintf(strend, " %d Bitmaps ", i );
  if( (i = file_counts[Xpm]) > 0 )
    sprintf(strend, " %d Pixmaps ", i );
  if( (i = file_counts[XpmBad]) > 0 )
    sprintf(strend, " %d Unloadable ", i );
  if( (i = file_counts[File] + file_counts[Text] + file_counts[Binary]) > 0 )
    sprintf(strend, " %d Others ", i );
  if( (i = file_counts[Dir] + file_counts[DirLink] + file_counts[DirBad]) > 0 )
    sprintf(strend, " %d Dirs ", i );
  if( (i = file_counts[Unknown]) > 0 )
    sprintf(strend, " %d Unknown ",    file_counts[Unknown] );

  XtVaSetValues(label, XtNlabel, (XtArgVal)label_info, NULL);
# undef strend

  /* check the results of the merger */
  assert( managed == num_visible );         /* number of widgets */
  assert( new_file_list == NULL );          /* both lists now empty */
  assert( old_file_list == NULL );
  assert( items_allocated == num_files );   /* allocation test */

}


static void
merge_file_lists()
/* Merge the old_file_list and the new_file_list. */
{
  int cmp;     /* count of bitmaps, filename comparasion */

  merge_init();

  while( old_file_list != NULL && new_file_list != NULL ) {
    /* both lists have items in them */

    cmp = strcmp( old_file_list->fname, new_file_list->fname );
    
    if( cmp == 0 ) {  /* the file names are the same */
      if ( new_file_list->type < File ||
           old_file_list->mtime != new_file_list->mtime ) {

        /* NON-FILE or -- mtime is non-sense so just add new item */
        /* MODIFIED    -- same name but it has been modified or replaced */
        old_file_list = free_item( old_file_list );  /* junk the old item */
        cmp = 1;    /* ok now pretend that this is a new item -- ADDITION */
        /* FALL THROUGH -- to ADDITION */

      } else {

        /* NO CHANGE -- same file in both lists without any modifications */
        new_file_list = free_item( new_file_list ); /* junk the new item */
        merge_item(old_file_list);         /* merge the old item */
	old_file_list = *last_link;        /* remove item from old list */
	continue;
      }
    }

    /* ADDITION -- a new file (or newly modified) to be merged */
    if( cmp > 0 ) {  /* name of new item is smaller than in old */
      merge_item(new_file_list);         /* merge the new item */
      new_file_list = *last_link;        /* remove item from new list */
      continue;
    }

    /* DELETION -- the old file has been deleted */
    if( cmp < 0 ) {  /* Hey old item name is smaller then new item name */
      old_file_list = free_item( old_file_list ); /* delete old item */
      continue;
    }

    /* this point should never be reached */
    /*NOTREACHED*/
    assert( FALSE );
  }

  /*                   --------------
  ** At this point either one or both of the merging file lists
  ** is empty. As such we can just finish of the merged list quickly 
  */
  assert( new_file_list == NULL || old_file_list == NULL );

  /* ADD any more new items in the new_file_list */

  while( new_file_list != NULL ) {
    merge_item(new_file_list);         /* merge the new item */
    new_file_list = *last_link;        /* remove item from new list */
  }
  new_file_list = NULL;    /* new_file_list is now empty */
     
  /* DELETE any old items left in the old_file_list */
  free_list(&old_file_list);

  /*                   --------------           */
  /* Merger of file lists complete -- finialize */
  merge_finish(); 
}


/*========================================================================*/
/* These routines are the external links into this modual */

void
scan_bitmaps()
/* scan and display all the bitmaps in the current directory.
** This is to be used when changing to new directories.
*/
{
  /* just free all the bitmaps and do a rescan which handles the
  ** reading of new bitmaps perfectly fine on its own. We need
  ** to unmanage them here, to avoid problems.
  */
  if( managed > 0 ) {
    XtUnmanageChildren(widget_array, managed);
    managed = 0;
  }

  free_list(&file_list);
  assert( items_allocated == 0 );  /* their should be no items left */

  rescan_bitmaps();
}

void
rescan_bitmaps()
/* Do a fast rescan and merger of the bitmaps in the current directory.
** The goal is to avoid re-reading files which were never changed.
*/
{
  new_file_list = get_files();  /* get items in the current directory */
  merge_file_lists();           /* merge any changes into file_list */
}

/*========================================================================*/
