#include <qmsgbox.h> 

#include "licqgui.h"
#include "icon.xpm"
#include "icon-newmsg.xpm"

#include <X11/Xlib.h>  // stuff for 
#include <X11/Xutil.h> // appicon generation

#ifdef USE_SCRNSAVER
#include <X11/extensions/scrnsaver.h>
#endif // USE_SCRNSAVER


NewLabel::NewLabel(QPopupMenu *m = NULL, QWidget *parent = 0, char *name = 0) : QLabel(parent, name)
{
   mnuPopUp = m;
}

void NewLabel::setBold(bool isBold)
{
   QFont newFont(font());
   newFont.setBold(isBold);
   setFont(newFont);
}

void NewLabel::setItalic(bool isItalic)
{
   QFont newFont(font());
   newFont.setItalic(isItalic);
   setFont(newFont);
}

void NewLabel::setNamedColor(char *theColor)
{
   QPalette pal(palette());
   QColorGroup normal(pal.normal());
   QColorGroup newNormal(normal.foreground(), normal.background(), normal.light(), normal.dark(),
                         normal.mid(), QColor(theColor), normal.base());
   setPalette(QPalette(newNormal, pal.disabled(), newNormal));
}


void NewLabel::mouseDoubleClickEvent(QMouseEvent *)
{
   emit doubleClicked();
}


void NewLabel::mouseReleaseEvent(QMouseEvent *e)
{
   QLabel::mouseReleaseEvent(e);
   if (mnuPopUp == NULL) return;
   if (e->button() == RightButton) 
   {
      QPoint clickPoint(e->x(), e->y());
      mnuPopUp->popup(mapToGlobal(clickPoint));      
   }
}



//-----LicqGUI::constructor------------------------------------------------------------------------
LicqGUI::LicqGUI(ICQ *theServer, QWidget *parent, const char *name) : QWidget(parent, name)
{
   // set up appicon and docking, code supplied by Mark Deneed
   WId win = winId();     // get the window
   Display *dsp = x11Display();  // get the display
   XWMHints *hints;  // hints
   XClassHint classhint;  // class hints
   classhint.res_name = "licq";  // res_name
   classhint.res_class = "Licq";  // res_class   
   XSetClassHint(dsp, win, &classhint); // set the class hints
   hints = XGetWMHints(dsp, win);  // init hints
   hints->window_group = win;  // set set the window hint
   hints->flags = WindowGroupHint;  // set the window group hint
   XSetWMHints(dsp, win, hints);  // set the window hints for WM to use.
   XFree( hints ); 

   // set the icon
   setIconText("Licq");
   setIcon(QPixmap(xpmIcon));

   // read in info from file
   char filename[MAX_FILENAME_LEN];
   unsigned short logonNow;

   if (DEBUG_LEVEL >= 1) cout << "-> GUI configuration." << endl;
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   File licqConf(filename);

   if (licqConf.error()) { cerr << "Fatal error!  Unable to open " << filename << "." << endl; exit(1); }
   licqConf.section("appearance").key("FontSize").read(fontSize, true);
   licqConf.section("appearance").key("FontFamily").read(fontFamily, true);
   licqConf.section("appearance").key("FontCharSet").read(fontCharSetInt, true);
   QFont::CharSet fontCharSet;
   switch (fontCharSetInt)
   {
   case 1: fontCharSet = QFont::Latin1; break;
   case 2: fontCharSet = QFont::Latin2; break;
   case 3: fontCharSet = QFont::Latin3; break;
   case 4: fontCharSet = QFont::Latin4; break;
   case 5: fontCharSet = QFont::Latin5; break;
   case 6: fontCharSet = QFont::Latin6; break;
   case 7: fontCharSet = QFont::Latin7; break;
   case 8: fontCharSet = QFont::Latin8; break;
   case 9: fontCharSet = QFont::Latin9; break;
   case 10: fontCharSet = QFont::KOI8R; break;
   default: fontCharSet = QFont::Latin1; break;
   }
   if (DEBUG_LEVEL >= 1) cout << "   Font: " << fontFamily << " / " << fontSize << "." << endl;
   qApp->setFont(QFont(fontFamily, fontSize, QFont::Normal, false, fontCharSet), true);

   licqConf.section("appearance").key("ColorOnline").read(colorOnline, true);
   licqConf.section("appearance").key("ColorOffline").read(colorOffline, true);
   licqConf.section("appearance").key("ColorAway").read(colorAway, true);
   licqConf.section("appearance").key("ColorNew").read(colorNew, true);
   licqConf.section("appearance").key("ColorBackground").read(colorBackground, true);
   if (DEBUG_LEVEL >= 1) cout << "   Colors: " << colorOnline << " / " << colorOffline << " / " << colorAway << " / " << colorNew << "." << endl;

   if (!licqConf.section("appearance").key("BgPixmap").read(pmBgName))
      { cerr << "Assuming no background." << endl; strcpy(pmBgName, "none"); }
   licqConf.section("appearance").key("GridLines").read(gridLines);
   if (DEBUG_LEVEL >= 1) cout << "   Grid Lines " << (gridLines ? "enabled." : "disabled.") << endl;
   licqConf.section("appearance").key("ShowHeader").read(showHeader);
   if (DEBUG_LEVEL >= 1) cout << "   Header is " << (showHeader ? "enabled." : "disabled.") << endl;

   licqConf.section("startup").key("Logon").read(logonNow, true);
   if (DEBUG_LEVEL >= 1) cout << "   Autologon " << (logonNow == 0 ? "disabled." : "enabled.") << endl;
   licqConf.section("startup").key("AutoAway").read(autoAwayTime, true);
   if (DEBUG_LEVEL >= 1) 
      if (autoAwayTime == 0) 
         cout << "   Auto-away disabled." << endl;
      else
         cout << "   Auto-away enabled after " << autoAwayTime << " minutes." << endl;
   licqConf.section("startup").key("AutoNA").read(autoNATime, true);
   if (DEBUG_LEVEL >= 1) 
      if (autoNATime == 0) 
         cout << "   Auto-not-available disabled." << endl;
      else
         cout << "   Auto-not-available enabled after " << autoNATime << " minutes." << endl;

   licqConf.section("functions").key("AutoClose").read(autoClose);
   if (DEBUG_LEVEL >= 1) cout << "   Autoclose " << (autoClose ? "disabled." : "enabled.") << endl;
   
   
   // load up position and size from file
   if (DEBUG_LEVEL >= 1) cout << "-> Geometry configuration." << endl;
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "geometry.conf");
   File geoConf(filename);
   if (!geoConf.error()) 
   {
      unsigned short xPos, yPos, hVal, wVal;
      geoConf.section("geometry").key("x").read(xPos);
      geoConf.section("geometry").key("y").read(yPos);
      geoConf.section("geometry").key("h").read(hVal, true);
      geoConf.section("geometry").key("w").read(wVal, true);
      if (DEBUG_LEVEL >= 1) cout << "   Using width " << wVal << ", height " << hVal << " at (" << xPos << ", " << yPos << ")." << endl;
      setGeometry(xPos, yPos, wVal, hVal);
   }
   else 
   {
      if (DEBUG_LEVEL >= 1) cout << "   Using defaults: width 170, height 440 at (0, 0)." << endl;
      setGeometry(0, 0, 150, 400);
   }
   setMinimumSize(100, 200);
   
   // load the pixmaps
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "online.ppm");
   pmOnline = new QPixmap(filename);
   if (pmOnline == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;
   
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "offline.ppm");
   pmOffline = new QPixmap(filename);
   if (pmOffline == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;
   
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "away.ppm");
   pmAway = new QPixmap(filename);
   if (pmAway == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;
   
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "na.ppm");
   pmNa = new QPixmap(filename);
   if (pmNa == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;
   
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "occupied.ppm");
   pmOccupied = new QPixmap(filename);
   if (pmOccupied == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;
   
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "dnd.ppm");
   pmDnd = new QPixmap(filename);
   if (pmDnd == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;

   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "private.ppm");
   pmPrivate = new QPixmap(filename);
   if (pmPrivate == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;
   
   sprintf(filename, "%s%s%s", BASE_DIR, DATA_DIR, "message.ppm");
   pmMessage = new QPixmap(filename);
   if (pmMessage == NULL) cerr << "Warning!  Unable to open pixmap " << filename << ".  Using blank." << endl;

   
   // create the child widgets
   server = theServer;
   netWin = new OutputWin;
   server->setOutputWin(netWin);
   awayMsgDlg = new AwayMsgDlg(&server->icqOwner);
   addUserDlg = new AddUserDlg(server);
   authUserDlg = new AuthUserDlg(server);
   searchUserDlg = new SearchUserDlg(server);
   optionsDlg = new OptionsDlg;
   
   char title[128];
   sprintf(title, "Licq (%s)", server->icqOwner.alias());
   setCaption(title);

   menu = new QMenuBar(this);
   menu->setGeometry(0, 0, 230, 20);
   menu->setFrameStyle(QFrame::Panel | QFrame::Raised);
   
   mnuStatus = new QPopupMenu(NULL);
   mnuStatus->insertItem("&Online");
   mnuStatus->insertItem("&Away");
   mnuStatus->insertItem("&Not Available");
   mnuStatus->insertItem("O&ccupied");
   mnuStatus->insertItem("&Do Not Disturb");
   mnuStatus->insertItem("&Private (Invisible)");
   mnuStatus->insertItem("&Free for Chat");
   mnuStatus->insertItem("O&ffline");
   connect(mnuStatus, SIGNAL(activated(int)), this, SLOT(changeStatusManual(int)));
   mnuDebug = new QPopupMenu(NULL);
   mnuDebug->insertItem("No debugging information");
   mnuDebug->insertItem("Startup comments only");
   mnuDebug->insertItem("Show all packets");
   mnuDebug->setCheckable(true);
   connect(mnuDebug, SIGNAL(activated(int)), this, SLOT(changeDebug(int)));
   changeDebug(DEBUG_LEVEL);
   mnuFile = new QPopupMenu(NULL);
   mnuFile->insertItem(server->icqOwner.alias(), this, SLOT(callICQOwnerFunction()));
   mnuFile->insertItem("&Status", mnuStatus);
   mnuFile->insertItem("Set &Away Message", awayMsgDlg, SLOT(show()));
   mnuFile->insertSeparator();
   mnuFile->insertItem("Show &Network Window", netWin, SLOT(show()));
   mnuFile->insertItem("Toggle &Mini Mode", this, SLOT(miniMode()));
   mnuFile->insertItem("Options", this, SLOT(showOptions()));
   mnuFile->insertItem("Save Options", this, SLOT(saveOptions()));
   mnuFile->insertItem("Debug Level", mnuDebug);
   mnuFile->insertSeparator();
   mnuFile->insertItem("Next &Server", this, SLOT(nextServer()));
   mnuFile->insertSeparator();
   mnuFile->insertItem("&About", this, SLOT(aboutBox()));
   mnuFile->insertItem("E&xit", qApp, SLOT(quit()));
   menu->insertItem("S&ystem", mnuFile);
   mnuUserAdm = new QPopupMenu(NULL);
   mnuUserAdm->insertItem("&Update Contact List", server, SLOT(updateContactList()));
   mnuUserAdm->insertItem("Update All Users", server, SLOT(updateAllUsers()));
   mnuUserAdm->insertItem("&Redraw User Window", this, SLOT(updateUserWin()));
   mnuUserAdm->insertSeparator();
   mnuUserAdm->insertItem("&Add User", addUserDlg, SLOT(show()));
   mnuUserAdm->insertItem("&Remove User", this, SLOT(removeUser()));
   mnuUserAdm->insertSeparator();
   mnuUserAdm->insertItem("&Search for User", searchUserDlg, SLOT(show()));
   mnuUserAdm->insertItem("A&uthorize User", authUserDlg, SLOT(show()));
   menu->insertItem("&Users", mnuUserAdm);
   
   mnuUser = new QPopupMenu(NULL);
   mnuUser->insertItem("&Read Event");
   mnuUser->insertItem("&Send Event");
   mnuUser->insertItem("&Info");
   mnuUser->insertItem("View &History");
   connect (mnuUser, SIGNAL(activated(int)), this, SLOT(callICQUserFunction(int)));
   
   char colKey[16];
   unsigned short colTitle, colWidth, colAlign, numColumns;
   licqConf.section("appearance").key("NumColumns").read(numColumns, true);
   for (unsigned short i = 1; i <= numColumns; i++) 
   {
      sprintf(colKey, "Column%dInfo", i);
      licqConf.section("appearance").key(colKey).read(colTitle, true);
      sprintf(colKey, "Column%dWidth", i);
      licqConf.section("appearance").key(colKey).read(colWidth, true);
      sprintf(colKey, "Column%dAlign", i);
      licqConf.section("appearance").key(colKey).read(colAlign, true);
      colInfo.push_back(ColInfo(colTitle, colWidth, colAlign));
   }
   
   //splUser = new QSplitter(QSplitter::Vertical, this);
   //splUser->setStyle(MotifStyle);
   //userViewOnline = new UserView(mnuUser, colInfo, true, splUser);
   //userViewOffline = new UserView(mnuUser, colInfo, false, splUser);
   userView = new UserView(mnuUser, colInfo, showHeader, this);
   
   // setup the background pixmap if there is one
   if (strncmp(pmBgName, "none", 4) != 0) 
   {
      char pmBgNameFull[1024];
      sprintf(pmBgNameFull, "%s%s%s", BASE_DIR, DATA_DIR, pmBgName);
      QPixmap pmBg(pmBgNameFull);
      setBackgroundPixmap(pmBg);
   }

   statusBar = new QGroupBox(this);
   statusBar->setFrameStyle(QFrame::Panel | QFrame::Raised);
   lblSysMsg = new NewLabel(NULL, statusBar);
   lblSysMsg->setFrameStyle(QFrame::Panel | QFrame::Sunken);
   lblSysMsg->setText("SysMsg");
   lblSysMsg->setBold(false);
   connect(lblSysMsg, SIGNAL(doubleClicked()), this, SLOT(callICQOwnerFunction()));
   lblStatus = new NewLabel(mnuStatus, statusBar);
   lblStatus->setFrameStyle(QFrame::Panel | QFrame::Sunken);
   lblMsg = new NewLabel(NULL, statusBar);
   lblMsg->setFrameStyle(QFrame::Panel | QFrame::Sunken);
   
   autoAwayTimer.start(10000);  // start the inactivity timer for auto away
   
   connect (&autoAwayTimer, SIGNAL(timeout()), this, SLOT(autoAway()));
   connect (server, SIGNAL(updatedUsers()), this, SLOT(updateUserWin()));
   connect (server, SIGNAL(updatedStatus()), this, SLOT(updateStatus()));
   connect (server, SIGNAL(doneOwnerFcn(bool, unsigned short)), this, SLOT(done(bool, unsigned short)));
   connect (server, SIGNAL(eventResult(ICQUser *, unsigned short, bool, unsigned long)), this, SLOT(eventResult(ICQUser *, unsigned short, bool, unsigned long)));
   //connect (userViewOnline, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(callDefaultICQFunction()));
   //connect (userViewOffline, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(callDefaultICQFunction()));
   connect (userView, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(callDefaultICQFunction()));
   connect (addUserDlg, SIGNAL(updatedUsers()), this, SLOT(updateUserWin()));
   connect (optionsDlg, SIGNAL(applyButtonPressed()), this, SLOT(getOptions()));
   
   inMiniMode = false;
   updateStatus();
   updateUserWin();
   manualAway = 0;
   
   resizeEvent(NULL);

   if (logonNow != 0) changeStatus(logonNow - 1);  // automatically logon if requested in conf file
}


//-----LicqGUI::destructor-------------------------------------------------------------------------
LicqGUI::~LicqGUI(void)
{
   // save window position and size
   char buf[MAX_FILENAME_LEN];
   sprintf(buf, "%s%s%s", BASE_DIR, CONF_DIR, "geometry.conf");
   QFile geoConf(buf);
   geoConf.open(IO_WriteOnly);
   sprintf(buf, "[geometry]\nx = %d\ny = %d\nw = %d\nh = %d\n", x(), y(), width(), height());
   geoConf.writeBlock(buf, strlen(buf));
   geoConf.close();
}


//-----LicqGUI::resizeEvent------------------------------------------------------------------------
void LicqGUI::resizeEvent (QResizeEvent *)
{
   if (width() >= 245) 
   {
      //splUser->setGeometry(0, 25, width(), height() - 56);
      userView->setGeometry(0, 25, width(), height() - 56);
      statusBar->setGeometry(0, height() - 30, width(), 30);
      lblSysMsg->setGeometry(5, 5, 55, 20);
      lblMsg->setGeometry(65, 5, 55, 20);
      lblStatus->setGeometry(125, 5, 115, 20);   
   }
   else
   {
      //splUser->setGeometry(0, 25, width(), height() - 81);
      userView->setGeometry(0, 25, width(), height() - 81);
      statusBar->setGeometry(0, height() - 55, width(), 55);
      lblSysMsg->setGeometry(5, 5, 55, 20);
      lblMsg->setGeometry(65, 5, width() - 70, 20); //60
      lblStatus->setGeometry(5, 30, width() - 10, 20);     //115
   }
}


//-----LicqGUI::updateUserWin----------------------------------------------------------------------
void LicqGUI::updateUserWin()
{
   unsigned short i;
   unsigned short msgs = 0;
   
   QPixmap *pmStatus;
   QColor userColor, backColor;
   
   //userViewOnline->clear();
   //userViewOffline->clear();
   userView->clear();

   // set the pixmap and color for each user and add them to the view
   for (i = 0; i < server->users.size(); i++)
   {
      switch (server->users[i]->status())
      {
      case ICQ_STATUS_ONLINE: 
         pmStatus = pmOnline; 
         userColor.setNamedColor(colorOnline); 
         break;
      case ICQ_STATUS_AWAY: 
         pmStatus = pmAway; 
         userColor.setNamedColor(colorAway); 
         break;
      case ICQ_STATUS_OCCUPIED: 
         pmStatus = pmOccupied; 
         userColor.setNamedColor(colorAway); 
         break;
      case ICQ_STATUS_DND: 
         pmStatus = pmDnd; 
         userColor.setNamedColor(colorAway); 
         break;
      case ICQ_STATUS_PRIVATE: 
         pmStatus = pmPrivate; 
         userColor.setNamedColor(colorAway); 
         break;
      case ICQ_STATUS_NA: 
         pmStatus = pmNa; 
         userColor.setNamedColor(colorAway); 
         break;
      case ICQ_STATUS_FREEFORCHAT: 
         pmStatus = pmOnline; 
         userColor.setNamedColor(colorOnline); 
         break;
      case ICQ_STATUS_OFFLINE: 
         pmStatus = pmOffline; 
         userColor.setNamedColor(colorOffline); 
         break;
      default: 
         pmStatus = NULL;
         userColor.setNamedColor(colorAway); 
         break;
      }
      backColor.setNamedColor(colorBackground);

      // special cases
      if (server->users[i]->numMessages() > 0) 
      {
         pmStatus = pmMessage;
         msgs += server->users[i]->numMessages();
      }
      if (server->users[i]->newUser()) userColor.setNamedColor(colorNew);
            
      //(void) new UserViewItem(server->users[i], i, pmStatus, userColor, backColor, gridLines, (server->users[i]->status() == ICQ_STATUS_OFFLINE ? userViewOffline : userViewOnline));
      (void) new UserViewItem(server->users[i], i, pmStatus, userColor, backColor, gridLines, userView);
   }
   
   lblSysMsg->setEnabled(server->icqOwner.numMessages() > 0);
   lblSysMsg->setBold(server->icqOwner.numMessages() > 0);

   if (msgs > 0)
   {
      setIcon(QPixmap(xpmIconNewMsg));
      char msgsText[16];
      sprintf (msgsText, "%d msg%c", msgs, (msgs == 1 ? ' ' : 's'));
      lblMsg->setText(msgsText);
   }
   else 
   {
      setIcon(QPixmap(xpmIcon));
      lblMsg->setText("No msgs");
   }

   //resizeEvent(NULL);
}


//-----LicqGUI::updateStatus-----------------------------------------------------------------------
void LicqGUI::updateStatus()
{  
   char *theColor = colorOffline;
   switch (server->icqOwner.status())
   {
   case ICQ_STATUS_ONLINE: lblStatus->setText("Online"); theColor = colorOnline; break;
   case ICQ_STATUS_AWAY: lblStatus->setText("Away"); theColor = colorAway; break;
   case ICQ_STATUS_NA: lblStatus->setText("Not Available"); theColor = colorAway; break;
   case ICQ_STATUS_OCCUPIED: lblStatus->setText("Occupied"); theColor = colorAway; break;
   case ICQ_STATUS_DND: lblStatus->setText("Do Not Disturb"); theColor = colorAway; break;
   case ICQ_STATUS_PRIVATE: lblStatus->setText("Invisible"); theColor = colorAway; break;
   case ICQ_STATUS_FREEFORCHAT: lblStatus->setText("Free for Chat"); theColor = colorOnline; break;
   case ICQ_STATUS_OFFLINE: lblStatus->setText("Offline"); theColor = colorOffline; break;
   } 
   lblStatus->setNamedColor(theColor);
}

//----LicqGUI::changeStatusManual------------------------------------------------------------------
void LicqGUI::changeStatusManual(int index)
{
   manualAway = index;
   changeStatus(index);
}


//----LicqGUI::changeStatus------------------------------------------------------------------------
void LicqGUI::changeStatus(int index)
{
   unsigned short newStatus =  ICQ_STATUS_OFFLINE;
   
   switch(index)
   {
   case 0: newStatus = ICQ_STATUS_ONLINE; break;
   case 1: newStatus = ICQ_STATUS_AWAY; break;
   case 2: newStatus = ICQ_STATUS_NA; break;
   case 3: newStatus = ICQ_STATUS_OCCUPIED; break;
   case 4: newStatus = ICQ_STATUS_DND; break;
   case 5: newStatus = ICQ_STATUS_PRIVATE; break;
   case 6: newStatus = ICQ_STATUS_FREEFORCHAT; break;
   case 7: newStatus = ICQ_STATUS_OFFLINE; break;
   default: cerr << "LicqGUI::changeStatus(): bad index value " << index << endl;
   }
   
   // disable combo box, flip pixmap...
   lblStatus->setEnabled(false);
   
   // call the right function
   if (newStatus == ICQ_STATUS_OFFLINE) 
      server->logoff(false);
   else if (server->icqOwner.status() == ICQ_STATUS_OFFLINE) 
   {
      if (!server->logon(newStatus))  // unable to connect to mirabilis server
      {
         QMessageBox::critical( this, "Licq", "No route to host (not connected or no servers set)", QMessageBox::Ok, 0);
         lblStatus->setEnabled(true);
      }
   }
   else 
      server->setStatus(newStatus);
}


//-----LicqGUI::callDefaultICQFunction-------------------------------------------------------------
void LicqGUI::callDefaultICQFunction()      { callICQFunction(-1, true);   }
void LicqGUI::callICQOwnerFunction()        { callICQFunction((server->icqOwner.numMessages() > 0 ? 0 : 3), false);  }
void LicqGUI::callICQUserFunction(int fcn)  { callICQFunction(fcn, true); }


//-----LicqGUI::callICQFunction--------------------------------------------------------------------
void LicqGUI::callICQFunction(int fcn, bool isUser)
{
   ICQUser *user = NULL;
   if (!isUser) 
      user = &server->icqOwner;
   else
      //if ((user = userViewOnline->currentUser()) == NULL)
      //   user = userViewOffline->currentUser();
      user = userView->currentUser();
   
   if (user != NULL) 
   {
      if (fcn < 0) fcn = (user->numMessages() == 0 ? 1 : 0);  // set default function to read or send depending on whether or not there are new messages
      if (user->fcnDlg == NULL) 
      {
         user->fcnDlg = new ICQFunctions(server);
         connect (user->fcnDlg, SIGNAL(updatedUsers()), this, SLOT(updateUserWin()));
         user->fcnDlg->setup(user, fcn, autoClose);
         connect (user->fcnDlg, SIGNAL(finished(ICQUser *)), this, SLOT(killICQFunction(ICQUser *)));
      }
      //else raise it to the top
   }   
}

void LicqGUI::killICQFunction(ICQUser *u)
{
   if (u->fcnDlg != NULL) 
   {
      delete u->fcnDlg;
      u->fcnDlg = NULL;
   }
}


//-----LicqGUI::done-------------------------------------------------------------------------------
void LicqGUI::done(bool isOk, unsigned short cmd)
{
   updateStatus();
   switch (cmd) 
   {
   case ICQ_CMDxSND_LOGON: 
      lblStatus->setEnabled(true);
      if (!isOk) QMessageBox::critical( this, "Licq", "Error logging on.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_LOGOFF: 
      lblStatus->setEnabled(true);
      if (!isOk) QMessageBox::critical( this, "Licq", "Error logging off.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_SETxSTATUS: 
      lblStatus->setEnabled(true);
      if (!isOk) QMessageBox::critical( this, "Licq", "Error changing status.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_USERxLIST:
      updateUserWin();
      if (!isOk) QMessageBox::critical( this, "Licq", "Error retrieving contact list.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_SYSxMSGxREQ:
      if (!isOk) QMessageBox::critical( this, "Licq", "Error retrieving system messages.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_PING:
      //if (!isOk) QMessageBox::critical( this, "Licq", "Error pinging Mirabilis, see Network Window for details.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_SYSxMSGxDONExACK:
      if (!isOk) QMessageBox::critical( this, "Licq", "Error acknowledging system messages.", QMessageBox::Ok, 0);
      break;
   case ICQ_CMDxSND_USERxADD:
      if (!isOk) QMessageBox::critical( this, "Licq", "Error informing Mirabilis of new user.", QMessageBox::Ok, 0);
      break;   
   case ICQ_CMDxSND_AUTHORIZE:
      if (!isOk) QMessageBox::critical( this, "Licq", "Error sending authorization.", QMessageBox::Ok, 0);
      else QMessageBox::information( this, "Licq", "Authorization granted.", QMessageBox::Ok, 0);
      break;   
   default:
      break;
   }
}



//-----LicqGUI::removeUser-------------------------------------------------------------------------
void LicqGUI::removeUser()
{
   ICQUser *u;
   //if ((u = userViewOnline->currentUser()) == NULL)
   //   u = userViewOffline->currentUser();
   u = userView->currentUser();
   if (u == NULL) return;
   char warning[256];
   sprintf(warning, "Are you sure you want to remove\n%s (%ld)\nfrom your contact list?", u->alias(), u->uin());
   if( QMessageBox::warning( this, "Licq", warning, QMessageBox::Ok, QMessageBox::Cancel) == 1)
      server->removeUser(u);
}


//-----LicqGUI::showOptions------------------------------------------------------------------------
void LicqGUI::showOptions()
{
   optionsDlg->edtFontFamily->setText(fontFamily);
   optionsDlg->spnFontSize->setValue(fontSize);
   optionsDlg->cmbFontCharSet->setCurrentItem(fontCharSetInt - 1);

   optionsDlg->edtColorOnline->setText(colorOnline);
   optionsDlg->edtColorAway->setText(colorAway);
   optionsDlg->edtColorOffline->setText(colorOffline);
   optionsDlg->edtColorNew->setText(colorNew);
   optionsDlg->edtColorBackground->setText(colorBackground);
   
   optionsDlg->chkGridLines->setChecked(gridLines);
   optionsDlg->chkHeader->setChecked(showHeader);
   optionsDlg->chkAutoClose->setChecked(autoClose);

   optionsDlg->spnDefServerPort->setValue(server->defaultRemotePort);
   optionsDlg->spnTcpServerPort->setValue(server->tcpServerPort);
   optionsDlg->cmbServers->clear();
   unsigned short i;
   for (i = 0; i < server->icqServers.numServers(); i++)
      optionsDlg->cmbServers->insertItem(server->icqServers.servers[i]->name());
   optionsDlg->spnAutoAway->setValue(autoAwayTime);
   optionsDlg->spnAutoNa->setValue(autoNATime);

   // set up the columns stuff
   for (i = 0; i < colInfo.size(); i++) 
   {
      optionsDlg->chkColEnabled[i]->setChecked(true);
      optionsDlg->cmbColInfo[i]->setCurrentItem(colInfo[i].info - 1);
      optionsDlg->spnColWidth[i]->setValue(colInfo[i].width);
      optionsDlg->cmbColAlign[i]->setCurrentItem(colInfo[i].align);
      optionsDlg->chkColEnabled[i]->setEnabled(true);
      optionsDlg->cmbColInfo[i]->setEnabled(true);
      optionsDlg->spnColWidth[i]->setEnabled(true);
      optionsDlg->cmbColAlign[i]->setEnabled(true);
   }
   if (i < 4) 
   {
      optionsDlg->chkColEnabled[i]->setChecked(false);
      optionsDlg->cmbColInfo[i]->setEnabled(false);
      optionsDlg->spnColWidth[i]->setEnabled(false);
      optionsDlg->cmbColAlign[i]->setEnabled(false);
      i++;
   }
   for (; i < 4; i++) 
   {
      optionsDlg->chkColEnabled[i]->setChecked(false);
      optionsDlg->chkColEnabled[i]->setEnabled(false);
      optionsDlg->cmbColInfo[i]->setEnabled(false);
      optionsDlg->spnColWidth[i]->setEnabled(false);
      optionsDlg->cmbColAlign[i]->setEnabled(false);
   } 
   
   // set up the sound stuff
   optionsDlg->cmbSndType->setCurrentItem(server->soundEnabled());
   optionsDlg->edtSndPlayer->setText(server->soundPlayer);
   optionsDlg->edtSndMsg->setText(server->soundMsg);
   optionsDlg->edtSndUrl->setText(server->soundUrl);
   optionsDlg->edtSndChat->setText(server->soundChat);
   optionsDlg->edtSndFile->setText(server->soundFile);
   optionsDlg->edtSndNotify->setText(server->soundNotify);

   optionsDlg->show();
}


//-----LicqGUI::getOptions-------------------------------------------------------------------------
void LicqGUI::getOptions()
{
   // get all options from the options dialog
   strncpy(fontFamily, optionsDlg->edtFontFamily->text(), 32);
   fontSize = optionsDlg->spnFontSize->value();
   fontCharSetInt = optionsDlg->cmbFontCharSet->currentItem() + 1;
   QFont::CharSet fontCharSet;
   switch (fontCharSetInt)
   {
   case 1: fontCharSet = QFont::Latin1; break;
   case 2: fontCharSet = QFont::Latin2; break;
   case 3: fontCharSet = QFont::Latin3; break;
   case 4: fontCharSet = QFont::Latin4; break;
   case 5: fontCharSet = QFont::Latin5; break;
   case 6: fontCharSet = QFont::Latin6; break;
   case 7: fontCharSet = QFont::Latin7; break;
   case 8: fontCharSet = QFont::Latin8; break;
   case 9: fontCharSet = QFont::Latin9; break;
   case 10: fontCharSet = QFont::KOI8R; break;
   default: fontCharSet = QFont::Latin1; break;
   }
   qApp->setFont(QFont(fontFamily, fontSize, QFont::Normal, false, fontCharSet), true);

   strncpy(colorOnline, optionsDlg->edtColorOnline->text(), 32);
   strncpy(colorOffline, optionsDlg->edtColorOffline->text(), 32);
   strncpy(colorAway, optionsDlg->edtColorAway->text(), 32);
   strncpy(colorNew, optionsDlg->edtColorNew->text(), 32);
   strncpy(colorBackground, optionsDlg->edtColorBackground->text(), 32);
   
   gridLines = optionsDlg->chkGridLines->isChecked();
   showHeader = optionsDlg->chkHeader->isChecked();
   autoClose = optionsDlg->chkAutoClose->isChecked();

   //server->defaultRemotePort = optionsDlg->spnDefServerPort->value();
   //server->tcpServerPort = optionsDlg->spnTcpServerPort->value();
   
   //optionsDlg->cmbServers->clear();
   //unsigned short i;
   //for (i = 0; i < server->icqServers.numServers(); i++)
   //   optionsDlg->cmbServers->insertItem(server->icqServers.servers[i]->name());
   
   autoAwayTime = optionsDlg->spnAutoAway->value();
   autoNATime = optionsDlg->spnAutoNa->value();

   // set up the columns stuff
   unsigned short i, j = colInfo.size();
   for (i = 0; i < j; i++) colInfo.pop_back();  // erase the old array
   i = 0;
   while (optionsDlg->chkColEnabled[i]->isChecked()) 
   {
      colInfo.push_back(ColInfo(optionsDlg->cmbColInfo[i]->currentItem() + 1,
                                optionsDlg->spnColWidth[i]->value(),
                                optionsDlg->cmbColAlign[i]->currentItem()));
      i++;
   }
   delete userView;
   userView = new UserView(mnuUser, colInfo, showHeader, this);
   connect (userView, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(callDefaultICQFunction()));
   updateUserWin();
   resizeEvent(NULL);
   userView->show();
   
   // set up the sound stuff   
   server->setSoundEnabled(optionsDlg->cmbSndType->currentItem());
   server->setSounds(optionsDlg->edtSndPlayer->text(), optionsDlg->edtSndMsg->text(),
                     optionsDlg->edtSndUrl->text(), optionsDlg->edtSndChat->text(),
                     optionsDlg->edtSndFile->text(), optionsDlg->edtSndNotify->text());

}


//-----LicqGUI::saveOptions------------------------------------------------------------------------
void LicqGUI::saveOptions()
{
   // save all options
   char filename[MAX_FILENAME_LEN];
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   File licqConf(filename);
   if (licqConf.error()) 
   {
      QMessageBox::critical( this, "Licq", "Error opening licq.conf for writing.\nVerify permissions and directories.", QMessageBox::Ok, 0);
      return; 
   }
   
   licqConf.section("appearance").key("FontSize").write(fontSize);
   licqConf.section("appearance").key("FontFamily").write(fontFamily);
   licqConf.section("appearance").key("FontCharSet").write(fontCharSetInt);
   licqConf.section("appearance").key("ColorOnline").write(colorOnline);
   licqConf.section("appearance").key("ColorOffline").write(colorOffline);
   licqConf.section("appearance").key("ColorAway").write(colorAway);
   licqConf.section("appearance").key("ColorNew").write(colorNew);
   licqConf.section("appearance").key("ColorBackground").write(colorBackground);
   licqConf.section("appearance").key("GridLines").write(gridLines);
   licqConf.section("appearance").key("ShowHeader").write(showHeader);
   //licqConf.section("startup").key("Logon").write(logonNow);
   licqConf.section("startup").key("AutoAway").write(autoAwayTime);
   licqConf.section("startup").key("AutoNA").write(autoNATime);
   licqConf.section("functions").key("AutoClose").write(autoClose);

   
   //server->defaultRemotePort = optionsDlg->spnDefServerPort->value();
   //server->tcpServerPort = optionsDlg->spnTcpServerPort->value();
   
   //optionsDlg->cmbServers->clear();
   //unsigned short i;
   //for (i = 0; i < server->icqServers.numServers(); i++)
   //   optionsDlg->cmbServers->insertItem(server->icqServers.servers[i]->name());
   
   // save the column info
   licqConf.section("appearance").key("NumColumns").write((unsigned short)colInfo.size());
   char colKey[32];
   for (unsigned short i = 1; i <= colInfo.size(); i++) 
   {
      sprintf(colKey, "Column%dInfo", i);
      licqConf.section("appearance").key(colKey).write(colInfo[i - 1].info);
      sprintf(colKey, "Column%dWidth", i);
      licqConf.section("appearance").key(colKey).write(colInfo[i - 1].width);
      sprintf(colKey, "Column%dAlign", i);
      licqConf.section("appearance").key(colKey).write(colInfo[i - 1].align);
   }
      
   // save the sound stuff
   licqConf.section("sound").key("Enable").write(server->soundEnabled());
   licqConf.section("sound").key("Player").write(server->soundPlayer);
   licqConf.section("sound").key("Message").write(server->soundMsg);
   licqConf.section("sound").key("Url").write(server->soundUrl);
   licqConf.section("sound").key("Chat").write(server->soundChat);
   licqConf.section("sound").key("File").write(server->soundFile);
   licqConf.section("sound").key("OnlineNotify").write(server->soundNotify);

}


//-----LicqGUI::aboutBox---------------------------------------------------------------------------
void LicqGUI::aboutBox()
{
   char about[1024];
   sprintf (about, "Licq version %s.\n\n%s (%ld)\n%d contacts.", VERSION, server->icqOwner.alias(), server->icqOwner.uin(), server->numUsers()); 
   QMessageBox::information( this, "Licq", about, QMessageBox::Ok, 0);
}


//-----LicqGUI::changeDebug------------------------------------------------------------------------
void LicqGUI::changeDebug(int newLevel)
{
   DEBUG_LEVEL = newLevel;  // set the global debug level
   mnuDebug->setItemChecked(0, false);
   mnuDebug->setItemChecked(1, false);
   mnuDebug->setItemChecked(2, false);
   mnuDebug->setItemChecked(newLevel, true);
}


//-----LicqGUI::nextServer-------------------------------------------------------------------------
void LicqGUI::nextServer()
{
   if (!server->nextServer())
      QMessageBox::critical( this, "Licq", "Error setting destination to new server.\nTry another.", QMessageBox::Ok, 0);
}


//-----LicqGUI::miniMode---------------------------------------------------------------------------
void LicqGUI::miniMode()
{
   static unsigned short oldH;
   
   if (inMiniMode) 
   {
      userView->show();
      //splUser->show();
      setMaximumHeight(2048);
      resize(width(), oldH);
      setMinimumHeight(100);
   }
   else
   {
      userView->hide();
      //splUser->hide();
      oldH = height();
      setMinimumHeight(statusBar->height() + 23);
      resize(width(), statusBar->height() + 23);
      setMaximumHeight(statusBar->height() + 23);
   }
   inMiniMode = !inMiniMode;
}


//-----LicqGUI::autoAway-----------------------------------------------------------
void LicqGUI::autoAway()
{
   Time idleTime = 0;  /* millisecs since last input event */
   
   unsigned short status = server->icqOwner.status();
   // manualAway 1 is Away and 2 is NA (from the popup menu)
   if ((manualAway == 1 || manualAway == 2) || ( status != ICQ_STATUS_NA && status != ICQ_STATUS_ONLINE && status != ICQ_STATUS_AWAY && status != ICQ_STATUS_FREEFORCHAT))
       return;

#ifdef USE_SCRNSAVER
   static XScreenSaverInfo *mit_info = 0;
   if (!mit_info) mit_info = XScreenSaverAllocInfo ();
   XScreenSaverQueryInfo (x11Display(), DefaultRootWindow (x11Display()), mit_info);
   idleTime = mit_info->idle;
#endif // USE_SCRNSAVER

   if (((unsigned long)idleTime > (unsigned long)autoNATime * 60000 && autoNATime != 0) && (status != ICQ_STATUS_NA))
      changeStatus(2);   // set na mode
   else if (((unsigned long)idleTime > (unsigned long)autoAwayTime * 60000 && autoAwayTime != 0) && (status != ICQ_STATUS_AWAY) && (status != ICQ_STATUS_NA))
      changeStatus(1);  // go into away mode
   else if ((status == ICQ_STATUS_AWAY || status == ICQ_STATUS_NA) && ((unsigned long)idleTime < (unsigned long)autoAwayTime * 60000 && autoAwayTime != 0)) 
      changeStatus(0); // wake back up

}


//-----LicqGUI::eventResult------------------------------------------------------------------------
void LicqGUI::eventResult(ICQUser *u, unsigned short cmd, bool accepted, unsigned long port)
{
   char result[128];
   switch (cmd) 
   {
   case ICQ_CMDxTCP_CHAT:
      if (accepted) 
      {
         ChatDlg *chatDlg = new ChatDlg(u, &(server->icqOwner), false, port);
         chatDlg->show();
      }
      else
      {
         sprintf(result, "Chat request to %s refused.", u->alias());
         QMessageBox::information(this, "Licq", result, QMessageBox::Ok, 0);
      }
      break;
   case ICQ_CMDxTCP_FILE:
      sprintf(result, "File transfer to %s %s.", u->alias(), accepted ? "accepted" : "refused");
      QMessageBox::information(this, "Licq", result, QMessageBox::Ok, 0);
      break;
   default:
      break;
   }
}

