#include <sys/types.h>
#include <netinet/in.h> 

#include "user.h"


//-----ICQUser::constructor------------------------------------------------------------------------
ICQUser::ICQUser(unsigned long id, char *filename, bool isNewUser)
/* called when first constructing our known users and when a new user is received without them
   actually connecting through tcp */
{
   uinVal = id;
   sequenceVal = 0x01;
   sendServerVal = false;
   newUserVal = isNewUser;
   fcnDlg = NULL;
   if (filename != NULL) 
   {
      File userFile(filename);
      if (userFile.error()) 
      {
         cerr << "Warning!  Unable to open user info file " << filename << ".  Using defaults instead." << endl;
         setDefaults(id);
      }
      else  // read in the fields, checking for errors each time
      {
         if (!userFile.section("user").key("Alias").read(aliasVal))
         {
            cerr << "Using uin as alias." << endl; 
            sprintf(aliasVal, "%ld", id);
         }
         if (!userFile.section("user").key("FirstName").read(firstNameVal))
         {
            cerr << "Leaving first name blank." << endl; 
            strcpy(firstNameVal, "");
         }
         if (!userFile.section("user").key("LastName").read(lastNameVal))
         { 
            cerr << "Leaving last name blank." << endl; 
            strcpy(lastNameVal, "");
         }
         if (!userFile.section("user").key("EMail").read(emailVal))
         { 
            cerr << "Leaving email blank." << endl; 
            strcpy(emailVal, "");
         }
         if (!userFile.section("user").key("History").read(historyFileVal))
         { 
            cerr << "No history will be created." << endl; 
            strcpy(historyFileVal, "none");
         }
         if (!userFile.section("user").key("OnlineNotify").read(onlineNotifyVal))
         { 
            cerr << "Assuming no online notification." << endl; 
            onlineNotifyVal = false;
         }
         if (!userFile.section("user").key("NewMessages").read(newMessages))
         { 
            cerr << "Assuming no new messages." << endl; 
            newMessages = 0;
         }
         if (!userFile.section("user").key("NewUser").read(newUserVal))
         { 
            cerr << "Assuming not a new user." << endl; 
            newUserVal = false;
         }
         // if any of the fields contain a newline, erase it
         if (strncmp(lastNameVal, "\n", 1) == 0) strcpy(lastNameVal, "");
         if (strncmp(emailVal, "\n", 1) == 0) strcpy(emailVal, "");
         if (strncmp(firstNameVal, "\n", 1) == 0) strcpy(firstNameVal, "");
         if (strncmp(aliasVal, "\n", 1) == 0) strcpy(aliasVal, "");
         if (strncmp(historyFileVal, "\n", 1) == 0) strcpy(aliasVal, "none");
      }
   }
   else 
      setDefaults(id);

   if (newMessages > 0) 
   {
      char newMessageStr[128];
      sprintf(newMessageStr, "Licq: %d new messages.  See History for copies.", newMessages);
      message.push_back(new Message(newMessageStr)); 
   }
   openHistory();
   statusVal = ICQ_STATUS_OFFLINE;
   awayMessageVal = NULL;
   setAwayMessage(NULL);
}  


//-----ICQUser::constructor------------------------------------------------------------------------
ICQUser::ICQUser(unsigned long id, TCPSocket &sock, bool isNewUser)
/* called to transfer an incoming tcp connection over to a new user */
{
   uinVal = id;
   sequenceVal = 0x01;
   newUserVal = isNewUser;
   sendServerVal = false;
   onlineNotifyVal = false;
   fcnDlg = NULL;
   
   setDefaults(id);
   openHistory();
   
   tcpSocket.transferConnectionFrom(sock);
   statusVal = ICQ_STATUS_ONLINE;
   awayMessageVal = NULL;
   setAwayMessage(NULL);
}


//-----ICQUser::destructor-------------------------------------------------------------------------
ICQUser::~ICQUser(void)
{
   while (numMessages() > 0) clearMessage(0);
   if (history != NULL) delete history;
}


//-----ICQUser::setDefaults------------------------------------------------------------------------
void ICQUser::setDefaults(unsigned long id)
{
   // set default data for the user
   sprintf(aliasVal, "%ld", id);
   strcpy(firstNameVal, "Some");
   strcpy(lastNameVal, "Guy");
   strcpy(emailVal, "someguy@uwaterloo.ca");
   strcpy(historyFileVal, "default");
   newMessages = 0;
   onlineNotifyVal = false;
}


//-----ICQUser::openHistory------------------------------------------------------------------------
void ICQUser::openHistory(void)
/* open the history file */
{
   // default history filename, compare only first 7 chars in case of spaces
   if (strncmp(historyFileVal, "default", 7) == 0)
   {
      sprintf(historyFileVal, "%s%s%ld.history", BASE_DIR, HISTORY_DIR, uinVal);
      history = new QFile(historyFileVal);
      history->open(IO_Append | IO_WriteOnly);
      sprintf(historyFileVal, "default (%s%s%ld.history)", BASE_DIR, HISTORY_DIR, uinVal);
   }
   // no history file
   else if (strncmp(historyFileVal, "none", 4) == 0)  // no history file
   {
      history = NULL;
   }
   // use given name
   else
   {
      history = new QFile(historyFileVal);
      history->open(IO_Append | IO_WriteOnly);
   }
      
}


//-----ICQUser::data retrieval functions-----------------------------------------------------------
char *ICQUser::alias(void)            { return(aliasVal); }
unsigned short ICQUser::status(void)  { return(statusVal); }
unsigned long ICQUser::uin(void)      { return(uinVal); }
char *ICQUser::email(void)            { return(emailVal); }
unsigned long ICQUser::sequence(void) { return(sequenceVal++); }
char *ICQUser::historyFile(void)      { return(historyFileVal); }
char *ICQUser::awayMessage(void)      { return(awayMessageVal); }
bool ICQUser::newUser(void)           { return(newUserVal); }
bool ICQUser::onlineNotify(void)      { return(onlineNotifyVal); }
bool ICQUser::sendServer(void)        { return(sendServerVal); }
unsigned short ICQUser::numMessages(void)  { return(message.size()); }

void ICQUser::setNewUser(bool s) { newUserVal = s; }
void ICQUser::setOnlineNotify(bool s)  { onlineNotifyVal = s; }
void ICQUser::setSendServer(bool s)   { sendServerVal = s; }


//-----ICQUser::getStatusInfo----------------------------------------------------------------------
void ICQUser::getStatusInfo(struct UserStatusline &us)
{
   strcpy(us.alias, aliasVal);
   sprintf(us.uin, "%ld", uinVal);
   strcpy(us.firstname, firstNameVal);
   strcpy(us.lastname, lastNameVal);
   sprintf(us.name, "%s %s", us.firstname, us.lastname);
   strcpy(us.email, emailVal);
   switch(statusVal)
   {
   case ICQ_STATUS_OFFLINE:     strcpy(us.status, "Offline");  break;
   case ICQ_STATUS_ONLINE:      strcpy(us.status, "Online");   break;
   case ICQ_STATUS_AWAY:        strcpy(us.status, "Away");     break;
   case ICQ_STATUS_NA:          strcpy(us.status, "Not Available"); break;
   case ICQ_STATUS_OCCUPIED:    strcpy(us.status, "Occupied"); break;
   case ICQ_STATUS_DND:         strcpy(us.status, "Do Not Disturb"); break;
   case ICQ_STATUS_FREEFORCHAT: strcpy(us.status, "Free for Chat"); break;
   case ICQ_STATUS_PRIVATE:     strcpy(us.status, "Private");  break;
   default:                     sprintf(us.status, "(0x%04X)", statusVal); break;
   }
   strcpy(us.ip, iptoa(tcpSocket.remoteIP()));
   sprintf(us.port, "%d", tcpSocket.remotePort());
   sprintf(us.ip_port, "%s:%s", us.ip, us.port);
   strcpy(us.history, historyFileVal);
   strncpy(us.awayMessage, awayMessageVal, MAX_MESSAGE_SIZE);
}


//-----ICQUser::saveInfo---------------------------------------------------------------------------
void ICQUser::saveInfo(void)
{
   char filename[256];
   sprintf(filename, "%s%s%ld.uin", BASE_DIR, CONF_DIR, uinVal);
   FILE *userFile;
   userFile = fopen(filename, "w");
   fprintf(userFile, "[user]\nAlias = %s\nFirstName = %s\nLastName = %s\nEMail = %s\nHistory = %s\nOnlineNotify = %d\nNewUser = %d\nNewMessages = %d\n",
           aliasVal, firstNameVal, lastNameVal, emailVal, historyFileVal, onlineNotifyVal, newUserVal, numMessages());
   fclose(userFile);
}


//-----ICQUser::addMessage----------------------------------------------------------------------------
void ICQUser::addMessage(char *m, unsigned short theCmd, unsigned short theSubCmd, unsigned long theSequence, time_t t = 0, bool isLicq = false, unsigned short theVer = 0)
{ 
   message.push_back(new Message(m, theCmd, theSubCmd, theSequence, t, isLicq, theVer));    
   addToHistory(message[message.size() - 1]);
}


//-----ICQUser::addMessage----------------------------------------------------------------------------
void ICQUser::addMessage(char *m, time_t t = 0, bool isLicq = false, unsigned short theVer = 0)
{ 
   message.push_back(new Message(m, t, isLicq, theVer)); 
   addToHistory(message[message.size() - 1]);
}


void ICQUser::addToHistory(Message *newMsg)
{
   // add message to history file
   if (history != NULL) 
   {
      char *msg = new char[strlen(newMsg->msg()) + 128];
      sprintf(msg, "%s from %s received %s:\n%s\n\n", newMsg->cmdStr(), aliasVal, newMsg->timeRec(), newMsg->msg());
      history->writeBlock(msg, strlen(msg));
      history->flush();
   }
   saveInfo();
}



//-----ICQUser::getMessage----------------------------------------------------------------------------
Message *ICQUser::getMessage(unsigned short index)
{ 
   if (index >= numMessages() || numMessages() == 0) return (NULL);
   return (message[index]); 
}


//-----ICQUser::clearMessage--------------------------------------------------------------------------
void ICQUser::clearMessage(unsigned short index)    
{ 
   delete message[index];
   for (unsigned short i = index; i < message.size() - 1; i++) message[i] = message[i + 1];
   message.pop_back();
   saveInfo();
}


//-----ICQUser::setAwayMessage---------------------------------------------------------------------
void ICQUser::setAwayMessage(char *m)
{
   if (awayMessageVal != NULL) delete awayMessageVal;
   if (m == NULL) 
   {
      awayMessageVal = new char[1];
      awayMessageVal[0] = '\0';
   }
   else
   {
      awayMessageVal = new char[strlen(m) + 1];
      strcpy(awayMessageVal, m);
   }
}


char *ICQUser::iptoa(unsigned long ipAddr)
{
   return(inet_ntoa(*((in_addr *)&ipAddr)));
}


//-----ICQOwner::constructor-----------------------------------------------------------------------
ICQOwner::ICQOwner(void)
{
   char *filename = new char[64];
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   
   if (DEBUG_LEVEL >= 1) cout << "-> Owner configuration." << endl;

   File licqConf(filename);
   if (licqConf.error())  { cerr << "Fatal error!  Unable to open " << filename << "." << endl; exit(1); }
   
   licqConf.section("owner").key("Alias").read(aliasVal, true);
   licqConf.section("owner").key("Uin").read(uinVal, true);
   licqConf.section("owner").key("Password").read(passwdVal, true);
   
   if (DEBUG_LEVEL >= 1) cout << "   " << aliasVal << " (" << uinVal << ") password scanned." << endl;

   if (strlen(passwdVal) > 8) 
   {
      cout << "Trying to hack ICQ...?  Password must be 8 characters or less, truncating." << endl;
      passwdVal[8] = '\0';
   }

   sequenceVal = 0x01;
   statusVal = ICQ_STATUS_OFFLINE;
   awayMessageVal = NULL;
   fcnDlg = NULL;
   setAwayMessage(NULL);
   history = NULL;
}


//-----ICQOwner::getStatusInfo----------------------------------------------------------------------
void ICQOwner::getStatusInfo(struct UserStatusline &us)
{
   ICQUser::getStatusInfo(us);
   strcpy(us.ip, iptoa(tcpSocket.localIP()));
   sprintf(us.port, "%d", tcpSocket.localPort());
   sprintf(us.ip_port, "%s:%s", us.ip, us.port);
   strcpy(us.history, "none");
}


char *ICQOwner::passwd(void) { return(passwdVal); }
unsigned short ICQOwner::searchSequence(void) { return (searchSequenceVal++); }


