#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <linux/fdreg.h>
#include <linux/fd.h>

#include "../busybox/internal.h"
#include "dinstall.h"
#include INCLINGUA

#ifdef _TESTING_
#  define BOOT_MESSAGE	"../../scripts/rootdisk/messages/" LINGUA "/boot_message"
#else
#  define BOOT_MESSAGE	"/boot_message"
#endif

/* ED: floppy filesystem type is not the same for all architectures */
#if #cpu(sparc) || #cpu(powerpc)
static char *fs_type = "ext2";
#else
static char *fs_type = "msdos";
#endif

extern int fdflush(const char *filename);
extern int mount_one(char *blockDevice, char *directory, char *filesystemType, 
  unsigned long flags, char *string_flags, int noMtab, int fake);

#if #cpu(sparc)
# define USE_SILO
#elif #cpu(m68k)
# define USE_LILOM68K
#elif #cpu(powerpc)
# define USE_PLILO
#else
# define USE_LILO
#endif

#ifdef _TESTING_
#define BC_DIR_PREFIX ""
#define BC_LILO_CONF  "etc_lilo.conf_test"
#define BC_MBR        "/boot/mbr.b"
#define BC_BOOT_DFLT   "export LD_LIBRARY_PATH=\"/lib:/usr/lib\"; /sbin/activate %s %s"
#define BC_RUN_LILO    "export LD_LIBRARY_PATH=\"/lib:/usr/lib\"; /sbin/lilo -r / >/dev/null"
#define BC_SFORMAT     "/usr/bin/superformat"
#define BC_RUN_SFORMAT "export LD_LIBRARY_PATH=\"/lib:/usr/lib\"; "BC_SFORMAT" %s %s"

#else

#define BC_DIR_PREFIX "/target"
#define BC_LILO_CONF  "/target/etc/lilo.conf"
#define BC_MBR        "/target/boot/mbr.b"
#define BC_BOOT_DFLT   "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/activate %s %s"
#define BC_RUN_LILO    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/lilo -r /target >/dev/null"

#define BC_SILO_CONF  "/target/etc/silo.conf"
#define BC_RUN_SILO    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/silo -f -r /target >/dev/null"

#define BC_QUIK_CONF  "/target/etc/quik.conf"
#define BC_RUN_QUIK   "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/quik -f -r /target >/dev/null"

#define BC_VMELILO_CONF  "/target/etc/vmelilo.conf"
#define BC_RUN_VMELILO    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/vmelilo -f -w /target >/dev/null"

#define BC_SFORMAT     "/target/usr/bin/superformat"
#define BC_RUN_SFORMAT "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; "BC_SFORMAT" %s %s"
#endif

static int write_boot_floppy (char *device)
{
  struct stat tmp;
  FILE* filep;
  int size,status;

  int fd;
  struct floppy_struct floppy_type;
  int num;
  char *buf;

#if #cpu (m68k)
  if(strcmp(Arch2,"Amiga")==0) {
      problemBox(MSG_NO_AMIGA_BOOT_FLOPPY,MSG_PROBLEM);
      return 0;
  }
  if(strcmp(Arch2,"VME")==0) {
      problemBox(MSG_NO_VME_BOOT_FLOPPY,MSG_PROBLEM);
      return 0;
  }
#endif

#if #cpu (powerpc)  
      problemBox(MSG_NO_POWERPC_BOOT_FLOPPY,MSG_PROBLEM);
      return 0;
#endif

  do_umount(device,0);
  do_umount("/floppy",0);
#ifdef SCSI_FLOPPY
  if (strncmp(device,"/dev/sfd", 8)) /* not scsi floppy */
#endif
    fdflush(device);

  /*
   * LFG: Given that getfdprm was changed from debian-1.3.1 to debian-2.0
   * and that the changes are small, I have decided to include the code in
   * here.
   *
   * This code is taken from getfdprm, fdutils-4.3 by Alain Knaff
   * Alain.Knaff@inrialpes.fr
   * that released it under GPL.
   */
  fd=open(device,3);
  if ( fd < 0 )
    fd=open(device,O_RDONLY);
  if ( fd < 0 )
    return 1;
  
  if(ioctl(fd,FDGETPRM,&floppy_type)<0)
    {
      close(fd);
      return 1; 
    }
  close(fd);
  
  /*
   * End of the code taken from getfdprm.c in fdutils.
   */

  size=floppy_type.size/2;
/* We don't have (yet) 2.88 MB bootable floppy images to use with syslinux */
  if (size==2880) size=1440;


  if(stat(BC_SFORMAT,&tmp)==0)
    {
      boxSuspend();
      printf(CLEAR MSG_FORMATTING_FLOPPY);
/* We don't have (yet) 2.88 MB bootable floppy ... */
      sprintf(prtbuf,BC_RUN_SFORMAT,device,(size>720)?"hd":"dd");
      status=system(prtbuf);
      boxResume();
      if (status) return 1;
    }
    
  pleaseWaitBox(MSG_CREATE_FS_BOOT_FLOPPY);

#if #cpu (m68k)
  if (strcmp(Arch2, "Atari") == 0)
     sprintf(prtbuf,"zcat < %s/usr/lib/atari-bootstrap/img%d%s.gz > %s",
	     BC_DIR_PREFIX,size,Arch2,device);
#elif #cpu (i386)
  sprintf(prtbuf,"zcat < %s/usr/lib/syslinux/img%dk.gz > %s",
	  BC_DIR_PREFIX,size,device);
#elif #cpu (sparc)
  /* read it from miniroot fs */
  sprintf(prtbuf,"zcat < /usr/lib/dbootstrap/silo%dk%s.gz > %s",
	  size,suffix_arch,device);
#endif
  status=system(prtbuf);

  boxPopWindow();
  if (status) return 1;

  if( mount_one(device,"/floppy",fs_type,0,"\0",0,0) )
    return 1;

  pleaseWaitBox(MSG_COPY_OS_BOOT_FLOPPY);

  sprintf(prtbuf,"cp %s/vmlinuz /floppy/linux",BC_DIR_PREFIX);
  status=system(prtbuf);

  if (status) {
  	boxPopWindow();
	return 1;
  }

  sync();

  boxPopWindow();
  
  pleaseWaitBox(MSG_WRITE_CONF_BOOT_FLOPPY);
#if #cpu (m68k)
  if(strcmp(Arch2,"Atari")==0) {
    if((filep=fopen("/floppy/bootargs","w"))==NULL) {
      do_umount("/floppy",0);
      boxPopWindow();
      return 1;
    }
    fprintf(filep,"-s -k a:\\linux root=%s",Root->name);
    fclose(filep);
  }
#elif #cpu (i386)
  if((filep=fopen("/floppy/syslinux.cfg","w"))==NULL) {
    do_umount("/floppy",0);
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"DISPLAY message.txt\n");
  fprintf(filep,"TIMEOUT 40\n");
  fprintf(filep,"PROMPT 1\n");
  fprintf(filep,"DEFAULT linux\n");
  fprintf(filep,"APPEND root=%s ro\n",Root->name);
  fclose(filep);
#elif #cpu (sparc)
  if((filep=fopen("/floppy/etc/silo.conf","w"))==NULL) {
    do_umount("/floppy",0);
    boxPopWindow();
    eject_floppy(device);
    return 1;
  }
  fprintf(filep,"partition=3\n");
  fprintf(filep,"root=%s\n",device);
  fprintf(filep,"timeout=100\n");
  fprintf(filep,"image=/linux\n");
  fprintf(filep,"label=linux\n");
  fprintf(filep,"read-only\n");
  fprintf(filep,"append=\"root=%s\"\n",Root->name);
  fclose(filep);
  /* no need to run silo at this time; the floppy disk is already bootable */
#endif
  
  if((filep=fopen("/floppy/message.txt","w"))!=NULL) {
    if ( NAME_ISREG(BOOT_MESSAGE, &tmp) && 
	 ( (fd=open(BOOT_MESSAGE, O_RDONLY)) >= 0 ) ) {
      buf = (char *) malloc (tmp.st_size+1);
      num = read (fd, buf, tmp.st_size);
      close(fd);
      buf[tmp.st_size] = '\0';
      fprintf(filep,buf,Root->name);
      status = 0;
    } else {
      status = 1;
    }
    fclose(filep);
  } else 
    status = 1;
  
  boxPopWindow();
  do_umount("/floppy",0);

  /* eject the floppy disk if avail (see dinstall.h) */
  eject_floppy(device);

  return status;
}

int make_boot_floppy (void)
{
  char* device="/dev/fd0";
  char* designation=MSG_FIRST;
  char* formatted="";
  int result;
  struct stat tmp;

  if(strcmp(InstallationRootDevice,"/dev/fd0")==0) {
    sprintf(prtbuf,MSG_USE_SECOND_FLOPPY_L,Root->name);
    if(yesNoBox(prtbuf,MSG_USE_SECOND_FLOPPY)!=0)
      return 1;
    device="/dev/fd1";
    designation=MSG_SECOND;
  }

  if(stat(BC_SFORMAT,&tmp)!=0) {
    formatted=MSG_FORMATTED;
  }

  sprintf(prtbuf,MSG_CHANGE_DISK_L,formatted,designation);
  problemBox(prtbuf,MSG_CHANGE_DISK);

  result=write_boot_floppy(device);
  /* FIXME: it should be a write_it_down box */
  if(result!=0) {
      sprintf(prtbuf,MSG_BOOT_FLOPPY_FAILED,designation);
      problemBox(prtbuf,MSG_PROBLEM);
  } else {
    notCreatedBootFloppy=0;
  }  
  return result;
}

#ifdef USE_LILO
static int run_lilo(const char *boot)
{
  FILE* filep;
  pleaseWaitBox(MSG_RUNNING_LILO);

  if((filep=fopen(BC_LILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"boot=%s\n",boot);
  fprintf(filep,"root=%s\n",Root->name);
  fprintf(filep,"install=/boot/boot.b\nmap=/boot/map\n");
  fprintf(filep,"vga=normal\ndelay=20\nimage=/vmlinuz\n\tlabel=Linux\n");
  fprintf(filep,"\tread-only\n");
  fclose(filep);
  boxPopWindow();
  if(strlen(boot)==0) return 1;
  if(system(BC_RUN_LILO)) return 1;
  return 0;
}
#endif

static int firstnumber(const char* boot)
{
  int idx,ch;
  idx=strlen(boot);
  do {
    idx--;
    ch=boot[idx];
  } while ((ch>='0')&&(ch<='9'));
  idx++;
  return idx;
}

#ifdef USE_SILO
static int run_silo(char *boot)
{
  FILE* filep;
  int idx;
  char *device, *partition;
  int retcode;

  idx=firstnumber(boot);
  device=strdup(boot);
  device[idx]='\0';
  partition=strdup(boot+idx);
  pleaseWaitBox(MSG_RUNNING_SILO);
  if((filep=fopen(BC_SILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"partition=%s\n",partition);
  fprintf(filep,"root=%s\n",Root->name);
  fprintf(filep,"timeout=100\n");
  fprintf(filep,"image=/vmlinuz\nlabel=linux\n");
  fprintf(filep,"read-only\n");
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  retcode = (strlen(boot) > 0 && system(BC_RUN_SILO) == 0) ? 0 : 1;
  boxPopWindow();
  free(partition);
  free(device);
  return retcode;
}
#endif

#ifdef USE_PLILO  /* PowerPC-LILO: for the moment it is QUIK */
static int run_plilo(char *boot)
{
  FILE* filep;
  int idx;
  char *device, *partition;
  int retcode;

  idx=firstnumber(boot);
  device=strdup(boot);
  device[idx]='\0';
  partition=strdup(boot+idx);
  pleaseWaitBox(MSG_RUNNING_QUIK);
  if((filep=fopen(BC_QUIK_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"partition=%s\n",partition);
  fprintf(filep,"root=%s\n",Root->name);
  fprintf(filep,"timeout=100\n");
  fprintf(filep,"image=/vmlinuz\nlabel=linux\n");
  fprintf(filep,"read-only\n");
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  retcode = (strlen(boot) > 0 && system(BC_RUN_QUIK) == 0) ? 0 : 1;
  boxPopWindow();
  free(partition);
  free(device);
  return retcode;
}
#endif

#ifdef USE_LILOM68K
static int run_vmelilo(const char *boot)
{
  FILE* filep;

  pleaseWaitBox(MSG_RUNNING_VMELILO);
  if((filep=fopen(BC_VMELILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"[options]\n");
  fprintf(filep,"default   = Linux\n");
  fprintf(filep,"boot      = %s\n", boot);
  fprintf(filep,"delay     = 2\n");
  fprintf(filep,"\n");
  fprintf(filep,"[boot]\n");
  fprintf(filep,"label     = Linux\n");
  fprintf(filep,"image     = /vmlinuz\n");
  fprintf(filep,"cmdline   = root=%s ro console=/dev/ttyS0\n", Root->name);
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  boxPopWindow();
  if(strlen(boot)==0) return 1;
  if(system(BC_RUN_VMELILO)) return 1;
  return 0;
}
#endif

#ifdef USE_LILO
static int install_mbr(const char* boot)
{
  int idx,status=0;
  char *device;

  idx=firstnumber(boot);
  device=strdup(boot);
  device[idx]='\0';
  sprintf(prtbuf,MSG_CREATE_MBR_L,device);
  if(yesNoBox(prtbuf,MSG_CREATE_MBR)!=0) {
    sprintf(prtbuf,"cp %s %s",BC_MBR,device);
    status=system(prtbuf);
  }
  free(device);
  return status;
}

static int set_boot_default(const char* boot)
{
  int idx,status=0;
  char *device, *partition;

  idx=firstnumber(boot);
  device=strdup(boot);
  device[idx]='\0';
  partition=strdup(boot+idx);
  sprintf(prtbuf,MSG_LINUX_DEFAULT_BOOT_L,boot);
  if(yesNoBox(prtbuf,MSG_LINUX_DEFAULT_BOOT)!=0)
    {
      sprintf(prtbuf,BC_BOOT_DFLT,device,partition);
      status=system(prtbuf);
    }
  free(device);
  free(partition);
  return status;
}
#endif

int make_bootable(void) 
{
  int status=0;
#if #cpu (i386)
  char *boot,dtype,dnum,*part;
  dtype=*((Root->name)+5);
  dnum=*((Root->name)+7);
  part=(Root->name)+8;

  fprintf(stderr,"Root=%s\n",Root->name);

  if(dnum!='a') {
    /* The root partition is not on the first IDE or SCSI disk. */
    if (0==yesNoBox(MSG_NOT_1ST_DISK,MSG_PROBLEM)) {
      problemBox(MSG_BOOT_IMPOSSIBLE,MSG_PROBLEM);
      run_lilo("");
      return 0;
    }
  }

#ifdef _TESTING_
#define HDA_EXISTS 1
  if ( (dtype=='s') && (HDA_EXISTS) ) {
#else
  if ( (dtype=='s') && (fdisk_find_disk("/dev/hda")) ) {
#endif
    /* The root partition is on the first SCSI disk, but hda exists. */
    if (0==yesNoBox(MSG_HDA_EXIST,MSG_PROBLEM)) {
      problemBox(MSG_BOOT_IMPOSSIBLE,MSG_PROBLEM);
      run_lilo("");
      return 0;    }
  }

  if ( (part[0]>'4') || (part[1]!='\0') ){
    /* The root partition is a logical one. Let the user choose an 
     * extended one.
     */
    struct fdisk_partition *p=NULL;
    p=select_not_mounted( MSG_ROOT_IS_LOGICAL,MSG_SELECT_PARTITION,
                          1,FSTYPE_EXTPART);
    if(p) {
      boot=strdup(p->name);
    } else {
      problemBox(MSG_BOOT_IMPOSSIBLE,MSG_PROBLEM);
      run_lilo("");
      return 0;
    }
  } else {
    boot=strdup(Root->name);
  }

  if(run_lilo(boot)!=0) {
    /* FIXME: write_it_down. */
    problemBox(MSG_LILO_PROBLEM,MSG_PROBLEM);
    status=1;
  } else {
    if ((status=install_mbr(boot))==0)
      status=set_boot_default(boot);
    notInstalledLILO=0;
  }  
  free(boot);
#elif #cpu (m68k)
  char *boot;
  if (strcmp(Arch2, "VME") == 0) {
    boot=strdup(Root->name);
    boot[firstnumber(boot)] = '\0';
    status=run_vmelilo(boot);
    if(status!=0) {
      problemBox(MSG_VMELILO_PROBLEM,MSG_PROBLEM);
      free(boot);
      return 1;
    }
    notInstalledLILO=0;
    free(boot);
  } else {
    problemBox(MSG_M68K_LILO_IMPOSSIBLE,MSG_PROBLEM);
  }
#elif #cpu (powerpc)
  char *boot;

  boot=strdup(Root->name);
  status=run_plilo(boot);
  if(status!=0) {
    problemBox(MSG_POWERPC_QUIK_IMPOSSIBLE,MSG_PROBLEM);
    free(boot);
    return 1;
  }
  notInstalledLILO=0;
  free(boot);
#elif #cpu (sparc)
  char *boot;

  boot=strdup(Root->name);
  status=run_silo(boot);
  if(status!=0) {
    problemBox(MSG_SILO_PROBLEM,MSG_PROBLEM);
    free(boot);
    return 1;
  }
  notInstalledLILO=0;
  free(boot);
#elif #cpu (alpha)
  problemBox(MSG_ALPHA_MILO_IMPOSSIBLE,MSG_PROBLEM);
#else
# error no boot loader to configure
#endif
  return status; 
}

#ifdef _TESTING_
/*
 * To test, compile using: make bootconfig_test
 */

void main(int argc,char** argv)
{
  Root=calloc(1,sizeof(struct fdisk_partition));
  fdisk_disks=calloc(1,sizeof(struct fdisk_disk));
  Root->name=strdup("/dev/sda1");
  fdisk_disks->name=strdup("/dev/sda");
  InstallationRootDevice=strdup("/dev/sdc");

  if(argc!=2)
    {
      fprintf(stderr,"Boot Config test program\n%s (floppy|lilo)\n",argv[0]);
      exit(-1);
    }

  initScreen("Boot Config test program");
  
  if(strcmp(argv[1],"floppy")==0)
    make_boot_floppy();
  else if(strcmp(argv[1],"lilo")==0)
    make_bootable();
  else
    {
      finishScreen();
      fprintf(stderr,"Boot Config test program\n%s (floppy|lilo)\n",argv[0]);
      exit(-1);
    }
  finishScreen();
}
#endif
