/****************************************************************/
/* file aritx.c

ARIBAS interpreter for Arithmetic
Copyright (C) 1996 O.Forster

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Address of the author

	Otto Forster
	Math. Institut der LMU
	Theresienstr. 39
	D-80333 Muenchen, Germany

Email	forster@rz.mathematik.uni-muenchen.de
*/
/****************************************************************/
/*
** aritx.c
** extended arithmetic functions
**
** date of last change
** 95-01-07:	brill1
** 95-03-05:	ithbit
** 95-03-11:	mod_pemult, mod_coshmult
** 95-04-09:	rab_primetest
** 97-02-24:	rab_primetest, overflow handling
** 97-03-02:	cf_factorize: small speedup (pivots)
** 97-04-13:	reorg (newintsym)
** 97-11-26:    Fixed bug in Ffactorial
*/

#include "common.h"

PUBLIC void iniaritx	_((void));
PUBLIC truc modpowsym;

/*--------------------------------------------------------*/
/********* fuer rho-Faktorisierung **********/
typedef struct {
	word2 *x, *y, *Q;
	int xlen, ylen, Qlen;
} T_xyQ;

/********* fuer CF-Faktorisierung ************/

typedef struct {
	word2 *bitmat;
	word2 *piv;
	word2 *fbas;
	word2 *Aarr;
	word2 *Qarr;
	int rank;
	int matinc;
	int baslen;
	int easize;
	int v2len;
	int vlen;
	int ainc;
	int qinc;
} BMDATA;

typedef struct {
	word2 *kN;
	word2 *m2;
	word2 *R0;
	word2 *Q0;
	word2 *A0;
	word2 *QQ;
	word2 *AA;
	int qrsign;
} CFDATA;

#define ALEN		17
#define QLEN		9
#define MEMPIECE	32000
#define STACKRES	4092

/*--------------------------------------------------------*/
#define ithbit(k)	(1 << (k))
#define setbit(vv,i)	vv[(i)>>4] |= ithbit((i)&0xF)
#define testbit(vv,i)	(vv[(i)>>4] & ithbit((i)&0xF))
/*-------------------------------------------------------------------*/
PRIVATE void primsiev	_((word2 *vect, int n));
PRIVATE truc Fjacobi	_((void));
PRIVATE int jacobi	_((int sign, word2 *x, int n, word2 *y, int m,
			   word2 *hilf));
PRIVATE int jac		_((unsigned x, unsigned y));
PRIVATE truc Ffact16	_((int argn));
PRIVATE truc Fprime32	_((void));
PRIVATE unsigned trialdiv  _((word2 *x, int n, unsigned u0, unsigned u1));
PRIVATE int prime16	_((unsigned u));
PRIVATE int prime32	_((word2 *x, int n));
PRIVATE truc Frabtest	_((void));
PRIVATE int rabtest	_((word2 *x, int n));
PRIVATE truc Fgcd	_((int argn));
PRIVATE unsigned gcdfixnums  _((truc *argptr, int argn));
PRIVATE int gcdbignums	_((word2 *x, word2 *y, truc *argptr, int argn));
PRIVATE truc Fmodinv	_((void));
PRIVATE truc Sgcdx	_((void));
PRIVATE int gcdcx	_((word2 *x, int n, word2 *y, int m,
			   word2 *cx, int *cxlptr, word2 *hilf));
PRIVATE int gcdcxcy	_((word2 *x1, int n1, word2 *x, int n, word2 *y,int m,
			   word2 *cx, int cxlen, word2 *cy, word2 *hilf));
PRIVATE truc Fmodpower	_((int argn));
PRIVATE int modpower	_((word2 *x, int n, word2 *ex, int exlen,
			   word2 *mm, int modlen, word2 *p, word2 *hilf));
PRIVATE truc Fcoshmult	_((int argn));
PRIVATE int coshmult	_((word2 *x, int n, word2 *ex, int exlen,
			   word2 *mm, int modlen, word2 *z, word2 *hilf));
PRIVATE truc Fpemult	_((int argn));
PRIVATE int pemult	_((word2 *x, int n, word2 *ex, int exlen,
			   word2 *aa, int alen,
			   word2 *mm, int modlen, word2 *z, word2 *hilf));
PRIVATE truc Frhofact	_((int argn));
PRIVATE int rhocycle	_((int anz, T_xyQ *xyQ, word2 *N, int len,
			   word2 *hilf));
PRIVATE void rhomess	_((word4 i));

PRIVATE truc Fcffact	_((int argn));
PRIVATE int brillmorr	_((word2 *N, int len, unsigned v, word2 *fact,
			   word2 *hilf));
PRIVATE int bmcfinit	_((int len, word2 *buf1, size_t len1,
			 word2 *buf2, size_t len2, BMDATA *bmp, CFDATA *cfp));
PRIVATE int brill1	_((word2 *N, int len, unsigned u, word2 *fact,
			   BMDATA *bmp, CFDATA *cfp, word2 *hilf));
PRIVATE void factorbase	 _((int anz, word2 *prim, word2 *N, int len));
PRIVATE int cfracinit	_((word2 *N, int len, unsigned u, CFDATA *cfp,
			   word2 *hilf));
PRIVATE int cfracnext	_((word2 *N, int len, CFDATA *cfp, word2 *hilf));
PRIVATE int smooth	_((CFDATA *cfp, BMDATA *bmp, word2 *hilf));
PRIVATE int gausselim	_((BMDATA *bmp));
PRIVATE int getfactor	_((word2 *N, int len, BMDATA *bmp, word2 *fact,
			   word2 *hilf));
PRIVATE void workmess	_((void));
PRIVATE void tick	_((int c));
PRIVATE int counttick	_((int v));
PRIVATE void cf0mess	_((int p, int blen));
PRIVATE void cf1mess	_((long n, int nf));
PRIVATE truc Ffactorial	 _((void));
PRIVATE int factorial	_((unsigned a, word2 *x));

PRIVATE truc jacobsym, gcdsym, gcdxsym, modinvsym;
PRIVATE truc factorsym, primsym, rabtestsym;
PRIVATE truc rhosym, cffactsym, pemultsym, cshmultsym;
PRIVATE truc factsym;

/*------------------------------------------------------------------*/
PUBLIC void iniaritx()
{
	modpowsym = newintsym("** mod", sFBINARY,(truc)Fmodpower);

	gcdsym	  = newsymsig("gcd",sFBINARY,(truc)Fgcd, s_0uii);
	gcdxsym	  = newsymsig("gcdx",sSBINARY,(truc)Sgcdx, s_iiiII);
	modinvsym = newsymsig("mod_inverse",sFBINARY,(truc)Fmodinv, s_iii);

	jacobsym  = newsymsig("jacobi",	 sFBINARY,   (truc)Fjacobi, s_iii);
	primsym	  = newsymsig("prime32test",sFBINARY,(truc)Fprime32,s_ii);
	rabtestsym= newsymsig("rab_primetest",sFBINARY,(truc)Frabtest,s_1);
	factorsym = newsymsig("factor16",sFBINARY,(truc)Ffact16, s_13);

	rhosym = newsymsig("rho_factorize",sFBINARY,(truc)Frhofact, s_12ii);
	cffactsym = newsymsig("cf_factorize",sFBINARY,(truc)Fcffact, s_12ii);
	pemultsym = newsymsig("mod_pemult",sFBINARY,(truc)Fpemult, s_Viiii);
	cshmultsym= newsymsig("mod_coshmult",sFBINARY,(truc)Fcoshmult,
						s_iiii);
	factsym	  = newsymsig("factorial", sFBINARY, (truc)Ffactorial,s_ii);

	primsiev(PrimTab,PRIMTABSIZE);
}
/*-------------------------------------------------------------------*/
/*
** Setzt im Bit-Vektor vect der Laenge n*16 bit (8 <= n <= 2096)
** das bit i, falls 2*i + 1 eine Primzahl ist
*/
PRIVATE void primsiev(vect,n)
word2 *vect;
int n;
{
	word2 mask[16];
	unsigned i, k, inc;

	for(i=0; i<16; i++)
		mask[i] = ~ithbit(i);
	setarr(vect,n,0xFFFF);
	n <<= 4;			/* mal 16 */
	vect[0] &= 0xFFFE;		/* da 1 keine Primzahl */
	for(i=1; i<128; i++) {		/* Zahlen 3 bis 255 */
		if(testbit(vect,i)) {	/* falls 2*i + 1 Primzahl */
			inc = i + i + 1;
			for(k=i+inc; k<n; k+=inc)
				vect[k >> 4] &= mask[k & 0xF];
		}
	}
}
/*-------------------------------------------------------------------*/
/*
** Stellt fest, ob die 16-bit-Zahl u prim ist
*/
PRIVATE int prime16(u)
unsigned u;
{
	if(!(u & 1))
		return(u == 2);
	u >>= 1;
	if(testbit(PrimTab,u))
		return(1);
	else
		return(0);
}
/*-------------------------------------------------------------------*/
/*
** Rueckgabewert 1, falls (x,n) kleiner als 2**32 und Primzahl ist,
** -1, falls (x,n) groesser-gleich 2**32 ist, sonst 0
*/
PRIVATE int prime32(x,n)
word2 *x;
int n;
{
	word4 u;
	unsigned v;

	if(n == 1)
		return(prime16(x[0]));
	else if(n == 0)
		return(0);
	else if(n > 2)
		return(-1);
	else {
		u = big2long(x,n);
		v = intsqrt(u);
		return(trialdiv(x,n,2,v) ? 0 : 1);
	}
}
/*------------------------------------------------------------------*/
PRIVATE truc Fjacobi()
{
	int n, m, sign1, sign2;
	int j;
	word2 *x, *y, *hilf;

	if(chkints(jacobsym,argStkPtr-1,2) == ERROR)
		return(brkerr());
	x = AriBuf;
	y = AriScratch;
	hilf = y + aribufSize;
	n = bigretr(argStkPtr-1,x,&sign1);
	m = bigretr(argStkPtr,y,&sign2);
	if(m == 0 || (*y & 1) == 0) {
		error(jacobsym,err_odd,*argStkPtr);
		return(brkerr());
	}
	j = jacobi(sign1,x,n,y,m,hilf);
	return(mksfixnum(j));
}
/*-------------------------------------------------------------------*/
/*
** Berechnet das Jacobi-Symbol von (x,n) ueber (y,m).
** Es wird vorausgesetzt, dass y ungerade ist.
** Falls x und y nicht teilerfremd sind, wird 0 zurueckgegeben.
*/
PRIVATE int jacobi(sign,x,n,y,m,hilf)
word2 *x, *y, *hilf;
int sign, n, m;
{
	int res;
	int m8, tlen;
	word2 *temp;

	res = (sign && (y[0] & 2) ? -1 : 1);	/* (jacobi -1 y) */
	while(n >= 1) {
		while((x[0] & 1) == 0) {
			n = shrarr(x,n,1);
			m8 = y[0] & 0x7;
			if(m8 == 3 || m8 == 5)
				res = -res;
		}
		if(*x == 1 && n == 1)
			return(res);
		temp = x;
		tlen = n;
		x = y;
		n = m;
		y = temp;
		m = tlen;
		if((x[0] & 2) && (y[0] & 2))	/* beide = 3 mod 4 */
			res = -res;
		n = modbig(x,n,y,m,hilf);
	}
	return(0);
}
/*-------------------------------------------------------------------*/
/*
** Version von jacobi fuer kleine nichtnegative integers
** y muss ungerade sein
*/
PRIVATE int jac(x,y)
unsigned x, y;
{
	int res = 1;
	int m8;
	unsigned temp;

	while(x >= 1) {
		while((x & 1) == 0) {
			x >>= 1;
			m8 = y & 0x7;
			if(m8 == 3 || m8 == 5)
				res = -res;
		}
		if(x == 1)
			return(res);
		if((x & 2) && (y & 2))		/* beide = 3 mod 4 */
			res = -res;
		temp = x;
		x = y % temp;
		y = temp;
	}
	return(0);
}
/*------------------------------------------------------------------*/
PRIVATE truc Ffactorial()
{
	int n;
	word2 a;

	if(*FLAGPTR(argStkPtr) != fFIXNUM || *SIGNPTR(argStkPtr)) {
		error(factsym,err_pfix,*argStkPtr);
		return(brkerr());
	}
	a = *WORD2PTR(argStkPtr);
        if(a <= 1)
                return(constone);
	if(bitlen(a) > maxfltex/a + 1) {
		error(factsym,err_ovfl,voidsym);
		return(brkerr());
	}
	n = factorial(a,AriBuf);
	return(mkint(0,AriBuf,n));
}
/*------------------------------------------------------------------*/
/*
** Hypothesis: a > 1
** factorial of a is stored in x (which must be large enough),
** its length is returned
*/
PRIVATE int factorial(a,x)
unsigned a;
word2 *x;
{
	word4 u;
	unsigned small = 718;
	unsigned i, k, b;
	int len, sh;

	if(a & 1) {
		x[0] = a;
		a--;
	}
	else
		x[0] = 1;
	len = 1;
#ifdef M_3264
	if(!(a & 2)) {
		u = a;
		u *= a-1;
		len = mult4arr(x,len,u,x);
		a -= 2;
	} /* now a = 2 mod 4 */
	b = (a <= small ? a : small);
	for(i=3, k=b>>1; i<b; i+=4, k-=2) {
		u = i*k;
		u *= (i+2)*(k-1);
		len = mult4arr(x,len,u,x);
	}
	for(i=small+1, k=(small>>1)+1; i<a; i+=2, k++) {
		u = i;
		u *= k;
		len = mult4arr(x,len,u,x);
	}
#else
	b = (a <= small ? a : small);
	for(i=3, k=b>>1; i<b; i+=2, k--)
		len = multarr(x,len,i*k,x);
	for(i=small+1, k=(small>>1)+1; i<a; i+=2, k++) {
		len = multarr(x,len,i,x);
		len = multarr(x,len,k,x);
	}
#endif
	sh = a >> 1;
	len = shiftarr(x,len,sh);
	return(len);
}
/*------------------------------------------------------------------*/
PRIVATE truc Ffact16(argn)
int argn;		/* 1 <= argn <= 3 */
{
	truc *argptr;
	word2 *x;
	unsigned u;
	unsigned u0 = 2, u1 = 0xFFFD;
	int n, sign, flg;

	argptr = argStkPtr - argn + 1;
	if(chkints(factorsym,argptr,argn) == ERROR)
		return(brkerr());
	n = bigref(argptr,&x,&sign);
	if(argn >= 2) {
		argptr++;
		flg = *FLAGPTR(argptr);
		if(flg != fFIXNUM)
			return(zero);
		else
			u0 = *WORD2PTR(argptr);
	}
	if(argn >= 3 && *FLAGPTR(argStkPtr) == fFIXNUM)
		u1 = *WORD2PTR(argStkPtr);
	u = trialdiv(x,n,u0,u1);
	return(mkfixnum(u));
}
/*------------------------------------------------------------------*/
PRIVATE truc Fprime32()
{
	word2 *x;
	int sign, n;

	n = bigref(argStkPtr,&x,&sign);
	if(n == ERROR) {
		error(primsym,err_int,*argStkPtr);
		return(brkerr());
	}
	return(mksfixnum(prime32(x,n)));
}
/*-------------------------------------------------------------------*/
/*
** Probedivision von (x,n) durch 16-bit Primzahlen
** zwischen u0 >= 2 und u1 <= (x,n)/2
** Der erste Teiler wird zurueckgegeben; 
** falls keiner gefunden, Rueckgabewert 0
*/
PRIVATE unsigned trialdiv(x,n,u0,u1)
word2 *x;
int n;
unsigned u0, u1;
{
	extern word2 *PrimTab;

	word2 *pbits;
	unsigned u, mask = 1;

	u = x[0];
	if(n == 1) {
		if(prime16(u))
			return(0);
		else if(u1 > (u >> 1))
			u1 = (u >> 1);
	}
	if(u1 > 0xFFFD)
		u1 = 0xFFFD;
	if(u0 > u1 || u1 < 2 || !n)
		return(0);
	if(u0 <= 2 && !(u & 1))
		return(2);
	u0 >>= 1;
	pbits = PrimTab + (u0 >> 4);
	mask <<= (u0 & 0xF);
	for(u = 2*u0 + 1; u <= u1; u += 2) {
		if((*pbits & mask) && (modarr(x,n,u) == 0))
			return(u);
		mask <<= 1;
		if((mask & 0xFFFF) == 0) {
			pbits++;
			mask = 1;
		}
	}
	return(0);
}
/*------------------------------------------------------------------*/
/*
** Strong-Pseudo-Primzahltest nach Rabin
*/
PRIVATE truc Frabtest()
{
	word2 *x;
	int sign, n;

	n = bigref(argStkPtr,&x,&sign);
	if(n == ERROR) {
		error(rabtestsym,err_int,*argStkPtr);
		return(brkerr());
	}
	if(n == 0 || x[0] & 1 == 0)
		return(false);
	else if(n <= 2)
		return(prime32(x,n) > 0 ? true : false);
	else if(n < aribufSize/2) 
		return(rabtest(x,n) ? true : false);
	else {
		error(rabtestsym,err_ovfl,voidsym);
		return(brkerr());
	}
}
/*------------------------------------------------------------------*/
PRIVATE int rabtest(x,n)
word2 *x;
int n;
{
	word2 *base, *ex, *y, *x1, *hilf;
	int exlen, ylen, n1;
	int i, t;

	base = AriBuf;
	y = AriBuf + n;
	x1 = AriScratch;
	ex = AriScratch + n;
	hilf = AriScratch + aribufSize;

	base[0] = 2 + random2(64000);
	cpyarr(x,n,x1);
	n1 = decarr(x1,n,1);
	cpyarr(x1,n1,ex);
	exlen = n1;
	t = 0;
	while((ex[0] & 1) == 0) {
		t++;
		exlen = shrarr(ex,exlen,1);
	}
	ylen = modpower(base,1,ex,exlen,x,n,y,hilf);
	if((ylen == 1 && y[0] == 1) || (cmparr(y,ylen,x1,n1) == 0)) {
		return(1);
	}
	/* else */
	ex[0] = 2; exlen = 1;
	for(i=1; i<t; i++) {
		cpyarr(y,ylen,base);
		ylen = modpower(base,ylen,ex,exlen,x,n,y,hilf);
		if(cmparr(y,ylen,x1,n1) == 0)
			return(1);
	}
	return(0);
}
/*------------------------------------------------------------------*/
PRIVATE truc Fgcd(argn)
int argn;		/* 0 <= argn < infinity */
{
	truc *argptr;
	unsigned u;
	int n, flg;

	if(argn == 0)
		return(zero);
	argptr = argStkPtr - argn + 1;
	flg = chkints(gcdsym,argptr,argn);
	if(flg == ERROR)
		return(brkerr());
	if(argn == 1)
		return(*argStkPtr);
	if(flg == fFIXNUM) {
		u = gcdfixnums(argptr,argn);
		return(mkfixnum(u));
	}
	else {
		n = gcdbignums(AriBuf,AriScratch,argptr,argn);
		return(mkint(0,AriBuf,n));
	}
}
/*------------------------------------------------------------------*/
PRIVATE unsigned gcdfixnums(argptr,argn)
truc *argptr;
int argn;	/* argn > 0 */
{
	unsigned x;

	x = *WORD2PTR(argptr++);
	while(--argn > 0) {
		x = shortgcd(x,*WORD2PTR(argptr++));
		if(x == 1)
			break;
	}
	return(x);
}
/*------------------------------------------------------------------*/
PRIVATE int gcdbignums(x,y,argptr,argn)
word2 *x, *y;
truc *argptr;
int argn;	/* argn > 0 */
{
	word2 *hilf;
	int n, m, sign;

	n = bigretr(argptr++,x,&sign);
	while(--argn > 0) {
		m = bigretr(argptr++,y,&sign);
		if(m == 0)
			continue;
		hilf = y + m;
		n = biggcd(x,n,y,m,hilf);
		if(*x == 1 && n == 1)
			break;
	}
	return(n);
}
/*-------------------------------------------------------------------*/
PRIVATE truc Fmodinv()
{
	word2 *x, *y, *cx;
	int n, m, bound, cxlen, n1;
	int sign1, sign2;

	if(chkints(modinvsym,argStkPtr-1,2) == ERROR) {
		return(brkerr());
	}
	x = AriBuf;
	n = bigretr(argStkPtr-1,x,&sign1);
	m = bigref(argStkPtr,&y,&sign2);
	if(n == 0 || m == 0) {
		return(zero);
	}
	bound = (aribufSize >> 1) - 2;
	if((n >= bound) || (m >= bound)) {
		error(modinvsym,err_ovfl,voidsym);
		return(brkerr());
	}
	cx = x + n + 2;
	n1 = gcdcx(x,n,y,m,cx,&cxlen,AriScratch);
	if(n1 != 1 || x[0] != 1)	/* not relatively prime */
		return(zero);
	else if(sign1)
		cxlen = sub1arr(cx,cxlen,y,m);
	return(mkint(0,cx,cxlen));
}
/*-------------------------------------------------------------------*/
PRIVATE truc Sgcdx()
{
	truc res;
	word2 *x, *y, *cx, *cy, *x1;
	int n, m, cxlen, cylen, n1;
	int sign1, sign2;
	int flg;

	res = eval(ARGNPTR(evalStkPtr,1));
	ARGpush(res);
	res = eval(ARGNPTR(evalStkPtr,2));
	ARGpush(res);
	flg = chkints(gcdxsym,argStkPtr-1,2);
	if(flg == ERROR) {
		res = brkerr();
		goto cleanup;
	}
	x1 = AriBuf;
	n = bigref(argStkPtr-1,&x,&sign1);
	cpyarr(x,n,x1);
	m = bigref(argStkPtr,&y,&sign2);
	cx = x1 + n + 2;
	cy = cx + m + 2;
	if(n && m) {
		n1 = gcdcx(x1,n,y,m,cx,&cxlen,AriScratch);
		cylen = gcdcxcy(x1,n1,x,n,y,m,cx,cxlen,cy,AriScratch);
		sign2 = (sign2 ? 0 : MINUSBYTE);
	}
	else if(n) {	/* x != 0, y == 0 */
		n1 = n;
		cx[0] = 1;
		cxlen = 1;
		cylen = 0;
	}
	else if(m) {	/* x == 0, y != 0 */
		n1 = m;
		cpyarr(y,m,x1);
		cxlen = 0;
		cy[0] = 1;
		cylen = 1;
	}
	else {		/* x == 0, y == 0 */
		n1 = cxlen = cylen = 0;
	}
	Lvalassign(ARGNPTR(evalStkPtr,3),mkint(sign1,cx,cxlen));
	Lvalassign(ARGNPTR(evalStkPtr,4),mkint(sign2,cy,cylen));
	res = mkint(0,x1,n1);
  cleanup:
	ARGnpop(2);
	return(res);
}
/*-------------------------------------------------------------------*/
/*
** Berechnet den GGT von (x,n) und (y,m), destruktiv auf x
** x wird durch den GGT ersetzt, seine Laenge ist der Rueckgabewert.
** y wird nicht beruehrt
** Ausserdem wird in (cx, *cxlptr) ein Koeffizient != 0 abgelegt,
** so dass (x,n) * (cx,*cxlptr) = GGT mod (y,m).
** Es wird vorausgesetzt, dass (y,m) ungleich Null.
** Platz hilf muss 5 * max(n,m) + 10 lang sein
*/
PRIVATE int gcdcx(x,n,y,m,cx,cxlptr,hilf)
word2 *x, *y, *cx, *hilf;
int n, m;
int *cxlptr;
{
	word2 *q, *x1, *x2, *alfa, *beta, *prod, *temp;
	int n1, n2, N, qlen, rlen, alen, blen, plen, tlen;
	int asign, bsign, tsign;

	N = (n > m ? n + 2 : m + 2);

	alfa = cx;
	x1 = hilf + N;
	q = hilf + 2*N;
	beta = hilf + 3*N;
	prod = hilf + 4*N;

	if(m == 0) {
		return(0);
	}
	alen = 0;
	blen = 1;
	beta[0] = 1;
	asign = bsign = 0;
	qlen = divbig(x,n,y,m,q,&rlen,hilf);
	x2 = x;
	n2 = rlen;
	cpyarr(y,m,x1);
	n1 = m;
/*
** Schleifeninvarianten:
** (x1,n1) = (alfa,alen)*(x,n) mod (y,m)
** (x2,n2) = (beta,blen)*(x,n) mod (y,m)
*/
	while(n2 > 0) {
		qlen = divbig(x1,n1,x2,n2,q,&rlen,hilf);
		/* x1neu = x2alt, x2neu = rest */
		temp = x1;
		x1 = x2;
		n1 = n2;
		x2 = temp;
		n2 = rlen;
		plen = multbig(beta,blen,q,qlen,prod,hilf);
		/* alfaneu = betaalt, betaneu = alfaalt - q*betaalt */
		temp = alfa;
		tlen = alen;
		tsign = asign;
		alfa = beta;
		alen = blen;
		asign = bsign;
		beta = temp;
		if(tsign != bsign) {
			blen = addarr(beta,tlen,prod,plen);
			bsign = tsign;
		}
		else if(cmparr(beta,tlen,prod,plen) >= 0) {
			blen = subarr(beta,tlen,prod,plen);
			bsign = tsign;
		}
		else {
			blen = sub1arr(beta,tlen,prod,plen);
			bsign = (tsign ? 0 : MINUSBYTE);
		}
	}
	alen = modbig(alfa,alen,y,m,hilf);
	if(asign)
		alen = sub1arr(alfa,alen,y,m);
	if(!alen) {
		*alfa = 1;
		alen = 1;
	}
	if(alfa != cx)
		cpyarr(alfa,alen,cx);
	*cxlptr = alen;
	if(x1 != x)
		cpyarr(x1,n1,x);
	return(n1);
}
/*------------------------------------------------------------------*/
/*
** Berechnet den Koeffizienten cy fuer die Darstellung
**		 x1 = cx * x - cy * y.
** x, y, x1 und cx sind vorgegeben.
** Dabei wird vorausgesetzt, dass x1 der GGT von x und y ist und cx != 0.
** Die Laenge von cy ist der Rueckgabewert.
*/
PRIVATE int gcdcxcy(x1,n1,x,n,y,m,cx,cxlen,cy,hilf)
word2 *x1, *x, *y, *cx, *cy, *hilf;
int n1, n, m, cxlen;
{
	word2 *temp;
	int tlen, len, rlen;

	temp = hilf;
	hilf += n + cxlen + 2;

	tlen = multbig(x,n,cx,cxlen,temp,hilf);
	tlen = subarr(temp,tlen,x1,n1);
	len = divbig(temp,tlen,y,m,cy,&rlen,hilf);
	return(len);
}
/*------------------------------------------------------------------*/
/*
** Base ** Ex mod Modulus
*/
PRIVATE truc Fmodpower(argn)
int argn;
{
	word2 *x, *y, *z, *p, *hilf;
	int n, n1, n2, n3;
	int sign1, sign2, sign3;

	if(chkints(modpowsym,argStkPtr-2,3) == ERROR)
		return(brkerr());

	x = AriScratch;
	p = AriBuf;
	hilf = AriScratch + aribufSize;

	n2 = bigref(argStkPtr-1,&y,&sign2);
	if(sign2) {
		error(modpowsym,err_pnum,*(argStkPtr-1));
		return(brkerr());
	}
	n1 = bigretr(argStkPtr-2,x,&sign1);
	n3 = bigref(argStkPtr,&z,&sign3);
	if(!n3) {
		error(modpowsym,err_div,voidsym);
		return(brkerr());
	}
	n = modpower(x,n1,y,n2,z,n3,p,hilf);
	if(sign1 && n && n2 && (*y & 1))
		n = sub1arr(p,n,z,n3);
	return(mkint(0,p,n));
}
/*-------------------------------------------------------------------*/
/*
** p = (x,n) hoch (ex,exlen) modulo (mm,modlen)
** (x,n) wird destruktiv modulo (mm,modlen) reduziert
** hilf ist Platz fuer Hilfsvariable,
** muss 2*modlen + max(n,modlen) + 2 lang sein.
*/
PRIVATE int modpower(x,n,ex,exlen,mm,modlen,p,hilf)
word2 *x, *ex, *mm, *p, *hilf;
int n, exlen, modlen;
{
	word2 *temp;
	int k, plen;

	temp = hilf + 2*modlen + 2;

	if(exlen == 0) {
		p[0] = 1;
		return(1);
	}
	n = modbig(x,n,mm,modlen,hilf);
	if(n == 0)
		return(0);

	cpyarr(x,n,p);
	plen = n;		/* plen <= modlen */

	k = ((exlen-1) << 4) + bitlen(ex[exlen-1]) - 1;
	while(--k >= 0) {
		plen = multbig(p,plen,p,plen,temp,hilf);
		cpyarr(temp,plen,p);
		plen = modbig(p,plen,mm,modlen,hilf);
		if(testbit(ex,k)) {
			plen = multbig(p,plen,x,n,temp,hilf);
			cpyarr(temp,plen,p);
			plen = modbig(p,plen,mm,modlen,hilf);
		}
	}
	return(plen);
}
/*------------------------------------------------------------------*/
/*
** mod_coshmult(Base,Exponent,Module)
*/
PRIVATE truc Fcoshmult(argn)
int argn;	/* argn = 3 */
{
	word2 *x, *y, *z, *res, *hilf;
	int len, n1, n2, n3;
	int sign1, sign;

	if(chkints(cshmultsym,argStkPtr-2,3) == ERROR)
		return(brkerr());

	x = AriScratch;
	res = AriBuf;
	hilf = AriScratch + aribufSize;

	n1 = bigretr(argStkPtr-2,x,&sign1);
	n2 = bigref(argStkPtr-1,&y,&sign);
	n3 = bigref(argStkPtr,&z,&sign);
	if(!n3) {
		error(cshmultsym,err_div,voidsym);
		return(brkerr());
	}
	len = coshmult(x,n1,y,n2,z,n3,res,hilf);
	return(mkint(0,res,len));
}
/*------------------------------------------------------------------*/
/*
** Berechnet z = cosh(ex*xi) modulo mm, wobei x = cosh(xi).
** z = v(ex) berechnet sich durch folgende Rekursion:
** v(0) = 1; v(1) = x;
** v(2*n) = 2*v(n)*v(n) - 1;
** v(2*n+1) = 2*v(n+1)*v(n) - x.
*/
PRIVATE int coshmult(x,n,ex,exlen,mm,modlen,z,hilf)
word2 *x, *ex, *mm, *z, *hilf;
int n, exlen, modlen;
{
	word2 *v, *w, *u, *temp;
	int k, vlen, wlen, ulen, tlen, zlen;
	int bit, bit0;

	v = z;
	w = hilf;
	u = w + modlen + 2;
	hilf = u + 2*modlen + 2;

	if(exlen == 0) {
		z[0] = 1;
		return(1);
	}
	n = modbig(x,n,mm,modlen,hilf);
	v[0] = 1;
	vlen = 1;
	cpyarr(x,n,w);
	wlen = n;

	bit = 0;
	k = ((exlen-1) << 4) + bitlen(ex[exlen-1]);
	while(--k >= 0) {
		bit0 = bit;
		bit = (testbit(ex,k) ? 1 : 0);
		if(bit != bit0) {
			temp = v;	tlen = vlen;
			v = w;		vlen = wlen;
			w = temp;	wlen = tlen;
		}
			/* w := 2*v*w - x */
		ulen = multbig(w,wlen,v,vlen,u,hilf);
		ulen = modbig(u,ulen,mm,modlen,hilf);
		cpyarr(u,ulen,w);
		wlen = addarr(w,ulen,u,ulen);
		if(cmparr(w,wlen,x,n) < 0)
			wlen = addarr(w,wlen,mm,modlen);
		wlen = subarr(w,wlen,x,n);

			/* v := 2*v*v - 1 */
		ulen = multbig(v,vlen,v,vlen,u,hilf);
		ulen = modbig(u,ulen,mm,modlen,hilf);
		cpyarr(u,ulen,v);
		vlen = addarr(v,ulen,u,ulen);
		if(vlen == 0) {
			cpyarr(mm,modlen,v);
			vlen = modlen;
		}
		vlen = decarr(v,vlen,1);
	}
	zlen = (bit == 0 ? vlen : wlen);
	zlen = modbig(z,zlen,mm,modlen,hilf);
	return(zlen);
}
/*------------------------------------------------------------------*/
/*
** mod_pemult(Base,Exponent,Param,Module)
*/
PRIVATE truc Fpemult(argn)
int argn;	/* argn = 4 */
{
	truc *ptr;
	truc vector, obj;
	word2 *x, *y, *aa, *z, *res, *hilf;
	int len, n1, n2, n3, alen;
	int sign1, sign2, sign;

	if(chkints(pemultsym,argStkPtr-3,4) == ERROR)
		return(brkerr());

	x = AriScratch;
	res = AriBuf;
	hilf = AriScratch + aribufSize;

	n1 = bigretr(argStkPtr-3,x,&sign1);
	n2 = bigref(argStkPtr-2,&y,&sign);
	alen = bigref(argStkPtr-1,&aa,&sign2);
	n3 = bigref(argStkPtr,&z,&sign);
	if(!n3) {
		error(pemultsym,err_div,voidsym);
		return(brkerr());
	}
	len = pemult(x,n1,y,n2,aa,alen,z,n3,res,hilf);

	vector = mkvect0(2);
	WORKpush(vector);
	if(len >= 0) {
		obj = mkint(0,res,len);
	}
	else {
		obj = mkint(0,res,-len-1);
	}
	ptr = VECTORPTR(workStkPtr);
	ptr[0] = obj;
	ptr[1] = (len >= 0 ? constone : zero);
	vector = WORKretr();
	return(vector);
}
/*------------------------------------------------------------------*/
/*
** Berechnet fuer die elliptische Kurve B*y*y = x*x*x + aa*x*x + x
** in z die x-Koordinate modulo mm von ex*P,
** wobei P ein Punkt der Kurve mit x-Koordinate (x,n) ist.
** Falls die Berechnung scheitert, weil ein Teiler von
** mm auftaucht, wird dieser Teiler in z abgelegt;
** der Rueckgabewert ist dann -(lenz + 1).
*/
PRIVATE int pemult(x,n,ex,exlen,aa,alen,mm,modlen,z,hilf)
word2 *x, *ex, *aa, *mm, *z, *hilf;
int n, exlen, alen, modlen;
{
	word2 *yy1, *yy2, *pp1, *pp2, *uu1, *uu2;
	int zlen, y1len, y2len, p1len, p2len, u1len, u2len;
	int k, n0, n1, m0, m1;
	int sign0, sign1;
	int odd;
	int ll;
	int first;

	struct {
		word2 *num;
		word2 *den;
		int nlen;
		int dlen;
	} vv, ww, *act0, *act1;

	ll = 2*modlen + 2;
	vv.num = z;
	vv.den = hilf;
	ww.num = hilf + ll;
	ww.den = hilf + 2*ll;
	yy1 = hilf + 3*ll;
	yy2 = hilf + 4*ll;
	pp1 = hilf + 5*ll;
	pp2 = hilf + 6*ll;
	uu1 = hilf + 7*ll;
	uu2 = hilf + 8*ll;
	hilf = hilf + 9*ll;

	n = modbig(x,n,mm,modlen,hilf);
	if(exlen == 0) {
		cpyarr(mm,modlen,z);
		return(-modlen);
	}
	cpyarr(x,n,vv.num);
	vv.nlen = n;
	vv.den[0] = 1;
	vv.dlen = 1;
	cpyarr(x,n,ww.num);
	ww.nlen = n;
	ww.den[0] = 1;
	ww.dlen = 1;

	first = 1;
	k = ((exlen-1) << 4) + bitlen(ex[exlen-1]);
	while(--k >= 0) {
		odd = testbit(ex,k);
		act0 = (odd ? &ww : &vv);
		act1 = (odd ? &vv : &ww);
		n0 = act0->nlen;
		m0 = act0->dlen;
		n1 = act1->nlen;
		m1 = act1->dlen;

		if(first) {
			first = 0;
			goto duplic;
		}
/* 
** i --> 2*i + 1 
** X(2*i+1) = ((X(i+1)-Z(i+1))*(X(i)+Z(i)) + (X(i+1)+Z(i+1))*(X(i)-Z(i)))**2
** Z(2*i+1) = x*(((X(i+1)-Z(i+1))*(X(i)+Z(i))-(X(i+1)+Z(i+1))*(X(i)-Z(i)))**2
*/
		sign0 = cmparr(act0->num,n0,act0->den,m0);
		sign1 = cmparr(act1->num,n1,act1->den,m1);
		cpyarr(act1->num,n1,yy1);
		if(sign1 >= 0)
			y1len = subarr(yy1,n1,act1->den,m1);
		else
			y1len = sub1arr(yy1,n1,act1->den,m1);
		cpyarr(act0->num,n0,yy2);
		y2len = addarr(yy2,n0,act0->den,m0);
		p1len = multbig(yy1,y1len,yy2,y2len,pp1,hilf);
		p1len = modbig(pp1,p1len,mm,modlen,hilf);
/* pp1 = (X(i+1)-Z(i+1))*(X(i)+Z(i)) */

		cpyarr(act1->num,n1,yy1);
		y1len = addarr(yy1,n1,act1->den,m1);
		cpyarr(act0->num,n0,yy2);
		if(sign0 >= 0)
			y2len = subarr(yy2,n0,act0->den,m0);
		else
			y2len = sub1arr(yy2,n0,act0->den,m0);
		p2len = multbig(yy1,y1len,yy2,y2len,pp2,hilf);
		p2len = modbig(pp2,p2len,mm,modlen,hilf);
/* pp2 = (X(i+1)+Z(i+1))*(X(i)-Z(i)) */

		cpyarr(pp1,p1len,uu1);
		if(sign0 * sign1 >= 0)
			u1len = addarr(uu1,p1len,pp2,p2len);
		else if(cmparr(pp1,p1len,pp2,p2len) >= 0)
			u1len = subarr(uu1,p1len,pp2,p2len);
		else 
			u1len = sub1arr(uu1,p1len,pp2,p2len);
		u2len = multbig(uu1,u1len,uu1,u1len,uu2,hilf);
		u2len = modbig(uu2,u2len,mm,modlen,hilf);
		cpyarr(uu2,u2len,act1->num);
		act1->nlen = u2len;

		cpyarr(pp1,p1len,uu1);
		if(sign0 * sign1 <= 0)
			u1len = addarr(uu1,p1len,pp2,p2len);
		else if(cmparr(pp1,p1len,pp2,p2len) >= 0)
			u1len = subarr(uu1,p1len,pp2,p2len);
		else 
			u1len = sub1arr(uu1,p1len,pp2,p2len);
		u2len = multbig(uu1,u1len,uu1,u1len,uu2,hilf);
		u2len = modbig(uu2,u2len,mm,modlen,hilf);
		p1len = multbig(uu2,u2len,x,n,pp1,hilf);
		p1len = modbig(pp1,p1len,mm,modlen,hilf);
		cpyarr(pp1,p1len,act1->den);
		act1->dlen = p1len;

  duplic:
/*
** i --> 2*i 
** X(2*i) = (X(i)*X(i) - Z(i)*Z(i))**2
** Z(2*i) = 4*X(i)*Z(i)*(X(i)*X(i) + A*X(i)*Z(i) + Z(i)*Z(i))
*/
		y1len = multbig(act0->num,n0,act0->num,n0,yy1,hilf);
		y1len = modbig(yy1,y1len,mm,modlen,hilf);
/* yy1 = X(i)*X(i) */
		y2len = multbig(act0->den,m0,act0->den,m0,yy2,hilf);
		y2len = modbig(yy2,y2len,mm,modlen,hilf);
/* yy2 = Z(i)*Z(i) */
		u1len = multbig(act0->num,n0,act0->den,m0,uu1,hilf);
		u1len = modbig(uu1,u1len,mm,modlen,hilf);
/* uu1 = X(i)*Z(i) */
		u2len = multbig(uu1,u1len,aa,alen,uu2,hilf);
		u2len = modbig(uu2,u2len,mm,modlen,hilf);
/* uu2 = A*X(i)*Z(i) */

		cpyarr(yy1,y1len,pp1);
		if(cmparr(yy1,y1len,yy2,y2len) >= 0)
			p1len = subarr(pp1,y1len,yy2,y2len);
		else
			p1len = sub1arr(pp1,y1len,yy2,y2len);
		p2len = multbig(pp1,p1len,pp1,p1len,pp2,hilf);
		p2len = modbig(pp2,p2len,mm,modlen,hilf);
		cpyarr(pp2,p2len,act0->num);
		act0->nlen = p2len;

		cpyarr(yy1,y1len,pp1);
		p1len = addarr(pp1,y1len,uu2,u2len);
		p1len = addarr(pp1,p1len,yy2,y2len);
		p1len = shlarr(pp1,p1len,2);
/* pp1 = 4*(X(i)*X(i) + A*X(i)*Z(i) + Z(i)*Z(i)) */
		p2len = multbig(pp1,p1len,uu1,u1len,pp2,hilf);
		p2len = modbig(pp2,p2len,mm,modlen,hilf);
		cpyarr(pp2,p2len,act0->den);
		act0->dlen = p2len;
	}
	m0 = vv.dlen;
	cpyarr(vv.den,m0,uu1);

	m0 = gcdcx(uu1,m0,mm,modlen,pp1,&p1len,hilf);
	if(m0 != 1 || uu1[0] != 1) {
		cpyarr(uu1,m0,z);
		return(-m0-1);
	}
	p2len = multbig(vv.num,vv.nlen,pp1,p1len,pp2,hilf);
	zlen = modbig(pp2,p2len,mm,modlen,hilf);
	cpyarr(pp2,zlen,z);
	return(zlen);
}
/*------------------------------------------------------------------*/
/*
** Pollardsche rho-Methode zur Faktorisierung;
** Aufruf rho_factorize(N,anz) oder rho_factorize(N); 
** dabei ist anz die maximale Anzahl der Iterationen,
** default anz = 2**16
*/
PRIVATE truc Frhofact(argn)
int argn;
{
	T_xyQ xyQ;

	truc *argptr;
	word2 *N, *z, *d, *Q, *hilf;
	word4 u, i;
	size_t m;
	unsigned rr;
	int cyclen = 256;
	int k, n, len, sign, ret;

	argptr = argStkPtr-argn+1;
	if(chkints(rhosym,argptr,argn) == ERROR)
		return(brkerr());

	len = bigref(argptr,&N,&sign);
	m = aribufSize/4;
	if(len >= (m-1)/2) {
		error(rhosym,err_ovfl,*argptr);
		return(brkerr());
	}

	d = AriBuf;
	xyQ.x = d + m;
	xyQ.y = d + 2*m;
	xyQ.Q = d + 3*m;
	Q = AriScratch;
	hilf = AriScratch + aribufSize;

	rr = random2(64000);
	xyQ.x[0] = xyQ.y[0] = rr;
	xyQ.xlen = xyQ.ylen = (rr ? 1 : 0);
	xyQ.Q[0] = 1;
	xyQ.Qlen = 1;

	if(argn == 2) {
		n = bigref(argStkPtr,&z,&sign);
		if(n <= 2 && n)
			u = big2long(z,n);
		else
			u = 0x80000000;
	}
	else
		u = 0x10000;
	workmess();

	for(i=0; i<u; i+=cyclen) {
		tick('.');
		ret = rhocycle(cyclen,&xyQ,N,len,hilf);
		if(ret) {
			k = xyQ.Qlen;
			cpyarr(xyQ.Q,k,Q);
			cpyarr(N,len,d);
			k = biggcd(d,len,Q,k,hilf);
			if(k > 1 || *d > 1) {
				rhomess(i+cyclen);
				return(mkint(0,d,k)); 
			}
		}
		if(INTERRUPT) {
			setinterrupt(0);
			break;
		}
	}
	return(zero);
}
/*-------------------------------------------------------------------*/
PRIVATE void workmess()
{
	fnewline(tstdout);
	fprintstr(tstdout,"working ");
}
/*-------------------------------------------------------------------*/
PRIVATE void tick(c)
int c;
{
	char tt[2];

	tt[0] = c;
	tt[1] = 0;
	fprintstr(tstdout,tt);
	fflush(stdout);
}
/*-------------------------------------------------------------------*/
PRIVATE void rhomess(anz)
word4 anz;
{
	char messbuf[80];

	s1form(messbuf,"~%factor found after ~D iterations",anz);
	fprintline(tstdout,messbuf);
	fnewline(tstdout);
}
/*-------------------------------------------------------------------*/
/*
** Berechnet anz mal 
** x -> x*x+2; x -> x*x+2; y -> y*y+2 mod N
** Q -> Q*(x-y) mod N
** Rueckgabewert Laenge von Q
*/
PRIVATE int rhocycle(anz,xyQ,N,len,hilf)
int anz;
T_xyQ *xyQ;
word2 *N;
int len;
word2 *hilf;
{
#define INC	2
	word2 *x1, *y1, *Q1, *z, *z1;
	int n, m, k, cmp;
	int zlen, z1len, nn;

	nn = 2*len + 2;
	z = hilf;
	z1 = hilf + nn;
	hilf = z1 + nn;

	x1 = xyQ->x;
	n = xyQ->xlen;
	y1 = xyQ->y;
	m = xyQ->ylen;
	Q1 = xyQ->Q;
	*Q1 = 1;
	k = 1;

	while(--anz >= 0) {
		zlen = multbig(x1,n,x1,n,z,hilf);
		zlen = incarr(z,zlen,INC);
		zlen = modbig(z,zlen,N,len,hilf);
		z1len = multbig(z,zlen,z,zlen,z1,hilf);
		z1len = incarr(z1,z1len,INC);
		n = modbig(z1,z1len,N,len,hilf);
		cpyarr(z1,n,x1);
		zlen = multbig(y1,m,y1,m,z,hilf);
		zlen = incarr(z,zlen,INC);
		m = modbig(z,zlen,N,len,hilf);
		cpyarr(z,m,y1);
		cmp = cmparr(z,m,z1,n);
		if(cmp > 0)
			zlen = subarr(z,m,z1,n);
		else if(cmp < 0)
			zlen = sub1arr(z,m,z1,n);
		else
			continue;
		z1len = multbig(Q1,k,z,zlen,z1,hilf);
		k = modbig(z1,z1len,N,len,hilf);
		cpyarr(z1,k,Q1);
	}
	xyQ->xlen = n;
	xyQ->ylen = m;
	xyQ->Qlen = k;
	return(k);
#undef INC
}
/*------------------------------------------------------------------*/
PRIVATE truc Fcffact(argn)
int argn;
{
	word2 *N, *x;
	long mm;
	size_t buflen;
	unsigned u = 1;
	int len0, len, n;
	int sign;

	if(chkints(cffactsym,argStkPtr-argn+1,argn) == ERROR)
		return(brkerr());
#if defined(ATARIST) || defined(MsDOS)
	mm = stkcheck() - STACKRES;
#else
	mm = auxbufSize * sizeof(word2);
#endif
	if(mm < MEMPIECE) {
		error(cffactsym,err_memev,voidsym);
		return(brkerr());
	}
	buflen = auxbufSize * sizeof(word2);
	if(buflen >= MEMPIECE)
		len0 = ALEN - 1;
	else
		len0 = 10;

	len = bigref(argStkPtr-argn+1,&N,&sign);
	if(len > len0 || (len == len0 && bitlen(N[len-1]) > 4)) {
		error(cffactsym,err_2big,*argStkPtr);
		return(brkerr());
	}
	if(argn == 2) {
		n = bigref(argStkPtr,&x,&sign);
		u = *x;
		if(u > 1023 || u == 0)
			u = 1;
	}
	n = brillmorr(N,len,u,AriBuf,AriScratch);

	return(mkint(0,AriBuf,n));
}
/*------------------------------------------------------------------*/
PRIVATE int brillmorr(N,len,v,fact,hilf)
word2 *N, *fact, *hilf;
unsigned v;
int len;
{
#if defined(ATARIST) || defined(MsDOS)
	/*** for ATARI, auto memory may not exceed 32 KB ***/
	word2 stackpiece[MEMPIECE/sizeof(word2)];
	size_t slen = MEMPIECE/sizeof(word2);
#else
	size_t slen = 0;
#endif

	BMDATA bm;
	CFDATA cf;

	word4 u;
	int len1, maxbas, b0, b1, b, ret;

	b1 = bitlen(N[len-1]);
	len1 = (b1 > 4 ? len+1 : len);
#if defined(ATARIST) || defined(MsDOS)
	maxbas = bmcfinit(len1,stackpiece,slen,AuxBuf,auxbufSize,&bm,&cf);
#else
	maxbas = bmcfinit(len1,AuxBuf,slen,AuxBuf,auxbufSize,&bm,&cf);
#endif
	b0 = maxbas/16;

	b = (len - 1)*16 + b1;
	b = (b - 32)/3;
	if(b < 3)
		b = 3;
	if(b > b0)
		b = b0;
	bm.vlen = b;
	bm.v2len = b * 2;
	bm.baslen = b * 16 - 1;
	u = b;
	bm.easize = intsqrt(u*96);

	ret = brill1(N,len,v,fact,&bm,&cf,hilf);

	return(ret);
}
/*-------------------------------------------------------------------*/
/*
** In (buf1,len1) und (buf2,len2) werden zwei Puffer uebergeben, von
** denen einer leer sein darf.
** Aus diesen wird den Strukturen *bmp und *cfp Speicher zugewiesen
**
** Bedarf fuer bmp:
**	bmp->bitmat	Platz fuer eine Bitmarix (maxbas+1)*(2*maxbas)
**	bmp->piv	Platz fuer maxbas+1 word2's
**	bmp->fbas	Platz fuer maxbas word2's
**	bmp->Aarr	Platz fuer (maxbas + 1) bigints der Laenge len,
**			einschliesslich Laengen-Angabe
**	bmp->Qarr	Platz fuer (maxbas + 1) bigints der Laenge len/2,
**			einschliesslich Laengen-Angabe
**
** Rueckgabewert ist maxbas; dies ist durch 16 teilbar
*/
PRIVATE int bmcfinit(len,buf1,len1,buf2,len2,bmp,cfp)
int len;
word2 *buf1, *buf2;	/* at least one must be non empty */
size_t len1, len2;
BMDATA *bmp;
CFDATA *cfp;
{
	word2 *temp, *xx, *yy;
	word4 u;
	size_t ll;
	int maxbas, alen, qlen;

	if(len1 == 0) {
		buf1 = buf2;
		len1 = len2/2;
		buf2 += len1;
		len2 = len1;
	}
	else if(len2 == 0) {
		len2 = len1/2;
		len1 = len2;
		buf2 = buf1 + len1;
	}
	else if(len2 > len1) {
		temp = buf1;
		buf1 = buf2;
		buf2 = temp;
		ll = len1;
		len1 = len2;
		len2 = ll;
	}
	/* now buf1 and buf2 are non-empty and len1 >= len2 */

	/* allocation for cfp */
	alen = len + 1;
	qlen = alen/2 + 1;

	xx = buf2 + 1;
	yy = xx + 3*alen;

	cfp->kN = xx;
	cfp->A0 = xx + alen;
	cfp->AA = xx + 2*alen;

	cfp->m2 = yy;
	cfp->R0 = yy + qlen;
	cfp->Q0 = yy + 2*qlen;
	cfp->QQ = yy + 3*qlen;

	ll = 3*alen + 4*qlen;
	buf2 += ll;
	len2 -= ll;

	/* allocation for bmp */
	u = len1;
	u *= 8;
	maxbas = (int)intsqrt(u);
	maxbas = (maxbas/16) * 16;
	u = maxbas+1;
	u *= maxbas/8;
	if(u > len1)
		maxbas -= 16;

	bmp->bitmat = buf1;
	bmp->matinc = maxbas/8;
	bmp->rank = 0;

	ll = len2 / (2 + alen + qlen);
	if(ll <= maxbas)
		maxbas = ((ll - 1)/16) * 16;

	xx = buf2 + alen * (maxbas + 1);
	yy = xx + qlen * (maxbas + 1);
	bmp->Aarr = buf2 + 1;
	bmp->Qarr = xx + 1;
	bmp->ainc = alen;
	bmp->qinc = qlen;

	bmp->piv = yy;
	bmp->fbas = yy + maxbas + 1;

	return(maxbas);
}
/*-------------------------------------------------------------------*/
PRIVATE int brill1(N,len,u,fact,bmp,cfp,hilf)
word2 *N, *fact, *hilf;
unsigned u;
int len;
BMDATA *bmp;
CFDATA *cfp;
{
	word2 *zz;
	int k, qrlen, blen;
	int count = 0, count1 = 0;
	long v;

	qrlen = cfracinit(N,len,u,cfp,hilf);
	if(qrlen == 0) {
		k = cfp->AA[-1];
		cpyarr(cfp->AA,k,fact);
		zz = cfp->kN;
		cpyarr(N,len,zz);
		return(biggcd(fact,k,zz,len,hilf));
	}
	blen = bmp->baslen;
	factorbase(blen,bmp->fbas,cfp->kN,cfp->kN[-1]);
	for(k=0; k<=blen; k++)
		bmp->piv[k] = blen-k;
		/* bmp->piv[k] = k; */

	cf0mess(bmp->fbas[blen-1],blen);

	for(v=0; qrlen; v++) {
	    if(smooth(cfp,bmp,hilf)) {
		if((++count & 0x3) == 0)
			tick('.');
		if(gausselim(bmp)) {
			tick('!');
			if(++count1 > blen)
				break;
			k = getfactor(N,len,bmp,fact,hilf);
			if(k > 0) {
				cf1mess(v+1,count);
				return(k);
			}
		}
	    }
	    if((v & 0xFF) == 0) {
		if(INTERRUPT) {
			setinterrupt(0);
			break;
		}
		if((v & 0x3FF) == 0) {
			tick(counttick(v >> 10));
		}
	    }
	    qrlen = cfracnext(N,len,cfp,hilf);
	}
	return(0);
}
/*-------------------------------------------------------------------*/
PRIVATE int counttick(v)
int v;
{
	if((v&0x3FF) == 0) {
		v = (v >> 10) & 0xF;
		return('A' + v);
	}
	else if((v&0x7F) == 0) {
		v = (v >> 7) & 0x7;
		return('0' + v);
	}
	else
		return('_');
}
/*-------------------------------------------------------------------*/
PRIVATE void cf0mess(p,blen)
int p, blen;
{
	char messbuf[80];

	s2form(messbuf,"~%CF-algorithm: factorbase 2 ... ~D of length ~D",
		(word4)p,(word4)blen);
	fprintline(tstdout,messbuf);
	fprintstr(tstdout,"working ");
}
/*-------------------------------------------------------------------*/
PRIVATE void cf1mess(n,nf)
long n;
int nf;
{
	char messbuf[80];

	s2form(messbuf,
	    "~%~D quadratic residues calculated, ~D completely factorized",
	    n,(word4)nf);
	fprintline(tstdout,messbuf);
	fnewline(tstdout);
}
/*------------------------------------------------------------------*/
/*
** Schreibt in das Array prim die Primzahl 2 und weitere (anz-1)
** ungerade Primzahlen, fuer die jacobi((N,len),p) = 1
*/
PRIVATE void factorbase(anz,prim,N,len)
int anz;
word2 *prim, *N;
int len;
{
	int count;
	unsigned m, p;

	prim[0] = 2;
	count = 1;
	for(p = 3; count < anz; p += 2) {
		if(!prime16(p))
			continue;
		m = modarr(N,len,p);
		if(jac(m,p) == 1) {
			prim[count] = p;
			count++;
		}
	}
}
/*-------------------------------------------------------------------*/
PRIVATE int cfracinit(N,len,u,cfp,hilf)
word2 *N;
int len;
unsigned u;
CFDATA *cfp;
word2 *hilf;
{
	word2 *temp, *temp1;
	int k, n, rlen;
	int ll = 2*len + 2;

	temp = hilf;
	temp1 = temp + ll;
	hilf = temp1 + ll;

	len = multarr(N,len,u,temp);
	cpyarr(temp,len,cfp->kN);
	cfp->kN[-1] = len;

	k = bigsqrt(temp,len,temp1,&rlen,hilf);
	cpyarr(temp1,k,cfp->AA);
	cfp->AA[-1] = k;

	n = multbig(temp1,k,temp1,k,temp,hilf);
	rlen = sub1arr(temp,n,cfp->kN,len);
	cpyarr(temp,rlen,cfp->QQ);
	cfp->QQ[-1] = rlen;

	k = shlarr(temp1,k,1);
	cpyarr(temp1,k,cfp->m2);
	cfp->m2[-1] = k;

	cfp->R0[-1] = 0;
	cfp->A0[0] = 1;
	cfp->A0[-1] = 1;
	cfp->Q0[0] = 1;
	cfp->Q0[-1] = 1;

	cfp->qrsign = -1;
	return(rlen);
}
/*-------------------------------------------------------------------*/
/*
** m2 - R0 = bb * QQ + rest;
** Qnew = Q0 + (rest - R0) * bb;
** Anew = (AA * bb + A0) mod N
** next R0 = rest
** next Q0 = QQ
** next A0 = AA
** next QQ = Qnew
** next AA = Anew
*/
PRIVATE int cfracnext(N,len,cfp,hilf)
word2 *N;
int len;
CFDATA *cfp;
word2 *hilf;
{
	static word2 rr[QLEN], bb[QLEN];
	static word2 Qtemp[2*QLEN], temp1[QLEN], Atemp[2*ALEN];

	word2 *QQ;
	int m2len, rlen, blen, qtlen, atlen, t1len;
	int cmp;

	QQ = cfp->QQ;

	m2len = cfp->m2[-1];
	cpyarr(cfp->m2,m2len,rr);
	m2len = subarr(rr,m2len,cfp->R0,cfp->R0[-1]);
	blen = divbig(rr,m2len,QQ,QQ[-1],bb,&rlen,hilf);
	cpyarr(rr,rlen,temp1);
	t1len = rlen;

	cmp = cmparr(temp1,t1len,cfp->R0,cfp->R0[-1]);
	if(cmp >= 0)
		t1len = subarr(temp1,t1len,cfp->R0,cfp->R0[-1]);
	else
		t1len = sub1arr(temp1,t1len,cfp->R0,cfp->R0[-1]);

	cpyarr(rr,rlen,cfp->R0);
	cfp->R0[-1] = rlen;

	qtlen = multbig(temp1,t1len,bb,blen,Qtemp,hilf);
	if(cmp >= 0)
		qtlen = addarr(Qtemp,qtlen,cfp->Q0,cfp->Q0[-1]);
	else
		qtlen = sub1arr(Qtemp,qtlen,cfp->Q0,cfp->Q0[-1]);

	cpyarr(QQ,QQ[-1],cfp->Q0);
	cfp->Q0[-1] = QQ[-1];
	cpyarr(Qtemp,qtlen,QQ);
	QQ[-1] = qtlen;

	atlen = multbig(cfp->AA,cfp->AA[-1],bb,blen,Atemp,hilf);
	atlen = addarr(Atemp,atlen,cfp->A0,cfp->A0[-1]);
	atlen = modbig(Atemp,atlen,N,len,hilf);

	cpyarr(cfp->AA,cfp->AA[-1],cfp->A0);
	cfp->A0[-1] = cfp->AA[-1];
	cpyarr(Atemp,atlen,cfp->AA);
	cfp->AA[-1] = atlen;

	cfp->qrsign = -cfp->qrsign;

	return(qtlen);
}
/*-------------------------------------------------------------------*/
/*
** Rueckgabe = 1, falls neuer quadratischer Rest smooth; sonst = 0
*/
PRIVATE int smooth(cfp,bmp,hilf)
CFDATA *cfp;
BMDATA *bmp;
word2 *hilf;
{
	word2 *Q, *A, *vect, *prim;
	word2 r;
	unsigned p;
	int qn, bound, bound0, i, v, bitl;

	Q = hilf;
	qn = cfp->QQ[-1];
	cpyarr(cfp->QQ,qn,Q);
	bitl = (qn - 1)*16 + bitlen(Q[qn-1]);
	vect = bmp->bitmat + (bmp->rank)*(bmp->matinc);
	setarr(vect,bmp->vlen,0);
	if(cfp->qrsign == -1)
		vect[0] = 1;	/* setbit(vect,0); */
	prim = bmp->fbas;
	bound0 = bmp->baslen;
	bound = bmp->easize;
	i = 0;
  nochmal:
	while(++i < bound) {
		v = 0;
		p = *prim++;
		while(modarr(Q,qn,p) == 0) {
			v++;
			qn = divarr(Q,qn,p,&r);
		}
		if(v & 1)
			setbit(vect,i);
		if(qn == 1 && Q[0] == 1) {
			Q = bmp->Qarr + (bmp->rank)*(bmp->qinc);
			Q[-1] = cfp->QQ[-1];
			cpyarr(cfp->QQ,Q[-1],Q);
			A = bmp->Aarr + (bmp->rank)*(bmp->ainc);
			A[-1] = cfp->AA[-1];
			cpyarr(cfp->AA,A[-1],A);

			return(1);
		}
	}
	if(bound < bound0) {
		if(bitl - 16*(qn - 1) - bitlen(Q[qn-1]) >= 10) {
			bound = bound0;
			goto nochmal;
		}
		/* else early abort */
	}
	return(0);
}
/*-------------------------------------------------------------------*/
/*
** Rueckgabe = 0, falls letzte Zeile unabhaengig; sonst = 1
*/
PRIVATE int gausselim(bmp)
BMDATA *bmp;
{
	word2 *v, *vect, *vectb, *piv;
	unsigned pivot;
	int i, rk;
	int vn, v2n, minc;

	minc = bmp->matinc;
	rk = bmp->rank;
	vn = bmp->vlen;
	v2n = bmp->v2len;

	vect = bmp->bitmat + rk * minc;
	vectb = vect + vn;
	setarr(vectb,vn,0);
	setbit(vectb,rk);

	v = bmp->bitmat;
	piv = bmp->piv;
	for(i=0; i<rk; i++) {
		pivot = piv[i];
		if(testbit(vect,pivot))
			xorarr(vect,v2n,v);
		v += minc;
	}
	for(i=0; i<vn; i++)
		if(vect[i]) break;
	if(i == vn)	/* then vect is identically zero */
		return(1);

	for(i=rk; pivot=piv[i],!testbit(vect,pivot); i++)
		;
	piv[i] = piv[rk];
	piv[rk] = pivot;
	bmp->rank = rk + 1;

	return(0);
}
/*-------------------------------------------------------------------*/
PRIVATE int getfactor(N,len,bmp,fact,hilf)
word2 *N, *fact, *hilf;
int len;
BMDATA *bmp;
{
	word2 *relat, *temp, *A, *AA, *Q, *QQ, *XX, *X1, *X2;
	int cmp, k, rk, ainc, qinc;
	int alen, qlen, tlen, xlen, x1len, x2len;
	int ll, lll;

	ll = 2*ALEN + 2;
	lll = 4*ALEN;

	A = hilf;
	XX = A + ll;
	X1 = XX + ll;
	X2 = X1 + ll;
	Q = X2 + ll;
	temp = Q + lll;
	hilf = temp + lll;

	rk = bmp->rank;
	relat = bmp->bitmat + rk * (bmp->matinc) + (bmp->vlen);
	ainc = bmp->ainc;
	qinc = bmp->qinc;

	AA = bmp->Aarr + rk*ainc;
	alen = AA[-1];
	cpyarr(AA,alen,A);
	AA = bmp->Aarr;
	for(k=0; k<rk; k++) {
		if(testbit(relat,k)) {
		    alen = multbig(A,alen,AA,AA[-1],temp,hilf);
		    cpyarr(temp,alen,A);
		    alen = modbig(A,alen,N,len,hilf);
		}
		AA += ainc;
	}

	XX[0] = 1;
	xlen = 1;
	QQ = bmp->Qarr + rk*qinc;
	qlen = QQ[-1];
	cpyarr(QQ,qlen,Q);
	QQ = bmp->Qarr;
	for(k=0; k<rk; k++) {
		if(testbit(relat,k)) {
		    cpyarr(QQ,QQ[-1],X1);
		    cpyarr(Q,qlen,temp);
		    x1len = biggcd(X1,QQ[-1],temp,qlen,hilf);
		    cpyarr(XX,xlen,X2);
		    xlen = multbig(X2,xlen,X1,x1len,XX,hilf);
		    xlen = modbig(XX,xlen,N,len,hilf);

		    cpyarr(QQ,QQ[-1],temp);
		    x2len = divbig(temp,QQ[-1],X1,x1len,X2,&tlen,hilf);
		    tlen = divbig(Q,qlen,X1,x1len,temp,&qlen,hilf);
		    qlen = multbig(temp,tlen,X2,x2len,Q,hilf);
		    if(qlen + ALEN >= lll) {
			lll += 2*ALEN;
			temp = Q + lll;
			hilf = temp + lll;
		    }
		}
		QQ += qinc;
	}
	tlen = bigsqrt(Q,qlen,temp,&qlen,hilf);
	qlen = multbig(temp,tlen,XX,xlen,Q,hilf);
	xlen = modbig(Q,qlen,N,len,hilf);
	cpyarr(Q,xlen,XX);

	cmp = cmparr(A,alen,XX,xlen);
	if(cmp == 0)
		return(0);
	cpyarr(A,alen,fact);
	if(cmp > 0)
		k = subarr(fact,alen,XX,xlen);
	else
		k = sub1arr(fact,alen,XX,xlen);
	cpyarr(N,len,temp);
	k = biggcd(fact,k,temp,len,hilf);
	if(k > 1 || (k == 1 && fact[0] != 1))
		return(k);

	cpyarr(A,alen,fact);
	k = addarr(fact,alen,XX,xlen);
	if(cmparr(fact,k,N,len) == 0)
		return(0);
	cpyarr(N,len,temp);
	k = biggcd(fact,k,temp,len,hilf);
	if(k > 1 || (k == 1 && fact[0] != 1))
		return(k);
	else
		return(0);
}
/********************************************************************/
